package ohd.hseb.ohdutilities.ffg.griddedFfg;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import ohd.hseb.grid.RfcGridWithTime;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.ohdutilities.ffg.FFG_RUNOFF_ADJUST_OPTION;
import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgRainfallRunoffCurve;
import ohd.hseb.ohdutilities.ffg.FfgThreshRCalculator;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.LineSeg;
import ohd.hseb.util.fews.OHDUtilities.StateLocation;
import ohd.hseb.util.fews.ParameterType.Row;

public class GriddedFFGDriver extends FFGDriver
{

    private final GriddedFFGParameters _griddedFfgParams;

    public GriddedFFGDriver()
    {
        super(); //calling FFGDriver constructor

        _griddedFfgParams = new GriddedFFGParameters();

        super._parameters = _griddedFfgParams;

    }

    @Override
    protected void execute() throws Exception
    {
        this.runDriverValidation();

        final int[] modelIntervalArray = _griddedFfgParams.getModelIntervalArray();

        String inputFfgNetcdfFile = _runInfo.getInputNetCdfFile();

        /*
         * If <inputNetcdfFile> element in run_info.xml is missing, use <input> from states.xml, otherwise, throw an
         * Exception; If both <inputNetcdfFile> in run info and <input> in states.xml are present, use
         * <inputNetcdfFile>.
         */

        if(inputFfgNetcdfFile == null)
        {//do not have  <inputNetcdfFile> element in run_info.xml, get from <input> from states.xml

            inputFfgNetcdfFile = getInputStateFile();

            if(inputFfgNetcdfFile == null)
            {
                throw new Exception("Input rfc netcdf template file is not defined in run_info.xml or states.xml.");
            }

            _logger.log(Logger.DEBUG, "As the first basin of RFC, use input netcdf file defined in state meta file.");
        }
        else
        {
            _logger.log(Logger.DEBUG, "Input netcdf: " + inputFfgNetcdfFile);
        }

        final RfcGridWithTime ffgGrid = new RfcGridWithTime(inputFfgNetcdfFile,
                                                            _runInfo.getRunLastObservationTimeLong(),
                                                            _logger);

        if(_griddedFfgParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.EXCLUDE_FROM_GRID_COMPUTATIONS_AS_9)
        {//no computation, no writing to nc file, so outputs.nc is same as the input netcdf file

            // Generates output/outputs.nc
            ffgGrid.writeOutNetcdfFile(_runInfo.getOutputNetCdfFile());

            ffgGrid.close();

            return;
        }

        final double[] ffgArray = new double[modelIntervalArray.length];

        final double[] minPrecipOfAreas = new double[ffgArray.length];//1st element is the smallest precip among the multiple areas' precip values at 1HR model interval, 2nd element is 3HR, etc.

        final double arbitraryBigNumber = 300.0;

        for(int i = 0; i < minPrecipOfAreas.length; i++)
        {
            minPrecipOfAreas[i] = arbitraryBigNumber; //initialize with a great number
        }

        final double[] pcfq = new double[modelIntervalArray.length];

        if(_griddedFfgParams.adjustHighFlow())
        { //high flow adjust option is true("FFG_HIGH_FLOW_ADJUST_OPTION" and "FFG_USER_CTRL_HIGH_FLOW_ADJUST")

            final FfgThreshRCalculator threshRCal = new FfgThreshRCalculator(_griddedFfgParams,
                                                                             _dischargeTs,
                                                                             _runInfo.getRunLastObservationTimeLong(),
                                                                             _logger);

            final double[] qb = threshRCal.getMaxForecastFlowArrayInCFS();

            if(qb != null)
            {//calculate pcfq

                //get flood flow from discharge ts header "threshold"
                final String floodFlowId = _runInfo.getProperties().getProperty(FfgConstants.FLOOD_FLOW_ID_TAG);

                if(floodFlowId == null)
                {
                    throw new Exception("FloodFlowId property is missing from run_info.xml. It is needed to retrieve the threshold from discharge TS header.");
                }

                final double floodFlowInCFS = _dischargeTs.getThreshold(floodFlowId).getValue(MeasuringUnit.cfs);

                for(int i = 0; i < pcfq.length; i++)
                {
                    pcfq[i] = qb[i] / floodFlowInCFS; //pcfq is unitless

                    if(pcfq[i] > 1.0)
                    {
                        pcfq[i] = 1.0;
                    }

                }//close for loop

            } //close if(qb != null)
        }

        final List<LineSeg>[] threshRArrayLineSegListArray = _griddedFfgParams.getGridThreshRunoffListArray();

        final double[] intensityArray = _griddedFfgParams.getIntensityArray();

        final Map<String, List<FfgRainfallRunoffCurve>> rainFallRunoffCurveMap = getRainFallRunoffCurveMap();

        final List<FfgRainfallRunoffCurve> listOfCurvesForVarDurations = rainFallRunoffCurveMap.get(_areaIdArray[0]); //Gridded FFG only has one area

        final int gridFillNum = _griddedFfgParams.getGridFillControlAsNum();

        for(int durationIndex = 0; durationIndex < modelIntervalArray.length; durationIndex++)
        {
            final double[] minMaxPair = _griddedFfgParams.getMinMaxValues(durationIndex);

            //get ffg value
            final FfgRainfallRunoffCurve rainFallRunOffCurve = listOfCurvesForVarDurations.get(durationIndex);

            final List<LineSeg> lineSegList = threshRArrayLineSegListArray[durationIndex];

            double ffg = FfgConstants.FFG_MISSING_VALUE; //initialized to -9.8

            double prevThesholdRunoff = 0.0; //stores most recent non-missing threshold runoff, ffg stores its corresponding result

            for(int lineSegCount = 0; lineSegCount < lineSegList.size(); lineSegCount++)
            {//each row has the same threshold runoff, or all pixels on this has missing threshold runoff

                final LineSeg lineSeg = lineSegList.get(lineSegCount);

                for(final Row row: lineSeg.getRowList())
                {

                    final int hrapRow = Integer.valueOf(row.getB());

                    final int startColumnNum = Integer.valueOf(row.getC());

                    final int endColumnNum = Integer.valueOf(row.getD());

                    final double troz = Double.valueOf(row.getE()); //get threshold runoff

                    if(troz < 0.0)
                    {//threshold runoff is missing for this pixel(and this row too), move on to next row
                        _logger.log(Logger.WARNING, "Threshold runoff is missing for HRAP Row " + hrapRow
                            + " HRAP Columns from " + startColumnNum + " to " + endColumnNum);

                        continue; //move on to the next row
                    }

                    if(_griddedFfgParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.USE_FIELDS_AS_FFG_AS_2)
                    {//when parameter "RUNOFF_ADJUST_OPTION" is "USE_FIELDS_AS_FFG", set ffg values directly: from FFG_INTENSITY_x_HOUR value
                        ffg = intensityArray[durationIndex];
                    }
                    else if(troz != prevThesholdRunoff)
                    {//do computation since threshold runoff is different

                        prevThesholdRunoff = troz;

                        final double uro = troz * (1.0 - pcfq[durationIndex]);

                        final double ro = uro * _griddedFfgParams.getBank();

                        ffg = rainFallRunOffCurve.getFFGValueByInterpolation(ro);

                        if(_griddedFfgParams.getRunOffAdjustOption() == FFG_RUNOFF_ADJUST_OPTION.ADJUST_FFG_AS_5)
                        {
                            ffg *= intensityArray[durationIndex];
                        }

                        //apply min/max logic here
                        if(ffg < minMaxPair[0])
                        {
                            ffg = minMaxPair[0];
                        }
                        else if(ffg > minMaxPair[1])
                        {
                            ffg = minMaxPair[1];
                        }

                    } //if(troz != prevThesholdRunoff)

                    row.setF(String.valueOf(ffg));//ffg value could be from newly calculated or from old value due to same threshold runoff

                }//end looping one row

                //now, has finished every row in this line seg
                ffgGrid.setFfgGridValuesFromLineSeg(durationIndex, lineSeg);

                //grid filling for between 1 and 5
                if(gridFillNum > 0 && gridFillNum <= 5)
                {
                    //fill left and right pixels
                    ffgGrid.fillFfgLineSegLeftAndRightPixels(durationIndex, lineSeg, gridFillNum);

                    //fill top or bottom linesegs
                    if(lineSegCount == 0)
                    {//top lineseg

                        final int origHrapRowNum = lineSeg.getHrapRowNum();

                        for(int i = 1; i <= gridFillNum; i++)
                        {
                            final int newHrapRow = origHrapRowNum + i;

                            _logger.log(Logger.DEBUG, "Grid filling the row. HRAP Row= " + newHrapRow + ", row index= "
                                + (newHrapRow - ffgGrid.getRfcOrigHrapRow()));

                            lineSeg.setHrapRowNum(newHrapRow); //stepping over the top lineseg

                            ffgGrid.setFfgGridValuesFromLineSeg(durationIndex, lineSeg);
                        }

                    }
                    else if(lineSegCount == lineSegList.size() - 1)
                    {//bottom lineseg

                        final int origHrapRowNum = lineSeg.getHrapRowNum();

                        for(int i = 1; i <= gridFillNum; i++)
                        {
                            final int newHrapRow = origHrapRowNum - i;

                            _logger.log(Logger.DEBUG, "Grid filling the row. HRAP Row= " + newHrapRow + ", row index= "
                                + (newHrapRow - ffgGrid.getRfcOrigHrapRow()));

                            lineSeg.setHrapRowNum(newHrapRow); //stepping below the lowest lineseg

                            ffgGrid.setFfgGridValuesFromLineSeg(durationIndex, lineSeg);
                        }
                    }
                }//close if(_griddedFfgParams.getGridFillControl() > 0)

            }//end looping one line seg

        } //end looping duration

        if(gridFillNum == 6)
        {
            ffgGrid.ffgGridFill6();
        }

        // Generates output/outputs.nc
        ffgGrid.writeOutNetcdfFile(_runInfo.getOutputNetCdfFile());

        _logger.log(Logger.DEBUG, "Output result netcdf file to " + _runInfo.getOutputNetCdfFile());

        /*
         * from FEWS Configuration, we found: 1)they don't really save output state(no exportStateActivity);
         * 2)<writeLocation> in states.xml has value output/outputs.nc, same as <outputNetcdfFile> value, so we
         * commented out the following code to avoid doing same thing twice.
         */
//        if(getOutputStateFile() != null)
//        {
//            ffgGrid.setRedefineMode(true);
//            ffgGrid.writeOutNetcdfFile(getOutputStateFile()); //one extra copy of netcdf file to <output> location
//
//            _logger.log(Logger.DEBUG, "Output result netcdf file to " + getOutputStateFile());
//        }

        ffgGrid.close();
    }

    /**
     * Parse Gridded Ffg parameter file, input time series files and states meta file, if it exists--only does for 1st
     * area. The parsing were skipped in OHDFewsAdapter. Call super class runDriverValidation. Get discharge TS if it is
     * required.
     */
    @Override
    final protected void runDriverValidation() throws Exception
    {

        super.runDriverValidation(); //parse input time series file and parameter file. Check every ts in _tsList has enough data 
                
        // for FogBugz 1616: Prepend logs with location info
        // this is handled by OHDFewsadapter for most drivers, but GriddedFFG doesn't always have a TS
        String currentPrependString =  ((Diagnostics)_logger).getPrependStr();
        String areaId = _areaIdArray[0]; //Gridded FFG only has one area
        ((Diagnostics)_logger).setPrependStr(currentPrependString+"("+areaId+")");
        
        
        this.parseGriddedFfgStates(); //only running Gridded FFG 1st area of RFC has states meta file. 2-N areas or FFH do not have. 

        if(_griddedFfgParams.adjustHighFlow())
        {//now, discharge TS is required

            if(_dischargeTs == null)
            {
                throw new Exception("Discharge time series is required for high flow adjust option, but it is absent.");
            }

            //it is present. trim it and check it has enough data
            _logger.log(Logger.DEBUG, "Using input forecast discharge time series(" + _dischargeTs.getTimeSeriesType()
                + ").");

            _dischargeTs.trimTimeSeriesAtStartWithCheck(getSwitchFromObservedToForecastTime());

            _dischargeTs.trimTimeSeriesAtEndWithCheck(getComputationEndTime());
        }
        else
        {//no need of discharge TS, so set it to null
            _dischargeTs = null;
        }

    }

    /**
     * This method only for Gridded FFG and only applies to the 1st area. For the 1st area, <inputNetcdfFile> is missing
     * from run_info.xml, but is defined in states.xml <readLocation> element; for 2-N areas, <inputNetcdfFile>is
     * present and no need to have states meta file.
     * 
     * @throws Exception
     */
    private void parseGriddedFfgStates() throws Exception
    {

        final String griddedFfgStateFileName = _runInfo.getProperties()
                                                       .getProperty(GriddedFFGConstants.GRIDDED_FFG_STATE_FILE_NAME);

        if(griddedFfgStateFileName != null && griddedFfgStateFileName.length() > 0)
        {//first area
            final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

            xmlParser.parseStatesMetaFileAndLoadState(griddedFfgStateFileName, _state);

            final Collection<StateLocation> collection = _state.getStateLocation().values();

            if(collection.size() != 1)
            {
                throw new Exception("Error: Gridded FFG has more than one states meta file.");
            }

            final StateLocation stateLoc = (StateLocation)collection.toArray()[0];

            super.setInputStateFile(stateLoc.getReadLocation()); //this is the input netcdf template file
            super.setOutputStateFile(stateLoc.getWriteLocation()); //this is the output netcdf file, analogy to statesO.txt

            _logger.log(Logger.DEBUG,
                        "This is the first area of the RFC. Using the input netcdf template file defined in "
                            + griddedFfgStateFileName);

        }
        else
        {
            _logger.log(Logger.DEBUG,
                        "This is not the first area of the RFC. The input netcdf template file is not defined in states meta file. It is defined from <inputNetcdfFile> element.");
        }
    } //close method

}
