package ohd.hseb.ohdutilities.ffg.model;

import java.io.File;
import java.io.FileFilter;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ohd.hseb.ohdutilities.UtilityParameters;
import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgParameters;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdmodels.ModelState;

abstract class FFGModelController
{
    protected FFGDriver _ffgDriver;

    /*
     * holding one or multiple GroupOfParameters.
     */
    protected final UtilityParameters _modelParams = new UtilityParameters();

    /*
     * holding one or multiple ModelStates. The key is the groupId, the value is ModelState obj.
     */
    protected final Map<String, ModelState> _statesMap = new HashMap<String, ModelState>();

    protected Logger _logger;

    private final FileFilter _dirFileFilter = new FileFilter()
    {
        public boolean accept(final File file)
        {
            return (file.isDirectory() && !file.isHidden());
        }
    };

    //This filter only returns xml files(avoid svn hidden files)
    private final FileFilter _xmlFileFilter = new FileFilter()
    {
        public boolean accept(final File file)
        {
            return ((file.isFile() && !file.isHidden() && file.getName().endsWith("xml")));
        }
    };

    /**
     * Returns an object of {@link GroupOfParameters} which corresponds to area ID. If cannot find it, throws an
     * Exception.
     */
    protected GroupOfParameters getGroupOfParameters(final String areaId) throws Exception
    {

        final GroupOfParameters groupOfParams = _modelParams.getParameters(areaId);

        if(groupOfParams == null)
        {
            throw new Exception("Cannot find the parameters for the area id " + areaId);
        }

        return groupOfParams;
    }

    /**
     * Parse each area's model parameter xml file(SNOW17 or SACSMA) or each area's states meta xml file.
     * 
     * @param propertyRootDirStr - can be one of the four: {@link FfgConstants#SNOW_MODEL_PARAMETERS_ROOT_DIR},
     *            {@link FfgConstants#SNOW_MODEL_STATES_ROOT_DIR}, {@link FfgConstants#RUNOFF_MODEL_PARAMETERS_ROOT_DIR}
     *            and {@link FfgConstants#RUNOFF_MODEL_STATES_ROOT_DIR}
     * @throws Exception
     */
    protected void parseAreasParamsOrStatesMetaXmlFile(final String propertyRootDirStr) throws Exception
    {
        boolean isParams;
        if(propertyRootDirStr == FfgConstants.SNOW_MODEL_PARAMETERS_ROOT_DIR
            || propertyRootDirStr == FfgConstants.RUNOFF_MODEL_PARAMETERS_ROOT_DIR)
        {
            isParams = true;
        }
        else if(propertyRootDirStr == FfgConstants.SNOW_MODEL_STATES_ROOT_DIR
            || propertyRootDirStr == FfgConstants.RUNOFF_MODEL_STATES_ROOT_DIR)
        {
            isParams = false;
        }
        else
        {
            throw new Exception("The property root directory[" + propertyRootDirStr + "] is wrong.");
        }

        final File rootDir = new File(_ffgDriver.getRunInfo().getProperties().getProperty(propertyRootDirStr));

        final File[] directoriesAsGroupId = rootDir.listFiles(_dirFileFilter);
        //if running multiple areas, will have multiple sub-directories; its value(directoriesAsAreaId[j].getName().toUpperCase()) equals to "area id"

        if(directoriesAsGroupId == null || directoriesAsGroupId.length == 0)
        {
            throw new Exception("There are non subdirectories in the path: " + rootDir.getPath());
        }

        final List<String> groupIdList = Arrays.asList(((FfgParameters)(_ffgDriver.getParameters())).getFfgAreaIdArray());

        for(int j = 0; j < directoriesAsGroupId.length; j++)
        {
            final File[] xmlFiles = directoriesAsGroupId[j].listFiles(_xmlFileFilter);

            String xmlFileNameToBeParsed = "";

            //going through the array to find the parameter xml file or states.xml, exclude inputs.xml
            for(int i = 0; i < xmlFiles.length; i++)
            {
                /*
                 * if the xml file is one of those inputs.xml, do not parse this file(here is to parse parameter or
                 * states meta xml files). To single out these files, however, there is some complexity due to the
                 * different file separator on linux versus on PC.
                 */
                final boolean isInputsXml = _ffgDriver.getRunInfo()
                                                      .getInputTimeSeriesFileList()
                                                      .contains(xmlFiles[i].toString()
                                                                           .replace(OHDConstants.BACKWARD_SLASH_CHAR,
                                                                                    OHDConstants.FORWARD_SLASH_CHAR));
                //.replace(OHDConstants.BACKWARD_SLASH_CHAR, OHDConstants.FORWARD_SLASH_CHAR) will do nothing when running on linux; it only helps when running on PC

                if(!isInputsXml)
                {//if enter here, xmlFiles[i] is not one of those input time series file

                    if(xmlFileNameToBeParsed.isEmpty() == false)
                    {//if enter here, it means that xmlFileNameToBeParsed has been assigned during previous looping, so there are multiple files to be parsed, which is wrong
                        throw new Exception("There are more than one parameter xml files(or states meta xml files) to be parsed in the directory: "
                            + Arrays.toString(xmlFiles));
                    }

                    xmlFileNameToBeParsed = xmlFiles[i].toString();

                }

            } //close for loop

            if(xmlFileNameToBeParsed.isEmpty())
            {
                throw new Exception("Did not find the xml file to be parsed under " + directoriesAsGroupId[j]);
            }

            final String groupId = directoriesAsGroupId[j].getName().toUpperCase(); //group id is same as area id, but may not be same as model id

            if(!groupIdList.contains(groupId))
            {
                throw new Exception("The parameter directory or states directory for the group id '" + groupId
                    + "' is missing.");
            }

            _logger.log(Logger.DEBUG, "Parsing the xml file: " + xmlFileNameToBeParsed);

            final FewsXMLParser xmlParser = new FewsXMLParser(_logger);

            if(isParams)
            {
                final UtilityParameters utilParams = new UtilityParameters();

                xmlParser.parseParameters(xmlFileNameToBeParsed, utilParams);

                _modelParams.insertParameters(groupId, utilParams.getParameters(OHDConstants.DEFAULT_STRING));

            }
            else
            {

                final ModelState aState = new ModelState();

                xmlParser.parseStatesMetaFileAndLoadState(xmlFileNameToBeParsed, aState);

                aState.setId(groupId);

                _statesMap.put(groupId, aState);
            }

        }//close for loop

    }

}
