package ohd.hseb.ohdutilities.ffg.model;

import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgParameters;
import ohd.hseb.ohdutilities.ffg.model.rainfall.ApiContFFGRainfallRunoffModel;
import ohd.hseb.ohdutilities.ffg.model.rainfall.FFGRainfallRunoffModel;
import ohd.hseb.ohdutilities.ffg.model.rainfall.SacFFGRainfallRunoffModel;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDUtilities;

/**
 *This class holds all the parameters and states for the Rainfall Runoff models(SAC-SMA or API-Cont). It also controls
 * which rainfall runoff model to be run based on the modelName passes to its method.
 */
public class FFGRainfallRunoffModelController extends FFGModelController
{
    private FFGRainfallRunoffModel _rainFallModel;

    /**
     * Constructor. Besides, parse and load Rainfall Runoff Model's parameters and states, including multiple segments
     * when multiple segs are used.
     */
    public FFGRainfallRunoffModelController(final FFGDriver driver) throws Exception
    {
        _ffgDriver = driver;

        _logger = driver.getLogger();

        super.parseAreasParamsOrStatesMetaXmlFile(FfgConstants.RUNOFF_MODEL_PARAMETERS_ROOT_DIR);

        super.parseAreasParamsOrStatesMetaXmlFile(FfgConstants.RUNOFF_MODEL_STATES_ROOT_DIR);

    }

    /**
     * Set the controller to be ready for either SAC-SMA model or API-Cont model computation, including setting the
     * parameter and the state. The code here can handle that one area runs SACSMA and another area runs API-Cont,
     * though I believe all areas in FFH should be consistent regarding to the rainfall runoff model.
     */
    public void initControllerForSegment(final String areaId) throws Exception
    {
        final String rainFallModel = ((FfgParameters)_ffgDriver.getParameters()).getFfgRainfallRunoffModelName(areaId);

        final GroupOfParameters groupOfParams = getGroupOfParameters(areaId);

        if(rainFallModel.equalsIgnoreCase(FfgConstants.SAC_MODEL_NAME))
        {
            _rainFallModel = new SacFFGRainfallRunoffModel(_ffgDriver, groupOfParams, _statesMap.get(areaId));
        }
        else
        {//API-Cont

            _rainFallModel = new ApiContFFGRainfallRunoffModel(_ffgDriver, groupOfParams, _statesMap.get(areaId));
        }

        _logger.log(Logger.DEBUG, "Run the model " + rainFallModel + " with area id " + areaId);

    }

    /**
     * Call one of {@link FFGRainfallRunoffModel} subclass' getRunoff method, depending on what {@link #_rainFallModel}
     * is.
     * 
     * @param precipAmount - unit of MM
     * @param modelTimeInterval - HR
     * @return runoff - unit of MM
     */
    public double getRunoff(final double precipAmount, final int modelTimeInterval) throws Exception
    {
        final double runOff = _rainFallModel.getRunoff(precipAmount, modelTimeInterval);

        _logger.log(Logger.DEBUG, "Rainfall Runoff model: precip(MM)=" + OHDUtilities.getFortranPrecison(precipAmount)
            + " interval(HR)=" + modelTimeInterval + " runoff(MM)=" + OHDUtilities.getFortranPrecison(runOff));

        return runOff;
    }

}
