package ohd.hseb.ohdutilities.ffg.model;

import ohd.hseb.ohdmodels.snow17.Snow17ModelConstants;
import ohd.hseb.ohdmodels.snow17.Snow17Technique;
import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.ohdutilities.ffg.FfgConstants;
import ohd.hseb.ohdutilities.ffg.FfgParameters;
import ohd.hseb.ohdutilities.ffg.model.snow.FFGSnowModel;
import ohd.hseb.ohdutilities.ffg.model.snow.Snow17FFGSnowModel;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDUtilities;

/**
 * This class controls which SNOW model to run. Right now, it has SNOW-17 only. SNOW model computation is optional
 * before running Rainfall Runoff model(SAC-SMA or API-CONT).
 */
public class FFGSnowModelController extends FFGModelController
{
    private FFGSnowModel _snowModel;

    private boolean _runSnow = false; //default value

    private final Snow17Technique _snow17Technique;

    /**
     * Constructor, regardless run SNOW model or not.
     */
    public FFGSnowModelController(final FFGDriver driver) throws Exception
    {

        _logger = driver.getLogger();

        _ffgDriver = driver;

        _snow17Technique = new Snow17Technique(_ffgDriver.getDriverProperties(), _logger);

    }

    /**
     * Decide if to run SNOW model or not for this area. If run SNOW model, parse its parameter xml file and input state
     * file. The code here can handle one area runs SNOW17 and another area does not run SNOW17, though I believe all
     * the areas in FFH be consistent regarding to run SNOW17 or not.
     * 
     * @param snowModel - if it is null, do not run SNOW model and if run info wants to run the SNOW model by technique
     *            SNOW, send out a warning message; if it is not null, then check if SNOW technique is present in run
     *            info and it it allows to run SNOW model or not.
     */
    public void initControllerForSegment(final String areaId) throws Exception
    {
        final String snowModel = ((FfgParameters)_ffgDriver.getParameters()).getFfgSnowModelName(areaId);

        if(snowModel == null)
        {
            _logger.log(Logger.DEBUG, "FFG does not run SNOW-17 model based on the parameters.");

            // TECHNIQUE SNOW tries to run SNOW-17, give a warning message
            if(_snow17Technique.getBoolValue(Snow17ModelConstants.TECHNIQUE_SNOW))
            {
                _logger.log(Logger.WARNING,
                            "WARNING: FFH parameters does not have SNOW-17, but Run Info properties SNOW TECHNIQUE set to TRUE.");

            }

            _runSnow = false;
        }
        else
        {//snowModel == SNOW_MODEL.SNOW_17

            if(_snow17Technique.getBoolValue(Snow17ModelConstants.TECHNIQUE_SNOW) == false)
            {// parameter wants to run SNOW-17, but TECHNIQUE SNOW turns it off

                _logger.log(Logger.DEBUG, "Run Info properties SNOW TECHNIQUE set to FALSE, so do NOT run the model "
                    + snowModel);

                _runSnow = false;
            }
            else
            { // parameter wants to run SNOW-17, and TECHNIQUE SNOW allows it

                _logger.log(Logger.DEBUG, "Run the model " + snowModel + " with area id " + areaId);

                _runSnow = true;

                if(_modelParams.isEmpty())
                { //since the following parsing covers all areas and the code here is looped through each area. To avoid repeated parsing

                    super.parseAreasParamsOrStatesMetaXmlFile(FfgConstants.SNOW_MODEL_PARAMETERS_ROOT_DIR);

                    super.parseAreasParamsOrStatesMetaXmlFile(FfgConstants.SNOW_MODEL_STATES_ROOT_DIR);
                } //finished parsing

                final GroupOfParameters groupOfParams = getGroupOfParameters(areaId);

                _snowModel = new Snow17FFGSnowModel(_ffgDriver, groupOfParams, _statesMap.get(areaId));

            }
        }
    }

    /**
     * @return the parameter precip value if not running SNOW-17; return rain plus melt if running SNOW-17
     */
    public double getSnowRaim(final double precip, final int modelInterval) throws Exception
    {
        if(_runSnow == false)
        { // do not run SNOW model, just returns precip

            return precip;
        }

        final double raim = _snowModel.getSnowRaim(precip, modelInterval);

        _logger.log(Logger.DEBUG, "SNOW model computation: precip(MM)=" + OHDUtilities.getFortranPrecison(precip)
            + " RAIM(MM)=" + OHDUtilities.getFortranPrecison(raim));

        return raim;
    }

}
