package ohd.hseb.ohdutilities.ffg.model.rainfall;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdmodels.apicont.APIContModelDriver;
import ohd.hseb.ohdmodels.apicont.APIContModelParameters;
import ohd.hseb.ohdmodels.apicont.APIContModelState;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.Driver;
import ohd.hseb.util.fews.FewsRegularTimeSeries;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.State;
import ohd.hseb.util.fews.ohdmodels.LegacyModelAdapter;

public class ApiContFFGRainfallRunoffModel extends FFGRainfallRunoffModel
{

    private LegacyModelAdapter _apiContAdapter;
    public static final String FFG_DURATION_HR = "FFGdurationHR";

    private APIContModelDriver _apiContModelDriver = null;

    public ApiContFFGRainfallRunoffModel(final Driver driver, final GroupOfParameters groupOfParams, final State state) throws Exception
    {
        _logger = driver.getLogger();

        _rainFallRunoffParam = new APIContModelParameters();

        final List<GroupOfParameters> paramsList = new ArrayList<GroupOfParameters>(1);
        paramsList.add(groupOfParams);

        _rainFallRunoffParam.setNumberOfValidPeriods(1);
        _rainFallRunoffParam.setParametersFromList(paramsList);

        _state = new APIContModelState();

        _state.setStatesMap(state.getStateMap());
        /*
         * For Fogbugz case 616, MARFC run workflow for MARFC_FFH_##Z, where ## is 00,06,12 and 18 in different T0 (also
         * 00, 06, 12 and 18) each day. Since APICont model State must starts at previous 12Z,(but not greater than past
         * 24 hours). But FFHDrive only passes the initial T0. Therefore, we need to do some adjustment here. This block
         * was added by RHC, in May 2012
         */

        // This will get the T0 time
        final DateTime stateTime = new DateTime(driver.getInitialStateTime(), OHDConstants.GMT_TIMEZONE);

        final int startHourOfDay = driver.getStartHourOfDay(); // this always gets 12Z
        final int stateTimeHour = stateTime.hour(); // at T0 hour 00, 06, 12 and 18 Z
        final int difference = -startHourOfDay - stateTimeHour; // The difference should be -12, -18, -24 and -30 hours 
        // The if-else was changed by RHC for Fogbugz case 1070, in April 15, 2013 during tax date
        if(stateTime.hour() > 12)
        {//when T0 > 12Z 
            stateTime.addHours(startHourOfDay - stateTimeHour); // set to 12Z when T0 is 18Z (18 - 6 = 12)
        }
        else
        {//when T0 in 00Z, 06 and 12Z (<= 12Z)
            stateTime.addHours(difference); // set to previous 12Z, (0 - 12 = -12); (6 - 18 = -12); (12 - 24 = -12)
        }

        _state.setDateTime(stateTime.getTimeInMillis()); // set it in _state object
        _logger.log(Logger.DEBUG, "API-CONT Model Initial " + _state.toString());

        _runInfo = driver.getRunInfo();

        _apiContModelDriver = new APIContModelDriver();
        _apiContModelDriver.setLogger(_logger);

        _apiContModelDriver.setRunInfo(_runInfo);

        _apiContModelDriver.setWorkDir(driver.getWorkDir());

        _apiContModelDriver.setLogger(_logger);

        _apiContModelDriver.setStopWatch(driver.getStopWatch());

        this.processInputData();

    }

    private void processInputData() throws Exception
    {
        final APIContModelParameters apiContParams = (APIContModelParameters)_rainFallRunoffParam;

        _apiContModelDriver.setParameters(apiContParams);
        _apiContModelDriver.setState(_state);

        _runInfo.setRunEndTime(_runInfo.getRunLastObservationTimeLong() + 1 * Timer.ONE_HOUR);

        /*
         * initialize driving input (RAIM) TS at first time to pass validate input Ts. It needs to be
         * FewsRegularTimeSeries because _tsList.get(0) is assumed to be FewsRegularTimeSeries and its LongName,
         * SoureOrganization etc are used for writing out TSs in resultMap to outputs.xml
         */
        final FewsRegularTimeSeries raimTS = new FewsRegularTimeSeries(_runInfo.getRunLastObservationTimeLong()
            + Timer.ONE_HOUR, _runInfo.getRunLastObservationTimeLong() + Timer.ONE_HOUR, 1, MeasuringUnit.mm);

        raimTS.setTimeSeriesType(DataType.RAIM_DATATYPE);
        _logger.log(Logger.DEBUG, "TS-id: " + apiContParams.getStringDataParameter(OHDConstants.TS_ID));
        raimTS.setQualifierIds(Arrays.asList(apiContParams.getStringDataParameter(OHDConstants.TS_ID)));

        _apiContModelDriver.getTsList().add(raimTS);

        _apiContModelDriver.setDrivingRTS(raimTS);

        _apiContAdapter = new LegacyModelAdapter(_apiContModelDriver, _apiContModelDriver.getWorkDir(), _logger);//the constructor will call APIContModelDriver runModelValidation too
    }

    /**
     * Use API-CONT model to calculate runoff(unit of MM).
     * 
     * @param precipAmount - unit of MM
     */
    @Override
    public double getRunoff(final double precipAmount, final int modelTimeInterval) throws Exception
    {

        _logger.log(Logger.DEBUG, "modelTimeInterval:" + modelTimeInterval + "  precipAmount(IN) = " + precipAmount
            * OHDConstants.INCHES_PER_MM);

        //modify _tsList.get(0) (e.g. _drivingTs) 's interval and time step value
        _apiContModelDriver.getTsList().get(0).setIntervalInHours(modelTimeInterval);
        _apiContModelDriver.getTsList().get(0).setMeasurementByIndex(precipAmount, 0);

        /** ---------------------calculate runoff in the time step---------------------- */

        // apicont for FFG needs an extra argument - FFG TIME INTERVAL
        final SortedProperties properties = new SortedProperties();

        properties.put(FFG_DURATION_HR, String.valueOf(modelTimeInterval));

        _apiContAdapter.writeStandAloneFiles();

        _apiContAdapter.executeModel(_apiContModelDriver.getDriverProperties()
                                                        .getProperty(OHDConstants.LEGACY_LOCATION_DIR)
                                         + "/"
                                         + "apicont",
                                     properties);

        // load the results
        final List<RegularTimeSeries> resultsList = _apiContAdapter.readOutputFlatFilesToLists();

        // get TCI (INFW) in unit MM
        return (resultsList.get(0).getMeasurementValueByIndex(0, MeasuringUnit.mm));

    }

}
