package ohd.hseb.ohdutilities.ffg.model.rainfall;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdmodels.sacsma.SacSmaModelConstants;
import ohd.hseb.ohdmodels.sacsma.SacSmaModelDriver;
import ohd.hseb.ohdmodels.sacsma.SacSmaModelParameters;
import ohd.hseb.ohdmodels.sacsma.SacSmaModelState;
import ohd.hseb.ohdmodels.sacsma.SacSmaRainfallRunoffModelAdapter;
import ohd.hseb.ohdmodels.sacsma.SacSmaTechnique;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.Driver;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.ParameterType;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.State;

public class SacFFGRainfallRunoffModel extends FFGRainfallRunoffModel
{
    private SacSmaRainfallRunoffModelAdapter _sacSmaAdapter;

    private RegularTimeSeries _matTs; //only required for FRZE

    private final SacSmaTechnique _sacSmaTechnique;

    public SacFFGRainfallRunoffModel(final Driver driver, final GroupOfParameters groupOfParams, final State state) throws Exception
    {
        _logger = driver.getLogger();

        _rainFallRunoffParam = new SacSmaModelParameters();

        _rainFallRunoffParam.setLogger(_logger);

        /*
         * SACSMA running in FFG does not use the parameter "ET_DEMAND_CURVE" at all, so it can be absent in
         * paramsSacSma.xml; To pass SacSmaModelParameter extractValuesFromMap() later, the dummy parameter is added
         */
        final ParameterType etDemandParam = new ParameterType();
        etDemandParam.setId(SacSmaModelConstants.ET_DEMAND_CURVE_TAG);
        etDemandParam.setTable(new Table());

        groupOfParams.setParameter(etDemandParam);

        final List<GroupOfParameters> paramsList = new ArrayList<GroupOfParameters>(1);
        paramsList.add(groupOfParams);

        _rainFallRunoffParam.setNumberOfValidPeriods(1);
        _rainFallRunoffParam.setParametersFromList(paramsList);

        _runInfo = driver.getRunInfo();

        _state = new SacSmaModelState();

        _state.setStatesMap(state.getStateMap());

        _state.extractValuesFromMap();

        _state.setId(state.getId());

        _sacSmaTechnique = new SacSmaTechnique(_runInfo.getProperties(), _logger);

        this.processInputData(driver.getTsList());
    }

    private void processInputData(final List<RegularTimeSeries> tsList) throws Exception
    {
        final SacSmaModelParameters sacSmaParams = (SacSmaModelParameters)_rainFallRunoffParam;

        final SacSmaModelDriver sacSmaModelDriver = new SacSmaModelDriver();

        sacSmaModelDriver.setParameters(sacSmaParams);
        sacSmaModelDriver.setState(_state);
        sacSmaModelDriver.setRunInfo(_runInfo);
        sacSmaModelDriver.setLogger(_logger);

        if(sacSmaParams.useFrozenGroundCalc())
        {
            //TECHNIQUE FROST only be meaningful when using FRZE
            if(_sacSmaTechnique.getBoolValue(SacSmaModelConstants.FROST_TAG) == false)
            {//FROST TECHNIQUE turn off FRZE

                sacSmaParams.setUseFrozenGroundCalc(false); //FROST is false, turn off FRZE
            }
            else
            {//now really use FRZE for sure

                for(final RegularTimeSeries ts: tsList)
                {
                    if(ts.getTimeSeriesType().equalsIgnoreCase(DataType.MAT_DATATYPE))
                    {
                        if(_state.getId() == null || _state.getId().equalsIgnoreCase(ts.getLocationId()))
                        {//for multiple areas, there are multiple input MAT TSs in inputs.xml, its locationId corresponds to modelId
                            _matTs = ts;

                            break;
                        }
                    }
                }

                //MAT TS is needed for SAC-SMA FRZE
                if(_matTs == null)
                {
                    throw new Exception("To run SAC-SMA Frozen Ground, input MAT time series is required.");
                }

                sacSmaModelDriver.setMatTs(_matTs);
            }
        } //close if(sacSmaParams.useFrozenGroundCalc())
        else
        {
            if(_sacSmaTechnique.getBoolValue(SacSmaModelConstants.FROST_TAG))
            {
                _logger.log(Logger.WARNING,
                            "WARNING: SAC-SMA does not have Frozen Ground based on its parameter, so could not apply"
                                + " FROST technique set equals to TRUE in run_info.xml");
            }
        }

        sacSmaParams.setUseMAPEInput(false); //do not use input MAPE TS for FFH
        sacSmaParams.setUseSascInputTS(false); //do not use input SASC TS for FFH

        _sacSmaAdapter = new SacSmaRainfallRunoffModelAdapter(sacSmaModelDriver);

    }

    /**
     * Run SAC-SMA model to get runoff(unit of MM).
     * 
     * @param precipAmount - unit of MM
     */
    @Override
    public double getRunoff(final double precipAmount, final int modelTimeInterval) throws Exception
    {
        return _sacSmaAdapter.getRunoffBySacSma(precipAmount, modelTimeInterval);
    }

}
