package ohd.hseb.ohdutilities.ffg.model.snow;

import java.util.ArrayList;
import java.util.List;
import java.util.TimeZone;

import javax.management.timer.Timer;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdmodels.snow17.RAINSNOW_MOD_VALUE;
import ohd.hseb.ohdmodels.snow17.Snow17Model;
import ohd.hseb.ohdmodels.snow17.Snow17ModelParameters;
import ohd.hseb.ohdmodels.snow17.Snow17ModelState;
import ohd.hseb.ohdmodels.snow17.Snow17Results;
import ohd.hseb.ohdutilities.ffg.FFGDriver;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.fews.ohdmodels.ModelState;

public class Snow17FFGSnowModel extends FFGSnowModel
{

    private final Snow17Model _snow17Model;
    private final Snow17ModelState _snow17States;
    private final Snow17ModelParameters _snow17Params;

    private final DateTime _dateTime;

    private final RunInfo _runInfo;

    private RegularTimeSeries _matTs;

    private final int _startOfDay;

    /**
     * Constructor: all the information needed to run SNOW-17 model. Check matTs is present. If not, throws an
     * Exception.
     */
    public Snow17FFGSnowModel(final FFGDriver ffgDriver,
                              final GroupOfParameters groupOfParams,
                              final ModelState modelState) throws Exception
    {
        _logger = ffgDriver.getLogger();

        _runInfo = ffgDriver.getRunInfo();

        //find MAT ts for this area(for multiple areas, there are multiple input MAT TSs, one for each area)
        for(final RegularTimeSeries ts: ffgDriver.getTsList())
        {
            if(ts.getTimeSeriesType().equalsIgnoreCase(DataType.MAT_DATATYPE))
            {
                if(ts.getLocationId().equalsIgnoreCase(modelState.getId()))
                {//states.getId() == areaId, it was set during parsing statesI.txt
                    _matTs = ts;

                    break; //jump out of for loop since it has been found.
                }
            }
        }

        //throws an Exception if MAT TS for this area was not found 
        if(_matTs == null)
        {
            throw new Exception("To run SNOW-17, input MAT time series for " + modelState.getId() + " is not found.");
        }
        //every ts in _tsList has been checked to have enough data in UtilityDriver runDriverValidation(). No need to check _matTs has enough data

        _snow17Params = new Snow17ModelParameters();
        _snow17Params.setLogger(_logger);

        final List<GroupOfParameters> paramsList = new ArrayList<GroupOfParameters>(1);
        paramsList.add(groupOfParams);

        _snow17Params.setNumberOfValidPeriods(1);
        _snow17Params.setParametersFromList(paramsList);

        _snow17Params.setUseRainSnowElevationInput(false); //not use RSEL option input TS. The value is hard-coded -999.0 for pack19

        _snow17States = new Snow17ModelState();

        _snow17States.setStatesMap(modelState.getStateMap());

        _startOfDay = ffgDriver.getStartHourOfDay();

        _snow17Model = new Snow17Model(_snow17Params, _snow17States, null, null, _startOfDay, _logger);

        _dateTime = new DateTime(_runInfo.getRunLastObservationTimeLong(), _startOfDay);

    } //close constructor

    @Override
    public double getSnowRaim(final double precip, final int modelInterval) throws Exception
    {
        final long time = _runInfo.getRunLastObservationTimeLong() + modelInterval * Timer.ONE_HOUR; //the time step to be calculated by models. only one time step.

        /*--------------- Refresh parameters and re-modify it based on new interval ---------*/
        _snow17Params.extractValuesFromMap(); //the way to refresh params back to params.xml
        /*--------------- Refresh states and re-modify it based on new interval ---------*/
        _snow17States.extractValuesFromMap();//the way to refresh states back to statesI.txt values
        
        _snow17Params.modifyParametersForDifferentTimestep(modelInterval);
        
        _snow17States.modifyStatesForDifferentTimestep(modelInterval);        

        /*--------------- get the correct air temperature based on new interval ---------*/
        double airTemp = 0.0;

        if(modelInterval <= _matTs.getIntervalInHours())
        {
            airTemp = _matTs.getMeasurementValueByTime(_runInfo.getRunLastObservationTimeLong()
                                                           + _matTs.getIntervalInMillis(),
                                                       MeasuringUnit.degreesCelsius);
        }
        else
        {
            final int num = modelInterval / _matTs.getIntervalInHours();

            for(int i = 1; i <= num; i++)
            {
                airTemp += _matTs.getMeasurementValueByTime(_runInfo.getRunLastObservationTimeLong() + i
                                                                * _matTs.getIntervalInMillis(),
                                                            MeasuringUnit.degreesCelsius);
            }

            airTemp /= num; //average out
        }

        /*--------------- if needed, use lapse rate to adjust temperature ---------*/
        if(_snow17Params.getElevationWithAirTemp() != _snow17Params.getElevation())
        { //use lapse rate

            final double talMax = _snow17Params.getTalmax();
            final double talMin = _snow17Params.getTalmin();

            final TimeZone runInfoTZ = _runInfo.getTimeZone();

            if(_runInfo.getTimeZoneRawOffsetInHours() == 0)
            {//run_info.xml is GMT time zone, log it as warning

                _logger.log(Logger.WARNING,
                            "run_info.xml is in GMT time zone. Please confirm that this is correct. "
                                + "The output results is affected by the local time zone when the two parameters ELEV and TAELEV are not equal.");
            }

            //get local hour
            _dateTime.setTime(time);

            double tl = _dateTime.getNwsrfsHour() + OHDUtilities.getLocalTimeAt12Z(runInfoTZ);

            if(tl > 24.0)
            {
                tl = tl - 24.0;
            }

            double lapseRate;

            if(tl >= 15.0)
            {
                lapseRate = talMax - (tl - 15.0) / 15.0 * (talMax - talMin);

            }
            else if(tl > 6.0)
            {
                lapseRate = talMin + (tl - 6.0) / 9.0 * (talMax - talMin);
            }
            else
            {
                lapseRate = talMin + (6.0 - tl) / 15.0 * (talMax - talMin);
            }

            //TAELEV - ELEV
            final double diffTaElevAndElev = _snow17Params.getElevationWithAirTemp() - _snow17Params.getElevation();

            airTemp += diffTaElevAndElev * 0.01 * lapseRate;

            _logger.log(Logger.DEBUG, "Using lapse rate to adjust the air temperature.");
        }

        _logger.log(Logger.DEBUG, "AIR TEMPERATURE=" + airTemp + "\tmodelInterval=" + modelInterval + "\tmap(mm)="
            + precip);

//      C  SET INITIAL VALUES
//      CEA NOTE: BY SETTING PCTS(1) TO 0.0 THIS RESULTS IN ALL THE PRECIPITATION
//      CEA  INPUT TO PACK19 TO BE IN THE FORM OF RAIN.  I BELIEVE THIS IS CORRECT
//      CEA  FOR FFG COMPUTATIONS.  RAIN-SNOW ELEVATION VALUE COMPUTED IN EX32
//      CEA  WILL NEVER BE USED AS IF PERCENT SNOWFALL (PCTS) IS DEFINED, IT
//      CEA  OVERRIDES OTHER LOGIC FOR DETERMINING THE FORM OF PRECIPITATION.
//            PCTS(1) = 0.0
//            OWE = -999.0
//            OSC = -999.0
//      CEA SET OBSERVED SNOW DEPTH TO MISSING
//            ODPT = -999.0

        final Snow17Results snow17Results = _snow17Model.pack19(airTemp,
                                                                new double[]{precip},
                                                                new double[]{OHDConstants.MISSING_DATA}, //%rain or snow set by RAINSNOW MOD
                                                                OHDConstants.MISSING_DATA, //always ignore RAIN-SNOW-ELEVATIOM TS(RSEL)
                                                                time,
                                                                OHDConstants.MISSING_DATA,
                                                                OHDConstants.MISSING_DATA,
                                                                OHDConstants.MISSING_DATA,
                                                                RAINSNOW_MOD_VALUE.RAIN); //always 100% rain

        _logger.log(Logger.DEBUG, "SNOW17 precip =" + precip + " RAIM=" + snow17Results.getRaim()[0]);

        return snow17Results.getRaim()[0];
    }	 
}
