package ohd.hseb.ohdutilities.prodgen;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Properties;

import ohd.hseb.grid.GridUtilities;
import ohd.hseb.grid.NetcdfConstants;
import ohd.hseb.grid.RfcGridWithTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.SortedProperties;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdutilities.LegacyUtilitiesAdapter;

public class GribProdGenerator
{
    private final ProductGeneratorDriver _productGenDriver;
    private final HashMap<String, String> _subcenterHashmap;

    public GribProdGenerator(final ProductGeneratorDriver productGenDriver)
    {
        super();
        _productGenDriver = productGenDriver;

//        put the siteId and subcenter data for 13 RFCs in the hashmap
        _subcenterHashmap = new HashMap<String, String>();

        _subcenterHashmap.put("KTUA", "150");
        _subcenterHashmap.put("PACR", "151");
        _subcenterHashmap.put("KSTR", "152");
        _subcenterHashmap.put("KRSA", "153");
        _subcenterHashmap.put("KORN", "154");
        _subcenterHashmap.put("KRHA", "155");
        _subcenterHashmap.put("KKRF", "156");
        _subcenterHashmap.put("KMSR", "157");
        _subcenterHashmap.put("KTAR", "158");
        _subcenterHashmap.put("KPTR", "159");
        _subcenterHashmap.put("KTIR", "160");
        _subcenterHashmap.put("KALR", "161");
        _subcenterHashmap.put("KFWR", "162");
        _subcenterHashmap.put("NOHR", "163");
        _subcenterHashmap.put("KNES", "170");
        _subcenterHashmap.put("KOHD", "200");
    }

    /**
     * Add all fields into properties in order to generate arguments.txt file
     * 
     * @throws IOException
     */
    private void createFfgGribProductProperties(final Properties properties) throws Exception
    {

        final String inputNetCdfFile = _productGenDriver.getRunInfo().getInputNetCdfFile();
        properties.put(TextProductGeneratorConstants.INPUTNETCDFFILE, inputNetCdfFile);

        final String outputGrib1FilePref = _productGenDriver._outputDir + "/"
            + _productGenDriver.get_singleProdId().substring(8);
        properties.put(TextProductGeneratorConstants.OUTPUTGRIB1FILEPREF, outputGrib1FilePref);

        final String diagFile = _productGenDriver.getRunInfo().getWorkDir() + "/" + OHDConstants.DIAGNOSTICS_FILENAME;
        properties.put(TextProductGeneratorConstants.DIAGFILENAME, diagFile);
        properties.put(TextProductGeneratorConstants.DEBUGFLAG,
                       String.valueOf(_productGenDriver.getLogger().getPrintDebugInfo()));

        final String siteId = _productGenDriver.get_ffhProductGeneratorParameter().get_processProd().get_prodOffice();
        properties.put(TextProductGeneratorConstants.SITEID, siteId);

        final String processProd = _productGenDriver.get_ffhProductGeneratorParameter().get_processProd().get_prodWmo();
        properties.put(TextProductGeneratorConstants.WMOID, processProd);

        properties.put(TextProductGeneratorConstants.PROCESSID, TextProductGeneratorConstants.FFG_PROCESSID_VAL);

        final Date lastObs = new Date();
        lastObs.setTime(_productGenDriver.getRunInfo().getRunLastObservationTimeLong());
        final DateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmm");
        dateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);
        properties.put(TextProductGeneratorConstants.COMPUTATIONYEAR, dateFormat.format(lastObs).substring(0, 4));
        properties.put(TextProductGeneratorConstants.COMPUTATIONMON, dateFormat.format(lastObs).substring(4, 6));
        properties.put(TextProductGeneratorConstants.COMPUTATIONDAY, dateFormat.format(lastObs).substring(6, 8));
        properties.put(TextProductGeneratorConstants.COMPUTATIONHRMIN, dateFormat.format(lastObs).substring(8, 12));

        properties.put(TextProductGeneratorConstants.FORECASTTIMEUNIT,
                       TextProductGeneratorConstants.FFG_FORECASTTIMEUNIT_VAL);
        properties.put(TextProductGeneratorConstants.TIMERANGEINDICATOR,
                       TextProductGeneratorConstants.FFG_TIMERANGINDICATOR_VAL);

        if(_subcenterHashmap.containsKey(siteId))
        {
            properties.put(TextProductGeneratorConstants.SUBCENTERID, _subcenterHashmap.get(siteId));
        }
        else
        {
            _productGenDriver.getLogger().log(Logger.WARNING, "The sub-center id is missing for site " + siteId);
        }

        properties.put(TextProductGeneratorConstants.DECIMALSCALEFACTOR,
                       TextProductGeneratorConstants.FFG_DECIMALSCALEFACTOR_VAL);
        properties.put(TextProductGeneratorConstants.PROJECTIONTYPE, TextProductGeneratorConstants.PROJECTIONTYPE_VAL);

        properties.put(NetcdfConstants.LONORIGIN, NetcdfConstants.LONORIGIN_VAL);
        properties.put(NetcdfConstants.XCELLSIZE, NetcdfConstants.XCELLSIZE_VAL);
        properties.put(NetcdfConstants.YCELLSIZE, NetcdfConstants.YCELLSIZE_VAL);

        final RfcGridWithTime myNetcdfFile = new RfcGridWithTime(inputNetCdfFile, _productGenDriver.getLogger());
        final int southRow = myNetcdfFile.getRfcOrigHrapRow();
        final int westCol = myNetcdfFile.getRfcOrigHrapColumn();

        final double[] latLon = GridUtilities.convertHRAPRowColToLatLon(southRow, westCol);

        properties.put(NetcdfConstants.LATOFFIRSTGRIDPOINT, Double.toString(latLon[0] * 1000));
        properties.put(NetcdfConstants.LONOFFIRSTGRIDPOINT, Double.toString(latLon[1] * 1000));
    }

    public void createGribProduct() throws Exception
    {
        final LegacyUtilitiesAdapter ffgGribProductAdapter = new LegacyUtilitiesAdapter(_productGenDriver.getRunInfo()
                                                                                                         .getWorkDir(),
                                                                                        _productGenDriver.getLogger());
        final SortedProperties properties = new SortedProperties();

        //add all the fields into properties
        this.createFfgGribProductProperties(properties);

        // Execute ffgGribProduct program in C code
        ffgGribProductAdapter.executeUtility(_productGenDriver.getDriverProperties()
                                                              .getProperty(OHDConstants.LEGACY_LOCATION_DIR)
                                                 + "/"
                                                 + TextProductGeneratorConstants.FFGGRIBPRODUCT_EXE_NAME,
                                             properties);

    }

}
