package ohd.hseb.ohdutilities.prodgen;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdutilities.ffg.ffh.FFHConstants;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdutilities.UtilityDriver;
import ohd.hseb.util.fews.ohdutilities.UtilityState;

/**
 * This class will retrieve parameters from params.xml file, process these parameters, generate text product based on
 * these parameters. Similarly to FFG/FFH prodgen in legecy NWSRFS.
 * 
 * @author deng
 */
public class ProductGeneratorDriver extends UtilityDriver
{
    //variables
    private boolean _printLog = false;
    private String _noDateStr = null;
    protected String _outputDir;
    private String _inputProdId = null;
    private String _groupId = null;
    private String _singleProdId = null;

    private String _groupProdValue;

    private final TextPrdGeneratorParameter _ffhProductGeneratorParameter;
    private long _time0;
    private String _forecasterInit = null;
    private int _num_of_products = 0;
    private boolean _onePrdPerFileFlag = false;
    private String _outputFile = "";
    PrintWriter _outFile = null;

    /**
     * Constructor for FFHPRoductGeneratorDriver
     */
    public ProductGeneratorDriver()
    {
        super();
        super._state = new UtilityState(); //FFHProductGenerator does not have a state at all. However, instantiating it to avoid null pointer exception only.
        super._parameters = new TextPrdGeneratorParameter();

        _ffhProductGeneratorParameter = (TextPrdGeneratorParameter)super._parameters;
        _runInfo = super.getRunInfo();
    }

    /**
     * close the output file
     * 
     * @throws IOException
     */
    public void closeGrpOutputTextFile() throws IOException
    {
        if(_outFile != null)
        {
            _outFile.flush();
            //close the text file after all group of product written in the master text file.
            _outFile.close();
        }

    }

    /**
     * create output text file for group of products
     * 
     * @throws IOException
     */
    public void createGrpOutputTextFile() throws IOException
    {
        //create textfile to contain the product, for group of product and all products in one
        // text file, generate a master file. All group products written in one file

        if(_noDateStr != null)
        {
            //create textfile without date string in the file name
            _outputFile = _outputDir + "/" + TextProductGeneratorConstants.PRODUCT_PREFIX_TAG + _groupId.substring(6)
                + ".out";
        }
        else
        {
            //create textfile with date string in the file name
            final String dateStr = dateSF("yyyyMMddHHmmss", 0);
            _outputFile = _outputDir + "/" + TextProductGeneratorConstants.PRODUCT_PREFIX_TAG + _groupId.substring(6)
                + "_" + dateStr + ".out";
        }

        _logger.log(Logger.INFO, "The output generated product is " + _outputFile);

        try
        {
            if(_noDateStr != null)
            {
                _outFile = new PrintWriter(new BufferedWriter(new FileWriter(_outputFile)));
            }
            else
            {
                _outFile = new PrintWriter(new BufferedWriter(new FileWriter(_outputFile, true)));
            }
        }
        catch(final FileNotFoundException e)
        {
            _logger.log(Logger.ERROR, "The generated output file is not found:" + _outputFile
                + " check the property set for outputDirectory.");
            System.out.print("The generated output file is not found:" + _outputFile + "\n");
            e.printStackTrace(_logger.getPrintWriter());
            return;
        }
    }

    /**
     * create output text file and write to it
     * 
     * @throws IOException
     */
    public void createSingleOutputTextFile() throws IOException
    {
        String outputFile = "";
        PrintWriter outFile = null;

        if(_noDateStr != null)
        {
            //create textfile without date string in the file name
            outputFile = _outputDir + "/" + _singleProdId + ".out";
        }
        else
        {
            //create textfile with date string in the file name
            final String dateStr = dateSF("yyyyMMddHHmmss", 0);
            outputFile = _outputDir + "/" + _singleProdId + "_" + dateStr + ".out";
        }

        _logger.log(Logger.INFO, "The single output generated product is " + outputFile);

        try
        {
            if(_noDateStr != null)
            {
                outFile = new PrintWriter(new BufferedWriter(new FileWriter(outputFile)));
            }
            else
            {
                outFile = new PrintWriter(new BufferedWriter(new FileWriter(outputFile, true)));
            }
        }
        catch(final FileNotFoundException e)
        {
            _logger.log(Logger.ERROR, "The generated output file is not found:" + outputFile
                + " check the property set for outputDirectory.");
            System.out.print("The generated output file is not found: " + outputFile + "\n");
            e.printStackTrace(_logger.getPrintWriter());
            return;
        }

        for(final String sb: _ffhProductGeneratorParameter.get_prodInfoById())
        {
            outFile.write(sb);
        }
        outFile.flush();
        outFile.close();
    }

    private String dateSF(final String sf, final long ldate)
    {
        final DateFormat dateFormat = new SimpleDateFormat(sf);
        Date date;

        if(ldate > 0)
        {
            dateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);
            date = new Date(ldate);
        }
        else
        {
            date = new Date();
        }

        return (dateFormat.format(date));

    } //close dateSF method ----------------------------------------------------

    /**
     * implement inherited abstract method execute(). It will retrive data from run_info.xml process product and output
     * generated text products.
     */
    @Override
    public void execute() throws Exception
    {
        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

        _logger.log(Logger.INFO, "Running FFH/FFG Product Generator.");

        //get the root directories from run_info.xml file
        _outputDir = super.getDriverProperties().getProperty("outputDirectory");

        //Added 4.1.a 07/09/2014 by champ
        final File theDir = new File(_outputDir);

        // If the directory does not exist, create it
        if(!theDir.exists())
        {
            final Path pathDir = Paths.get(_outputDir);
            Files.createDirectories(pathDir.getParent()); //1st dir after parent path
            theDir.mkdir(); //sub dir
            _logger.log(Logger.DEBUG, "Creating directory: " + theDir.getAbsolutePath());
        }
        //end 07/09/14

        _inputProdId = super.getDriverProperties().getProperty("productId");
        _groupId = super.getDriverProperties().getProperty("groupId");
//        _lastObservationDateTime = super.getRunInfo().getRunLastObservationTimeLong();
        _time0 = super.getRunInfo().getTime0Long();
        _forecasterInit = super.getDriverProperties().getProperty("forecasterInitials");
        _noDateStr = super.getDriverProperties().getProperty("noDateString");
        _ffhProductGeneratorParameter.set_lastObservationDateTime(_time0);

        final GribProdGenerator gribProdGenerator = new GribProdGenerator(this);

        if(_forecasterInit != null)
        {
            if(_logger.getPrintDebugInfo() > 0)
            {
                _logger.log(Logger.DEBUG, "The forecaster initials is " + _forecasterInit);
            }
        }
        _ffhProductGeneratorParameter.set_forecasterInit(_forecasterInit);

        if(((_inputProdId == null) && (_groupId == null)) || ((_inputProdId != null) && (_groupId != null)))
        {
            _logger.log(Logger.ERROR, "Either productId or groupId in run_info.xml is mandatory.");
            throw new Exception("Error:" + getClass().getName()
                + "-Either productId or groupId in run_info.xml is mandatory.");
        }

        if(_inputProdId != null)
        {
            _logger.log(Logger.INFO, "The product ID is " + _inputProdId);
            _ffhProductGeneratorParameter.set_inputProdId(_inputProdId);
            _singleProdId = _inputProdId;

            //extract the data from params.xml file for the specified inputProdId
            _ffhProductGeneratorParameter.extractValueForSpecificProdId(this);

            //skip this invalid product is return as null from _ffhProductGeneratorParameter
            if(_ffhProductGeneratorParameter.get_inputProdId() == null)
            {
                return;
            }

            //create GRIB product
            if(_ffhProductGeneratorParameter.get_processProd()
                                            .get_prodFormat()
                                            .equals(TextProductGeneratorConstants.GRIBPRODUCT))
            {
                gribProdGenerator.createGribProduct();

            }
            else
            //create SHEF product
            {
                //create and output product into text file
                createSingleOutputTextFile();
            }
        }
        else if(_groupId != null)
        {
            _logger.log(Logger.INFO, "The group ID is " + _groupId);

            // get block for group product groupId
            _groupProdValue = _ffhProductGeneratorParameter.getStringDataParameter(TextProductGeneratorConstants.GROUP_PRODUCT_PRODGEN_TAG,
                                                                                   _groupId);
            if(_logger.getPrintDebugInfo() > 0)
            {
                _logger.log(Logger.DEBUG, "The content of the groupId:" + _groupProdValue);
            }

            _ffhProductGeneratorParameter.set_groupProdValue(_groupProdValue);
            _ffhProductGeneratorParameter.processGroupProdValue();

            _num_of_products = _ffhProductGeneratorParameter.get_groupProdId().size();
            final String grpPrdOutputFile = _ffhProductGeneratorParameter.getStringDataParameter(TextProductGeneratorConstants.GROUP_PRODUCT_PRODGEN_TAG,
                                                                                                 TextProductGeneratorConstants.GROUP_PRODUCT_OUTPUT_FILE_TAG);

            if(grpPrdOutputFile.equalsIgnoreCase(TextProductGeneratorConstants.ONE_PRODUCT_PER_FILE_TAG))
            {
                _onePrdPerFileFlag = true;
            }
            else
            {
                _onePrdPerFileFlag = false;
            }

            if(!_onePrdPerFileFlag)
            {
                createGrpOutputTextFile();
            }
            for(int i = 0; i < _num_of_products; i++)
            {
                _ffhProductGeneratorParameter.set_inputProdId("PRODUCT_"
                    + _ffhProductGeneratorParameter.get_groupProdId().get(i));
                _singleProdId = "PRODUCT_" + _ffhProductGeneratorParameter.get_groupProdId().get(i);

                //extract the data from params.xml file for the specified inputProdId
                _ffhProductGeneratorParameter.extractValueForSpecificProdId(this);

                //skip this invalid product if return as null from _ffhProductGeneratorParameter
                if(_ffhProductGeneratorParameter.get_inputProdId() == null)
                {
                    continue;
                }

                if(_ffhProductGeneratorParameter.get_processProd()
                                                .get_prodFormat()
                                                .equals(TextProductGeneratorConstants.GRIBPRODUCT))
                {
                    gribProdGenerator.createGribProduct();

                }
                else
                {
                    if(_onePrdPerFileFlag)
                    {
                        //create and output each product into individual text file
                        createSingleOutputTextFile();
                    }
                    else
                    {
                        //create and output a group of products into one text file
                        outputGrpTextFile();
                    }
                }
            }

            if(!_onePrdPerFileFlag)
            {
                closeGrpOutputTextFile();
            }
        }
    }

    public TextPrdGeneratorParameter get_ffhProductGeneratorParameter()
    {
        return _ffhProductGeneratorParameter;
    }

    public String get_inputProdId()
    {
        return _inputProdId;
    }

    public String get_singleProdId()
    {
        return _singleProdId;
    }

    /*
     * public long get_lastObservationDateTime() { return _lastObservationDateTime; }
     */

    public long get_time0()
    {
        return _time0;
    }

    /**
     * return the output directory
     */
    public String getOutputDir()
    {
        return _outputDir;
    }

    /**
     * write group of products into text file
     * 
     * @throws IOException
     */
    public void outputGrpTextFile() throws IOException
    {
        if(_outFile != null)
        {
            //write product contents into text file _outputFile
            for(final String sb: _ffhProductGeneratorParameter.get_prodInfoById())
            {
                _outFile.write(sb);
            }
            //add new lines between the product
            _outFile.write("\n\n");
        }

    }

    //Besides validation, also retrieve individual input time series from _tsList.
    @Override
    protected void runDriverValidation() throws Exception
    {
        super.runDriverValidation();
        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        while(ite.hasNext())
        {
            final RegularTimeSeries inputRts = ite.next();
            if(!inputRts.getTimeSeriesType().equals(FFHConstants.FFH_DATATYPE))
            {
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "The input time series type" + inputRts.getTimeSeriesType()
                        + " is not FFH. So it is ignored.");
                }
                ite.remove();
            }

        }
    }

    /*
     * public void set_lastObservationDateTime(final long observationDateTime) { _lastObservationDateTime =
     * observationDateTime; }
     */

    public void set_inputProdId(final String prodId)
    {
        _inputProdId = prodId;
    }

    public void set_singleProdId(final String prodId)
    {
        _singleProdId = prodId;
    }

    public void set_time0(final long _time0)
    {
        this._time0 = _time0;
    }

}
