package ohd.hseb.ohdutilities.prodgen;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.TimeZone;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdutilities.ffg.ffh.FFHConstants;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.ohdutilities.UtilityParameters;

/**
 * Process parameters defined in params.xml file
 * 
 * @author dengj
 */
public class TextPrdGeneratorParameter extends UtilityParameters
{
    //variables
    private String _prodValue;
    private String _inputProdId = null;
    private String _groupProdValue = null;
    private final List<String> _groupProdId = new ArrayList<String>();
    private TextProductGeneratorProcessProd _processProd = null;

    private String[] _locationIdInTable;
    private String[] _descriptionInTable;
    private String[] _streamNameInTable;
    private String[] _forecasterIntTable;
    private String[] _forecasterNameTable;
    private String[] _zoneIdInTable;
    private String[] _zoneNameInTable;
    private List<String> _prodInfoById = null;
    private long _lastObservationDateTime;
    private String _forecasterInit;
    private String _forecasterInfo;
    ProductGeneratorDriver _driver;
    private String _physicalElement;
    private String _yearFormat;

    /**
     * constructor
     */
    public TextPrdGeneratorParameter()
    {
        super();
    }

    /**
     * For specified product Id, retrieve and process the data fields.
     * 
     * @param driver
     * @throws Exception
     */
    public void extractValueForSpecificProdId(final ProductGeneratorDriver driver) throws Exception
    {
        _driver = driver;
        String firstLine = null;
        String secondLine = null;
        String thirdLine = null;
        String commsStr = null;
        String textStr = null;
        String hffgStr = null;
        String forecasterNameStr = null;
        String officeNameStr = null;
        boolean commentFlag=false;

        _prodInfoById = new ArrayList<String>();

        try
        {
            //get block for product inputProdId
            _prodValue = getStringDataParameter(TextProductGeneratorConstants.PROD_PRODGEN_TAG, _inputProdId);

            if(_logger.getPrintDebugInfo() > 0)
            {
                _logger.log(Logger.DEBUG, "The content of the product:" + _prodValue);
            }

            //process the block for product and return with _processProd
            _processProd = new TextProductGeneratorProcessProd(_prodValue);
            _processProd.setLogger(_logger);
            _processProd.processProd();

            //retrieve location table contents from params.xml
            retrieve_headwater_stream();

            //retrieve zone table from params.xml
            retrieveZoneTable();

            //retrieve forecast duty flag and forecast table data
            retrieveForecastInfo();

            //first line of FFH/FFG product
            firstLine = "ZCZC " + _processProd.get_prodId() + " " + _processProd.get_prodCircuits() + "\n";
            _prodInfoById.add(firstLine);

            //second line of FFH/FFG product
            secondLine = _processProd.get_prodWmo() + " " + String.format("%-4s", _processProd.get_prodOffice())
            + " DDHHMM\n";
            _prodInfoById.add(secondLine);

            //AWIPS or AFOS header
            commsStr = super.getStringDataParameter(TextProductGeneratorConstants.FORECASTER_PRODGEN_GROUP_TAG,
                                                    TextProductGeneratorConstants.COMMS_HEADER_END_TAG);

            if(commsStr.equalsIgnoreCase(TextProductGeneratorConstants.COMMS_AWIPS_TAG))
            {
                thirdLine = _inputProdId.substring(8) + "\n";
                _prodInfoById.add(thirdLine);
            }

            //Physical elements PP or PF
            _physicalElement = super.getStringDataParameter(TextProductGeneratorConstants.FORECASTER_PRODGEN_GROUP_TAG,
                                                            TextProductGeneratorConstants.PHYSICAL_ELEMENTS_TAG);

            //year format e.g. 2010 or just 10
            _yearFormat = super.getStringDataParameter(TextProductGeneratorConstants.FORECASTER_PRODGEN_GROUP_TAG,
                                                       TextProductGeneratorConstants.YEAR_DATE_FORMAT_TAG);

            //the rest of lines of FFH/FFG product are retrieved from arrayList _prodInfoById
            for(int i = 0; i < _processProd.get_prodIdValuePair().size(); i++)
            {
                //if the value is "TEXT"
                if(_processProd.get_prodIdValuePair().get(i).get_prodValue().equals("TEXT"))
                {
                    //currently do not handle water supply text such as WSUPDESC, sending warning message
                    if(_processProd.get_prodIdValuePair().get(i).get_prodId().startsWith("WSUP"))
                    {
                        _logger.log(Logger.WARNING, "Currently do not output water supply information.");
                    }
                    else if(_processProd.get_prodIdValuePair().get(i).get_prodId().equalsIgnoreCase("comment"))
                    {
                        commentFlag = true;
                    }
                    else
                    {
                        textStr = processText(_processProd.get_prodIdValuePair().get(i).get_prodId());
                        if(textStr.length() > 0)
                        {
                            _prodInfoById.add(textStr);
                        }
                    }

                }

                //if the value is "hffg" for FFH location id, "affg" for FFG zone id
                else if(_processProd.get_prodIdValuePair().get(i).get_prodValue().equalsIgnoreCase("hffg")
                || _processProd.get_prodIdValuePair().get(i).get_prodValue().equalsIgnoreCase("affg")
                )
                {
                    hffgStr = processHffg(_processProd.get_prodIdValuePair().get(i).get_prodId());
                    if (commentFlag == true)
                    {
                        //handle text string with COMMENT label.  NERFC is often created this scenario
                        hffgStr = ":"+hffgStr;
                        commentFlag = false;
                    }

                    _prodInfoById.add(hffgStr);
                }


            }

            // add .E and $$
            _prodInfoById.add(".END\n");

            if(commsStr.equalsIgnoreCase(TextProductGeneratorConstants.COMMS_AFOS_TAG))
            {
                _prodInfoById.add("$$\n");
            }

            // add duty forecaster is needed
            if(_forecasterInit != null)
            {
                for(int i = 0; i < _forecasterIntTable.length; i++)
                {
                    if((_forecasterIntTable[i] != null) && _forecasterIntTable[i].equalsIgnoreCase(_forecasterInit))
                    {
                        forecasterNameStr = "\nDUTY FORECASTER:  " + _forecasterNameTable[i] + "\n";
                    }
                    else if((_forecasterIntTable[i] != null) && _forecasterIntTable[i].equalsIgnoreCase("office"))
                    {
                        officeNameStr = _forecasterNameTable[i];
                    }
                }
            }

            if((_forecasterInfo != null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.FORECASTER_NOT_INCLUDE_TAG))
            {
                if(_logger.getPrintDebugInfo() > 0)
                {
                    _logger.log(Logger.DEBUG, "The FORCASTER DUTY FLAG is set as not include in params.xml.");
                }
            }
            else if((_forecasterInfo != null) && (officeNameStr != null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.INCLUDE_OFFICE_TAG))
            {
                _prodInfoById.add("\n" + officeNameStr + "\n");
            }
            else if((forecasterNameStr != null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.INCLUDE_FORECASTER_TAG))
            {
                _prodInfoById.add(forecasterNameStr);
            }
            else if((forecasterNameStr != null) && (officeNameStr == null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.INCLUDE_OFFICE_FORECASTER_TAG))
            {
                _prodInfoById.add(forecasterNameStr);
            }
            else if((officeNameStr != null) && (forecasterNameStr != null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.INCLUDE_OFFICE_FORECASTER_TAG))
            {
                _prodInfoById.add("\n" + officeNameStr);
                _prodInfoById.add(forecasterNameStr);

            }
            else if((forecasterNameStr == null) && (officeNameStr != null)
            && _forecasterInfo.equalsIgnoreCase(TextProductGeneratorConstants.INCLUDE_OFFICE_FORECASTER_TAG))
            {
                _prodInfoById.add("\n" + officeNameStr + "\n");
            }
            else
            {
                if(_logger.getPrintDebugInfo() > 0)
                {
                    _logger.log(Logger.DEBUG, "No forecaster/office info is appended.");
                }
            }

            //add NNNN based on comms definition
            if((commsStr != null) && commsStr.equalsIgnoreCase(TextProductGeneratorConstants.COMMS_AFOS_TAG))
            {
                _prodInfoById.add("NNNN");
            }
            else
            {
                _prodInfoById.add("\n$$\n");
            }

        }
        catch(final NullPointerException e)
        {
            _logger.log(Logger.WARNING, getClass().getName() +
                        "- Catch nullPointerException. Either product or product contents are not valid for product "
                        + _inputProdId);
            //set _inputProdId as null;
            _inputProdId = null;

            return;

        }
        catch(final Exception e)
        {
            _logger.log(Logger.WARNING, getClass().getName() +
                        "- Catch Exception. Either product or product contents are not valid for product "
                        + _inputProdId);
            _inputProdId = null;
            return;
        }
    }

    /**
     * overide the extractValuesFromMap()
     */
    @Override
    protected void extractValuesFromMap()
    {

    }

    public String[] get_descriptionInTable()
    {
        return _descriptionInTable;
    }

    public String get_forecasterInit()
    {
        return _forecasterInit;
    }

    public List<String> get_groupProdId()
    {
        return _groupProdId;
    }

    public String get_groupProdValue()
    {
        return _groupProdValue;
    }

    public String get_inputProdId()
    {
        return _inputProdId;
    }

    public long get_lastObservationDateTime()
    {
        return _lastObservationDateTime;
    }

    public String[] get_locationIdInTable()
    {
        return _locationIdInTable;
    }

    public TextProductGeneratorProcessProd get_processProd()
    {
        return _processProd;
    }

    public List<String> get_prodInfoById()
    {
        return _prodInfoById;
    }

    public String get_prodValue()
    {
        return _prodValue;
    }

    public String[] get_streamNameInTable()
    {
        return _streamNameInTable;
    }

    public String getLidHnameSnameProdgen(final String tableGroupId) throws Exception
    {
        return super.getStringDataParameter(tableGroupId, TextProductGeneratorConstants.LID_HNAME_SNAME_PRODGEN_TAG);
    }

    public String getLidHnameSnameTable(final String tableGroupId) throws Exception
    {
        return super.getStringDataParameter(tableGroupId, TextProductGeneratorConstants.LID_HNAME_SNAME_TABLE_TAG);
    }

    public String getProdProdgen(final String prodGroupId) throws Exception
    {
        return super.getStringDataParameter(prodGroupId, TextProductGeneratorConstants.PROD_PRODGEN_TAG);
    }

    public String getTableHname(final String tableGroupId) throws Exception
    {
        return super.getStringDataParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_HEADWATER_NAME_TAG);
    }

    public String getTableLid(final String tableGroupId) throws Exception
    {
        return super.getStringDataParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_LOCATION_ID_TAG);
    }

    public String getTableSname(final String tableGroupId) throws Exception
    {
        return super.getStringDataParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_STREAM_NAME_TAG);
    }

    /*
     * set/get methods
     */
    public String getTextProdgen(final String textGroupId) throws Exception
    {
        return super.getStringDataParameter(textGroupId, TextProductGeneratorConstants.TEXT_PRODGEN_TAG);
    }

    /**
     * Define the issue date time format
     * 
     * @return
     */
    public String makeIssuedDateTime()
    {
        String returnIssuedDateTime = "";
        TimeZone tz;

        /*get the timezone config data from xxrfc.ffg.products file
        e.g. C for central, E for Eastern */

        String tzStr= _processProd.get_prodTz();

        /* get the current local sytem time */

        Calendar cal = Calendar.getInstance();
        Date now = cal.getTime();

        /* The central timezone can be CDT during daylight saving time,
         * and CST during standard time zone based on the location US/Centeral
         */
        if (tzStr.equals("C"))
        {
            tz = TimeZone.getTimeZone("US/Central");
        }
        else if (tzStr.equals("E"))
        {
            tz = TimeZone.getTimeZone("US/Eastern");
        }
        else if (tzStr.equals("P"))
        {
            tz = TimeZone.getTimeZone("US/Pacific");
        }
        else if (tzStr.equals("M"))
        {
            tz = TimeZone.getTimeZone("US/Mountain");
        }
        else if (tzStr.equals("A"))
        {
            tz = TimeZone.getTimeZone("US/Alaska");
        }
        else if (tzStr.equals("H"))
        {
            tz = TimeZone.getTimeZone("US/Hawaii");
        }
        else if (tzStr.equals("N"))
        {
            tz = TimeZone.getTimeZone("America/Nome");
        }

        else
        {
            /* if tzStr is not identified in TimeZone id, then
             * will use GMT timezone
             */
            tz = TimeZone.getTimeZone(tzStr);
        }

        final SimpleDateFormat dateFormat = new SimpleDateFormat("hmm a z EEE MMM d yyyy");

        dateFormat.setTimeZone(tz);
        returnIssuedDateTime = dateFormat.format(now);

        return returnIssuedDateTime;
    }

    /**
     * process group of products
     */
    public void processGroupProdValue()
    {
        String[] textSegments;
        textSegments = _groupProdValue.split(" ");
        for(int i = 0; i < textSegments.length; i++)
        {
            if(!textSegments[i].trim().equalsIgnoreCase("") && !textSegments[i].equalsIgnoreCase("ENDID") && (i > 1))
            {
                _groupProdId.add(textSegments[i]);
            }
        }
    }

    /**
     * process Hffg marker for each location in specified product id.
     * 
     * @param locationId
     * @return
     */
    public String processHffg(final String locationId)
    {
        String locationMessage = "";
        String locationStr = "";
        String precipStr = "";
        String headwaterStr = "";
        String streamStr = "";
        Double oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
        Double threeHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
        Double sixHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
        Double twelveHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
        Double twentyfourHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
        int headwaterStrFlag = 0;
        int streamNameFlag = 0;
        double missingVal = -999.0;

        final Iterator<RegularTimeSeries> ite = _driver.getTsList().iterator();

        while(ite.hasNext())
        {
            String productType = null;

            if(_processProd.get_prodName().contains(FFHConstants.FFH_DATATYPE))
            {
                productType = FFHConstants.FFH_DATATYPE;
            }
            else if(_processProd.get_prodName().contains(DataType.FFG_DATATYPE))
            {
                productType = DataType.FFG_DATATYPE;
            }
            else if(_processProd.get_prodName().contains("BASINS")) //ABRFC has PRODUCT_BASINS for affg
            {
                productType = DataType.FFG_DATATYPE;
            }

            final RegularTimeSeries inputRts = ite.next();
            missingVal = inputRts.getMissingValue();

            if(inputRts.getTimeSeriesType().equals(productType) && inputRts.getLocationId().equals(locationId))
            {

//                if((inputRts.getIntervalInHours() == Integer.parseInt(TextProductGeneratorConstants.PROD_LONGEST_DUR1))
//                    || (inputRts.getIntervalInMillis() == Double.parseDouble(TextProductGeneratorConstants.PROD_MINUTES_DUR1)))
//                {
//                    oneHrPrecip = inputRts.getMeasurementValueByIndex(0, inputRts.getMeasuringUnit());
//                }
//                else if((inputRts.getIntervalInHours() == Integer.parseInt(TextProductGeneratorConstants.PROD_LONGEST_DUR3))
//                    || (inputRts.getIntervalInMillis() == Double.parseDouble(TextProductGeneratorConstants.PROD_MINUTES_DUR3)))
//                {
//                    threeHrPrecip = inputRts.getMeasurementValueByIndex(0, inputRts.getMeasuringUnit());
//                }
//                else if((inputRts.getIntervalInHours() == Integer.parseInt(TextProductGeneratorConstants.PROD_LONGEST_DUR6))
//                    || (inputRts.getIntervalInMillis() == Double.parseDouble(TextProductGeneratorConstants.PROD_MINUTES_DUR6)))
//                {
//                    sixHrPrecip = inputRts.getMeasurementValueByIndex(0, inputRts.getMeasuringUnit());
//                }
//                else if((inputRts.getIntervalInHours() == Integer.parseInt(TextProductGeneratorConstants.PROD_LONGEST_DUR12))
//                    || (inputRts.getIntervalInMillis() == Double.parseDouble(TextProductGeneratorConstants.PROD_MINUTES_DUR12)))
//                {
//                    twelveHrPrecip = inputRts.getMeasurementValueByIndex(0, inputRts.getMeasuringUnit());
//                }
//                else if((inputRts.getIntervalInHours() == Integer.parseInt(TextProductGeneratorConstants.PROD_LONGEST_DUR24))
//                    || (inputRts.getIntervalInMillis() == Double.parseDouble(TextProductGeneratorConstants.PROD_MINUTES_DUR24)))
//                {
//                    twentyfourHrPrecip = inputRts.getMeasurementValueByIndex(0, inputRts.getMeasuringUnit());
//                }

                final long time = _driver.get_time0();
                oneHrPrecip = inputRts.getMeasurementValueByTimeNoException(time + 1 * 3600 * 1000,
                                                                            MeasuringUnit.inches);
                threeHrPrecip = inputRts.getMeasurementValueByTimeNoException((time + 3 * 3600 * 1000),
                                                                              MeasuringUnit.inches);
                sixHrPrecip = inputRts.getMeasurementValueByTimeNoException((time + 6 * 3600 * 1000),
                                                                            MeasuringUnit.inches);
                twelveHrPrecip = inputRts.getMeasurementValueByTimeNoException((time + 12 * 3600 * 1000),
                                                                               MeasuringUnit.inches);
                twentyfourHrPrecip = inputRts.getMeasurementValueByTimeNoException((time + 24 * 3600 * 1000),
                                                                                   MeasuringUnit.inches);
            }
            else
            {

            }
        }

        locationStr = locationId;

        if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR1))
        {
            if((oneHrPrecip == missingVal) || (oneHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "One hour FFG(or FFH) is missing for " + locationId);
            }
            precipStr = String.format("%5.1f", oneHrPrecip);
        }
        else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR3))
        {
            if((oneHrPrecip == missingVal) || (oneHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "One hour FFG(or FFH) is missing for " + locationId);
            }
            if((threeHrPrecip == missingVal) || (threeHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                threeHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Three hour FFG(or FFH) is missing for " + locationId);
            }
            precipStr = String.format("%5.1f", oneHrPrecip) + "/" + String.format("%5.1f", threeHrPrecip);
        }
        else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR6))
        {

            if((oneHrPrecip == missingVal) || (oneHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "One hour FFG(or FFH) is missing for " + locationId);
            }
            if((threeHrPrecip == missingVal) || (threeHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                threeHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Three hour FFG(or FFH) is missing for " + locationId);
            }
            if((sixHrPrecip == missingVal) || (sixHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                sixHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Six hour FFG(or FFH) is missing for " + locationId);
            }

            precipStr = String.format("%5.1f", oneHrPrecip) + "/" + String.format("%5.1f", threeHrPrecip) + "/"
            + String.format("%5.1f", sixHrPrecip);
        }
        else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR12))
        {
            if((oneHrPrecip == missingVal) || (oneHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "One hour FFG(or FFH) is missing for " + locationId);
            }
            if((threeHrPrecip == missingVal) || (threeHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                threeHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Three hour FFG(or FFH) is missing for " + locationId);
            }
            if((sixHrPrecip == missingVal) || (sixHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                sixHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Six hour FFG(or FFH) is missing for " + locationId);
            }
            if((twelveHrPrecip == missingVal) || (twelveHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                twelveHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Twelve hour FFG(or FFH) is missing for " + locationId);
            }

            precipStr = String.format("%5.1f", oneHrPrecip) + "/" + String.format("%5.1f", threeHrPrecip) + "/"
            + String.format("%5.1f", sixHrPrecip) + "/" + String.format("%5.1f", twelveHrPrecip);
        }
        else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR24))
        {
            if((oneHrPrecip == missingVal) || (oneHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                oneHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "One hour FFG(or FFH) is missing for " + locationId);
            }
            if((threeHrPrecip == missingVal) || (threeHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                threeHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Three hour FFG(or FFH) is missing for " + locationId);
            }
            if((sixHrPrecip == missingVal) || (sixHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                sixHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Six hour FFG(or FFH) is missing for " + locationId);
            }
            if((twelveHrPrecip == missingVal) || (twelveHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                twelveHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Twelve hour FFG(or FFH) is missing for " + locationId);
            }
            if((twentyfourHrPrecip == missingVal)
            || (twentyfourHrPrecip == TextProductGeneratorConstants.MISSING_VALUE))
            {
                twentyfourHrPrecip = TextProductGeneratorConstants.MISSING_VALUE;
                _logger.log(Logger.WARNING, "Twentyfour hour FFG(or FFH) is missing for " + locationId);
            }
            precipStr = String.format("%5.1f", oneHrPrecip) + "/" + String.format("%5.1f", threeHrPrecip) + "/"
            + String.format("%5.1f", sixHrPrecip) + "/" + String.format("%5.1f", twelveHrPrecip) + "/"
            + String.format("%5.1f", twentyfourHrPrecip);

        }

        //for FFH product, output headwater if flag is set as true
        if(_processProd.get_prodDescriptionFlag().equals("1")
        && _processProd.get_prodName().contains(FFHConstants.FFH_DATATYPE))
        {
            headwaterStrFlag = 1;
            for(int i = 0; i < _locationIdInTable.length; i++)
            {
                if(_locationIdInTable[i].equalsIgnoreCase(locationStr))
                {
                    headwaterStr = ":" + _descriptionInTable[i];
                    break;
                }
            }
        }

        //for FFG product, output zone name if flag is set as true
        if(_processProd.get_prodDescriptionFlag().equals("1")
        && _processProd.get_prodName().contains(DataType.FFG_DATATYPE))
        {
            headwaterStrFlag = 1;
            for(int i = 0; i < _zoneIdInTable.length; i++)
            {
                if(_zoneIdInTable[i].equalsIgnoreCase(locationStr))
                {
                    headwaterStr = ":" + _zoneNameInTable[i];
                    break;
                }
            }
        }

        //output stream name if flag is set as true
        if(_processProd.get_prodStreamNameFlag().equals("1"))
        {
            streamNameFlag = 1;
            for(int i = 0; i < _locationIdInTable.length; i++)
            {
                if(_locationIdInTable[i].equalsIgnoreCase(locationStr))
                {
                    streamStr = _streamNameInTable[i];
                    break;
                }
            }
        }

        if((headwaterStrFlag == 0) && (streamNameFlag == 0))
        {
            locationMessage = String.format("%-8s", locationStr) + precipStr + "\n";
        }
        else if(headwaterStrFlag == 0)
        {
            locationMessage = String.format("%-8s", locationStr) + precipStr + "   " + streamStr + "\n";
        }
        else if(streamNameFlag == 0)
        {
            locationMessage = String.format("%-8s", locationStr) + precipStr + " "
            + String.format("%-20s", headwaterStr) + "\n";
        }
        else
        {
            locationMessage = String.format("%-8s", locationStr) + precipStr + " "
            + String.format("%-20s", headwaterStr) + "   " + streamStr + "\n";
        }

        return locationMessage;
    }

    /**
     * Process text marker defined in specified product id
     * 
     * @param textId
     * @return
     */
    public String processText(final String textId)
    {
        String returnTextMessage = "";
        String strTag = "";
        String textStr = "";
        String[] textSegments;
        String issueOffice = "";
        String dotbStr = "";
        String issuedStr = "";
        String lablStr1 = "";
        String lablStr2 = "";
        String lablHeadStr = "";
        String skipLineStr = "";
        String regularTextStr = "";
        String lastObservationDate = "";
        String lastObservationHr = "";
        String nowUGCStr = "";
        String ppStr = "";

        strTag = "TEXT_" + textId.toUpperCase();

        try
        {
            textStr = super.getStringDataParameter(TextProductGeneratorConstants.TEXT_PRODGEN_TAG, strTag);

        }
        catch(final Exception e)
        {
            return "";

        }

        //given string textStr will be split by delimiter provided.
        textSegments = textStr.split("\n");

        for(int i = 0; i < textSegments.length; i++)
        {
            if(textSegments[i].toUpperCase().contains(TextProductGeneratorConstants.TEXT_DOTB_TAG))
            {
                issueOffice = textSegments[i].substring(5, textSegments[i].trim().length());
                final Date lastObs = new Date();
                final Date nowUGC = new Date();
                lastObs.setTime(_lastObservationDateTime);

                if(_yearFormat.equals(TextProductGeneratorConstants.YYYY_DATE_FORMAT_TAG))
                {
                    final DateFormat dateFormat1 = new SimpleDateFormat("yyyyMMddHH");
                    dateFormat1.setTimeZone(OHDConstants.GMT_TIMEZONE);
                    lastObservationDate = dateFormat1.format(lastObs).substring(0, 8);
                    lastObservationHr = dateFormat1.format(lastObs).substring(8, 10);
                }
                else
                {
                    final DateFormat dateFormat1 = new SimpleDateFormat("yyMMddHH");
                    dateFormat1.setTimeZone(OHDConstants.GMT_TIMEZONE);
                    lastObservationDate = dateFormat1.format(lastObs).substring(0, 6);
                    lastObservationHr = dateFormat1.format(lastObs).substring(6, 8);
                }

                if(_yearFormat.equals(TextProductGeneratorConstants.YYYY_DATE_FORMAT_TAG))
                {
                    final DateFormat dateFormat2 = new SimpleDateFormat("yyyyMMddHHmm");
                    dateFormat2.setTimeZone(OHDConstants.GMT_TIMEZONE);
                    nowUGCStr = dateFormat2.format(nowUGC);
                }
                else
                {
                    final DateFormat dateFormat2 = new SimpleDateFormat("yyMMddHHmm");
                    dateFormat2.setTimeZone(OHDConstants.GMT_TIMEZONE);
                    nowUGCStr = dateFormat2.format(nowUGC);
                }

                if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR6))
                {
                    if(_physicalElement.equals(TextProductGeneratorConstants.PF_PHYSICAL_ELEMENTS_TAG))
                    {
                        ppStr = TextProductGeneratorConstants.SIXHR_PF_TAG;
                    }
                    else
                    {
                        ppStr = TextProductGeneratorConstants.SIXHR_PP_TAG;
                    }
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR12))
                {
                    if(_physicalElement.equals(TextProductGeneratorConstants.PF_PHYSICAL_ELEMENTS_TAG))
                    {
                        ppStr = TextProductGeneratorConstants.TWELVEHR_PF_TAG;
                    }
                    else
                    {
                        ppStr = TextProductGeneratorConstants.TWELVEHR_PP_TAG;
                    }
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR24))
                {
                    if(_physicalElement.equals(TextProductGeneratorConstants.PF_PHYSICAL_ELEMENTS_TAG))
                    {
                        ppStr = TextProductGeneratorConstants.TWENTYFOUR_PF_TAG;
                    }
                    else
                    {
                        ppStr = TextProductGeneratorConstants.TWENTYFOUR_PP_TAG;
                    }
                }
                //at least six hr PP is required
                else
                {
                    if(_physicalElement.equals(TextProductGeneratorConstants.PF_PHYSICAL_ELEMENTS_TAG))
                    {
                        ppStr = TextProductGeneratorConstants.SIXHR_PF_TAG;
                    }
                    else
                    {
                        ppStr = TextProductGeneratorConstants.SIXHR_PP_TAG;
                    }
                }

                dotbStr = ".B " + issueOffice + " " + lastObservationDate + " Z " + "DH" + lastObservationHr + "/DC"
                + nowUGCStr + " /DUE" + ppStr;

                returnTextMessage = returnTextMessage + dotbStr + "\n";

            }
            else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_ISSUED_TAG))
            {
                issuedStr = makeIssuedDateTime();

                returnTextMessage = returnTextMessage + issuedStr.toUpperCase() + "\n";
            }
            else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_LABL_TAG))
            {
                //output title  ":IDENT    1HR   3HR   6HR  12HR  24HR  HEADWATER NAME        STREAM\n";
                //output symbol ":======= ====  ====  ====  ====  ====  ====================  ==================\n";

                if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR1))
                {
                    lablStr1 = TextProductGeneratorConstants.ONEHR_LABLE_TITLE;
                    lablStr2 = TextProductGeneratorConstants.ONEHR_LABLE_SYMBOL;
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR3))
                {
                    lablStr1 = TextProductGeneratorConstants.THREEHR_LABLE_TITLE;
                    lablStr2 = TextProductGeneratorConstants.THREEHR_LABLE_SYMBOL;
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR6))
                {
                    lablStr1 = TextProductGeneratorConstants.SIXHR_LABLE_TITLE;
                    lablStr2 = TextProductGeneratorConstants.SIXHR_LABLE_SYMBOL;
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR12))
                {
                    lablStr1 = TextProductGeneratorConstants.TWELVEHR_LABLE_TITLE;
                    lablStr2 = TextProductGeneratorConstants.TWELVEHR_LABLE_SYMBOL;
                }
                else if(_processProd.get_prodLongestDur().equals(TextProductGeneratorConstants.PROD_LONGEST_DUR24))
                {
                    lablStr1 = TextProductGeneratorConstants.TWENTYFOUR_LABLE_TITLE;
                    lablStr2 = TextProductGeneratorConstants.TWENTYFOURHR_LABLE_SYMBOL;
                }
                else
                {
                    if(_logger.getPrintDebugInfo() > 0)
                    {
                        _logger.log(Logger.DEBUG, "The longest duration value %s" + _processProd.get_prodLongestDur()
                                    + "is not valid.");
                    }
                }

                //flags for headwater/zone and stream fields are false
                if(!_processProd.get_prodDescriptionFlag().equals("1")
                && (!_processProd.get_prodStreamNameFlag().equals("1")))
                {
                    returnTextMessage = returnTextMessage + lablStr1 + "\n" + lablStr2 + "\n";
                }
                // flag for headwater is false
                else if(!_processProd.get_prodDescriptionFlag().equals("1"))
                {
                    returnTextMessage = returnTextMessage + lablStr1 + "  "
                    + TextProductGeneratorConstants.STREAM_LABLE_TITLE + "\n" + lablStr2 + "  "
                    + TextProductGeneratorConstants.STREAM_LABLE_SYMBOL;
                }
                // flag for stream is false
                else if(!_processProd.get_prodStreamNameFlag().equals("1"))
                {
                    if(_processProd.get_prodName().contains(FFHConstants.FFH_DATATYPE))
                    {
                        returnTextMessage = returnTextMessage + lablStr1 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_TITLE + "\n" + lablStr2 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_SYMBOL;
                    }
                    else if(_processProd.get_prodName().contains(DataType.FFG_DATATYPE))
                    {
                        returnTextMessage = returnTextMessage + lablStr1 + "  "
                        + TextProductGeneratorConstants.ZONE_LABLE_TITLE + "\n" + lablStr2 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_SYMBOL;
                    }
                }
                else
                {
                    if(_processProd.get_prodName().contains(FFHConstants.FFH_DATATYPE))
                    {
                        returnTextMessage = returnTextMessage + lablStr1 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_TITLE + "        "
                        + TextProductGeneratorConstants.STREAM_LABLE_TITLE + "\n" + lablStr2 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_SYMBOL + "  "
                        + TextProductGeneratorConstants.STREAM_LABLE_SYMBOL;
                    }
                    else if(_processProd.get_prodName().contains(DataType.FFG_DATATYPE))
                    {
                        returnTextMessage = returnTextMessage + lablStr1 + "  "
                        + TextProductGeneratorConstants.ZONE_LABLE_TITLE + "        "
                        + TextProductGeneratorConstants.STREAM_LABLE_TITLE + "\n" + lablStr2 + "  "
                        + TextProductGeneratorConstants.HEADWATER_LABLE_SYMBOL + "  "
                        + TextProductGeneratorConstants.STREAM_LABLE_SYMBOL;
                    }
                }
                returnTextMessage = returnTextMessage + "\n";

            }
            else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_LABLHEAD_TAG))
            {
                lablHeadStr = "HEADWATER FLASH FLOOD GUIDANCE\n";
                returnTextMessage = returnTextMessage + lablHeadStr + "\n";
            }
            else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_SKIPLINE_TAG))
            {
                skipLineStr = "\n";
                returnTextMessage = returnTextMessage + skipLineStr;
            }
            /*else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_COMMENT_TAG))
            {
            	return returnTextMessage;

            }*/
            else if(textSegments[i].toUpperCase().startsWith(TextProductGeneratorConstants.TEXT_ENDTEXT_TAG))
            {
                return returnTextMessage;

            }

            else
            {

                regularTextStr = textSegments[i] + "\n";

                returnTextMessage = returnTextMessage + regularTextStr;
            }

        }

        return returnTextMessage;
    }

    /**
     * retrieve table data from LID_HNAME_SNAME_PRODGEN group, the data will include PRODGEN_LOCATION_ID,
     * PRODGEN_HEADWATER_NAME and PRODGEN_STREAM_NAME
     */
    public void retrieve_headwater_stream()
    {
        Table textProductGeneratorTable = null;

        try
        {
            textProductGeneratorTable = super.getTableDataParameter(TextProductGeneratorConstants.LID_HNAME_SNAME_PRODGEN_TAG,
                                                                    TextProductGeneratorConstants.LID_HNAME_SNAME_TABLE_TAG);
        }
        catch(final Exception e)
        {
            return;
        }

        if((textProductGeneratorTable != null) && (textProductGeneratorTable.getRow() != null))
        {
            _locationIdInTable = new String[textProductGeneratorTable.getRow().size()];
            _descriptionInTable = new String[textProductGeneratorTable.getRow().size()];
            _streamNameInTable = new String[textProductGeneratorTable.getRow().size()];

            int i = 0;
            for(final Row myRow: textProductGeneratorTable.getRow())
            {
                _locationIdInTable[i] = myRow.getA();
                _descriptionInTable[i] = myRow.getB();
                _streamNameInTable[i] = myRow.getC();
                i++;
            }
        }

    }

    /**
     * retrieve forecaster duty data from FORECASTER_DUTY_FLAG, retrieve forecaster initial and forecaster name from
     * FORECAST_TABLE
     * 
     * @param inTable
     */
    public void retrieveForecastInfo()
    {
        Table textProductForecasterTable = null;

        try
        {
            _forecasterInfo = super.getStringDataParameter(TextProductGeneratorConstants.FORECASTER_PRODGEN_GROUP_TAG,
                                                           TextProductGeneratorConstants.FORECASTER_DUTY_FLAG_TAG);

            //retrieve forecaster table contents from params.xml
            textProductForecasterTable = super.getTableDataParameter(TextProductGeneratorConstants.FORECASTER_PRODGEN_GROUP_TAG,
                                                                     TextProductGeneratorConstants.FORECASTER_TABLE_TAG);
        }
        catch(final Exception e)
        {
            return;
        }

        if((textProductForecasterTable != null) && (textProductForecasterTable.getRow() != null))
        {
            _forecasterIntTable = new String[textProductForecasterTable.getRow().size()];
            _forecasterNameTable = new String[textProductForecasterTable.getRow().size()];

            int i = 0;
            for(final Row myRow: textProductForecasterTable.getRow())
            {
                _forecasterIntTable[i] = myRow.getA();
                _forecasterNameTable[i] = myRow.getB();

                i++;
            }
        }
    }

    /**
     * retrieve table data from LID_ZONE_PRODGEN group, the data will include PRODGEN_LOC_ID and PRODGEN_ZONE_NAME
     */
    public void retrieveZoneTable()
    {
        Table textProductGeneratorZoneTable = null;

        try
        {
            textProductGeneratorZoneTable = super.getTableDataParameter(TextProductGeneratorConstants.LID_ZONE_PRODGEN_TAG,
                                                                        TextProductGeneratorConstants.LID_ZONE_TABLE_TAG);
        }
        catch(final Exception e)
        {
            return;

        }
        if((textProductGeneratorZoneTable != null) && (textProductGeneratorZoneTable.getRow() != null))
        {
            _zoneIdInTable = new String[textProductGeneratorZoneTable.getRow().size()];
            _zoneNameInTable = new String[textProductGeneratorZoneTable.getRow().size()];

            int i = 0;
            for(final Row myRow: textProductGeneratorZoneTable.getRow())
            {
                _zoneIdInTable[i] = myRow.getA();
                _zoneNameInTable[i] = myRow.getB();

                if(_zoneNameInTable[i].startsWith("'"))
                {
                    _zoneNameInTable[i] = _zoneNameInTable[i].substring(1);
                }
                if(_zoneNameInTable[i].endsWith("'"))
                {
                    _zoneNameInTable[i] = _zoneNameInTable[i].substring(0, _zoneNameInTable[i].length() - 1);
                }

                i++;
            }
        }
    }

    public void set_descriptionInTable(final String[] inTable)
    {
        _descriptionInTable = inTable;
    }

    public void set_forecasterInit(final String forecasterInit)
    {
        _forecasterInit = forecasterInit;
    }

    public void set_groupProdValue(final String groupProdValue)
    {
        _groupProdValue = groupProdValue;
    }

    public void set_inputProdId(final String prodId)
    {
        _inputProdId = prodId;
    }

    public void set_lastObservationDateTime(final long observationDateTime)
    {
        _lastObservationDateTime = observationDateTime;
    }

    public void set_locationIdInTable(final String[] idInTable)
    {
        _locationIdInTable = idInTable;
    }

    public void set_processProd(final TextProductGeneratorProcessProd prod)
    {
        _processProd = prod;
    }

    public void set_prodValue(final String value)
    {
        _prodValue = value;
    }

    public void set_streamNameInTable(final String[] nameInTable)
    {
        _streamNameInTable = nameInTable;
    }

    public void setLidHnameSnameProdgen(final String tableGroupId, final String lidHnameSname)
    {
        super.insertParameter(tableGroupId, TextProductGeneratorConstants.LID_HNAME_SNAME_PRODGEN_TAG, lidHnameSname);
    }

    public void setLidHnameSnameTable(final String tableGroupId, final String tLidHnameSname)
    {
        super.insertParameter(tableGroupId, TextProductGeneratorConstants.LID_HNAME_SNAME_TABLE_TAG, tLidHnameSname);
    }

    public void setProdProdgen(final String prodGroupId, final String prodProdgen)
    {
        super.insertParameter(prodGroupId, TextProductGeneratorConstants.PROD_PRODGEN_TAG, prodProdgen);
    }

    public void setTableHname(final String tableGroupId, final String tableHname)
    {
        super.insertParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_HEADWATER_NAME_TAG, tableHname);
    }

    public void setTableLid(final String tableGroupId, final String tableLid)
    {
        super.insertParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_LOCATION_ID_TAG, tableLid);
    }

    public void setTableSname(final String tableGroupId, final String tableSname)
    {
        super.insertParameter(tableGroupId, TextProductGeneratorConstants.PRODGEN_STREAM_NAME_TAG, tableSname);
    }

    public void setTextProdgen(final String textGroupId, final String textProdgen)
    {
        super.insertParameter(textGroupId, TextProductGeneratorConstants.TEXT_PRODGEN_TAG, textProdgen);
    }
}
