package ohd.hseb.ohdutilities.ratingCurve;

import java.io.File;

import ohd.hseb.util.CodeTimer;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsAdapterDAO;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.OHDConfigurationInfoReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.fews.ohdutilities.UtilityParameters;
import ohd.hseb.util.fews.ratingCurve.RatingCurveConstants;
import ohd.hseb.util.fews.ratingCurve.RatingCurveDAO;
import ohd.hseb.util.fews.ratingCurve.RatingCurves;
import ohd.hseb.util.io.ExceptionParser;

/**
 * This class use output Xml from FEWS and convert it to ArcGIS for Fargo ND mapping. This output data will be used as
 * input to the ArcGIS mapping project. OHD convert the information in the PI-xml file (output_profiles.xml) to the file
 * formats desired by NCRFC (text files). <br>
 * This program is going to be execute inside FEWS as part of an existing Workflow. No user interaction will be required
 * besides the call to "main" method from FEWS.<br>
 * <br>
 * NOTE: Knowledge of FEWS, XML, XML Schema and XML parsers is assume.
 * 
 * @author Freddy.Camacho
 */
public class OHDRatingCurveTool
{
    private final Logger _logger = new Diagnostics();
    private String _diagnosticsFile;

    private RunInfo _runInfo = null;
    private final UtilityParameters _parameters = new UtilityParameters();
    RatingCurves _ratingCurves = new RatingCurves();

    /* -------- for record each period time consuming ------------- */
    private final CodeTimer _stopWatch = new CodeTimer();

    public OHDRatingCurveTool()
    {
    }

    /**
     * This program was designed to be execute from a FEWS/CHPS WorkFlow. Never was intended to run in stand alone mode.
     * An exception is throw if the program fails to inform FEWS/CHPS that there was a problem. WE can not capture and
     * create a user friendly error message as it will not be reflected into FEWS/CHPS interface and user will never
     * known that the program failed.<br>
     * The main method read the arguments, it will assume the data is passed in correct order: 1 run_info.xml
     * path+filename. <br>
     * If number of arguments is incorrect the program will throw an exception and end.
     * 
     * @param args The must be 1 value: run_info.xml input data
     * @throws Exception An exception will be the way to notify fews that the program did not work properly.
     */

    public static void main(final String[] args)
    {

        final OHDRatingCurveTool ratingCurveTool = new OHDRatingCurveTool();

        ratingCurveTool._stopWatch.start();// start the stop watch

        try
        {
            ratingCurveTool.readConfigFilesAndSetValues(args);

            /** ----------------------write the rating curve xml file -------------------------- */
            ratingCurveTool.writeOuputRatingCurves();
            /** --------log timing states ------------------ */
            ratingCurveTool._stopWatch.stop();
            final StringBuffer logTime = new StringBuffer();
            logTime.append(OHDConstants.NEW_LINE)
                   .append("******************************Timing Stats*************************")
                   .append(OHDConstants.NEW_LINE);
            logTime.append("Total run time: ")
                   .append(ratingCurveTool._stopWatch.getElapsedTimeInSeconds())
                   .append(" second(s)")
                   .append(OHDConstants.NEW_LINE);
            logTime.append("***********************************************************************")
                   .append(OHDConstants.NEW_LINE);
            ratingCurveTool._logger.log(Logger.DEBUG, logTime.toString());

        }
        catch(final Exception e)
        { //TODO: need to delete e.printStackTrace after we are done.
            e.printStackTrace();
            /*
             * inserted "ERROR in CHPSRatingCurverTool: " into diag.xml so that junit error test can detect it
             */
            ratingCurveTool._logger.log(Logger.FATAL, "ERROR in CHPS Rating Curve Converter tool");

            ratingCurveTool._logger.log(Logger.FATAL, ExceptionParser.multiLineStackTrace(e));
        }
        finally
        {
            try
            {
                /** ---------------- write diagnostics out to diagnostics file --------------------- */
                FewsAdapterDAO.writeLog(ratingCurveTool._logger, ratingCurveTool._diagnosticsFile);
            }
            catch(final Exception e)
            {
                System.err.println("ERROR: The diagnostics could not be printed to the diagnostic file: "
                    + e.getMessage());
                System.err.println("The contents that were to be logged are for diagnostics:"
                    + ((Diagnostics)ratingCurveTool._logger).getListOfDiagnosticsAsString());
            }
        }
    } // close main

    /**
     * Check that the specified file exist in the system and can be accessed.
     * 
     * @param fileName
     * @return
     */
    private boolean canReadFile(final String fileName)
    {
        boolean result = false;

        final File file = new File(fileName);
        if(file.canRead())
        {
            result = true;
        }

        return result;
    }

    /**
     * Get the run info details from the args[] array. Validate that the run info is valid,read the input files and set
     * the rating curve values
     * 
     * @param args
     * @throws Exception
     */
    private synchronized void readConfigFilesAndSetValues(final String[] args) throws Exception
    {
        // read the config file containing version number of CHPS
        final OHDConfigurationInfoReader config = new OHDConfigurationInfoReader(_logger, "version/"
            + OHDConstants.OHD_FEWSADAPTER_CONFIG); //access nonsrc/version/ohdcommonchps_config.xml

        _logger.log(Logger.INFO, "OHDRatingCurveTool - Using OHD Fews Adapter version: "
            + config.getVersionControl().getVersion());

        final FewsXMLParser xmlParser = new FewsXMLParser(_logger);
        final RatingCurveController ratingCurveController = new RatingCurveController(_logger);

        if(args != null && args.length == 1)
        {
            String run_info_file_name = args[0];

            if(run_info_file_name == null)
            {
                throw new Exception("run info file name does not exist!");
            }

            run_info_file_name = run_info_file_name.trim();

            if(!canReadFile(run_info_file_name))
            {
                throw new Exception(" Cannot read from the run info file: " + run_info_file_name);
            }

            _runInfo = new RunInfo(_logger);

            xmlParser.parseRunInfoFile(run_info_file_name, _runInfo);

            _diagnosticsFile = _runInfo.getDiagnosticFile(); //since now, diag.xml is available

            final String printDebugInfo = _runInfo.getProperties().getProperty(OHDConstants.PRINT_DEBUG_INFO);

            if(printDebugInfo != null)
            {
                try
                {
                    final int debugLevel = Integer.valueOf(printDebugInfo);
                    _logger.setPrintDebugInfo(debugLevel);
                }
                catch(final Exception e)
                {
                    _logger.log(Logger.DEBUG,
                                "A problem ocurrs when reading the print debug information flag from run file "
                                    + e.getMessage());
                }
                _runInfo.logRunInfo(Logger.DEBUG);

            }
            // Validate run info is correct
            ratingCurveController.validateRunInfo(_runInfo);

            // Parse the input parameter file             
            final String parameterFileName = _runInfo.getInputParameterFile();
            xmlParser.parseParameters(parameterFileName, _parameters);

            _logger.log(Logger.DEBUG, "OHDRatingCurveTool parsed input parameter file.");

            ratingCurveController.validateRatingCurvesInput(_ratingCurves, _runInfo);

            // Read rating curve input file and set rating curve object             
            ratingCurveController.readTextFileAndSetRatingCurveInfo(_runInfo, _ratingCurves);
            //Validate output data
            ratingCurveController.validateRatingCurvesOutput(_ratingCurves, _parameters, _runInfo);

        }

        // stop to take a reading
        _stopWatch.stop();
        _logger.log(Logger.DEBUG,
                    "Time spent parsing input xml files (input, params and run_info): "
                        + _stopWatch.getElapsedTimeInSeconds() + " sec."); // timeOnePeriod
        // restart to continue reading
        _stopWatch.restart();
    }

    /**
     * Write the rating curve information to the output XML file. File names are predefined only the output directory
     * can be defined in the run_info.xml.<br>
     * The input files and the output files also are copy to the work directory.This avoid changes in the CHPS workflow
     * to log the input/output data for debugging.
     * 
     * @throws Exception
     */
    private void writeOuputRatingCurves() throws Exception
    {
        // file name is predefined for each input format.
        final StringBuffer outputFile = new StringBuffer();
        outputFile.append(_runInfo.getProperties().getProperty(RatingCurveConstants.OUTPUT_RATING_CURVE_DIR))
                  .append(File.separator);

        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(_runInfo.getProperties()
                                                              .getProperty(RatingCurveConstants.INPUT_TYPE)))
            outputFile.append(RatingCurveConstants.RATING_CURVES_FILE_NAME_RDB);
        else
            outputFile.append(RatingCurveConstants.RATING_CURVES_FILE_NAME_TXT);

        if(!_ratingCurves.getRatingCurveList().isEmpty())
        {
            RatingCurveDAO.writeRatingCurves(_ratingCurves, outputFile.toString(), _logger);

            final RatingCurveController ratingCurveController = new RatingCurveController(_logger);
            // copy input and output files to be used for support.
            // if performance is affected the two copy methods could be commented really easy.
            ratingCurveController.copyInputFilesToWorkDir(_runInfo);
            ratingCurveController.copyOutputFilesToWorkDir(_runInfo);
        }
        else
        {
            _logger.log(Logger.WARNING, "No valid RDB or TXT rating curve input files were found.");
        }
    }
}
