package ohd.hseb.ohdutilities.ratingCurve;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

import java.io.File;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.fews.ohdutilities.UtilityParameters;
import ohd.hseb.util.fews.ratingCurve.RatingCurve;
import ohd.hseb.util.fews.ratingCurve.RatingCurveConstants;
import ohd.hseb.util.fews.ratingCurve.RatingCurves;

/**
 * Business logic Only valid Rating curve objects could be save to the final list If a rating curve object is wrong it
 * would be discarded a warning log message should be write and continue with the next object. If information in the run
 * info is incorrect or missing an error log message should be write and an exception must be throw.
 * 
 * @author camachof
 */
public class RatingCurveController
{
    private final Logger _logger;

    /**
     * RatingCurveTool uses this constructor
     * 
     * @param logger Diagnostics object to log errors in
     * @throws Exception
     */
    public RatingCurveController(final Logger logger) throws Exception
    {
        _logger = logger;
    }

    /**
     * Read Rating Curve input files and generated a ratingCurve java object containing the information read from the
     * input file. There could be more than one file as input and in two different formats. RDB ad TXT formats. This
     * class support both formats and multiple files.
     * 
     * @param runInfo the run info object containing information like input directory, input type and metadatafield
     *            information
     * @param ratingCurves The rating curve objects were information will be stored.
     * @throws Exception
     */
    public void readTextFileAndSetRatingCurveInfo(final RunInfo runInfo,
                                                  final RatingCurves ratingCurves) throws Exception
    {
        final RatingCurveFileConverter ratingCurveInputFile = new RatingCurveFileConverter(_logger);
        //set the time zone from the run info as the default value. if time zone is found in input files later it will be replaced.
        ratingCurves.setTimeZone(runInfo.getTimeZone());
        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(runInfo.getProperties()
                                                             .getProperty(RatingCurveConstants.INPUT_TYPE)))
            ratingCurveInputFile.readUSGSTextFileAndSetRatingCurveInfo(runInfo, ratingCurves);
        else
            ratingCurveInputFile.readKeyValueTextFileAndSetRatingCurveInfo(runInfo, ratingCurves);
    }

    /**
     * Validate the rating curve precision is set in the run info file. If value is present it will check that the value
     * is one of the three valid values allowed. IF value is not valid. a warning message will be log and the default
     * value will be set to be used. If run info does not have rating curve precision defined then the default values
     * will be used. This method assume run info was already validated and exist.
     * 
     * @param ratingCurves
     * @param runInfo
     * @throws Exception - No exception is expected as the error should be logged and move to the next ratingCurve
     *             object.
     */
    public void validateRatingCurvesInput(final RatingCurves ratingCurves, final RunInfo runInfo) throws Exception
    {

        this.validateRatingCurvePrecisionExist(ratingCurves, runInfo);
        this.validateZeroFlowStage(ratingCurves, runInfo);
        _logger.log(Logger.DEBUG, "Rating Curve input validation passes.");
    }

    /**
     * This method perform two task instead of one to improve performance, if break down it will requited to process the
     * list of rating curves twice. <br>
     * 1: Validate the list of rating curve against business logic <br>
     * 2: Update the location Id and set the startDate value for each one of rating curve objects.<br>
     * Also, remove the rating curve objects that are not valid from the list of rating curve objects, instead of throw
     * an exception, a log messages is recorded if no valid rating curve object is found.
     * 
     * @param ratingCurves - the rating curves object were information will be saved.
     * @param parameters - the parameters information
     * @param runInfo - the run information provide by the user to process the rating curve
     * @throws Exception - No exception is expected as the error should be logged and move to the next ratingCurve
     *             object.
     */
    public void validateRatingCurvesOutput(final RatingCurves ratingCurves,
                                           final UtilityParameters parameters,
                                           final RunInfo runInfo) throws Exception
    {
        // This list contain the ratingCurve objects to be removed, as they will have wrong data.
        List<RatingCurve> wrongRatingCurveList = new ArrayList<RatingCurve>();
        for(final RatingCurve ratingCurve: ratingCurves.getRatingCurveList())
        {
            if(this.validateRatingCurvesNegativeFlowsExist(ratingCurve, wrongRatingCurveList))
                if(this.validateRatingCurvesLocationIdExist(ratingCurve, parameters, wrongRatingCurveList))
                    if(this.validateRatingCurvesStageDischargePairExist(ratingCurve, wrongRatingCurveList))
                        if(this.validateRatingCurvesDischargeUnitExist(ratingCurve, wrongRatingCurveList))
                            if(this.validateRatingCurvesInterpolationMethodExist(ratingCurve, wrongRatingCurveList))
                                if(this.validateRatingCurvesStageUnitExist(ratingCurve, wrongRatingCurveList))
                                {
                                    this.validateRatingCurvesZeroFlowStageData(ratingCurves.getComputeZeroFlowStage(),
                                                                               ratingCurve);
                                    this.setWarningMessageMinStage(ratingCurve);
                                    this.setWarningMessageNoOffSetAndZeroFlowOrdinates(ratingCurve);
                                    this.setRatingCurveStartDate(ratingCurve, runInfo);
                                    this.validateUseMaxStageInRatingCurveExist(ratingCurve, runInfo);
                                }
        }
        // Remove the rating curve object that are not valid.
        for(final RatingCurve ratingCurve: wrongRatingCurveList)
        {
            ratingCurves.getRatingCurveList().remove(ratingCurve);
        }
        wrongRatingCurveList = null;
        _logger.log(Logger.DEBUG, "Rating Curve output validation passes.");
    }

    /**
     * Validate that the information in the run info object is valid or not missing. IF the data in the runInfo object
     * is not valid a exception should be throw. This will cause the program to stop and report the error.
     * 
     * @param runInfo
     * @throws Exception
     */
    public void validateRunInfo(final RunInfo runInfo) throws Exception
    {
        // check all required elements
        this.validateParameterFileExist(runInfo);
        this.validateDiagnosticFileExist(runInfo);
        this.validateInputRatingCurverFilesExist(runInfo);
        this.validateOutputRatingCurveDirectoryExist(runInfo);
        this.validateRatingCurveTypeExist(runInfo);
        this.validateStartTimeExist(runInfo);

        //if reach here, no Exception has been thrown, the validation passes
        _logger.log(Logger.DEBUG, "Run info for Rating Curve validation passes.");
    }

    /**
     * This method check that the LocationId set in the ratingCurve object is contained in the parameter table object.
     * If found the locationId in parameter then replace it with the value found. If not found the rating curve object
     * need to removed from the list.
     * 
     * @param ratingCurve
     * @param parameters
     * @param rcl
     * @return
     * @throws Exception - No error is thrown but a ERROR message is set in the log file.
     */
    private boolean validateRatingCurvesLocationIdExist(final RatingCurve ratingCurve,
                                                        final UtilityParameters parameters,
                                                        final List<RatingCurve> wrongRatingCurveList) throws Exception
    {
        boolean found = false;
        String locationId = null;
        Table tableParam = null;
        found = false;
        locationId = ratingCurve.getLocationId();
        tableParam = parameters.getTableDataParameter(RatingCurveConstants.USGS_NUMBER_TO_LOCATION_ID);
        for(final Row row: tableParam.getRow())
        {
            if(locationId.equals(row.getA().toString()))
            {
                ratingCurve.setLocationId(row.getB().toString());
                found = true;
                break;
            }
        }
        if(!found)
        {
            _logger.log(Logger.ERROR,
                        "LocationId: " + locationId + " for file: " + ratingCurve.getFileName()
                            + " could not be found in parameters xml file. Please check OHDRatingCurveToolParams.xml under ModuleParFiles/ratingcurve directory.");
            _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
            // Remove the rating curve object as it will contain wrong data.
            wrongRatingCurveList.add(ratingCurve);
        }
        return found;
    }

    /**
     * Copy the RDB or TXT input(s) file(s) from the input directory into the work directory. Both input and work
     * directory paths are read from the runInfo object.<br>
     * Note: this method and copyOutputFilesToWorkDir() were created to be used for debug.
     * 
     * @param runInfo
     */
    public void copyInputFilesToWorkDir(final RunInfo runInfo)
    {
        String inputFileType = null;
        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(runInfo.getProperties()
                                                             .getProperty(RatingCurveConstants.INPUT_TYPE)))
            inputFileType = RatingCurveConstants.INPUT_TYPE_RDB.toLowerCase();
        else
            inputFileType = RatingCurveConstants.INPUT_TYPE_TEXT.toLowerCase();

        final Path source = Paths.get(runInfo.getProperties().getProperty(RatingCurveConstants.INPUT_RATING_CURVE_DIR));

        try (DirectoryStream<Path> stream = Files.newDirectoryStream(source, "*.{" + inputFileType + "}"))
        {
            Path target = Paths.get(runInfo.getWorkDir());
            for(final Path entry: stream)
            {
                target = Paths.get(runInfo.getWorkDir() + File.separator + entry.getFileName());
                Files.copy(entry, target, REPLACE_EXISTING);
            }
        }
        catch(final Exception x)
        {
            _logger.log(Logger.WARNING,
                        "Input Files in directory: "
                            + runInfo.getProperties().getProperty(RatingCurveConstants.INPUT_RATING_CURVE_DIR)
                            + " could not be saved to work dir: '" + runInfo.getWorkDir()
                            + "', it does not exist or cannot be written to.");
        }
    }

    /**
     * Copy the generated Rating Curves XML file from the output directory into the work directory. Both output and work
     * directory paths are read from the runInfo object.<br>
     * Note: this method and copyInputFilesToWorkDir() were created to be used for debug.
     * 
     * @param runInfo
     */
    public void copyOutputFilesToWorkDir(final RunInfo runInfo)
    {

        final Path source =
                          Paths.get(runInfo.getProperties().getProperty(RatingCurveConstants.OUTPUT_RATING_CURVE_DIR));

        try (
        DirectoryStream<Path> stream = Files.newDirectoryStream(source,
                                                                "*.{" + RatingCurveConstants.FILE_EXTENSION_XML + "}"))
        {
            Path target = Paths.get(runInfo.getWorkDir());
            for(final Path entry: stream)
            {
                target = Paths.get(runInfo.getWorkDir() + File.separator + entry.getFileName());
                Files.copy(entry, target, REPLACE_EXISTING);
            }
        }
        catch(final Exception x)
        {
            _logger.log(Logger.WARNING,
                        "Output Files in directory: "
                            + runInfo.getProperties().getProperty(RatingCurveConstants.OUTPUT_RATING_CURVE_DIR)
                            + " could not be saved to work dir: '" + runInfo.getWorkDir()
                            + "', it does not exist or cannot be written to.");

        }
    }

    /**
     * If rating curve is generated from a RDB input data then we need to validate that the input data (DischargeUnit)
     * is valid and update the value to CFS. For input data from TXT no need to validate it.
     * 
     * @param ratingCurve
     * @param rcl - a list that will contain the RatingCuve objects that are not valid
     * @return
     * @throws Exception
     */
    private boolean validateRatingCurvesDischargeUnitExist(final RatingCurve ratingCurve,
                                                           final List<RatingCurve> rcl) throws Exception
    {
        boolean found = false;
        if(RatingCurveConstants.INPUT_TYPE_TEXT.equals(ratingCurve.getFileType()))
            return true;
        if(RatingCurveConstants.key_discharge1.equals(ratingCurve.getDischargeUnit())
            || RatingCurveConstants.key_discharge2.equals(ratingCurve.getDischargeUnit()))
        {
            ratingCurve.setDischargeUnit(RatingCurveConstants.CFS);
            found = true;
        }
        else
        {
            //Redmine 47363 - Allow the discharge unit to not be present in the RDB input file as it is an optional value. 
            // If not present the default value is set.
            if(ratingCurve.getDischargeUnit() == null || ratingCurve.getDischargeUnit().isEmpty())
            {

                _logger.log(Logger.WARNING,
                            "DischageUnit: " + ratingCurve.getDischargeUnit() + " for file: "
                                + ratingCurve.getFileName() + " is not defined. Using the default value of: "
                                + RatingCurveConstants.CFS);
                ratingCurve.setDischargeUnit(RatingCurveConstants.CFS);
                found = true;
            }
            else
            {
                if(ratingCurve.getDischargeUnit().toUpperCase().contains(RatingCurveConstants.CFS)
                    || ratingCurve.getDischargeUnit().toUpperCase().contains(RatingCurveConstants.FEET))
                {
                    _logger.log(Logger.WARNING,
                                "DischageUnit: " + ratingCurve.getDischargeUnit() + " for file: "
                                    + ratingCurve.getFileName()
                                    + " is not defined as valid. Using the default value of: "
                                    + RatingCurveConstants.CFS);
                    ratingCurve.setDischargeUnit(RatingCurveConstants.CFS);
                    found = true;
                }
                else
                {
                    _logger.log(Logger.ERROR,
                                "DischageUnit: " + ratingCurve.getDischargeUnit() + " for file: "
                                    + ratingCurve.getFileName() + " is not defined as valid. ");
                    _logger.log(Logger.ERROR,
                                "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
                    // Remove the rating curve object as it will contain wrong data.
                    rcl.add(ratingCurve);
                }
            }

        }
        return found;
    }

    /**
     * If rating curve is generated from a RDB input data then we need to validate that the interpolationMethod is
     * valid. For input data from TXT no need to validate it.
     * 
     * @param ratingCurve
     * @param rcl - a list that will contain the RatingCuve objects that are not valid
     * @return true if stageUnit in rating curve object is valid.
     * @throws Exception
     */
    private boolean validateRatingCurvesInterpolationMethodExist(final RatingCurve ratingCurve,
                                                                 final List<RatingCurve> rcl) throws Exception
    {
        boolean found = false;
        if(RatingCurveConstants.INPUT_TYPE_TEXT.equals(ratingCurve.getFileType()))
            return true;
        if(RatingCurveConstants.LINEAR_INTERPOLATION.equals(ratingCurve.getInterpolationMethod())
            || RatingCurveConstants.LOGARITHMIC_INTERPOLATION.equals(ratingCurve.getInterpolationMethod()))
        {
            found = true;
        }
        else
        {
            if(ratingCurve.getInterpolationMethod() == null || ratingCurve.getInterpolationMethod().isEmpty())
            {
                _logger.log(Logger.ERROR,
                            "interpolationMethod: " + ratingCurve.getInterpolationMethod() + " for file: "
                                + ratingCurve.getFileName() + " is not defined. ");
                _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
                // Remove the rating curve object as it will contain wrong data.                
                rcl.add(ratingCurve);
            }
            else
            {
                _logger.log(Logger.ERROR,
                            "interpolationMethod: " + ratingCurve.getInterpolationMethod() + " for file: "
                                + ratingCurve.getFileName() + " is not defined as valid. Must be:"
                                + RatingCurveConstants.LINEAR_INTERPOLATION + " or "
                                + RatingCurveConstants.LOGARITHMIC_INTERPOLATION);
                _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
                // Remove the rating curve object as it will contain wrong data.
                rcl.add(ratingCurve);
            }
        }
        return found;
    }

    /**
     * If rating curve is generated from a RDB input data then we need to validate that the input data (StageUnit) is
     * valid and update the value to FT. For input data from TXT no need to validate it.
     * 
     * @param ratingCurve
     * @param rcl - a list that will contain the RatingCuve objects that are not valid
     * @return true if stageUnit in rating curve object is valid.
     * @throws Exception
     */
    private boolean validateRatingCurvesStageUnitExist(final RatingCurve ratingCurve,
                                                       final List<RatingCurve> rcl) throws Exception
    {
        boolean found = false;
        if(RatingCurveConstants.INPUT_TYPE_TEXT.equals(ratingCurve.getFileType()))
            return true;
        if(RatingCurveConstants.key_gage_height.equals(ratingCurve.getStageUnit()))
        {
            ratingCurve.setStageUnit(RatingCurveConstants.FEET);
            found = true;
        }
        else
        {
            //FB 1729. - We will use a default value instead of removing the rating curve from the list.
            //A warning message will be logged stating that the value was replaced.
            _logger.log(Logger.WARNING,
                        "StageUnit: " + ratingCurve.getStageUnit() + " for file: " + ratingCurve.getFileName()
                            + " is not defined as valid. Using the default value of: "
                            + RatingCurveConstants.key_gage_height);
            ratingCurve.setStageUnit(RatingCurveConstants.key_gage_height);
            found = true;
        }
        return found;
    }

    /**
     * If rating curve is generated from a RDB input data then we need to validate that the input data (flow values) are
     * valid (positive). For input data from TXT no need to validate it.
     * 
     * @param ratingCurve
     * @param rcl - a list that will contain the RatingCuve objects that are not valid
     * @return true if stageUnit in rating curve object is valid.
     * @throws Exception
     */
    private boolean validateRatingCurvesNegativeFlowsExist(final RatingCurve ratingCurve,
                                                           final List<RatingCurve> rcl) throws Exception
    {
        final boolean found = false;
        if(RatingCurveConstants.INPUT_TYPE_TEXT.equals(ratingCurve.getFileType())
            || Boolean.TRUE.equals(ratingCurve.isValidRatingCurveData()))
            return true;
        else
        {
            _logger.log(Logger.ERROR,
                        "The file: " + ratingCurve.getFileName()
                            + " could not be process as it contains negatives flow values.");
            _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
            // Remove the rating curve object as it will contain wrong data.
            rcl.add(ratingCurve);

        }
        return found;
    }

    /**
     * Validate that the Zero Flow Stage flag is set to TRUE if yes, check that the input data is valid to calculate the
     * Zero Flow Stage.
     * 
     * @param computeZeroFlowStage
     * @param ratingCurve
     * @return
     * @throws Exception
     */
    private void validateRatingCurvesZeroFlowStageData(final boolean computeZeroFlowStage,
                                                       final RatingCurve ratingCurve) throws Exception
    {
        //boolean found = false;
        if(Boolean.TRUE.equals(computeZeroFlowStage))
        {
            if(Boolean.FALSE.equals(ratingCurve.isValidZeroFlowStageData()))
            {
                _logger.log(Logger.WARNING,
                            "For locationId: " + ratingCurve.getLocationId() + " the first two Stages values in the "
                                + ratingCurve.getFileName()
                                + " file are equals. An error will ocurr if Zero Flow Stage value is computed. Input file processed without Zero Flow Stage enable.");

                ratingCurve.setStageUnit(RatingCurveConstants.key_gage_height);
                //found = true;
            }
        }
    }

    /**
     * Set the startDate value. If could be a fixed date or a number of dates before T0<br>
     * Note:These values were already validate. No exception should be expected
     * 
     * @param ratingCurves
     * @param runInfo
     * @throws Exception
     */
    private void setRatingCurveStartDate(final RatingCurve ratingCurve, final RunInfo runInfo) throws Exception
    {

        String startTimeBeforeT0 = runInfo.getProperties().getProperty(RatingCurveConstants.START_TIME_BEFORE_T0);

        if(startTimeBeforeT0 != null && !startTimeBeforeT0.isEmpty())
        {
            // Need to subtract the number of days represented by startTimeBeforeT0
            startTimeBeforeT0 = "-" + startTimeBeforeT0;
            final DateTime t0 = new DateTime(runInfo.getTime0Long());
            ratingCurve.setStartDate(t0.addDaysTo(Integer.parseInt(startTimeBeforeT0)));
        }
        else
        {
            final String startTime = runInfo.getProperties().getProperty(RatingCurveConstants.START_TIME_MMDDYYYYHHZ);
            if(startTime != null && !startTime.isEmpty())
            {
                ratingCurve.setStartDate(DateTime.parseMMDDYYYYHHZ(startTime));
            }
        }
    }

    /**
     * Check that there are no Offset or zero-flow ordinates defined. When we set the ShiftMinStage value we validate
     * that not offset values are present.
     * 
     * @param ratingCurve
     */
    private void setWarningMessageNoOffSetAndZeroFlowOrdinates(final RatingCurve ratingCurve)
    {
        if(ratingCurve.getShiftMinStage() > 0)
            _logger.log(Logger.WARNING,
                        "For locationId: " + ratingCurve.getLocationId()
                            + " There are no Offset or zero-flow ordinates defined. You may want to redefine this rating with an estimated zero-flow ordinate. MinStage set to "
                            + ratingCurve.getMinStage());
    }

    /**
     * Offset are defined, but a shift has been applied and the lowest ordinate is below the lowest offset.
     * 
     * @param ratingCurve
     */
    private void setWarningMessageMinStage(final RatingCurve ratingCurve)
    {
        if(ratingCurve.isWarningMinStage())
            _logger.log(Logger.WARNING,
                        "For locationId: " + ratingCurve.getLocationId()
                            + " Offset are defined, but a shift has been applied and the lowest ordinate is below the lowest offset. You may want to redefine this rating with an estimated zero-flow ordinate and an additional lower offset. MinStage set to "
                            + ratingCurve.getMinStage());
    }

    /**
     * Check that the rating curve object have at least two stage-discharge values defined (row objects) in the list of
     * rows
     * 
     * @param ratingCurves
     * @throws Exception
     */
    private boolean validateRatingCurvesStageDischargePairExist(final RatingCurve ratingCurve,
                                                                final List<RatingCurve> wrongRatingCurveList) throws Exception
    {
        boolean found = true;
        if(ratingCurve.getRow().size() < 2)
        {

            _logger.log(Logger.ERROR,
                        "LocationId: " + ratingCurve.getLocationId() + " for file: " + ratingCurve.getFileName()
                            + " has less than two stage-discharge values defined. Need at least 2 stage-discharge values defined for minStage and maxStage");
            _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
            wrongRatingCurveList.add(ratingCurve);
            found = false;
        }
        return found;
    }

    /**
     * Check that the parameter file is defined in the run info object. If defined validate that the file exist in the
     * path provided. Otherwise throw an error.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateParameterFileExist(final RunInfo runInfo) throws Exception
    {
        if(runInfo.getInputParametersFileList() != null && runInfo.getInputParametersFileList().size() > 0)
        {
            for(final String inputParameterFile: runInfo.getInputParametersFileList())
                if(inputParameterFile != null && !validateFileExists(inputParameterFile))
                {
                    _logger.log(Logger.ERROR,
                                "Please provide correct information from run info in the model parameter file name: "
                                    + inputParameterFile + " does not exist!");
                    throw new Exception("Please provide correct information from run info in the model parameter file name: "
                        + inputParameterFile + " does not exist!");
                }
        }
        else
        {
            _logger.log(Logger.ERROR, "Parameter XML file is not define in the runInfo file");
            throw new Exception("Parameter XML file is not define in the runInfo file");
        }
    }

    /**
     * Set the rating curve Zero flow stage value to true if the property RatingCurveConstants.ZERO_FLOW_STAGE is found
     * in the runIfo file and it is set as RatingCurveConstants.TRUE. Otherwise, not set or value different to any other
     * value, it will be set to false.
     * 
     * @param ratingCurves
     * @param runInfo
     * @throws Exception
     */
    private void validateZeroFlowStage(final RatingCurves ratingCurves, final RunInfo runInfo) throws Exception
    {
        final String zeroFlowStage = runInfo.getProperties().getProperty(RatingCurveConstants.ZERO_FLOW_STAGE);
        if(zeroFlowStage != null && !zeroFlowStage.isEmpty() && zeroFlowStage.equals(RatingCurveConstants.TRUE))
        {
            ratingCurves.setComputeZeroFlowStage(true);
        }
        else
        {
            ratingCurves.setComputeZeroFlowStage(false);
        }
    }

    /**
     * Check that the diagnostics file (log) is defined in the run info object.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateDiagnosticFileExist(final RunInfo runInfo) throws Exception
    {
        if(runInfo.getDiagnosticFile() != null)
        {
            final File dirCheckTemp = new File(runInfo.getDiagnosticFile());
            final String fewsDataOutputLocation = dirCheckTemp.getParent();

            final File dirCheck = new File(fewsDataOutputLocation);
            if(!dirCheck.canWrite())
            {
                throw new Exception("Output directory, " + dirCheck.getAbsolutePath()
                    + ", does not exist or cannot be written to.");
            }
        }
    }

    /**
     * Validate the input rating curve file exist. It would return a warning message is the input directory exist but
     * does not contain any file of the input type required. It would throw an exception if the input file directory
     * does not exist in the file system or is not defined in the runinfo object.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateInputRatingCurverFilesExist(final RunInfo runInfo) throws Exception
    {
        final String inputRatingCurvePath = runInfo.getProperties()
                                                   .getProperty(RatingCurveConstants.INPUT_RATING_CURVE_DIR);
        // Get list of files in directory
        if(inputRatingCurvePath != null)
        {
            final List<String> fileList = getFilePathNameList(inputRatingCurvePath,
                                                              RatingCurveConstants.INPUT_TYPE_RDB.toLowerCase() + ","
                                                                  + RatingCurveConstants.INPUT_TYPE_TEXT.toLowerCase());
            final File dirCheck = new File(inputRatingCurvePath);
            if(!dirCheck.canWrite())
            {
                throw new Exception("Input directory, " + dirCheck.getAbsolutePath()
                    + ", does not exist or cannot be written to.");
            }
            if(fileList.isEmpty())
            {
                if(RatingCurveConstants.INPUT_TYPE_RDB.equals(runInfo.getProperties()
                                                                     .getProperty(RatingCurveConstants.INPUT_TYPE)))
                    _logger.log(Logger.INFO,
                                "No RDB input files to convert. Please check ratingsPreprocessorRDB  under Import directory "
                                    + dirCheck.getAbsolutePath());
                else
                    _logger.log(Logger.INFO,
                                "No TXT input files to convert. Please check ratingsPreprocessorTXT under Import directory "
                                    + dirCheck.getAbsolutePath());
            }
        }
        else
        {
            _logger.log(Logger.ERROR,
                        RatingCurveConstants.INPUT_RATING_CURVE_DIR + " is not define in the runInfo file");
            throw new Exception(RatingCurveConstants.INPUT_RATING_CURVE_DIR + " is not define in the runInfo file");
        }

    }

    /**
     * Validate the output directory where the rating curve results will be write exist and can be accessed.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateOutputRatingCurveDirectoryExist(final RunInfo runInfo) throws Exception
    {
        final String outputRatingCurvePath = runInfo.getProperties()
                                                    .getProperty(RatingCurveConstants.OUTPUT_RATING_CURVE_DIR);
        if(outputRatingCurvePath != null)
        {
            final File dirCheck = new File(outputRatingCurvePath);
            if(!dirCheck.canWrite())
            {
                _logger.log(Logger.ERROR,
                            "Output directory, " + dirCheck.getAbsolutePath()
                                + ", does not exist or cannot be written to.");
                throw new Exception("Output directory, " + dirCheck.getAbsolutePath()
                    + ", does not exist or cannot be written to.");
            }
        }
        else
        {
            _logger.log(Logger.ERROR,
                        "Rating Curve output directory is not defined or empty, the following property must exist:"
                            + RatingCurveConstants.OUTPUT_RATING_CURVE_DIR);
            throw new Exception("Rating Curve output directory is not defined or empty, the following property must exist:"
                + RatingCurveConstants.OUTPUT_RATING_CURVE_DIR);
        }
    }

    /**
     * Validate the inputType for the rating curve file exist in the run info object. <br>
     * It would throw a error message if the inputType is not present in the run info object as it is required.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateRatingCurveTypeExist(final RunInfo runInfo) throws Exception
    {
        final String ratingCurveType = runInfo.getProperties().getProperty(RatingCurveConstants.INPUT_TYPE);
        if(ratingCurveType != null && !ratingCurveType.isEmpty())
        {
            if(!RatingCurveConstants.INPUT_TYPE_RDB.equals(ratingCurveType)
                && !RatingCurveConstants.INPUT_TYPE_TEXT.equals(ratingCurveType))
            {
                _logger.log(Logger.ERROR,
                            "Input Rating Curve file type: \"" + ratingCurveType + "\", must be one of: "
                                + RatingCurveConstants.INPUT_TYPE_RDB + " or " + RatingCurveConstants.INPUT_TYPE_TEXT);
                throw new Exception("Input Rating Curve file type: \"" + ratingCurveType + "\", must be one of: "
                    + RatingCurveConstants.INPUT_TYPE_RDB + " or " + RatingCurveConstants.INPUT_TYPE_TEXT);
            }
        }
        else
        {
            _logger.log(Logger.ERROR,
                        "Input Rating Curve file type is not defined or empty, must be one of:"
                            + RatingCurveConstants.INPUT_TYPE_RDB + " or " + RatingCurveConstants.INPUT_TYPE_TEXT);
            throw new Exception("Input Rating Curve file type is not defined or empty, must be one of:"
                + RatingCurveConstants.INPUT_TYPE_RDB + " or " + RatingCurveConstants.INPUT_TYPE_TEXT);
        }
    }

    /**
     * Validate the useMaxStageInRatingCurveAsMaxStage for the rating curve file exist in the run info object. <br>
     * If the useMaxStageInRatingCurveAsMaxStage property value is not present or set to FALSE, set the maxStage value
     * to {@link Double.POSITIVE_INFINITY}. If the useMaxStageInRatingCurveAsMaxStage property value is present and set
     * to TRUE keep current value.
     * 
     * @param ratingCurve
     * @param runInfo
     * @throws Exception
     */
    private void validateUseMaxStageInRatingCurveExist(final RatingCurve ratingCurve,
                                                       final RunInfo runInfo) throws Exception
    {
        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(runInfo.getProperties()
                                                             .getProperty(RatingCurveConstants.INPUT_TYPE)))
        {
            final String useMaxStage = runInfo.getProperties()
                                              .getProperty(RatingCurveConstants.USE_MAX_STAGE_IN_RATING_CURVE);
            if(useMaxStage != null && !useMaxStage.isEmpty())
            {
                if(RatingCurveConstants.FALSE.equalsIgnoreCase(useMaxStage))
                {
                    ratingCurve.setMaxStage(Double.POSITIVE_INFINITY);
                }
            }
            else
            {
                ratingCurve.setMaxStage(Double.POSITIVE_INFINITY);
            }
        }
    }

    /**
     * Validate that the start time is defined in the run info object. IF defined that it has valid format otherwise
     * throws an error
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateStartTimeExist(final RunInfo runInfo) throws Exception
    {
        final String ratingCurveType = runInfo.getProperties().getProperty(RatingCurveConstants.INPUT_TYPE);
        // Check 
        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(ratingCurveType))
        {
            final String startTimeBeforeT0 = runInfo.getProperties()
                                                    .getProperty(RatingCurveConstants.START_TIME_BEFORE_T0);
            final String startTime = runInfo.getProperties().getProperty(RatingCurveConstants.START_TIME_MMDDYYYYHHZ);
            if(startTimeBeforeT0 == null && startTime == null)
            {
                _logger.log(Logger.ERROR,
                            "Rating Curve start time is not defined or empty, one of following properties must exist:"
                                + RatingCurveConstants.START_TIME_BEFORE_T0 + " or "
                                + RatingCurveConstants.START_TIME_MMDDYYYYHHZ);
                throw new Exception("Rating Curve start time is not defined or empty, one of following properties must exist:"
                    + RatingCurveConstants.START_TIME_BEFORE_T0 + " or " + RatingCurveConstants.START_TIME_MMDDYYYYHHZ);
            }
            else
            {
                if(startTimeBeforeT0 != null && startTime != null)
                {
                    _logger.log(Logger.ERROR,
                                "Both '" + RatingCurveConstants.START_TIME_BEFORE_T0 + "' and '"
                                    + RatingCurveConstants.START_TIME_MMDDYYYYHHZ
                                    + " properties defined, only one property must exist.");
                    throw new Exception("Both '" + RatingCurveConstants.START_TIME_BEFORE_T0 + "' and '"
                        + RatingCurveConstants.START_TIME_MMDDYYYYHHZ
                        + "' properties defined, only one property must exist.");
                }
                if(startTimeBeforeT0 != null)
                {
                    try
                    {
                        Integer.parseInt(startTimeBeforeT0);
                    }
                    catch(final Exception e)
                    {
                        _logger.log(Logger.ERROR,
                                    "Rating Curve start time: " + RatingCurveConstants.START_TIME_BEFORE_T0
                                        + " does not have a valid format: " + startTimeBeforeT0);
                        throw new Exception("Rating Curve start time: " + RatingCurveConstants.START_TIME_BEFORE_T0
                            + " does not have a valid format: " + startTimeBeforeT0);
                    }
                }
                if(startTime != null)
                {
                    try
                    {
                        DateTime.parseMMDDYYYYHHZ(startTime);
                    }
                    catch(final Exception e)
                    {
                        _logger.log(Logger.ERROR,
                                    "Rating Curve start time: " + RatingCurveConstants.START_TIME_MMDDYYYYHHZ
                                        + " does not have a valid format: " + startTime);
                        throw new Exception("Rating Curve start time: " + RatingCurveConstants.START_TIME_MMDDYYYYHHZ
                            + " does not have a valid format: " + startTime);
                    }
                } // if
            } // else
        } // if
    }

    /**
     * Validate that the rating curve precision value is define in the run Info object as a property. If defined
     * validate that is a valid format. If not defined set the default value.
     * 
     * @param runInfo
     * @throws Exception
     */
    private void validateRatingCurvePrecisionExist(final RatingCurves ratingCurves,
                                                   final RunInfo runInfo) throws Exception
    {
        if(RatingCurveConstants.INPUT_TYPE_RDB.equals(runInfo.getProperties()
                                                             .getProperty(RatingCurveConstants.INPUT_TYPE)))
        {
            final String ratingCurvePrecision = runInfo.getProperties()
                                                       .getProperty(RatingCurveConstants.RATING_CURVE_PRECISION);
            if(ratingCurvePrecision != null && !ratingCurvePrecision.isEmpty())
            {
                switch(ratingCurvePrecision)
                {
                    case RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT:
                        ratingCurves.setRatingCurvePrecision(RatingCurveConstants.RATING_CURVE_PRECISION_0_POINT_5);
                        break;
                    case RatingCurveConstants.RATING_CURVE_PRECISION_TENTH_FOOT:
                        ratingCurves.setRatingCurvePrecision(RatingCurveConstants.RATING_CURVE_PRECISION_0_POINT_1);
                        break;
                    case RatingCurveConstants.RATING_CURVE_PRECISION_WHOLE_FOOT:
                        ratingCurves.setRatingCurvePrecision(RatingCurveConstants.RATING_CURVE_PRECISION_1_POINT_0);
                        break;
                    default:
                        _logger.log(Logger.WARNING,
                                    "Rating Curve precision: " + RatingCurveConstants.RATING_CURVE_PRECISION
                                        + " does not have a valid format. " + "It must be one of the following: "
                                        + RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT + ", "
                                        + RatingCurveConstants.RATING_CURVE_PRECISION_TENTH_FOOT + ", "
                                        + RatingCurveConstants.RATING_CURVE_PRECISION_WHOLE_FOOT
                                        + " Using default value: "
                                        + RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT);
                        ratingCurves.setRatingCurvePrecision(RatingCurveConstants.RATING_CURVE_PRECISION_DEFAULT);
                        break;
                }
            }
            else
            {
                _logger.log(Logger.WARNING,
                            "Rating Curve precision: '" + RatingCurveConstants.RATING_CURVE_PRECISION
                                + "' is not defined or empty. Using default value: "
                                + RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT);
                ratingCurves.setRatingCurvePrecision(RatingCurveConstants.RATING_CURVE_PRECISION_DEFAULT);
            }
        }
    }

    /**
     * Check that the file exist in the system
     * 
     * @param fileName
     * @return true if file exist, false otherwise
     */
    private boolean validateFileExists(final String fileName)
    {
        boolean result = false;
        final File file = new File(fileName);

        if(file.exists())
        {
            result = true;
        }
        return result;
    }

    /**
     * @param filePath
     * @param fileExtension
     * @return
     */

    //TODO: duplicated method.
    private List<String> getFilePathNameList(final String filePath, final String fileExtension)
    {
        final List<String> filePathName = new ArrayList<String>();
        Path dir = null;

        if(filePath != null && !filePath.isEmpty())
        {
            try
            {
                // http://docs.oracle.com/javase/tutorial/essential/io/dirs.html    
                dir = Paths.get(filePath);
            }
            catch(final InvalidPathException ipe)
            {
                _logger.log(Logger.ERROR, "Error in reading the input file path" + ipe.getMessage());
                throw ipe;
            }
            try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir, "*.{" + fileExtension + "}"))
            {
                for(final Path entry: stream)
                {
                    filePathName.add(entry.getParent() + File.separator + entry.getFileName());
                }
            }
            catch(final IOException x)
            {
                // IOException can never be thrown by the iteration.
                // In this snippet, it can // only be thrown by newDirectoryStream.
                System.err.println(x);
            }
        }
        else
        {
            _logger.log(Logger.ERROR, "Error reading the input file path, path null or empty");
            throw new InvalidPathException(filePath, "Error reading the input file path, path null or empty");
        }
        return filePathName;
    }

}
