package ohd.hseb.ohdutilities.ratingCurve;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.InvalidPathException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.TreeMap;

import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.MathHelper;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.fews.ratingCurve.RatingCurve;
import ohd.hseb.util.fews.ratingCurve.RatingCurveConstants;
import ohd.hseb.util.fews.ratingCurve.RatingCurveRow;
import ohd.hseb.util.fews.ratingCurve.RatingCurves;

/**
 * Read Rating Curve input files and generated a ratingCurve java object containing the information read from the input
 * files directory. There could be more than one input file and two different formats: RDB ad TXT format. This class
 * support both formats and multiple files.
 */
public class RatingCurveFileConverter
{
    private final Logger _logger;

    public RatingCurveFileConverter(final Logger logger)
    {
        _logger = logger;
    }

    /**
     * Read a text file containing data in a key=value format. Each key must match the elements of the ratingCurve.xsd
     * file. This method read the text file and stored each line in String object. Each string is stored in a List that
     * is used to process the information one by one. Each line is break into a Map containing a key-value pair, the key
     * is used to find the matching XML tag and stored in the rating curve object.
     * 
     * @param runInfo The run info object containing the file name and any other needed information
     * @param ratingCurves the rating curves object where information will be stored.
     * @throws Exception
     */
    protected void readKeyValueTextFileAndSetRatingCurveInfo(final RunInfo runInfo,
                                                             final RatingCurves ratingCurves) throws Exception
    {
        List<String> qualifierIdList = null;
        RatingCurve ratingCurve = null;
        Map<String, String> resultMap = null;
        RatingCurveRow row = null;
        List<RatingCurveRow> rowList = null;
        final List<RatingCurve> ratingCurveList = new ArrayList<RatingCurve>();

        List<Double> breakPointList = null;
        List<Double> offSetList = null;

        for(final String inputFileName: this.getFilePathNameList(runInfo.getProperties()
                                                                        .getProperty(RatingCurveConstants.INPUT_RATING_CURVE_DIR),
                                                                 RatingCurveConstants.INPUT_TYPE_TEXT.toLowerCase()))
        {
            qualifierIdList = new ArrayList<String>();
            ratingCurve = new RatingCurve();
            rowList = new ArrayList<RatingCurveRow>();

            ratingCurve.setFileName(inputFileName);
            ratingCurve.setFileType(RatingCurveConstants.INPUT_TYPE_TEXT);

            for(final String docline: this.readInputFileLines(inputFileName))
            {
                breakPointList = new ArrayList<Double>();
                offSetList = new ArrayList<Double>();

                resultMap = this.parseSimpleElement(docline, "", 0);
                for(final String key: resultMap.keySet())
                {
                    switch(key)
                    {
                        case RatingCurveConstants.TIME_ZONE:
                            final TimeZone tz = TimeZone.getTimeZone(resultMap.get(RatingCurveConstants.TIME_ZONE));
                            ratingCurves.setTimeZone(tz);
                            break;
                        case RatingCurveConstants.LOCATION_ID:
                            ratingCurve.setLocationId(resultMap.get(RatingCurveConstants.LOCATION_ID).trim());
                            break;
                        case RatingCurveConstants.START_DATE:
                            ratingCurve.setStartDate(DateTime.parseMMDDYYYYHHZ(resultMap.get(RatingCurveConstants.START_DATE)
                                                                                        .trim()));
                            break;
                        case RatingCurveConstants.END_DATE:
                            ratingCurve.setEndDate(DateTime.parseMMDDYYYYHHZ(resultMap.get(RatingCurveConstants.END_DATE)
                                                                                      .trim()));
                            break;
                        case RatingCurveConstants.LONG_NAME:
                            ratingCurve.setLongName(resultMap.get(RatingCurveConstants.LONG_NAME).trim());
                            break;
                        case RatingCurveConstants.STATION_NAME:
                            ratingCurve.setStationName(resultMap.get(RatingCurveConstants.STATION_NAME).trim());
                            break;
                        case RatingCurveConstants.STAGE_UNIT:

                            ratingCurve.setStageUnit(resultMap.get(RatingCurveConstants.STAGE_UNIT).trim());
                            break;
                        case RatingCurveConstants.DISCHARGE_UNIT:
                            ratingCurve.setDischargeUnit(resultMap.get(RatingCurveConstants.DISCHARGE_UNIT).trim());
                            break;
                        case RatingCurveConstants.SOURCE_ORGANISATION:
                            ratingCurve.setSourceOrganisation(resultMap.get(RatingCurveConstants.SOURCE_ORGANISATION)
                                                                       .trim());
                            break;
                        case RatingCurveConstants.SOURCE_SYSTEM:
                            ratingCurve.setSourceSystem(resultMap.get(RatingCurveConstants.SOURCE_SYSTEM).trim());
                            break;
                        case RatingCurveConstants.FILE_DESCRIPTION:
                            ratingCurve.setFileDescription(resultMap.get(RatingCurveConstants.FILE_DESCRIPTION).trim());
                            break;
                        case RatingCurveConstants.COMMENT:
                            ratingCurve.setComment(resultMap.get(RatingCurveConstants.COMMENT).trim());
                            break;
                        case RatingCurveConstants.CREATION_DATE:
                            ratingCurve.setCreationDate(resultMap.get(RatingCurveConstants.CREATION_DATE).trim());
                            break;
                        case RatingCurveConstants.CREATION_TIME:
                            ratingCurve.setCreationTime(resultMap.get(RatingCurveConstants.CREATION_TIME).trim());
                            break;
                        case RatingCurveConstants.REGION:
                            ratingCurve.setRegion(resultMap.get(RatingCurveConstants.REGION).trim());
                            break;
                        case RatingCurveConstants.INTERPOLATION_METHOD:
                            ratingCurve.setInterpolationMethod(resultMap.get(RatingCurveConstants.INTERPOLATION_METHOD)
                                                                        .trim());
                            break;
                        case RatingCurveConstants.MIN_STAGE:
                            ratingCurve.setMinStage(Double.parseDouble(resultMap.get(RatingCurveConstants.MIN_STAGE)
                                                                                .trim()));
                            break;
                        case RatingCurveConstants.MAX_STAGE:
                            this.setRatingCurveMaxStage(ratingCurve, resultMap.get(RatingCurveConstants.MAX_STAGE));
                            break;
                        default:
                            if(key.startsWith(RatingCurveConstants.QUALIFIER_ID))
                            {
                                qualifierIdList.add(resultMap.get(key).trim());
                            }
                            else if(key.startsWith(RatingCurveConstants.STAGE)
                                && !RatingCurveConstants.STAGE_UNIT.equals(key))
                            { //data is expected in certain order, first stage, second discharge and if available logScaleStageOffset
                                row = new RatingCurveRow();
                                rowList.add(row);
                                row.setStage(Double.parseDouble(resultMap.get(key).trim()));
                            }
                            else if(key.startsWith(RatingCurveConstants.DISCHARGE)
                                && !RatingCurveConstants.DISCHARGE_UNIT.equals(key))
                            {
                                row.setDischarge(Double.parseDouble(resultMap.get(key).trim()));
                            }
                            else if(key.startsWith(RatingCurveConstants.LOG_SCALE_STAGE_OFFSET))
                            {
                                row.setLogScaleStageOffset(Double.parseDouble(resultMap.get(key).trim()));
                            }
                            break;
                    } // end switch
                } //End for loop  - resultMap iteration
                _logger.log(Logger.DEBUG, docline);

            } // End for loop - read list document lines
            ratingCurve.setQualifierIds(qualifierIdList);
            ratingCurve.setRow(rowList);
            ratingCurveList.add(ratingCurve);
            if(ratingCurves.getComputeZeroFlowStage())
                this.computeStageValueForZeroFlow(ratingCurve, rowList, rowList, breakPointList, offSetList);
        } // End for loop - read files list
        ratingCurves.setRatingCurveList(ratingCurveList);
    }

    /**
     * Set the value of MaxStage for each rating curve. Value can be the a double number or the INF string that
     * represent the infinite value.
     * 
     * @param ratingCurve
     * @param resultMap
     */
    private void setRatingCurveMaxStage(final RatingCurve ratingCurve, final String maxStage)
    {
        if(maxStage != null)
            if(maxStage.trim().equalsIgnoreCase(RatingCurveConstants.XML_INFINITY_VALUE))
                ratingCurve.setMaxStage(Double.POSITIVE_INFINITY);
            else
                ratingCurve.setMaxStage(Double.parseDouble(maxStage.trim()));
    }

    /**
     * Read a text file containing data in a RDB format. This method read the text file and stored each line in String
     * object. Each string is stored in a List that is used to process the information one by one. Each line is break
     * into a Map containing multiple key-value pairs, the key is used to find the matching predefined tag from the text
     * file, the tag is translate to a XML tag and stored in the rating curve object.
     * 
     * @param runInfo The run info object containing the file name and any other needed information
     * @param ratingCurves the rating curves object where information will be stored.
     * @throws Exception
     */
    protected void readUSGSTextFileAndSetRatingCurveInfo(final RunInfo runInfo,
                                                         final RatingCurves ratingCurves) throws Exception
    {
        List<String> metaDataFieldsList = new ArrayList<String>();
        RatingCurve ratingCurve = null;
        Map<String, String> resultMap = null;
        List<Double> breakPointList = null;
        List<Double> offSetList = null;
        List<RatingCurveRow> rowList = null;

        final List<RatingCurve> ratingCurveList = new ArrayList<RatingCurve>();

        final String metaDataFields = runInfo.getProperties().getProperty(RatingCurveConstants.METADATA_FIELDS);
        final double ratingCurvePrecision = ratingCurves.getRatingCurvePrecision();
        if(metaDataFields != null && !metaDataFields.isEmpty())
            metaDataFieldsList = Arrays.asList(metaDataFields.split(","));
        // Iterate the list of RDB files
        for(final String inputFileName: this.getFilePathNameList(runInfo.getProperties()
                                                                        .getProperty(RatingCurveConstants.INPUT_RATING_CURVE_DIR),
                                                                 RatingCurveConstants.INPUT_TYPE_RDB.toLowerCase()))
        {
            ratingCurve = new RatingCurve();
            breakPointList = new ArrayList<Double>();
            offSetList = new ArrayList<Double>();
            rowList = new ArrayList<RatingCurveRow>();
            // Read the input file
            final List<String> inputLinesList = this.readInputFileLines(inputFileName);
            ratingCurve.setFileName(inputFileName);
            ratingCurve.setFileType(RatingCurveConstants.INPUT_TYPE_RDB);
            for(final String docline: inputLinesList)
            { // line starts with # //
                if(docline.startsWith(RatingCurveConstants.key_number_slash))
                {
                    final StringTokenizer st = new StringTokenizer(docline);
                    // Evaluating each token in the string
                    while(st.hasMoreTokens())
                    {
                        final String strtoken = st.nextToken();
                        // Does this string have the "//" string? This is a marker
                        // for the beginning of a tag.
                        if(strtoken.startsWith(RatingCurveConstants.key_slash))
                        {
                            // remove the "//" characters from the String
                            final String strelement = strtoken.substring(2);
                            resultMap = this.parseSimpleElement(docline, RatingCurveConstants.key_slash, 2);
                            //Begin parsing based on Elements found...
                            switch(strelement)
                            {
                                case RatingCurveConstants.key_station:
                                    this.setRDBRatingCurveStationValues(resultMap, ratingCurves, ratingCurve);
                                    break;
                                case RatingCurveConstants.key_rating:
                                    this.setRDBRatingCurveRatingValues(resultMap,
                                                                       //metaDataFieldsList,
                                                                       breakPointList,
                                                                       offSetList,
                                                                       ratingCurve);
                                    break;
                                case RatingCurveConstants.key_rating_indep:
                                    if(resultMap.containsKey(RatingCurveConstants.key_parameter))
                                        ratingCurve.setStageUnit(resultMap.get(RatingCurveConstants.key_parameter)
                                                                          .trim()); // RATING_INDEP:PARAMETER
                                    break;
                                case RatingCurveConstants.key_rating_dep:
                                    if(resultMap.containsKey(RatingCurveConstants.key_parameter))
                                        ratingCurve.setDischargeUnit(resultMap.get(RatingCurveConstants.key_parameter)
                                                                              .trim()); //RATING_DEP:PARAMETER
                                    break;
                                default:
                                    // do nothing -- field not needed.
                                    _logger.log(Logger.DEBUG, docline);
                            } // end switch
                              // set the comments - if line is part of metadataFields list
                            this.setRDBRatingCurveCommentsValues(strelement,
                                                                 resultMap,
                                                                 metaDataFieldsList,
                                                                 ratingCurve);
                        }
                    } // end while
                } //end if
                  // Determine if this is data
                else
                {
                    this.setRDBRatingCurveRowValues(docline, rowList);
                }
            } // end for       
              //FB 1729 Ignored rating curve with negative flow.

            if(rowList.get(0).getDischarge() >= 0.0)
            {
                if(ratingCurves.getComputeZeroFlowStage() && this.isValidDataForZeroFlowStage(rowList))
                {

                    this.setRDBRatingCurveRowValuesWithZeroFlow(breakPointList,
                                                                offSetList,
                                                                rowList,
                                                                ratingCurvePrecision,
                                                                ratingCurve);
                    ratingCurve.setValidZeroFlowStageData(true);

                }
                else
                {
                    this.setRDBRatingCurveRowValuesWithWarningMessage(breakPointList,
                                                                      offSetList,
                                                                      rowList,
                                                                      ratingCurvePrecision,
                                                                      ratingCurve);
                }
                ratingCurve.setValidRatingCurveData(true);

            }
            else
            {
                ratingCurve.setValidRatingCurveData(false);
            }
            // Save the rating curve information.
            ratingCurveList.add(ratingCurve);

//            else
//            {
//                //FB 1729 Ignored rating curve with negative flow.
//                _logger.log(Logger.ERROR, "The file: " + ratingCurve.getFileName()
//                    + " could not be process as it contains negatives flow values.");
//                _logger.log(Logger.ERROR, "Rating curves for file: " + ratingCurve.getFileName() + " not processed.");
//
//            }
        } // end for
        ratingCurves.setRatingCurveList(ratingCurveList);
    }//close method

    /**
     * Get the rating curve station name, source organization and time zone values from the RDB input file.
     * 
     * @param resultMap
     * @param ratingCurves
     * @param ratingCurve
     */
    private void setRDBRatingCurveStationValues(final Map<String, String> resultMap,
                                                final RatingCurves ratingCurves,
                                                final RatingCurve ratingCurve)
    {
        if(resultMap.containsKey(RatingCurveConstants.key_name))
        {
            ratingCurve.setStationName(resultMap.get(RatingCurveConstants.key_name).trim()); // STATION NAME
        }
        else
        {
            ratingCurve.setSourceOrganisation(resultMap.get(RatingCurveConstants.key_agency).trim()); // STATION AGENCY
            ratingCurve.setLocationId(resultMap.get(RatingCurveConstants.key_number).trim()); // STATION NUMBER
            // Redmine ticket 47363 - Avoid null pointer exception when no time zone defined.
            if(resultMap.get(RatingCurveConstants.key_timeZone) != null)
            {
                final TimeZone tz = TimeZone.getTimeZone(resultMap.get(RatingCurveConstants.key_timeZone));
                ratingCurves.setTimeZone(tz); // TIME_ZONE
            }
        }
    }

    /**
     * Get the rating curve interpolation method, offset, breakpoint and comments values from the RDB input file.
     * 
     * @param resultMap
     * @param breakPointList
     * @param offSetList
     * @param ratingCurve
     */
    private void setRDBRatingCurveRatingValues(final Map<String, String> resultMap,
                                               //final List<String> metaDataFieldsList,
                                               final List<Double> breakPointList,
                                               final List<Double> offSetList,
                                               final RatingCurve ratingCurve)
    {
        if(resultMap.containsKey(RatingCurveConstants.key_expansion))
        {
            ratingCurve.setInterpolationMethod(resultMap.get(RatingCurveConstants.key_expansion).trim()); // RATING EXPANSION                                    
        }
        else
        {
            if(!resultMap.isEmpty())
            {
                for(final String key: resultMap.keySet())
                {
                    if(key.contains(RatingCurveConstants.key_breakpoint))
                    {
                        // Need to set the BREAKPOINT for each value    
                        breakPointList.add(Double.valueOf(resultMap.get(key)));
                    }
                    else if(key.contains(RatingCurveConstants.key_offset))
                    { // the only option is OFFSET so far, but they is change that other values are presents, that is the reason of the if condition
                        offSetList.add(Double.valueOf(resultMap.get(key)));
                    }
                }
            }
        }
    }

    /**
     * Get the rating curve comments values from the RDB input file. The comments values are based on the
     * metadataFieldList list provide by the user.
     * 
     * @param fieldElement - s string that identify the document line.
     * @param resultMap - a map of the values read from the document line.
     * @param metaDataFieldsList list of String that contain two values for each list element, like FILE TYPE, First
     *            value will identify the row, second value the field that we need to include in the comments.
     * @param ratingCurve - the rating curve were information is saved.
     */
    private void setRDBRatingCurveCommentsValues(final String fieldElement,
                                                 final Map<String, String> resultMap,
                                                 final List<String> metaDataFieldsList,
                                                 final RatingCurve ratingCurve)
    {
        String key = null;
        String value = null;
        StringTokenizer st = null;
        StringBuffer comments = null;
        for(final String metadataField: metaDataFieldsList)
        {
            st = new StringTokenizer(metadataField);
            key = st.nextToken().trim();
            value = st.nextToken().trim();
            if(key.equals(fieldElement) && resultMap.containsKey(value))
            {
                comments = new StringBuffer();
                comments.append(metadataField.trim()).append("=\"").append(resultMap.get(value)).append("\" ");
                ratingCurve.setComment(comments.toString());
            }
        }
    }

    /**
     * Get the rating curve stage and discharge values from the RDB input file.<br>
     * When defining stage and discharge we use the following rules to choose which values to include: <br>
     * 1.Always use lowest and highest pair (if offset present MinStage must be bigger than first offset) <br>
     * 2. Always use starred values <br>
     * 3. For other, use discharge values ending at: hole foot, half-foot, tenth-of-foot
     * 
     * @param docline
     * @param breakPointList
     * @param offSetList
     * @param rowList
     * @param ratingCurvePrecision - one of these three values: RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT,
     *            RatingCurveConstants.RATING_CURVE_PRECISION_TENTH_FOOT,
     *            RatingCurveConstants.RATING_CURVE_PRECISION_WHOLE_FOOT
     * @param ratingCurve
     */
    private void setRDBRatingCurveRowValues(final String docline, final List<RatingCurveRow> rowList)
    {
        RatingCurveRow row = null;
        if(docline.startsWith(RatingCurveConstants.key_indep) || docline.startsWith(RatingCurveConstants.key_16n))
        {
            // do nothing with these lines             
        }
        else
        {// data section of RDB file            
            final StringTokenizer ratingCurveDataSt = new StringTokenizer(docline);
            // read "indep" value from line and set as "stage"
            final String stage = ratingCurveDataSt.nextToken();
            // "shift" from input line is Not used - but need to move to next token
            ratingCurveDataSt.nextToken();
            // read "dep" value from line and set as "discharge"
            final String discharge = ratingCurveDataSt.nextToken();
            String logScaleStageOffset = null; //optional value
            // Look for the "*" value in line
            if(ratingCurveDataSt.hasMoreTokens())
                logScaleStageOffset = ratingCurveDataSt.nextToken();

            row = new RatingCurveRow();
            row.setDischarge(Double.parseDouble(discharge));
            row.setStage(Double.parseDouble(stage));
            // Set a flag value for starred values. "1.0" means starred value
            if(logScaleStageOffset != null && logScaleStageOffset.equals(RatingCurveConstants.key_star))
            {
                row.setLogScaleStageOffset(1.0);
            }
            rowList.add(row);

            _logger.log(Logger.DEBUG, docline);
        }
    }

    /**
     * This method will apply the business logic to the input rows to return the rows that only match the requirements.
     * If the first discharge value of the List of rows is not zero, then this method also will compute the discharge
     * value for zero flow.<br>
     * It will get the rating curve stage and discharge values from the input rows.<br>
     * When defining stage and discharge we use the following rules to choose which values to include: <br>
     * 1.Always use lowest and highest pair (if offset present MinStage must be bigger than first offset) <br>
     * 2. Always use starred values <br>
     * 3. For other, use discharge values ending at: hole foot, half-foot, tenth-of-foot
     * 
     * @param breakPointList - a list containing the breakpoint values.
     * @param offSetList - a list containing the offset values.
     * @param allRowList - a list contained all the rows values read from the input file. This list will be
     * @param ratingCurvePrecision
     * @param ratingCurve
     */
    private void setRDBRatingCurveRowValuesWithZeroFlow(final List<Double> breakPointList,
                                                        final List<Double> offSetList,
                                                        final List<RatingCurveRow> allRowList,
                                                        final double ratingCurvePrecision,
                                                        final RatingCurve ratingCurve)
    {
        // the final rowList that contain only RatingCurveRow that match the business logic. 
        final List<RatingCurveRow> rowList = new ArrayList<RatingCurveRow>();
        int rowCount = 0;
        int zeroValueDischargeCount = 0;

        this.computeStageValueForZeroFlow(ratingCurve, rowList, allRowList, breakPointList, offSetList);

        for(final RatingCurveRow row: allRowList)
        {
            final double discharge = row.getDischarge();
            final double stage = row.getStage();
            final Double logScaleStageOffset = row.getLogScaleStageOffset();
            //1.Always use lowest pair 
            if(rowCount == 0)
            {
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                if(discharge == 0.0)
                { // if there is more than one discharge with zero value we need to removed all except the last one.
                    zeroValueDischargeCount++;
                }
            }
            //1.Always use highest pair 
            else if(rowCount == (allRowList.size() - 1))
            {
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                ratingCurve.setMaxStage(stage);
                if(discharge == 0.0)
                {
                    zeroValueDischargeCount++;
                }
            }
            else
            // 2. Always use starred values
            if(logScaleStageOffset != null && logScaleStageOffset.equals(Double.valueOf("1.0")))
            {
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                if(discharge == 0.0)
                {
                    zeroValueDischargeCount++;
                }
            }
            //3. For other, use discharge values ending at: hole foot, half-foot, tenth-of-foot
            else
            {
                final Double stageD = Double.valueOf(stage);
                // Obtain the decimal part from stage; for instance, 2.3 - 2 = 0.3 
                final double stageDecimalPart = MathHelper.roundToNDecimalPlaces(stage - stageD.intValue(), 2);
                // 0.3 % 0.3 == 0
                final double remainderTemp = (stageDecimalPart * 100) % (ratingCurvePrecision * 100);//MathHelper.roundToNDecimalPlaces(stageDecimalPart % 0.1, 2);
                if(remainderTemp == 0.0)
                {
                    this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                    if(discharge == 0.0)
                    {
                        zeroValueDischargeCount++;
                    }
                }
            }
            rowCount++;
        }
        if(zeroValueDischargeCount != 0)
        {
            this.computeStageValueForMoreThanOneZeroFlow(zeroValueDischargeCount,
                                                         ratingCurve,
                                                         rowList,
                                                         breakPointList,
                                                         offSetList);
        }
        ratingCurve.setRow(rowList);
    }

    /**
     * Get the rating curve stage and discharge values from the RDB input file.<br>
     * When defining stage and discharge we use the following rules to choose which values to include: <br>
     * 1.Always use lowest and highest pair (if offset present MinStage must be bigger than first offset) <br>
     * 2. Always use starred values <br>
     * 3. For other, use discharge values ending at: hole foot, half-foot, tenth-of-foot
     * 
     * @param docline
     * @param breakPointList
     * @param offSetList
     * @param rowList
     * @param ratingCurvePrecision - one of these three values: RatingCurveConstants.RATING_CURVE_PRECISION_HALF_FOOT,
     *            RatingCurveConstants.RATING_CURVE_PRECISION_TENTH_FOOT,
     *            RatingCurveConstants.RATING_CURVE_PRECISION_WHOLE_FOOT
     * @param ratingCurve
     */
    private void setRDBRatingCurveRowValuesWithWarningMessage(final List<Double> breakPointList,
                                                              final List<Double> offSetList,
                                                              final List<RatingCurveRow> allRowList,
                                                              final double ratingCurvePrecision,
                                                              final RatingCurve ratingCurve)
    {
        double discharge = 0.0;
        double stage = 0.0;
        // the final rowList that contain only RatingCurveRow that match the business logic. 
        final List<RatingCurveRow> rowList = new ArrayList<RatingCurveRow>();
        int rowCount = 0;
        // get the number of rows read from input file.
        int lastRowCount = allRowList.size() - 1;

        //TODO: You can take out from loop the option 1 and 2, option 1 will need to have its own loop. Future enhancement.
        for(final RatingCurveRow row: allRowList)
        {
            discharge = row.getDischarge();
            stage = row.getStage();
            final Double logScaleStageOffset = row.getLogScaleStageOffset();
            //1.Always use lowest pair 
            if(rowCount == 0)
            {
                if(!offSetList.isEmpty() && offSetList.get(0).doubleValue() > stage)
                {
                    // Ignore all the values until first stage is bigger than first offset value.
                    ratingCurve.setWarningMinStage(true);
                    // Reduce the number of rows by 1, as we skip the rows with offset bigger than the stage 
                    lastRowCount--;
                    continue;
                }
                if(offSetList.isEmpty() && stage > 0)
                {
                    // This is used to set the warning message, not other use.
                    ratingCurve.setShiftMinStage(stage);
                }
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                ratingCurve.setMinStage(stage);
            }
            //2.Always use highest pair 
            else if(rowCount == lastRowCount)
            {
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                ratingCurve.setMaxStage(stage);
            }
            else
            // 3. Always use starred values (*) from RDB file, represented internally by "1.0"
            if(logScaleStageOffset != null && logScaleStageOffset.equals(Double.valueOf("1.0")))
            {
                this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
            }
            //4. For other, use discharge values ending at: hole foot, half-foot, tenth-of-foot
            else
            {
                final Double stageD = Double.valueOf(stage);
                // Obtain the decimal part from stage; for instance, 2.3 - 2 = 0.3 
                final double stageDecimalPart = MathHelper.roundToNDecimalPlaces(stage - stageD.intValue(), 2);
                // 0.3 % 0.3 == 0
                final double remainderTemp = (stageDecimalPart * 100) % (ratingCurvePrecision * 100);//MathHelper.roundToNDecimalPlaces(stageDecimalPart % 0.1, 2);
                if(remainderTemp == 0.0)
                {
                    this.setRowValuesToRowList(stage, discharge, breakPointList, offSetList, rowList);
                }
            }
            rowCount++;
        }
        ratingCurve.setRow(rowList);
    }

    /**
     * Convert a giving stage to the associate stage using interpolation of a single-value rating curve.
     * 
     * @param ratingCurve
     * @param rowList Contains the stage and discharge values that comply with the business rules defined.
     * @param allRowList Contain the first and second stage and discharge values from the input file.
     * @param breakPointList
     * @param offSetList
     */
    private void computeStageValueForZeroFlow(final RatingCurve ratingCurve,
                                              final List<RatingCurveRow> rowList,
                                              final List<RatingCurveRow> allRowList,
                                              final List<Double> breakPointList,
                                              final List<Double> offSetList)
    {

        if(allRowList.get(0).getDischarge() > 0.0)
        {
            //FB 1501 - part 2 - if OffSet is defined and the lowest stage ordinate is above it's offset, 
            //we would like that offset to be used as Minimum Stage value.
            final RatingCurveRow row = allRowList.get(0);
            double stage = row.getStage();
            if(offSetList != null && offSetList.size() > 0 && stage > offSetList.get(0))
            {
                ratingCurve.setMinStage(offSetList.get(0));
            }
            else
            {
                final RatingCurveRow row1 = allRowList.get(0);
                final RatingCurveRow row2 = allRowList.get(1);

                double shiftFactor = 0.0;
                // convert to metric units
                final Measurement discharge1M = new Measurement(row1.getStage(), MeasuringUnit.feet);
                final Measurement stage1M = new Measurement(row1.getDischarge(), MeasuringUnit.ft3);
                final Measurement discharge2M = new Measurement(row2.getStage(), MeasuringUnit.feet);
                final Measurement stage2M = new Measurement(row2.getDischarge(), MeasuringUnit.ft3);

                double discharge1 = discharge1M.getValue(MeasuringUnit.meters);
                double stage1 = stage1M.getValue(MeasuringUnit.m3);
                double discharge2 = discharge2M.getValue(MeasuringUnit.meters);
                double stage2 = stage2M.getValue(MeasuringUnit.m3);

                if(discharge1 < 0.0 || ratingCurve.getMinStage() < 0.0)
                {
                    shiftFactor = 0.01 - Math.min(discharge1, ratingCurve.getMinStage());
                }

                discharge1 = discharge1 + shiftFactor;
                discharge2 = discharge2 + shiftFactor;

                double discharge = 0.0001;
                stage = 0.0;
                double slope = 0.0;

                if(RatingCurveConstants.LOGARITHMIC_INTERPOLATION.equals(ratingCurve.getInterpolationMethod()))
                {
                    stage1 = Math.log(stage1);
                    stage2 = Math.log(stage2);
                    discharge1 = Math.log(discharge1);
                    discharge2 = Math.log(discharge2);
                    discharge = Math.log(discharge);

                    slope = (discharge1 - discharge2) / (stage1 - stage2);
                    stage = (discharge - stage1) * slope + discharge1;
                    //DYDX=(Y1-Y2)/(X1-X2)
                    //YY=(XX-X1)*DYDX+Y1

                    slope = (discharge1 - discharge2) / (stage1 - stage2);
                    stage = (discharge - stage1) * slope + discharge1;
                    if(stage < -100.0)
                        stage = -100.0;
                    stage = Math.exp(stage);
                }
                else
                {
                    slope = (discharge1 - discharge2) / (stage1 - stage2);
                    stage = (discharge - stage1) * slope + discharge1;

                }

                stage = stage - shiftFactor;
                final Measurement stageM = new Measurement(stage, MeasuringUnit.meters);
                stage = stageM.getValue(MeasuringUnit.feet);
                stage = MathHelper.roundToNDecimalPlaces(stage, 2);
                //TODO: It was removed by Brain Request.
//            final RatingCurveRow myRow = new RatingCurveRow();
//            myRow.setDischarge(0); // Fixed value of 0.0
//            myRow.setStage(stage);
//
//            // set the offset value equals to computed stage.
//            myRow.setLogScaleStageOffset(0.0);
                // If First Stage value is smaller than the first offset a shift must be applied to the offset list.
                // a new offset value is computed base on the first stage value. Because of the new offset definition a new breakpoint needs to be defined 
                // as the first breakpoint from the original offset list. The original offset is shifted with new computed value in first place.  
                if(offSetList != null && offSetList.size() > 0 && (stage < offSetList.get(0)))
                {
                    breakPointList.add(0, offSetList.get(0));
                    offSetList.add(0, stage);
                }
                // Replace the OffSet value for any row in the rowList with the offset list definition.
                for(final RatingCurveRow rtr: rowList)
                {
                    rtr.setLogScaleStageOffset(this.setOffSetValue(stage, breakPointList, offSetList));
                }
                // add the new row into the fist position of the List
                //TODO: It was removed by Brain Request.
//            rowList.add(0, myRow);
                //Update minimum stage value
                ratingCurve.setMinStage(stage);
            }
        }
//        else if(allRowList.get(0).getDischarge() < 0.0)
//        {
//            //TODO: We will need to iterate until we find the fist value bigger than zero.
//            System.out.println("***** This message should not be printed as it is only for negative values *******");
//            System.out.println(ratingCurve.getFileName());
//        }

    }

    /**
     * If zero flow stage is already defined keep it (take the largest if more than one zero flow stage is defined). Use
     * the zero flow stage value as the offset value for all stages greater than or equal to the zero flow stage and
     * less than or equal to the original offset.
     * 
     * @param zeroValueDischargeCount - Define the number of rows with discharge value of zero
     * @param ratingCurve
     * @param rowList
     * @param breakPointList
     * @param offSetList
     */
    private void computeStageValueForMoreThanOneZeroFlow(final int zeroValueDischargeCount,
                                                         final RatingCurve ratingCurve,
                                                         final List<RatingCurveRow> rowList,
                                                         final List<Double> breakPointList,
                                                         final List<Double> offSetList)
    {
        // If more than one zero flow stage is defined removed all except the largest (last in the list)
        // here is were the zeroValueDischargeCount comes into play at it contain the number of zero flow stage in the Rating Curve list.
        for(int dischargeCount = 0; dischargeCount < zeroValueDischargeCount - 1; dischargeCount++)
        {
            rowList.remove(0);
        }
        // get the zero flow stage from the list of rating curve with discharge value equals to zero (first occurrence) 
        final RatingCurveRow myRow = rowList.get(0);
        final double stage = myRow.getStage();

        // If First Stage value is smaller than the first offset a shift must be applied to the offset list.
        // The new offset value is the first stage value. Because of the new offset definition a new breakpoint needs to be defined 
        // as the first breakpoint from the original offset list. The original offset is shifted with the new value in first place.  
        //FB 1729 null pointer error due to the list being empty 
        if(!offSetList.isEmpty() && (stage < offSetList.get(0)))
        {
            breakPointList.add(0, offSetList.get(0));
            offSetList.add(0, stage);

            // Replace the OffSet value for any row in the rowList with the new offset list definition.
            for(final RatingCurveRow rtr: rowList)
            {
                rtr.setLogScaleStageOffset(this.setOffSetValue(rtr.getStage(), breakPointList, offSetList));
                if(rtr.getStage() > offSetList.get(1).doubleValue())
                {
                    break;
                }
            }
        }
        myRow.setLogScaleStageOffset(stage);

        //Update minimum stage value
        ratingCurve.setMinStage(stage);

    }

    /**
     * Validate the the first two values of the Rating Curve are not zero and if not that they are not equals. To avoid
     * a division by zero error.
     * 
     * @param allRowList
     * @return
     */
    private boolean isValidDataForZeroFlowStage(final List<RatingCurveRow> allRowList)
    {
        boolean validZeroFlow = true;

        final RatingCurveRow row1 = allRowList.get(0);
        final RatingCurveRow row2 = allRowList.get(1);

        // Division by zero error.
        if(row1.getDischarge() > 0.0 && (row1.getDischarge() == row2.getDischarge()))
        {
            validZeroFlow = false;
        }
        return validZeroFlow;
    }

    /**
     * Set the offset value to the row object and insert it to the list of rows.
     * 
     * @param indep
     * @param dep
     * @param breakPointList
     * @param offSetList
     * @param rowList
     */
    private void setRowValuesToRowList(final double indep,
                                       final double dep,
                                       final List<Double> breakPointList,
                                       final List<Double> offSetList,
                                       final List<RatingCurveRow> rowList)
    {
        final RatingCurveRow row = new RatingCurveRow();
        row.setDischarge(dep);
        row.setLogScaleStageOffset(this.setOffSetValue(indep, breakPointList, offSetList));
        row.setStage(indep);
        rowList.add(row);
    }

    /**
     * Set the OffSet value for the giving stage value, it is calculated based on the list of breakPoint values.<br>
     * There is always one more offset value than breakpoint value defined.
     * 
     * @param stage the value used as key (to compare against it) to determine the offset value
     * @param breakPointList the break point list used to determined the offset value
     * @param offSetList List of off set values to be used.
     * @return an Double offset value to be used in the row or empty if not offset values are present (empty list of
     *         offset values).
     */
    private Double setOffSetValue(final double stage, final List<Double> breakPointList, final List<Double> offSetList)
    {
        Double logScaleStageOffset = null;
        if(breakPointList.isEmpty() && offSetList.isEmpty())
        {
            // No breakPoint or OffSet values
            logScaleStageOffset = null;
        }
        else if(breakPointList.isEmpty() && !offSetList.isEmpty())
        {
            //Only one OffSet values is present
            logScaleStageOffset = Double.valueOf(offSetList.get(0));
        }
        else
        {
            // Both list of values are present
            for(int i = 0; i < breakPointList.size(); i++)
            {
                if(stage < breakPointList.get(i).doubleValue())
                {
                    logScaleStageOffset = offSetList.get(i);
                    break;
                }
                else
                {
                    logScaleStageOffset = offSetList.get(i + 1);
                }
            }
        }
        return logScaleStageOffset;
    }

    /**
     * Given a single line of the input text document it will analyze it and break it into key value pairs delimited by
     * the equals sign, it will removed all double quotes in the value section if present and preserve any equal sign if
     * it is present in the value section.
     * 
     * @param documentLine
     * @param delimitor
     * @param offset
     * @return
     */
    private Map<String, String> parseSimpleElement(final String documentLine, final String delimitor, final int offset)
    {

        // parseSimpleElement assumes that this is a unique element which does not
        // get repeated.

        final Map<String, String> resultMap = new TreeMap<String, String>();

        // We will assume that there is one or more attributes and that the attribute
        // does not cross multiple lines.
        // This next line will iterate through each String and find the attributes.

        // FB 1960 - The substring operation was removing a valid value.
        //final String str_parse = documentLine.trim();
        final String str_parse = documentLine;
        //We will parse multiple lines as comments.
        final int x = str_parse.indexOf(delimitor);
        final String sub = str_parse.substring(x + offset);

        final StringBuffer sbsub = new StringBuffer();
        // Check for tokens inside comments.  We need to 
        // replace these with XML equivalents &eq;.  Then
        // replace those characters after the parsing.
        boolean inside = false;

        for(int k = 0; k < sub.length(); k++)
        {
            // Check for cases where the '=' sign appears
            // within a quote.  
            // Determine when we have an open qoute.
            if(sub.charAt(k) == '"' && inside == false)
            {
                // We are now inside
                inside = true;
            }
            else if(sub.charAt(k) == '"' && inside == true)
            {
                // We are now outside the quote
                inside = false;
            }

            // Now we have determined if we are inside or outside.
            // Replace the = character with the '&eq; string.
            if(inside && sub.charAt(k) == '=')
            {
                sbsub.append("&eq;");
            }
            else
            {
                sbsub.append(sub.charAt(k));
            }
        }
        //Find all instances of the = sign.
        final StringTokenizer st = new StringTokenizer(sbsub.toString(), "=");
        int p = 0;
        String strtype = "";
        String strvalue = "";

        while(st.hasMoreTokens())
        {
            final String strtoken = st.nextToken();
            final int pos = strtoken.lastIndexOf(" ");
            if(p == 0)
            {
                strtype = strtoken.substring(pos + 1);
            }
            if(p > 0)
            {
                final int valpos = strtoken.lastIndexOf("\"");
                strvalue = strtoken.substring(0, valpos + 1);
                strtype = strtype.trim();
                final int findex = strvalue.indexOf("\"");
                int lastindex = strvalue.lastIndexOf("\"");
                // We need to ignore any cases where an = sign is
                // used in a comment.

                // In cases where the value is surrounded by ""
                if(findex >= 0 && lastindex >= 0)
                {
                    strvalue = strtoken.substring(findex + 1, lastindex);
                    resultMap.put(strtype, strvalue.replaceAll("&eq;", "="));
                    strtype = strtoken.substring(lastindex + 1);
                }
                // cases where the value is not surrounded by quotes.
                if(findex == -1 && lastindex == -1)
                {
                    //determine the last index of the space.
                    lastindex = strtoken.lastIndexOf(" ");
                    // This may be the last value in the line.
                    // Last index may report -1 if no space exists.
                    if(lastindex != -1)
                    {
                        // FB 1960 - The substring operation was removing a valid value.
                        //strvalue = strtoken.substring(0, lastindex - 1);
                        strvalue = strtoken.substring(0, lastindex);
                        resultMap.put(strtype, strvalue.replaceAll("&eq;", "="));
                        strtype = strtoken.substring(lastindex + 1);
                    }
                    else
                    {
                        //There is a space after the value. A
                        //data type may follow.
                        strvalue = strtoken.substring(0);
                        resultMap.put(strtype, strvalue.replaceAll("&eq;", "="));
                    }
                }
                strvalue = strvalue.replaceAll("&eq;", "=");
            }
            p++;
        }

        return resultMap;
    }

    /**
     * Read the input file line by line and store them in the vector {@link #_inputLinesList}. The line is not altered
     * by trimming or padding with " " to some length. It is stored as it is on the opt file.
     * 
     * @param fileName - input file name
     * @exception throws to IOException
     */
    final private List<String> readInputFileLines(final String fileName) throws IOException
    {
        final List<String> inputLinesList = new ArrayList<String>();
        String line = null;
        try
        {
            final BufferedReader inputStream = new BufferedReader(new FileReader(fileName));

            while((line = inputStream.readLine()) != null)
            {
                inputLinesList.add(line);
                /*
                 * can not trim the line here, because if there are " " in the beginning, trimming will shift the line
                 * and all the index later is wrong
                 */
            }

            inputStream.close();
        }
        catch(final IOException e)
        {
            _logger.log(Logger.ERROR, "Error in reading the input file " + e.getMessage());
            throw e;
        }
        return inputLinesList;
    }

    /**
     * Get the list of files in the specified path directory. The result list will be filter base in the file extension.
     * There are two possible valid extension in the directory "RDB" and "TXT" any other file extension will be ignored.
     * 
     * @param filePath the directory path where input rating curve files exist in RDB or TXT format.
     * @return a List of path + file names for the given extension
     */
    private List<String> getFilePathNameList(final String filePath, final String fileExtension)
    {
        final List<String> filePathName = new ArrayList<String>();

        if(filePath != null && !filePath.isEmpty())
        {
            try
            {
                final File dirFiles = new File(filePath);
                final File[] files = dirFiles.listFiles(new FilenameFilter()
                {

                    @Override
                    public boolean accept(final File directory, final String fileName)
                    {
                        if(fileName.endsWith("." + fileExtension))
                        {
                            return true;
                        }
                        return false;
                    }

                });
                Arrays.sort(files);
                for(final File file: files)
                {
                    filePathName.add(file.getPath());
                }
            }
            catch(final Exception e)
            {
                _logger.log(Logger.ERROR, "Error reading the input file path, " + filePath + " is not valid");
                throw new InvalidPathException(filePath,
                                               "Error reading the input file path, " + filePath + " is not valid");
            }

        }
        else
        {
            _logger.log(Logger.ERROR, "Error reading the input file path, path null or empty");
            throw new InvalidPathException(filePath, "Error reading the input file path, path null or empty");
        }
        return filePathName;
    }
}
