/*
 * Created on Feb 25, 2004 This is the main program for the extraction of OFS data, conversion to XML, and then sending
 * to the WFO
 */
package ohd.hseb.ohdutilities.sshp.messaging;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.text.SimpleDateFormat;
import java.util.List;
import java.util.TimeZone;

import ohd.hseb.util.FileLogger;
import ohd.hseb.util.Logger;

import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.jdom.output.Format;
import org.jdom.output.XMLOutputter;

/**
 * @author GobsC
 */
public class SshpDataTransferMgr
{
    private static String _ohdBinDir = null;
    private static String _controlFilePath = null;
    private static String _extractedDirName = null;
    private static String _processedDirName = null;
    private static String _logDirName = null;
    private static String _rfcName = null;
    private static String _portNumber = null;
    private Document _controlDoc = null;
    private final SAXBuilder _builder = new SAXBuilder();
    private final XMLOutputter _xmlOutputter = new XMLOutputter(Format.getPrettyFormat());

    private FileLogger _logger = null;
    private String _logFilePath = null;

    // --------------------------------------------------------------
    public SshpDataTransferMgr(final String rfcName,
                               final String portNumber,
                               final String controlFilePath,
                               final String extractedDirName,
                               final String processedDirName,
                               final String logDirName)
    {
        _rfcName = rfcName;
        _portNumber = portNumber;
        _controlFilePath = controlFilePath;
        _extractedDirName = extractedDirName;
        _processedDirName = processedDirName;

        _logDirName = logDirName;

        _logFilePath = _logDirName + "/SshpDataTransferMgrCHPS.log";

        _logger = new FileLogger(_logFilePath);

        loadControlFile();

        return;
    }

//  --------------------------------------------------------------
    public void execute()
    {
//        String header = "SshpDataTransferMgr.execute(): ";

        System.out.println("SshpDataTransferMgr logging to " + _logFilePath);
        //using the control xml document
        //for each wfo

        //for each forecast point
        //extract all the segment data to a file  
        //translate the extract file and add contents to the live XML doc
        //end for each forecast point

        //output live XML document object to a file
        //send file using MHS
        // end for each wfo     

        final Element controlRoot = _controlDoc.getRootElement();
        final List wfoElementList = controlRoot.getChildren("wfo");
        final int wfoCount = wfoElementList.size();

        final long time = System.currentTimeMillis();
        final String timeString = getDateTimeStringFromLongTime(time);

        //for each wfo  
        for(int i = 0; i < wfoCount; i++)
        {
            final Element wfoElement = (Element)wfoElementList.get(i);
            final String wfoName = wfoElement.getChildText("name");
            final String mhsId = wfoElement.getChildText("mhsId");
            final String productId = wfoElement.getChildText("productId");

            _logger.log(Logger.DEBUG, "processing wfoName = " + wfoName);

            final List fcstPointElementList = wfoElement.getChildren("fcstPoint");
            final int fcstPointCount = fcstPointElementList.size();

            final Document outputXmlDoc = getBaseDocument();

            //for each forecast point 
            for(int j = 0; j < fcstPointCount; j++)
            {
                final Element fcstPointElement = (Element)fcstPointElementList.get(j);
                final String moduleInstanceId = fcstPointElement.getChildText("moduleInstanceId");

                if(moduleInstanceId.indexOf("_") <= 0)
                {
                    _logger.log(Logger.WARNING, "WARNING- Could not get the correct moduleInstanceId \""
                        + moduleInstanceId + "\" in the control file.  Skip this forecast point....");
                    continue;
                }

                final String locationId = fcstPointElement.getChildText("ihfsLocId");
                final String basinId = fcstPointElement.getChildText("ihfsBasinId");
                _logger.log(Logger.DEBUG, "moduleInstanceId = " + moduleInstanceId + " locationId = " + locationId);

                final String extractFileName = _extractedDirName + "/" + moduleInstanceId + "." + timeString + ".txt";

                // extract the segment data to a file  
                extractChpsData(moduleInstanceId, extractFileName);

                //translate the extract file and add contents to the live XML doc
                translateToXml(basinId, extractFileName, outputXmlDoc);
            }

            if(fcstPointCount < 1)
            {
                _logger.log(Logger.DEBUG, "There were no forecast points defined in the control file: "
                    + _controlFilePath);
            }

            //output the xml document to a file
            final String xmlFilePath = _processedDirName + "/" + wfoName + "." + timeString + ".xml";

            writeXmlToFile(outputXmlDoc, xmlFilePath);

            //send xml file to the site (WFO, usually) with this mhsId
            //run_SSHPCHPS_data_send script won't be executed by this method
            //sendXml(xmlFilePath, xmlFileName, mhsId, productId);
        }

        if(wfoCount < 1)
        {
            _logger.log(Logger.DEBUG, "There were no wfos defined in the control file: " + _controlFilePath);
        }

        _logger.log(Logger.DEBUG, "SshpDataTransferMgr Completed.");
        System.out.println("SshpDataTransferMgr Completed.");

    }

//  --------------------------------------------------------------

    private Document getBaseDocument()
    {
        final Document doc = new Document();

        final Element rootElement = new Element("SacModelMessage");

        doc.setRootElement(rootElement);

        final Element sacParamsListElement = new Element("SacParamsList");
        final Element sacStateListElement = new Element("SacStateList");
        final Element peTimeSeriesListElement = new Element("PeTimeSeriesList");
        final Element runoffTimeSeriesElement = new Element("RunoffTimeSeriesList");
        final Element monthlyValuesListElement = new Element("MonthlyValuesList");

        rootElement.addContent(sacParamsListElement);
        rootElement.addContent(sacStateListElement);
        rootElement.addContent(peTimeSeriesListElement);
        rootElement.addContent(runoffTimeSeriesElement);
        rootElement.addContent(monthlyValuesListElement);

        return doc;
    }

//  --------------------------------------------------------------

    private void loadControlFile()
    {
        try
        {
            _controlDoc = _builder.build(_controlFilePath);
        }
        catch(final IOException e)
        {
            e.printStackTrace();
        }
        catch(final JDOMException e2)
        {
            e2.printStackTrace();
        }
    }

//  --------------------------------------------------------------

    private void writeXmlToFile(final Document outputXmlDoc, final String xmlFileName)
    {

        try
        {
            final FileWriter writer = new FileWriter(xmlFileName);
            _xmlOutputter.output(outputXmlDoc, writer);
            writer.close();
        }
        catch(final IOException e)
        {
            e.printStackTrace();
        }
    }

//     --------------------------------------------------------------
    private static String getDateTimeStringFromLongTime(final long time)
    {
        final String timeString = getStringFromLongTime(time, "yyyy_MM_dd_HH_mm_ss");

        return timeString;
    }

    //  -------------------------------------------------------

    private static String getStringFromLongTime(final long time, final String dateFormat)
    {
        String timeString = null;

        //System.out.println("timeString = !" + timeString + "!");
        final SimpleDateFormat utcSdf2 = new SimpleDateFormat(dateFormat);
        utcSdf2.setTimeZone(TimeZone.getTimeZone("UTC"));
        timeString = utcSdf2.format(new java.util.Date(time));

        return timeString;
    }

//  --------------------------------------------------------------
    private void extractChpsData(final String moduleInstanceId, final String extractFileName)
    {
        final String commandString = _ohdBinDir + "/run_SSHPCHPS_data_extract " + moduleInstanceId + " "
            + extractFileName + " " + _rfcName + " " + _portNumber;
        
        ProcessBuilder pb = new ProcessBuilder(commandString.split(" "));
        pb.redirectOutput(ProcessBuilder.Redirect.DISCARD).redirectErrorStream(true);
        
        String result;
        String overall="";
        try {
            Process p = pb.start();
            p.waitFor();
            BufferedReader br = new BufferedReader(
                    new InputStreamReader(p.getInputStream()));
                while ((result = br.readLine()) != null){
                    overall = overall + "\n" + result;
                }
                p.destroy();
                //System.out.println(result);

        } catch (Exception e) {
            e.printStackTrace();
        }

    }

//  --------------------------------------------------------------
    private void translateToXml(final String basinId, final String extractFileName, final Document outputXmlDoc)
    {
        final File file = new File(extractFileName);

        final int sleepLimit = 15;
        int sleepCount = 0;

        boolean done = false;
        long oldLength = 0;
        long length = 0;

        while(!done && (sleepCount < sleepLimit))
        {
            try
            {
                Thread.sleep(1000);
            }
            catch(final InterruptedException e)
            {
                e.printStackTrace();
            }
            sleepCount++;

            if(file.exists())
            {
                length = file.length();
                if(length > 0)
                {
                    if(length == oldLength)
                    {
                        //_logger.log(Logger.DEBUG,"length = oldlength = " + length);
                        done = true;
                    }
                }
            }

            oldLength = length;
        }

        final SacXMLEncoder encoder = new SacXMLEncoder(basinId, extractFileName, outputXmlDoc, _logger);

        try
        {
            encoder.parse();
        }
        catch(final OfsFileParserException e)
        {
            exitOnError();
        }

    }

//	--------------------------------------------------------------

    private void exitOnError()
    {
        _logger.log(Logger.DEBUG, "Error parsing CHPS extract file");
        _logger.log(Logger.DEBUG, "Please check the CHPS extract files for any problems");
        _logger.log(Logger.DEBUG, "Exiting...");
        System.exit(0);
    }

//  --------------------------------------------------------------
//    private void sendXml(String xmlFilePath, String xmlFileName, String mhsId, String productId)
//    {
////        String header = "SshpDataTransferMgr.sendXml(): ";
//
//        /*
//         * # Send product to specific location # add the one-word description and the office id # #echo
//         * "Sending file:$FILENAME  product_ID:$PRODUCT_ID to *OFFICE* via distributeProduct" >> $LOGNAME
//         * #SUBJECT="*DESCRIPTION* $PRODUCT_ID" #/awips/fxa/bin/distributeProduct -c SHEFPRODUCT -s "$SUBJECT" -a
//         * *OFFICE* $PRODUCT_ID $FILENAME #RETURN_STATUS=$?
//         */
//
//        // String commandString = "/awips/fxa/bin/distributeProduct -c HYDRO_MODEL_DATA -a " +
//        //                         mhsId + " " + productId + " " + xmlFileName; 
//
//        EnvHelper envHelper = new EnvHelper();
//
//        String dirString = envHelper.getProperty("WHFS_BIN_DIR");
//        String commandString = dirString + "/run_SSHPCHPS_data_send " + xmlFilePath + " " + xmlFileName + " " + mhsId
//            + " " + productId + " " + "YES";
//
////        System.out.println(header + "making system call:" + commandString + ":");
//
//        try
//        {
//            Process process = Runtime.getRuntime().exec(commandString);
//            process.waitFor();
//        }
//        catch(IOException e)
//        {
//            e.printStackTrace();
//        }
//        catch(InterruptedException e)
//        {
//            e.printStackTrace();
//        }
//        return;
//    }

//  --------------------------------------------------------------

    public static void main(final String[] argStringArray)
    {
        if(argStringArray.length >= 6)
        {
            final String controlFilePath = argStringArray[0];
            final String extractedDirName = argStringArray[1];
            final String processedDirName = argStringArray[2];
            final String logDirName = argStringArray[3];
            final String rfcName = argStringArray[4];
            final String portNumber = argStringArray[5];

            //get the OHD BIN path 
            _ohdBinDir = System.getProperty("OHD_BIN_DIR");
            final SshpDataTransferMgr mgr = new SshpDataTransferMgr(rfcName,
                                                                    portNumber,
                                                                    controlFilePath,
                                                                    extractedDirName,
                                                                    processedDirName,
                                                                    logDirName);

            mgr.execute();
        }
        else
        {
            System.out.println("usage: ohd.hseb.sshp.message.SSHPDataTransferManager controlFilePath  extractedDirName processedDirName logDirName");
        }
    }

//  --------------------------------------------------------------

} //end class SshpDataTransferMgr
