package ohd.hseb.ohdutilities.sshp.sshpChpsExtractor;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import nl.wldelft.fews.system.pi.FewsPiService;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.io.ExceptionParser;

/**
 * @author Cham Pham
 * @since 12/21/2010
 */

public class ConfiguredQueryInputs
{
    FewsPiService _PiService = null;
    Logger _logger = null;
    boolean _printLog = false;
    Date _TO; //system time
    public Date _coldStateDate;
    private Map<String, String> _clientConfigMap = null;
    private String _taskId = null;

    /**
     * @param log
     * @param url - http://localHost:####/##rfc_pi/FewsPiService
     */
    public ConfiguredQueryInputs(final Logger log, final String url)
    {
        _logger = log;

        WrapFewsPiService.setLog(_logger);

        WrapFewsPiService.getFewsPiServiceImpl(url);

        _clientConfigMap = new HashMap<String, String>();

    }

    public String getSegmentTimeSeries(final String segId,
                                       final String clientId,
                                       final long startTime,
                                       final long endTime) throws Exception
    {
        this.getSystemTime();

        final String timeSeriesXML = _PiService.getTimeSeriesForSegment(segId,
                                                                        clientId,
                                                                        _TO,
                                                                        new Date(startTime),
                                                                        new Date(endTime),
                                                                        false);
        if(timeSeriesXML.isEmpty())
        {

            throw new Exception("ERROR- could not retrieve " + clientId + " Timeseries from database...");

        }

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "TimeSeriesXML-SEGMENTID: " + segId + " =====\n" + timeSeriesXML);
        }

        return timeSeriesXML;
    }

    public String getLocationIds(final String clientId) throws Exception
    {
        this.connectFewsPiService();

        final String filterId = _PiService.getFilters(null);

        final String locations = _PiService.getLocations(clientId, filterId, null);

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "===> LOCATIONS: " + locations);
        }

        return locations;
    }

    /**
     * This method will retrieve time series from Fews database
     * 
     * @param clientId - PiServiceConfig filename
     * @param timeseriesId - the id to retrieve ts data
     * @param startTime - starting time
     * @param endTime - ending time
     * @param paramIds - The list of Ts data type (i.e INFW, MAPE)
     * @param locIds - The list of location Ids
     * @param ensembleId - default NULL
     * @param ensembleIndex- default -1
     * @return String content of a PiTimeseries XML file
     * @throws Exception
     */
    public String getTimeSeriesStringXML(final String clientId,
                                         final String timeseriesId,
                                         final long startTime,
                                         final long endTime,
                                         final String[] paramIds,
                                         final String[] locIds,
                                         final String ensembleId,
                                         final int ensembleIndex) throws Exception
    {
        this.getSystemTime();

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "startTime = " + DateTime.getDateTimeStringFromLong(startTime, null));
            _logger.log(Logger.DEBUG, "       TO = " + DateTime.getDateTimeStringFromLong(_TO.getTime(), null));
            _logger.log(Logger.DEBUG, "  endTime = " + DateTime.getDateTimeStringFromLong(endTime, null));
            _logger.log(Logger.DEBUG, "clientId= " + clientId + " timsereisId= " + timeseriesId);

            for(int i = 0; i < paramIds.length; i++)
                _logger.log(Logger.DEBUG, "para = " + paramIds[i]);

            for(int i = 0; i < locIds.length; i++)
                _logger.log(Logger.DEBUG, " loc = " + locIds[i]);
        }

        final String timeSeriesXML = _PiService.getTimeSeries(clientId,
                                                              timeseriesId,
                                                              null,
                                                              new Date(startTime),
                                                              _TO,
                                                              new Date(endTime),
                                                              paramIds,
                                                              locIds,
                                                              ensembleId,
                                                              ensembleIndex,
                                                              false);
        if(timeSeriesXML == null || timeSeriesXML.isEmpty())
        {
            throw new Exception("ERROR- could not retrieve " + timeseriesId + " Timeseries from database...");

        }

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "TimeSeriesXML: =====\n" + timeSeriesXML);
        }

//         System.out.println("\nTimeSeries: \n" + timeSeriesXML);

        return timeSeriesXML;
    }

    /**
     * @param clientId
     * @param ensembleId
     * @return
     * @throws Exception
     */
    public int[] getEnsembleIndices(final String clientId, final String ensembleId) throws Exception
    {
        this.connectFewsPiService();

        return _PiService.getEnsembleMemberIndices(clientId, ensembleId);
    }

    /**
     * This method will retrieve module parameter from Fews database
     * 
     * @param clientId - PiServiceConfig filename
     * @param moduleId - the Id to retrieve parameter file
     * @return String content of the ModuleParameterSet file for requested ModuleState
     * @throws Exception
     */
    public String getModuleParameterStringXML(final String clientId, final String moduleId) throws Exception
    {
        this.connectFewsPiService();

        final String paramXML = _PiService.getModuleParameterSet(clientId, moduleId, null, -1);

        if(paramXML.isEmpty())
        {
            throw new Exception("ERROR- could not retrieve " + moduleId + " Parameter from database ...  ");

        }

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "ModuleParamXML ============\n" + paramXML);
        }

        return paramXML;
    }

    /**
     * This method will retrieve module parameter from Fews database
     * 
     * @param clientId - PiServiceConfig filename
     * @param moduleId - the Id to put back parameter string to fewsDB
     * @return String content of the ModuleParameterSet file for requested ModuleState
     * @throws Exception
     */
    public void putModuleParameterStringXML(final String clientId,
                                            final String moduleId,
                                            final String piParmSetXmlContent,
                                            final Date validStartTime,
                                            final Date validEndTime) throws Exception
    {
        this.connectFewsPiService();

        _PiService.putModuleParameterSet(clientId,
                                         moduleId,
                                         _taskId,
                                         piParmSetXmlContent,
                                         validStartTime,
                                         validEndTime,
                                         null,
                                         -1);
    }

    /**
     * This method will retrieve module states from fews database
     * 
     * @param clientId - PiServiceConfig filename
     * @param moduleId - the Id to retrieve parameter file
     * @param unzipDir - the path to place unzip stateI.txt and previous_params.txt files
     * @param ensembleId - default NULL
     * @param ensembleIndex - default -1
     * @throws Exception
     */
    public void getModuleStateBinary(final String clientId,
                                     final String moduleId,
                                     final String unzipDir,
                                     final String ensembleId,
                                     final int ensembleIndex) throws Exception
    {
        this.getSystemTime();

        final Date[] stateTimes = _PiService.getAvailableStateTimes(clientId, moduleId);

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "State times for Module " + moduleId);

            _logger.log(Logger.DEBUG, "Available State Times: " + stateTimes.length);
        }

        byte[] stateBinary = null;

        final int slotCarryover = stateTimes.length;

        Date stateDateTime = null;

        if(slotCarryover > 0)
        {
            stateDateTime = stateTimes[slotCarryover - 1];

            if(stateDateTime.getTime() <= _TO.getTime())
            {
                stateBinary = _PiService.getModuleStateBinary(clientId,
                                                              moduleId,
                                                              stateDateTime,
                                                              ensembleId,
                                                              ensembleIndex);
                if(stateBinary == null)
                {
                    final String ErrMessage = "ERROR- could not retrieve Model State id " + moduleId
                        + " from database ...  " + DateTime.getDateTimeStringFromLong(_TO.getTime(), null);

                    throw new Exception(ErrMessage);
                }

                setColdStateDate(stateDateTime);
            }
        }
        else
        {
            _logger.log(Logger.ERROR, "ERROR- getAvailableStateTimes equals " + stateTimes.length);
        }

        //unzip the module state 
        this.unzipStateByte(stateBinary, unzipDir);

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "FewsDB: " + DateTime.getDateTimeStringFromLong(stateDateTime.getTime(), null)
                + "  TO: " + DateTime.getDateTimeStringFromLong(_TO.getTime(), null));
        }
    }

    /**
     * Unzip state binary which retrieve from fews database (state binary contains statesI.txt and params_previous.xml
     * files)
     * 
     * @param byteArray
     * @param unzipDir
     */
    private void unzipStateByte(final byte[] byteArray, final String unzipDir)
    {
        try
        {
            final ZipInputStream zipInput = new ZipInputStream(new ByteArrayInputStream(byteArray));

            ZipEntry zip = zipInput.getNextEntry();

            final int BUFSIZE = 4096;

            while(zip != null)
            {
                final File f = new File(unzipDir + zip.getName());
                final FileOutputStream fout = new FileOutputStream(f);
                final byte inbuf[] = new byte[BUFSIZE];
                int n = 0;
                while((n = zipInput.read(inbuf, 0, BUFSIZE)) != -1)
                {
                    fout.write(inbuf, 0, n);
                }
                fout.close();
                zip = zipInput.getNextEntry();
            }

            zipInput.close();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.ERROR, e.getMessage());
            _logger.log(Logger.DEBUG, ExceptionParser.multiLineStackTrace(e));
        }

    }

    /**
     * This method will get cold state date
     * 
     * @return cold state date
     */
    public Date getColdStateDate()
    {
        return _coldStateDate;
    }

    /**
     * This method will reset cold state date
     * 
     * @param stateDate
     */
    public void setColdStateDate(final Date stateDate)
    {
        _coldStateDate = stateDate;
    }

    /**
     * This method will get current system time
     * 
     * @return the system time in Date
     * @throws Exception
     */
    public Date getSystemTime() throws Exception
    {
        this.connectFewsPiService();

        _TO = _PiService.getSystemTime("GMT");

        return _TO;
    }

    /**
     * This method will set current system time
     * 
     * @return the system time in Date
     * @throws Exception
     */
    public void setSystemTime(final Date newSystemTime) throws Exception
    {
        this.connectFewsPiService();

        _PiService.setSystemTime(newSystemTime);

    }

    /**
     * This method will connect to Pi Service
     * 
     * @throws Exception
     */
    private void connectFewsPiService() throws Exception
    {
        try
        {
            _PiService = WrapFewsPiService.getDefaultProxy();
        }
        catch(final Exception e)
        {
            _logger.log(Logger.FATAL, "ERROR- Service connection refused... Please check the PiServicePort");
            e.printStackTrace();
        }
    }

    /**
     * This method will obtain on command line when invoked work flow from Fews instead of execute fews GUI.
     * 
     * @param clientId - PiServiceConfig filename
     * @throws Exception
     */
    public void runningTask(final String clientId, final Date startTimeRun, final Date endTimeRun) throws Exception
    {//Need to use system.out for this methods. Don't need write informations into log file.
        this.connectFewsPiService();

        final String workFlow = clientId;

        _taskId = _PiService.createTask(clientId);

        final Date sysTime = _PiService.getSystemTime("");

        System.out.println("Starting task with id " + _taskId + " time=" + new Date(System.currentTimeMillis()));

        final String taskRunId = _PiService.runTask(clientId,
                                                    _taskId,
                                                    workFlow,
                                                    startTimeRun,
                                                    sysTime,
                                                    endTimeRun,
                                                    null,
                                                    null,
                                                    "OPT3 user",
                                                    "PiWebservice taskrun");

        final boolean result = _PiService.waitForTask(clientId, taskRunId, 120000); //30 seconds

        System.out.println("Task with id " + _taskId + (result ? " finished successfully" : " failed"));

        System.out.println("time=" + new Date(System.currentTimeMillis()));

    }

    public void cancelTask(final String clienId) throws Exception
    {
        this.connectFewsPiService();
        _PiService.cancelTask(clienId, _taskId);
    }

    /**
     * Parse the property format file and load the entries into {@link #_statesMap}. All *ModelState.java constructors
     * do not fill their states' values. It is this method that does from the input state file.
     */
    public void loadClientConfig(final String clientId, final String extensionClientFile) throws IOException, Exception
    {

        this.connectFewsPiService();

        final String clientConfigFileName = _PiService.getClientConfigFile(clientId, extensionClientFile);

        if(clientConfigFileName.isEmpty())
        {
            final String ErrMessage = "ERROR- could not retrieve the client config file " + clientId
                + " under /Config/PiClientConfigFiles dir ...  "
                + DateTime.getDateTimeStringFromLong(_TO.getTime(), null);

            throw new Exception(ErrMessage);
        }

        //read the client properties
        final Properties clientConfigProp = new Properties();

        final InputStream readClientConfigStream = new ByteArrayInputStream(clientConfigFileName.getBytes());

        clientConfigProp.load(readClientConfigStream);

        readClientConfigStream.close();

        //copy entries from client config String to the clientconfig map
        for(final Map.Entry<Object, Object> value: clientConfigProp.entrySet())
        {
            _clientConfigMap.put(value.getKey().toString(), value.getValue().toString());
        }

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "Loading the client config properties: " + clientConfigFileName);
        }

    }

    /**
     * Get the client config value as a String. Throws an Exception if the client config does not exist.
     */
    public String getClientConfigAsString(final String tag) throws Exception
    {
        if(isClientConfigExisting(tag) == false)
        {
            throw new Exception("The client config " + tag + " is not present.");
        }

        return _clientConfigMap.get(tag);
    }

    protected boolean isClientConfigExisting(final String tag)
    {
        if(_clientConfigMap.containsKey(tag))
        {
            return true;
        }

        return false;
    }

    public void setPrintLog(final boolean printLog)
    {
        _printLog = printLog;
    }
}
