package ohd.hseb.ohdutilities.sshp.sshpChpsExtractor;

import java.io.File;
import java.util.Date;
import java.util.Iterator;

import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.ohdmodels.unithg.UHGModelParameters;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.TimeHelper;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsAdapterDAO;
import ohd.hseb.util.fews.FewsXMLParser;
import ohd.hseb.util.fews.OHDConfigurationInfoReader;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ohdutilities.UtilityDriver;

/**
 * @author Cham Pham
 * @since 12/01/2010
 */

public class SshpChpsExtractorDriver extends UtilityDriver
{
    private boolean _printLog = false;
    private String _segId = null;
    private String _opName = null;
    private String _outputFileName = null;
    private final String _ensembleId = null;
    private final int _ensembleIndex = -1;

    //time in milliseconds
    static long _currStateTime;
    long _startTime;
    long _endTime;

    //clientid is PiServiceConfig filename; moduleId is used to retrieve module parameter and state
    private String _moduleId = null; //segmentId_operationName
    private String _clientId = null; //"SSHPDataTransfer";

    private final String _diagnosticsFile;

    private final Logger _logger = new Diagnostics();

    FewsXMLParser _xmlParser = null;

    private ConfiguredQueryInputs _queryInputs = null;

    private final SshpChpsExtractorParameters _sshpParameters = new SshpChpsExtractorParameters();

    private final SshpChpsExtractorStates _sshpStates = new SshpChpsExtractorStates();

    private final UHGModelParameters _unithgParams = new UHGModelParameters();

    // input TSs:
    private RegularTimeSeries _mapeTs = null; // optional TS

    private RegularTimeSeries _infwTs = null; // required TS

    private final String[] _paramIds = new String[]{"INFW", "MAPE"};

    /**
     * @param arguments - command lines (url, moduleinstanceid, output filename and printDebug)
     * @throws Exception
     */
    public SshpChpsExtractorDriver(final String[] arguments) throws Exception //String runInfoFilename, String segmentId, String rfcName, String outputFilename) throws Exception 
    {
        final OHDConfigurationInfoReader config = new OHDConfigurationInfoReader(_logger, "version/"
            + OHDConstants.OHD_FEWSADAPTER_CONFIG); //access nonsrc/version/ohdcommonchps_config.xml

        _logger.log(Logger.INFO, "Using OHD Fews Adapter version: " + config.getVersionControl().getVersion());

        _xmlParser = new FewsXMLParser(_logger);

        final String urlLink = arguments[0]; //URL (http://localHost:2001/##rfc_pi/FewsPiService)

        _moduleId = arguments[1]; //segment Id_operationName    

        final String[] splitModuleId = _moduleId.split("_");

        _segId = splitModuleId[0];

        _opName = splitModuleId[1];

        _outputFileName = arguments[2]; //output extract text filename

        _diagnosticsFile = _outputFileName + ".log"; //log file

        _clientId = arguments[3]; //client id means PiServiceConfig file name and it is the same as time series id

        _queryInputs = new ConfiguredQueryInputs(_logger, urlLink);

        int debugFlag = 0;

        if(arguments[4] != null && !arguments[4].isEmpty())
        {
            debugFlag = Integer.valueOf(arguments[4]);

            if(debugFlag > 0)
            {
                _logger.setPrintDebugInfo(debugFlag);
                _printLog = true;
                _queryInputs.setPrintLog(_printLog);
            }
        }
        else
        {
            throw new Exception("SSHP - Print debug flag passed by command line equals null/empty...");
        }

        if(_printLog)
        {
            _logger.log(Logger.DEBUG, "URL: " + urlLink);
            _logger.log(Logger.DEBUG, "segID: " + _segId);
            _logger.log(Logger.DEBUG, "output: " + _outputFileName);
        }
    }

    /**
     * This method reads UNITHG and SACSMA's parameter, states and time series from Fews database; and extracts the
     * parameters and carryover states for all SAC-SMA operations within a given moduleinstance id. It also extracts the
     * time series related to this operation.
     */
    @Override
    public void execute() throws Exception
    {
        SshpChpsExtractorText sshpAdapter = null;
        
        final String unzipDir = File.separator + "tmp" + File.separator;
        
        //**** Retrieve UNITHG parameter to compute start time that is
        //**** used by INFW
        final String unithgParamId = "UNITHG_" + _moduleId;

        if(_printLog)
        {
        	_logger.log(Logger.DEBUG, "==========================");
            _logger.log(Logger.DEBUG, "UnitHG param ID = " + unithgParamId);
        }

        // Retrieve parameter from moduleParameter
        // final String paramXML = _queryInputs.getModuleParameterStringXML(_clientId, unithgParamId);
        // _xmlParser.parseParametersFromStringXML(paramXML, _unithgParams);
       
        // Retrieve UNITHG module STATE from database -----------------------------------
        // This function call will return zip file that includes params_previous.xml and stateI.txt
        _queryInputs.getModuleStateBinary(_clientId, unithgParamId, unzipDir, _ensembleId, _ensembleIndex);
        	
        // Use parameter saved from the most recent run of the UpdateStates.
        _logger.log(Logger.INFO, "Retrieve UNITHG parameters from moduleState...");
        _xmlParser.parseParameters(unzipDir + "params_previous.xml", _unithgParams);
        

        final String[] _locIds = new String[]{_opName};

        // Get Module Id to retrieve parameter and states
        // Retrieve SACSMA PARAMETER from database ----------------------------------
        final String sacParamId = "SACSMA_" + _moduleId;

        if(_printLog)
        {
        	_logger.log(Logger.DEBUG, "==========================");        	
            _logger.log(Logger.DEBUG, "SACSMA param ID = " + sacParamId);
        }
        //final String paramStringXML = _queryInputs.getModuleParameterStringXML(_clientId, sacParamId);
        //_xmlParser.parseParametersFromStringXML(paramStringXML, _sshpParameters);
    	//_logger.log(Logger.DEBUG, "Retrieve SACSMA parameter from moduleParameter...");
        
        // Retrieve SACSMA module STATE from database -----------------------------------
        // This function call will return zip file that includes params_previous.xml and stateI.txt
        _queryInputs.getModuleStateBinary(_clientId, sacParamId, unzipDir, _ensembleId, _ensembleIndex);        
        
        // Parse params_previous.xml
        _logger.log(Logger.INFO, "Retrieve SACSMA parameters from moduleState..."); 
        _xmlParser.parseParameters(unzipDir + "params_previous.xml", _sshpParameters);
        
        
        // Get SACSMA statesI.txt
        _logger.log(Logger.INFO, "Retrieve SACSMA statesI.txt from moduleState...");
        _sshpStates.loadState(unzipDir + "statesI.txt", _logger);

        // States are validated
        _sshpStates.validateState(_sshpParameters);

        _currStateTime = _queryInputs._coldStateDate.getTime();

        _startTime = computeStartTimeMills();

        _endTime = computEndTimeMills();

        // Retrieve TIME SERIES from fews database
        final String tsStringXML = _queryInputs.getTimeSeriesStringXML(_clientId,
                                                                       "TimeSeries",
                                                                       _startTime,
                                                                       _endTime,
                                                                       _paramIds,
                                                                       _locIds,
                                                                       _ensembleId,
                                                                       _ensembleIndex);

        // -----------------------------------
        // store TIME SERIES to RegularTimeSeries -----------------------------------
        this.getTimeSeries(tsStringXML, _locIds[0]);

        /*------------------------------------------------------------------------ */
        sshpAdapter = new SshpChpsExtractorText(this);

        // Get extract data date/time (local time)
        sshpAdapter.getExtractDateTime();

        // Print segment name and operation name
        sshpAdapter.printSegmentandOpname();

        // Print PE time series (INFW and MAPE)
        sshpAdapter.printROtsPEts(_mapeTs, _infwTs);

        // Print Et demand time series
        sshpAdapter.printEtDemand();

        // Print parameters
        sshpAdapter.printParams();

        // Print carryover (states)
        sshpAdapter.printStates(_queryInputs._coldStateDate);

        // Write Timeseries, params and states information into
        sshpAdapter.writeTSParmsStateToTextFile(_outputFileName);

    } // close execute method --------------------------------------------------

    /**
     * Get the input time series (INFW - required and MAPE - optional)
     * 
     * @param tsStringXML
     * @param locationId
     * @throws Exception
     */
    private void getTimeSeries(final String tsStringXML, final String locationId) throws Exception
    {

        setAreMissingValuesAlwaysAllowed(true);

        _xmlParser.parseTimeSeriesFromStringXML(tsStringXML,
                                                getTsList(),
                                                getAreMissingValuesAlwaysAllowed(),
                                                _startTime);

        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        while(ite.hasNext())
        {
            final RegularTimeSeries inputRts = ite.next();
            //mandatory INFW ts
            if(inputRts.getTimeSeriesType().equals(DataType.NWSRFS_CHANNEL_INFLOW_DATATYPE)
                || inputRts.getTimeSeriesType().equals(DataType.CHANNEL_INFLOW_DATATYPE))
            {
                if(locationId.equalsIgnoreCase(inputRts.getLocationId()))
                {
                    _infwTs = inputRts;

                    _infwTs.trimTimeSeriesAtStartWithCheck(_startTime); //SD - unithgTime

                    _infwTs.trimTimeSeriesAtEndWithCheck(_currStateTime); //SD

                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Retrieve INFW time series: ");
                        _logger.log(Logger.DEBUG,
                                    _opName + " - StartTime: "
                                        + DateTime.getDateTimeStringFromLong(_infwTs.getStartTime(), null));
                        _logger.log(Logger.DEBUG,
                                    _opName + "         - EndTime: "
                                        + DateTime.getDateTimeStringFromLong(_infwTs.getEndTime(), null));
                    }
                }
            }
            //Optional MAPE ts
            else if(inputRts.getTimeSeriesType().equals(DataType.POTENTIAL_ET_DATATYPE))
            {
                if(_sshpParameters.useMAPEInput())
                {
                    _mapeTs = inputRts; // NWSRFS requires the interval be 24
                                        // hours
                    _mapeTs.trimTimeSeriesAtStartWithCheck(_currStateTime); //SD

                    _mapeTs.trimTimeSeriesAtEndWithCheck(_endTime);//setEndTime(_endTime); // +5 days after TO

                    _logger.log(Logger.DEBUG, "Using optional input MAPE time series:");

                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG,
                                    _opName + " - StartTime: "
                                        + DateTime.getDateTimeStringFromLong(_mapeTs.getStartTime(), null));
                        _logger.log(Logger.DEBUG,
                                    "         - EndTime: "
                                        + DateTime.getDateTimeStringFromLong(_mapeTs.getEndTime(), null));
                    }
                }
                else
                {// since not using it, delete it from _tsList
                    ite.remove();
                    _logger.log(Logger.DEBUG, "Not use optional input MAPE time series.");
                }
            }
            else
            {
                _logger.log(Logger.DEBUG, "The input time series type " + inputRts.getTimeSeriesType()
                    + " is not recognized. So it is ignored.");
                ite.remove();
            }
        } // close while loop
    }

    /**
     * Get Parameter
     * 
     * @return the global object
     */
    public SshpChpsExtractorParameters getParam()
    {
        return _sshpParameters;
    }

    /**
     * Get state
     * 
     * @return the global object
     */
    @Override
    public SshpChpsExtractorStates getState()
    {
        return _sshpStates;
    }

    /**
     * Get the segment Id
     * 
     * @return - the segment id
     */
    public String getSegId()
    {
        return _segId;
    }

    /**
     * Get operation name
     * 
     * @return the operation name
     */
    public String getOpName()
    {
        return _opName;
    }

    /**
     * Get current system time
     * 
     * @return the current system time (TO) is the date the states were retrieved for
     */
    public long getcurrentSystemTime()
    {
        return _currStateTime;
    }

    /**
     * Compute time based on UnitHG module
     * 
     * @return
     */
    private long computeUnitHGTime()
    {
//        _logger.log(Logger.INFO, "_unithgParams.getUHGOrdinatesAmount(): " + _unithgParams.getUHGOrdinatesAmount());
//        _logger.log(Logger.INFO, " _unithgParams.getUHGDurationInHours(): " + _unithgParams.getUHGDurationInHours());

        long uhgTimeMills = _unithgParams.getUHGOrdinatesAmount() * _unithgParams.getUHGDurationInHours();

        final long startAt12Z = uhgTimeMills % 24;

        if(startAt12Z != 0)
        {
            uhgTimeMills = uhgTimeMills - startAt12Z; //start at 12Z
        }

        return uhgTimeMills;
    }

    /**
     * Compute start time based on unitHg module
     * 
     * @return the start time in milliseconds
     */
    private long computeStartTimeMills()
    {
//        _logger.log(Logger.INFO,
//                    "start INFW: "
//                        + DateTime.getDateTimeStringFromLong((_currStateTime - (computeUnitHGTime() * TimeHelper.MILLIS_PER_HOUR)),
//                                                             null));
        return (_currStateTime - (computeUnitHGTime() * TimeHelper.MILLIS_PER_HOUR));

    }

    /**
     * Compute end time with 5 days after the date the states were retrieved for
     * 
     * @return the end time in milliseconds
     */
    private long computEndTimeMills()
    {
        return (_currStateTime + (5 * TimeHelper.MILLIS_PER_DAY));
    }

    /**
     * @return - The MAPE time series from list of input time series
     */
    RegularTimeSeries getMapeTs()
    {
        return _mapeTs;
    }

    /**
     * @return - The INFW (TCI) time series from list of input time series
     */
    RegularTimeSeries getInfwTs()
    {
        return _infwTs;
    }

    /**
     * @return the logger object
     */
    @Override
    public Logger getLogger()
    {
        return _logger;
    }

    /**
     * @return the flag to print out if level is DEBUG
     */
    public boolean isPrintLog()
    {
        return _printLog;
    }

    /**
     * Main function ==================================================
     */
    public static void main(final String[] args)
    {
        SshpChpsExtractorDriver sshpAdapter = null;

        if(args.length != 5)
        {
            System.err.println("Usage: ohd.hseb.ohdutilities.sshp.SshpChpsExtractorDriver URL segmentID outputFileName PiServiceConfigFile printDebug"
                + args.length + ".");
            System.exit(0);
        }

        try
        {
            sshpAdapter = new SshpChpsExtractorDriver(args);

            sshpAdapter._logger.log(Logger.DEBUG,
                                    "Running SSHP program from GMT time: " + new Date(System.currentTimeMillis()));

            /* Execute SSHP extract data ------------------------------- */
            sshpAdapter.execute();

            sshpAdapter._logger.log(Logger.DEBUG, "SSHP extract utility has successfully finished at "
                + new Date(System.currentTimeMillis()));
        }
        catch(final Exception e)
        {
            //sshpAdapter._logger.log(Logger.FATAL,
              //                      "ERROR- WebService connection refused. Please check the PiServicePort.");
            sshpAdapter._logger.log(Logger.FATAL, e.getMessage());
        }
        finally
        {
            try
            {
                /* ------ write diagnostics out to diagnostics file ------------- */
                FewsAdapterDAO.writeLog(sshpAdapter._logger, sshpAdapter._diagnosticsFile);
            }
            catch(final Exception e)
            {
                System.err.println("ERROR: SSHP: the diagnostics could not be printed to the diagnostic file: "
                    + e.getMessage());
                System.err.println("The contents that were to be logged are for diagnostics:"
                    + ((Diagnostics)sshpAdapter._logger).getListOfDiagnosticsAsString());
            }
        }
    }

}
