package ohd.hseb.ohdutilities.sshpdatatransfer;

import java.util.Iterator;

import ohd.hseb.db.DbTimeHelper;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.RUNOFF_DATATYPE;
import ohd.hseb.util.fews.ohdutilities.UtilityDriver;

/**
 * @author Cham Pham
 */
final public class SSHPDataExtractorDriver extends UtilityDriver
{

    /** alias to super._parameters to save type casting */
    private final SshpParameters _sshpParameters;

    /** alias to super._state to save type casting */
    private final SshpState _sshpState;

    //input TSs:
    private RegularTimeSeries _mapeTs = null; //optional, precipTs is the required and is the driving Ts
    private RegularTimeSeries _infwTs = null; //require
    private String _segId;
    private String _opName;
    private boolean _printLog = false;

    private String _outputDir;

    public SSHPDataExtractorDriver()
    {
        super(); //calling ModelDriver constructor

        super._state = new SshpState();
        super._parameters = new SshpParameters();

        _sshpParameters = (SshpParameters)super._parameters; // use alias to save casting

        _sshpState = (SshpState)super._state; // use alias to save casting

    } //close SshpDriver constructor class method -----------------------------

    @Override
    public void execute() throws Exception
    {

        if(_logger.getPrintDebugInfo() > 0)
        {
            _printLog = true;
        }

        /* return gracefully, errors are logged */
        runDriverValidation();

        _logger.log(Logger.DEBUG,
                    "Running SSHP program from GMT time: "
                        + DateTime.getDateTimeStringFromLong(getInitialStateTime()
                            + _tsList.get(0).getIntervalInMillis(), OHDConstants.GMT_TIMEZONE) + " to "
                        + DateTime.getDateTimeStringFromLong(getComputationEndTime(), OHDConstants.GMT_TIMEZONE));

        //Get root directory
        _outputDir = super.getDriverProperties().getProperty("outputDirectory");
        _segId = super.getDriverProperties().getProperty("segment");
        _opName = super.getDriverProperties().getProperty("operation");

        final SshpExtractText sshpAdapter = new SshpExtractText(this);

        //Get extract data date/time (local time)
        final String outputFile = sshpAdapter.getTextFilename();

        //Print segment name and operation name
        sshpAdapter.printSegmentandOpname();

        //Print PE time series (INFW and MAPE)        
        sshpAdapter.printROtsPEts(_mapeTs, _infwTs);

        //Print Et demand time series
        sshpAdapter.printEtDemand();

        //Print parameters
        sshpAdapter.printParams();

        //Print carryover (states)
        sshpAdapter.printStates();

        //Write Timeseries, params and states information into <segment_name>.YYYY_MM_DD_HHmm.txt 
        //file
        sshpAdapter.writeTSParmsStateToTextFile(outputFile);

        _logger.log(Logger.DEBUG, "SSHP extract utility has successfully finished.");

    } //close execute method --------------------------------------------------

    /**
     * Besides validation, also retrieve individual input time series from _tsList.
     */
    @Override
    protected void runDriverValidation() throws Exception
    {
        super.runDriverValidation(); //run super class first

        final Iterator<RegularTimeSeries> ite = getTsList().iterator();

        while(ite.hasNext())
        {
            final RegularTimeSeries inputRts = ite.next();

            if(RUNOFF_DATATYPE.isRunOffType(inputRts.getTimeSeriesType())
                && inputRts.getTimeSeriesType().equals(DataType.POTENTIAL_ET_DATATYPE) == false) //mandatory input TS
            {
                _infwTs = inputRts;
            }
            else if(inputRts.getTimeSeriesType().equals(DataType.POTENTIAL_ET_DATATYPE))
            {//optional input TS

                if(_sshpParameters.useMAPEInput())
                {
                    _mapeTs = inputRts;

                    /*
                     * NWSRFS requires the interval be 24 hours
                     */
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Using optional input MAPE time series.");

                    }
                }
                else
                {//since not using it, delete it from _tsList

                    ite.remove();
                    if(_printLog)
                    {
                        _logger.log(Logger.DEBUG, "Not use optional input MAPE time series.");
                    }
                }
            }
            else
            {
                if(_printLog)
                {
                    _logger.log(Logger.DEBUG, "The input time series type " + inputRts.getTimeSeriesType()
                        + " is not recognized. So it is ignored.");
                }

                ite.remove();
            }

        } //close while loop

        //states are validated
        _sshpState.validateState(_sshpParameters);

    } //close runDriverValidation method ---------------------------------

    /**
     * @return - The MAPE time series from list of input time series
     */
    RegularTimeSeries getMapeTs()
    {
        return _mapeTs;
    } //close getMapeTs method ------------------------------------------------

    /**
     * @return - The INFW (TCI) time series from list of input time series
     */
    RegularTimeSeries getInfwTs()
    {
        return _infwTs;
    } //close getInfwTs method ------------------------------------------------

    @Override
    public SshpParameters getParameters()
    {
        return _sshpParameters;
    } //close getParameters method --------------------------------------------

    @Override
    public SshpState getState()
    {
        return _sshpState;
    } //close getState method -------------------------------------------------

    /**
     * @return - the segment id
     */
    public String getSegId()
    {
        return _segId;
    } //close getSegId method -------------------------------------------------

    public String getOpName()
    {
        return _opName;
    } //close getSegId method -------------------------------------------------

    /**
     * @return - the output directory
     */
    public String getOutputDir()
    {
        return _outputDir;
    } //close getOutputDir method ---------------------------------------------

}
