package ohd.hseb.ohdutilities.sshpdatatransfer;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.fews.DataType;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.RUNOFF_DATATYPE;

import org.apache.commons.io.filefilter.WildcardFileFilter;

/**
 * @author Cham Pham
 */

class SshpExtractText
{
    private final SSHPDataExtractorDriver _fromModel;
    private final double[] _ettValues;
    private final SshpParameters _params;
    private final SshpState _states;

    /** Output directory */
    private final String _outputDir;
    private final String _segmentId;
    private final String _operationName;

    private final ArrayList<String> _extractList = new ArrayList<String>();

    /**
     * This method initializes SshpExtract constructor class
     * 
     * @param fromModel
     */
    SshpExtractText(final SSHPDataExtractorDriver fromModel)
    {
        _fromModel = fromModel; //alias, both are the same thing

        _outputDir = _fromModel.getOutputDir() + "/";
        _segmentId = _fromModel.getSegId();
        _operationName = _fromModel.getOpName();

        _params = _fromModel.getParameters();
        _states = _fromModel.getState();
        _ettValues = _fromModel.getParameters().getEtDemand();

    } //close SshpExtract constructor class method -----------------------------

    public String getTextFilename() throws IOException
    {
        String outputFile = null;

        final String dateStr1 = dateSF("yyyy_MM_dd_HH", 0);

        final String dateStr2 = dateSF("yyyy_MM_dd_HH_mm_ss", 0);

        final File dir = new File(_outputDir);
        final FileFilter fileFilter = new WildcardFileFilter(_segmentId + "." + dateStr1 + "*.txt");
        final File[] files = dir.listFiles(fileFilter);

        // if could not find "segId.yyyy_MM_dd_HH_ss*.txt"
        if(files.length <= 0)
        {
            outputFile = _outputDir + _segmentId + "." + dateStr2 + ".txt";

            //Get extract data date/time (local time)
            getExtractDateTime();

        }
        else
        {
            for(int i = 0; i < files.length; i++)
            {
                outputFile = files[i].getPath();
                System.out.println(outputFile);
            }

        }

        return outputFile;

    } //Close getTextFilename class method -------------------------------------

    /**
     * This method writes out Parameters and state values into format text file
     * 
     * @throws IOException
     */
    public void writeTSParmsStateToTextFile(final String oFilename) throws IOException
    {
        System.out.printf("\nOutput text file: %s\n\n", oFilename);

        final PrintWriter outFile = new PrintWriter(new BufferedWriter(new FileWriter(oFilename, true)));

        for(final String sb: _extractList)
        {
            outFile.write(sb);
        }

        outFile.flush();
        outFile.close();

    } //close writeTSParmsStateToTextFile method -------------------------------

    /**
     * This method writes out string date format depending on string sf
     * 
     * @param sf
     * @param ldate
     * @return
     */
    private String dateSF(final String sf, final long ldate)
    {
        final DateFormat dateFormat = new SimpleDateFormat(sf);
        Date date;

        if(ldate > 0)
        {
            dateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);
            date = new Date(ldate);
        }
        else
        {
            date = new Date();
        }

        return (dateFormat.format(date));

    } //close dateSF method ----------------------------------------------------

    /**
     * Get extracted utility runs date/time (local time)
     */
    public void getExtractDateTime()
    {
        final String datetime = dateSF("MM/dd/yyyy HHmm", 0);

        System.out.printf("\nExtracted on " + datetime + "Z\n");

        _extractList.add(String.format("Extracted on " + datetime + "Z\n"));

    } //close getExtractDateTime method ----------------------------------------

    /**
     * print out segment name and operation name
     */
    public void printSegmentandOpname()
    {
        System.out.printf("SEGID=%-8s OPNAM=%s\n", _segmentId, _operationName);

        _extractList.add(String.format("SEGID=%-8s OPNAM=%s\n", _segmentId, _operationName));

    } //close getSegmentandOpname method ---------------------------------------

    /**
     * This method will printout time series into format text file.
     * 
     * @param typeTs
     * @param Ts
     * @throws SshpException
     */
    private void printTimeSerires(final String typeTs, final RegularTimeSeries Ts) throws Exception
    {
        System.out.printf("%s Time Series Begin\n", typeTs);
        _extractList.add(String.format("%s Time Series Begin\n", typeTs));

        if(Ts != null)
        {
            int measureMentCount = Ts.getMeasurementCount();
            final MeasuringUnit regulationUnit = Ts.getMeasuringUnit();

            // If FEWs sends first time step is missing value, then ignore it
            if(Ts.getMeasurementValueByIndex(0, regulationUnit) == Ts.getMissingValue())
            {
                Ts.trimTimeSeriesAtStart(Ts.getMeasurementTimeByIndex(1));
                measureMentCount = Ts.getMeasurementCount();
            }

            // Get beginning date/time from GMT time for MAPE/INFW time series.                  
            final long stTime = Ts.getStartTime();
            final String begTime = dateSF("MM/dd/yyyy HH", stTime);

            _extractList.add(String.format("Identifier=%-8s Data Type=%-4s Time Step=%2d Data Units=%s\n",
                                           Ts.getLocationId(),
                                           Ts.getTimeSeriesType(),
                                           Ts.getIntervalInHours(),
                                           Ts.getMeasuringUnit()));
            _extractList.add(String.format("Beginning Time=%sZ    Value Count=%4d\n", begTime, measureMentCount));
            _extractList.add("Values=\n");

            System.out.printf("Identifier=%-8s Data Type=%-4s Time Step=%2d Data Units=%s\n",
                              Ts.getLocationId(),
                              Ts.getTimeSeriesType(),
                              Ts.getIntervalInHours(),
                              Ts.getMeasuringUnit());
            System.out.printf("Beginning Time=%sZ    Value Count=%4d\n", begTime, measureMentCount);
            System.out.printf("Values=\n");

            for(int index = 0; index < measureMentCount; index++)
            {
                final double tsValue = Ts.getMeasurementValueByIndex(index, regulationUnit);

                System.out.printf("%9.4f ", tsValue);
                _extractList.add(String.format("%9.4f ", tsValue));

                if(((index + 1) % 5) == 0)
                {
                    System.out.printf("\n");
                    _extractList.add("\n");
                }
                else
                {
                    if(index == measureMentCount - 1)
                    {
                        System.out.printf("\n");
                        _extractList.add("\n");
                    }
                }

            }
        }

        System.out.printf("%s Time Series End\n", typeTs);
        _extractList.add(String.format("%s Time Series End\n", typeTs));

    } //close printTimeSerires method ------------------------------------------

    /**
     * This method will print out RO (INFW) or PE (MAPE) time series
     * 
     * @param PE - MAPE
     * @param RO - INFW
     * @throws SshpException
     */
    public void printROtsPEts(final RegularTimeSeries PE, final RegularTimeSeries RO) throws Exception
    {
        if(RO.getTimeSeriesType().equals(RUNOFF_DATATYPE.TCI.getTypeName()))
        {
            RO.setTimeSeriesType(RUNOFF_DATATYPE.INFW.getTypeName());
        }

        if(!RO.isEmpty())
        {

            printTimeSerires("RO", RO);
        }
        else
        {
            System.out.printf("RO Time Series Begin\n");
            _extractList.add(String.format("RO Time Series Begin\n"));
            System.out.printf("No %s time series found for Identifier=%-8s Data Type=%s\n",
                              "RO",
                              RO.getLocationId(),
                              RO.getTimeSeriesType());
            _extractList.add(String.format("No %s time series found for Identifier=%-8s Data Type=%s\n",
                                           "RO",
                                           RO.getLocationId(),
                                           RO.getTimeSeriesType()));
            System.out.printf("RO Time Series End\n");
            _extractList.add(String.format("RO Time Series End\n"));

        }

        if(_params.useMAPEInput() == true && PE != null)
        {
            printTimeSerires("PE", PE);
        }
        else
        {
            final String typeid = "PE";

            System.out.printf("PE Time Series Begin\n");
            _extractList.add(String.format("PE Time Series Begin\n"));

            try
            {
                if(PE.getTimeSeriesType().equals(DataType.POTENTIAL_ET_DATATYPE))
                {

                    System.out.printf("No %s time series found for Identifier=%s Data Type=%s\n",
                                      typeid,
                                      PE.getLocationId(),
                                      PE.getTimeSeriesType());
                    _extractList.add(String.format("No %s time series found for Identifier=%s Data Type=%s\n",
                                                   typeid,
                                                   PE.getLocationId(),
                                                   PE.getTimeSeriesType()));
                }

            }
            catch(final NullPointerException e)
            {
                System.out.printf("No %s Time Series stored\n", typeid);
                _extractList.add(String.format("No %s Time Series stored\n", typeid));
            }

            System.out.printf("%s Time Series End\n", typeid);
            _extractList.add(String.format("%s Time Series End\n", typeid));
        }

    } //close printROtsPEts method --------------------------------------------

    /**
     * Print out ET-demand time series or PE adjust if MAPE input time series is used
     */
    public void printEtDemand()
    {
        System.out.printf("Monthly Values Begin\n");
        _extractList.add("Monthly Values Begin\n");

        // If MAPE input ts is used then values are PE adjustment.
        if(_params.useMAPEInput() == true)
        {
            System.out.printf("Value Type=PE Adjustment\n");
            _extractList.add("Value Type=PE Adjustment\n");
        }
        else
        {
            System.out.printf("Value Type=ET-Demand\n");
            _extractList.add("Value Type=ET-Demand\n");
        }

        for(int i = 0; i < 12; i++)
        {
            System.out.printf("%6.1f", _ettValues[i]);
            _extractList.add(String.format("%6.1f", _ettValues[i]));

        }

        System.out.printf("\nMonthly Values End\n");
        _extractList.add("\nMonthly Values End\n");

    } // close printEtDemand method --------------------------------------------

    /**
     * Print out Parameter values
     */
    public void printParams()
    {
        _extractList.add("SAC Params Begin\n");

        _extractList.add(String.format("PXADJ =%8.3f\n", _params.getPxAdj()));

        _extractList.add(String.format("PEADJ =%8.3f\n", _params.getPeAdj()));

        _extractList.add(String.format("UZTWM =%8.3f\n", _params.getUztwm()));

        _extractList.add(String.format("UZFWM =%8.3f\n", _params.getUzfwm()));

        _extractList.add(String.format("UZK   =%8.3f\n", _params.getUzk()));

        _extractList.add(String.format("PCTIM =%8.3f\n", _params.getPctim()));

        _extractList.add(String.format("ADIMP =%8.3f\n", _params.getAdimp()));

        _extractList.add(String.format("RIVA  =%8.3f\n", _params.getRiva()));

        _extractList.add(String.format("ZPERC =%8.3f\n", _params.getZperc()));

        _extractList.add(String.format("REXP  =%8.3f\n", _params.getRexp()));

        _extractList.add(String.format("LZTWM =%8.3f\n", _params.getLztwm()));

        _extractList.add(String.format("LZFSM =%8.3f\n", _params.getLzfsm()));

        _extractList.add(String.format("LZFPM =%8.3f\n", _params.getLzfpm()));

        _extractList.add(String.format("LZSK  =%8.3f\n", _params.getLzsk()));

        _extractList.add(String.format("LZPK  =%8.3f\n", _params.getLzpk()));

        _extractList.add(String.format("PFREE =%8.3f\n", _params.getPfree()));

        _extractList.add(String.format("RSERV =%8.3f\n", _params.getRserv()));

        _extractList.add(String.format("SIDE  =%8.3f\n", _params.getSide()));

        _extractList.add(String.format("IOPTET=%d\n", 0));

        _extractList.add(String.format("EFC   =%8.3f\n", _params.getEfc()));

        _extractList.add("SAC Params End\n");

        System.out.printf("SAC Params Begin\n");
        System.out.printf("PXADJ =%8.3f\n", _params.getPxAdj());
        System.out.printf("PEADJ =%8.3f\n", _params.getPeAdj());
        System.out.printf("UZTWM =%8.3f\n", _params.getUztwm());
        System.out.printf("UZFWM =%8.3f\n", _params.getUzfwm());
        System.out.printf("UZK   =%8.3f\n", _params.getUzk());
        System.out.printf("PCTIM =%8.3f\n", _params.getPctim());
        System.out.printf("ADIMP =%8.3f\n", _params.getAdimp());
        System.out.printf("RIVA  =%8.3f\n", _params.getRiva());
        System.out.printf("ZPERC =%8.3f\n", _params.getZperc());
        System.out.printf("REXP  =%8.3f\n", _params.getRexp());
        System.out.printf("LZTWM =%8.3f\n", _params.getLztwm());
        System.out.printf("LZFSM =%8.3f\n", _params.getLzfsm());
        System.out.printf("LZFPM =%8.3f\n", _params.getLzfpm());
        System.out.printf("LZSK  =%8.3f\n", _params.getLzsk());
        System.out.printf("LZPK  =%8.3f\n", _params.getLzpk());
        System.out.printf("PFREE =%8.3f\n", _params.getPfree());
        System.out.printf("RSERV =%8.3f\n", _params.getRserv());
        System.out.printf("SIDE  =%8.3f\n", _params.getSide());
        System.out.printf("IOPTET=%8d\n", 0);
        System.out.printf("EFC   =%8.3f\n", _params.getEfc());
        System.out.printf("SAC Params End\n");

    } //printParams method -----------------------------------------------------

    /**
     * Print out carryover (states) values
     */
    public void printStates()
    {
        //Get the most recent carryover date
        final long sTime = _fromModel.getSwitchFromObservedToForecastTime(); // getInitialStateTime();

        if(sTime > 0)
        {
            final String stateTime = dateSF("MM/dd/yyyy HH", sTime);
            _extractList.add("SAC State Begin\n");

            _extractList.add(String.format("Carryover date= %sZ\n", stateTime));

            _extractList.add(String.format("UZTWC=%7.1f\n", _states.getUztwc()));

            _extractList.add(String.format("UZFWC=%7.1f\n", _states.getUzfwc()));

            _extractList.add(String.format("LZTWC=%7.1f\n", _states.getLztwc()));

            _extractList.add(String.format("LZFSC=%7.1f\n", _states.getLzfsc()));

            _extractList.add(String.format("LZFPC=%7.1f\n", _states.getLzfpc()));

            _extractList.add(String.format("ADIMC=%7.1f\n", _states.getAdimc()));

            _extractList.add("SAC State End\n");

            System.out.printf("SAC State Begin\n");
            System.out.printf("Carryover date= %sZ\n", stateTime);
            System.out.printf("UZTWC=%7.1f\n", _states.getUztwc());
            System.out.printf("UZFWC=%7.1f\n", _states.getUzfwc());
            System.out.printf("LZTWC=%7.1f\n", _states.getLztwc());
            System.out.printf("LZFSC=%7.1f\n", _states.getLzfsc());
            System.out.printf("LZFPC=%7.1f\n", _states.getLzfpc());
            System.out.printf("ADIMC=%7.1f\n", _states.getAdimc());
            System.out.printf("SAC State End\n");
        }
        else
        {
            System.out.printf("No SAC States");
            _extractList.add("No SAC States");
        }

    } //close printStates method ---------------------------------------------- 
}