package ohd.hseb.time;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;

import javax.management.timer.Timer;

import ohd.hseb.util.HashUtility;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;

public class DateTime
{

    private static final int MILLIS_PER_SECOND = 1000;

    private String _dateString;
    private String _timeString;
    private String _dateTimeString;
    private long _timeInMillis; //milliseconds since January 1st, 1970, _startOfDay, GMT
    private TimeZone _timeZone = OHDConstants.GMT_TIMEZONE;

    /*
     * default is like NWSRFS forecasting mode: start of a day is 12Z. Note: when calculating epoch
     * time(getTimeInMillis()), it is still from 1970-01-01 00:00:00, not 12:00:00
     */
    private int _startOfDay = OHDConstants.NWSRFS_START_OF_DAY;

    public final static long DEFAULT_DATE_TIME_LONG = 0;

    private final FasterCustomDateTimeParser _parser = new FasterCustomDateTimeParser();

    private static final ThreadLocal<SimpleDateFormat> _gmtSdfThreadLocal = new ThreadLocal<SimpleDateFormat>()
    {
        @Override
        public SimpleDateFormat initialValue()
        {
            return OHDUtilities.getSdfGMT(OHDConstants.DATE_TIME_FORMAT_STR);
        }
    };

    static final ThreadLocal<SimpleDateFormat> _Date_Time_FormatterThreadLocal = new ThreadLocal<SimpleDateFormat>()
    {
        @Override
        public SimpleDateFormat initialValue()
        {
            return new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);
        }
    };

    static final ThreadLocal<SimpleDateFormat> _Date_Formatter = new ThreadLocal<SimpleDateFormat>()
    {
        @Override
        public SimpleDateFormat initialValue()
        {
//            System.out.println("_dateFormatter.initialValue() called");
            return new SimpleDateFormat(OHDConstants.DATE_FORMAT_STR);
        }
    };

    static final ThreadLocal<SimpleDateFormat> _Time_Formatter = new ThreadLocal<SimpleDateFormat>()
    {
        @Override
        public SimpleDateFormat initialValue()
        {
            return new SimpleDateFormat(OHDConstants.TIME_FORMAT_STR);
        }
    };

    public DateTime()
    {
        setTime(0);
    }

    /**
     * Take date String and time String to construct the object. The time zone is the default one(GMT).
     * 
     * @param dateStr - yyyy-MM-dd format
     * @param timeStr - HH:MM:SS format
     * @throws Exception
     */
    public DateTime(final String dateStr, final String timeStr) throws Exception
    {
        this(dateStr, timeStr, OHDConstants.GMT_TIMEZONE);
    }

    /**
     * Take date String and time String to construct the object.
     * 
     * @param dateStr - yyyy-MM-dd format
     * @param timeStr - HH:MM:SS format
     * @throws Exception
     */
    public DateTime(final String dateString, final String timeString, final TimeZone timeZone) throws Exception
    {
        _dateString = dateString;
        _timeString = timeString;
        _timeZone = timeZone;

        _dateTimeString = _dateString + " " + _timeString;
        _timeInMillis = calculateDateTimeLong(_dateTimeString, _timeZone);
    }

    /**
     * Construct an obj. which has the year, month, day and hour as the parameters respectively. Minutes, Seconds and
     * Milliseconds are 0.
     * <p>
     * 
     * @param month - 1 for Jan., 2 for Feb. etc
     * @param hour - can take 0 or 24. DateTime(2005, 2, 18, 24) and DateTime(2005, 2, 19, 0) are the same. Their
     *            {@link #getTimeInMillis()} are equal.
     */
    public DateTime(final int year, final int month, final int day, final int hour)
    {

        long dateTimeLong = 0;

        try
        {
            // final String dateTimeString = String.format("%04d-%02d-%02d %02d:%02d:%02d", year, month, day, hour, 0, 0, 0);
            //dateTimeLong = calculateDateTimeLong(dateTimeString, OHDConstants.GMT_TIMEZONE);
            dateTimeLong = _parser.getUniversalTimeInSeconds(year, month, day, hour, 0, 0) * (MILLIS_PER_SECOND);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Error(e.getMessage());
        }

        setTime(dateTimeLong);
    }

    /**
     * Take the milliseconds from the Epoch time(1970-01-01 00:00:00 GMT) and construct an obj. which has the right
     * Year, Month, DayOfMonth and Hour. Minute, Second and Millisecond are set to 0. This constructor maybe very useful
     * since milliseconds from the Epoch time is available in most places.
     * 
     * @param millis
     */
    public DateTime(final long millis)
    {
        setTime(millis);
    }

    public DateTime(final long millis, final TimeZone timeZone)
    {
        setTime(millis, timeZone);
    }

    /**
     * This constructor takes the second parameter to set {@link #_startOfDay}. However, the second parameter does not
     * affect the milliseconds stored internally({@link #_timeInMillis}).
     * 
     * @param millis
     * @param startOfDay - For NWSRFS, it should be 12.
     */
    public DateTime(final long millis, final int startOfDay)
    {
        this(millis);

        setStartOfDay(startOfDay);

    }

    public void setTime(final long timeInMillis)
    {

        setTime(timeInMillis, null);

        return;
    }

    public void setTime(final long timeInMillis, final TimeZone timeZone)
    {

        _timeInMillis = timeInMillis;
        _dateTimeString = getDateTimeStringFromLong(timeInMillis, timeZone);

        setDateAndTimeStringMembers();

        return;
    }

    /**
     * Set to the current time
     */
    public void setToCurrentTime()
    {
        final Calendar cal = new GregorianCalendar(_timeZone); //default is GMT
        this.setTime(cal.getTimeInMillis());
    }

    private void setDateAndTimeStringMembers()
    {
        _dateString = _dateTimeString.substring(0, 10);
        _timeString = _dateTimeString.substring(11, 19);
    }

    /**
     * Parse a String in a format of {@link OHDConstants#DATE_TIME_FORMAT_STR} and return an obj. of DateTime. It is
     * assumed that this is GMT time zone. For Strings like "yyyy-MM-dd HH:mm:ss GMT" or "yyyy-MM-dd HH:mm:ssZ", the
     * extra chars after "yyyy-MM-dd HH:mm:ss" are not used at all.
     * 
     * @param representation
     * @throws ParseException
     */
    public static DateTime parseYYYYMMDDHHZ(final String representation) throws ParseException
    {
        return new DateTime(_gmtSdfThreadLocal.get().parse(representation).getTime());
    }

    /**
     * Parse a String "MMDDYYYYHH" and returns a DateTime obj. The last letter "Z" is not needed.
     * 
     * @param representation
     * @throws ParseException
     */
    public static DateTime parseMMDDYYYYHHZ(final String representation) throws ParseException
    {
        final int month = Integer.parseInt(representation.substring(0, 2));
        final int day = Integer.parseInt(representation.substring(2, 4));
        final int year = Integer.parseInt(representation.substring(4, 8));
        final int hour = Integer.parseInt(representation.substring(8, 10));
        return new DateTime(year, month, day, hour);
    }

    /**
     * returns a DateTime obj. which is a number of hours from January 1, 1900 at 1200 UTC
     */
    public static DateTime convertFromJulianHoursRelativeTo12Z(final int julianHours)
    {
        final DateTime januaryFirst1900 = new DateTime(1900, 1, 1, 12);
        januaryFirst1900.addHours(julianHours);

        return januaryFirst1900;
    }

    /**
     * Take String with the format of "yyyy-MM-dd HH:MM:SS" and the time zone to convert to long.
     */
    private long calculateDateTimeLong(final String dateTimeString, final TimeZone timeZone) throws Exception
    {
        long returnValue;

        if((timeZone == null) || (timeZone.equals(OHDConstants.GMT_TIMEZONE)))
        {
            //parses GMT time only
            returnValue = _parser.parse(dateTimeString);
        }
        else
        //parse non-GMT times
        {
            final SimpleDateFormat dateFmt = _Date_Time_FormatterThreadLocal.get();
            dateFmt.setTimeZone(timeZone);
            returnValue = dateFmt.parse(dateTimeString).getTime();
        }

        return returnValue;
    }

    /**
     * Return the hour of the day, from 0 to 23. Note: no 24th hour.
     */
    public int hour()
    {
        final int hour = Integer.valueOf(getTimeString().substring(0, 2));
        return hour;
    }

    /**
     * Returns the hour of the day, from 1 to 24. (24 o'clock is equal to 0 o'clock next day.), relative to
     * {@link #_startOfDay}.
     */
    public int getNwsrfsHour()
    {
        int hour = this.hour();

        if(hour < _startOfDay)
        {
            hour += 24;
        }

        hour -= _startOfDay;

        if(hour == 0)
        {
            hour = 24;
        }

        return hour;
    }

    public int year()
    {
        final int year = Integer.valueOf(getDateString().substring(0, 4));
        return year;
    }

    /**
     * Unlike Java convention, here 1 stands for Jan.
     */
    public int month()
    {
        final int month = Integer.valueOf(getDateString().substring(5, 7));
        return month;
    }

    public int dayOfMonth()
    {

        final int day = Integer.valueOf(getDateString().substring(8));
        return day;

    }

    /**
     * Add several hours to current obj. The parameter can be positive or negative, causing increasing or decreasing the
     * hour respectively. If the hour goes out of the range [0,23], the DayOfMonth, or even Month, Year will be adjusted
     * automatically.
     */
    public void addHours(final int hours)
    {
        setTime(getTimeInMillis() + (hours * Timer.ONE_HOUR));

    }

    /**
     * Add hours to a copy of the current obj. The parameter can be positive or negative, causing increasing or
     * decreasing the hour respectively. If the hour goes out of the range [0,23], the DayOfMonth, or even Month, Year
     * will be adjusted automatically. Returns a new object
     */
    public DateTime addHoursTo(final int hours)
    {
        final DateTime sdt = new DateTime(getTimeInMillis());
        sdt.addHours(hours);

        return sdt;
    }

    /**
     * Add days to a copy of the current obj. The parameter can be positive or negative.
     */
    public DateTime addDaysTo(final int days)
    {
        final DateTime sdt = new DateTime(getTimeInMillis());
        sdt.addHours(days * 24);

        return sdt;
    }

    /**
     * Add a number of months to a copy of the current obj. The parameter can be positive or negative.
     */
    public DateTime addMonthsTo(final int numMons)
    {
        int year = this.year();

        int newMonth = this.month() + numMons;

        if(newMonth > 12)
        {//move into next year
            newMonth -= 12;
            year += 1;
        }
        else if(newMonth <= 0)
        {//shift back one year
            newMonth += 12;
            year -= 1;
        }

        final DateTime sdt = new DateTime(year, newMonth, this.dayOfMonth(), this.hour());
        return sdt;
    }

    public String flatten()
    {
        final String monthPart = twoRightCharacters("0" + month());
        final String dayPart = twoRightCharacters("0" + dayOfMonth());
        final String yearPart = String.valueOf(year());
        final String hourPart = twoRightCharacters("0" + hour());

        return monthPart + dayPart + yearPart + hourPart;
    }

    private String twoRightCharacters(final String string)
    {
        return string.substring(string.length() - 2);
    }

    /**
     * Returns a String representing Date and Time which format is based on the parameter dateTimePattern, in GMT.
     * 
     * @param dateTimePattern - if {@link OHDConstants#DATE_TIME_FORMAT_STR}, return like "2002-07-01 12:30:00"; if
     *            {@link OHDConstants#DATE_FORMAT_STR}, return like "2002-07-01"; if
     *            {@link OHDConstants#TIME_FORMAT_STR}, return like "12:30:00"; if "yyyyMMddHH", return like
     *            "2002070112"; if "yyyyMMddHHmm", return like "200207011230"
     */
    public String toString(final String dateTimePattern)
    {
        final SimpleDateFormat simpleDateFormat = OHDUtilities.getSdfGMT(dateTimePattern);
        return simpleDateFormat.format(new Date(this._timeInMillis));
    }

    @Override
    public boolean equals(final Object otherObject)
    {
        if(otherObject == null)
        {
            return false;
        }

        if(otherObject.getClass() != DateTime.class)
        {
            return false;
        }

        final DateTime otherDateTime = (DateTime)otherObject;

        return getTimeInMillis() == otherDateTime.getTimeInMillis();

    }

    /**
     * Compute the number of days between two DateTime. e.g. 1997-04-27 12:00:00 GMT is 2 dayUntil 1997-04-29 12:00:00,
     * but is 1 dayUntil 1997-04-29 11:00:00 GMT GMT
     */
    public int daysUntil(final DateTime other)
    {

        final long diffInMillis = other.getTimeInMillis() - this.getTimeInMillis();

        return (int)(diffInMillis / Timer.ONE_DAY);

    }

    /**
     * Return Julian hours(hours since January 1, 1900 at 00:00 UTC). Note: the returned number is related to
     * {@link #_startOfDay} value.
     */
    public int convertToJulianHours()
    {

        final DateTime januaryFirst1900 = new DateTime(1900, 1, 1, 0);

        return januaryFirst1900.daysUntil(this) * 24 + hour() - _startOfDay;
    }

    /**
     * returns a DateTime (always in Z - UTC) to a JulianDayAndHour object; the julian day is assumed to start at 12Z
     * and a 1 - 24 hour clock is used
     * 
     * @param SimpleDateAndTie dateAndTime
     * @return JulianDayAndHour
     */

    public JulianDayAndHour convertDateTimeToJulianDayAndHour()
    {
        final long dateTimeInJulianHours = convertToJulianHours();

        long julianDay = (dateTimeInJulianHours / 24) + 1;
        long julianHour = dateTimeInJulianHours % 24;

        if(julianHour == 0)
        {
            julianHour = 24;
            julianDay -= 1;
        }

        return new JulianDayAndHour(julianDay, julianHour);
    }

    /**
     * Returns the hours between two DateTime objs. For example, "07/01/2002 12z" has 120 hoursUntil "07/06/2002 12z".
     * 
     * @param throughTime
     */
    public int hoursUntil(final DateTime throughTime)
    {
//        System.out.println("HOURS UNTIL: " + (throughTime.convertToJulianHours() - this.convertToJulianHours()));
        return throughTime.convertToJulianHours() - this.convertToJulianHours();
    }

    public int[] convertToRelativeHours(final DateTime[] times)
    {
        final int[] hours = new int[times.length];
        for(int i = 0; i < hours.length; i++)
        {
            hours[i] = this.hoursUntil(times[i]);
        }
        return hours;
    }

    public DateTime getPreviousSynopticTime(final int synopticTimeInterval)
    {
        final int hoursToPreviousSynopticTime = this.hour() % synopticTimeInterval;

        final DateTime sdt = new DateTime(getTimeInMillis());

        sdt.addHours(-1 * hoursToPreviousSynopticTime);

        return sdt;
    }

    public long getTimeInMillis()
    {
        return _timeInMillis;
    }

    /**
     * Return the start of the day. For NWSRFS, it is 12Z. For our daily common sense, it is 0Z, which is also the
     * default value.
     */
    public int getStartOfDay()
    {
        return _startOfDay;
    }

    /**
     * Set the start of the day. For NWSRFS, it is 12Z. For our daily common sense, it is 0Z. The default value is
     * {@link OHDConstants#NWSRFS_START_OF_DAY}.
     */
    public void setStartOfDay(final int startOfDay)
    {
        _startOfDay = startOfDay;
    }

    @Override
    public int hashCode()
    {

        return _dateTimeString.hashCode();

    }

    /**
     * A helper method. Returns a string like "YYYY-MM-DD", or "HH:MM:SS", or "YYYY-MM-DD HH:MM:SS", depending on the
     * parameter sdf.
     * <p>
     * Note: the string returned is related to the parameter timeZone, even though the returned string does not show the
     * time zone information.
     * 
     * @param time(long): milliseconds since the Epoch Time(1970-01-01 00:00:00 GMT). It is a number in GMT time zone.
     * @param timeZone: if it is null, take the default time zone, GMT.
     */
    static String getStringFromLongUsingFormatter(final long time, final SimpleDateFormat sdf, final TimeZone timeZone)
    {
        TimeZone usedTimeZone = timeZone;
        if(usedTimeZone == null)
        {
            usedTimeZone = OHDConstants.GMT_TIMEZONE;
        }

        if(usedTimeZone.equals(OHDConstants.GMT_TIMEZONE))
        {
            sdf.setTimeZone(usedTimeZone);
        }
        else
        {
            sdf.setTimeZone(usedTimeZone);
        }

        final String dateTimeString = sdf.format(new Date(time));

        return dateTimeString;
    }

    /**
     * Returns a string like "HH:MM:SS".
     * <p>
     * Note: The string returned is related to the parameter timeZone, even though the returned string does not show the
     * time zone information. The hour count is 00 to 23. The last hour of the day is 23 o'clock. One hour after that is
     * 0 o'clock next day.
     * 
     * @param time(long): milliseconds since the Epoch Time(1970-01-01 00:00:00 GMT). It is a number in GMT time zone.
     * @param timeZone: if it is null, take the default time zone, GMT.
     */
    public static String getTimeStringFromLong(final long time, final TimeZone timeZone)
    {
        return DateTime.getStringFromLongUsingFormatter(time, _Time_Formatter.get(), timeZone);
    }

    /**
     * Returns a string like "YYYY-MM-DD".
     * <p>
     * Note: The string returned is related to the parameter timeZone, even though the returned string does not show the
     * time zone information.
     * 
     * @param time(long): milliseconds since the Epoch Time(1970-01-01 00:00:00 GMT). It is a number in GMT time zone.
     * @param timeZone: if it is null, take the default time zone, GMT.
     */
    public static String getDateStringFromLong(final long time, final TimeZone timeZone)
    {
        return DateTime.getStringFromLongUsingFormatter(time, _Date_Formatter.get(), timeZone);
    }

    /**
     * Returns a date and time string "YYYY-MM-DD HH:MM:SS GMT" or "**** GMT-HH:00" or "***** GMT+HH:00". The hour count
     * is 00 to 23. The last hour of the day is 23 o'clock. One hour after that is 0 o'clock next day.
     * 
     * @param time(long): milliseconds since the Epoch Time(1970-01-01 00:00:00 GMT). It is a number in GMT time zone.
     * @param timeZone: if it is null, take the default time zone, GMT.
     */
    public static String getDateTimeStringFromLong(final long time, final TimeZone timeZone)
    {
        final StringBuilder resultStrBuf = new StringBuilder(30);

        SimpleDateFormat dateTimeFormat = null;

        if(timeZone == null)
        {
            dateTimeFormat = _gmtSdfThreadLocal.get();
        }
        else
        {
            dateTimeFormat = _Date_Time_FormatterThreadLocal.get();
        }

        resultStrBuf.append(DateTime.getStringFromLongUsingFormatter(time, dateTimeFormat, timeZone)).append(" ");

        if(timeZone == null)
        {
            resultStrBuf.append(OHDConstants.TIMEZONE_GMT_STRING);
        }
        else
        {
            resultStrBuf.append(timeZone.getID());
        }

        return resultStrBuf.toString();
    }

    public static int hash(final int seed, final int anInt)
    {
        return HashUtility.hashCombine(seed, anInt);
    }

    public String getTimeString()
    {
        return _timeString;
    }

    public String getDateString()
    {
        return _dateString;
    }

    public TimeZone getTimeZone()
    {
        return _timeZone;
    }

    @Override
    public DateTime clone()
    {
        final DateTime clone = new DateTime(getTimeInMillis(), _timeZone);

        return clone;
    }
}
