package ohd.hseb.time;

public class FasterCustomDateTimeParser implements DateTimeParser
{
    private final static int YEAR = 0;
    private final static int MONTH = 1;
    private final static int DAY = 2;
    private final static int HOUR = 3;
    private final static int MIN = 4;
    private final static int SEC = 5;

    private final static int MILLIS_PER_SECOND = 1000;
    private final static int SECONDS_PER_MINUTE = 60;
    private final static int SECONDS_PER_HOUR = 3600;
    private final static int SECONDS_PER_DAY = SECONDS_PER_HOUR * 24;
//    private final static int SECONDS_PER_REGULAR_YEAR = SECONDS_PER_DAY * 365;
//    private final static int SECONDS_PER_LEAP_YEAR = SECONDS_PER_DAY * 366;

    private final static int BASE_YEAR = 1970;
    private final static int END_YEAR = 2200;
    private final static int EPOCH_YEAR = 1970;

    //contains the number of days in the year prior to the start of the month

    private static final int _daysInMonthArray[] = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
    private static int[] _totalDaysPriorToMonthArray = new int[12]; //leap years will be handled elsewhere
    private static int[] _totalDaysPriorToYearArray = new int[END_YEAR];

    static
    {
        //compute total Days Prior to Month Array
        _totalDaysPriorToMonthArray[0] = 0;
        for(int i = 1; i < _daysInMonthArray.length; i++)
        {
            _totalDaysPriorToMonthArray[i] = _totalDaysPriorToMonthArray[i - 1] + _daysInMonthArray[i - 1];
        }

        //compute total Days Prior to Year Array
        initTotalDaysPriorToYearArray(BASE_YEAR, END_YEAR, EPOCH_YEAR);
    }

    // -----------------------------------------------------------------------------------------------------------------
    private String[] getParts(final String dateTimeString)
    {
        // year, month, day, hour, minute, second
        final String[] partsArray = new String[6];

        try
        {

            partsArray[YEAR] = dateTimeString.substring(0, 4);
            partsArray[MONTH] = dateTimeString.substring(5, 7);

            partsArray[DAY] = dateTimeString.substring(8, 10);
            partsArray[HOUR] = dateTimeString.substring(11, 13);

            partsArray[MIN] = dateTimeString.substring(14, 16);
            partsArray[SEC] = dateTimeString.substring(17, 19);
        }
        catch(final Exception e)
        {

            System.out.println("original dateTimeString = " + dateTimeString);
            System.out.println(e.getMessage());
        }

        return partsArray;
    }

    @Override
    protected Object clone() throws CloneNotSupportedException
    {
        return super.clone();
    }

    // -----------------------------------------------------------------------------------------------------------------
    @Override
    public long parse(final String dateTimeString)
    {
        long millis = 0;

        final String[] dateTimePartsArray = getParts(dateTimeString);

        millis = getUniversalTime(dateTimePartsArray);

        return millis;

    }

    // -----------------------------------------------------------------------------------------------------------------

    private long getUniversalTime(final String[] dateTimePartsArray)
    {
        final int year = Integer.parseInt(dateTimePartsArray[YEAR]);
        final int month = Integer.parseInt(dateTimePartsArray[MONTH]);
        final int day = Integer.parseInt(dateTimePartsArray[DAY]);

        final int hour = Integer.parseInt(dateTimePartsArray[HOUR]);
        final int min = Integer.parseInt(dateTimePartsArray[MIN]);
        final int sec = Integer.parseInt(dateTimePartsArray[SEC]);

        final long millis = getUniversalTimeInSeconds(year, month, day, hour, min, sec) * MILLIS_PER_SECOND;

        return millis;
    }

    // -----------------------------------------------------------------------------------------------------------------

    public long getUniversalTimeInSeconds(final int year,
                                          final int month,
                                          final int day,
                                          final int hour,
                                          final int min,
                                          final int sec)
    {

        //monthIndex =  month - 1, so that I can look up month in an array with positions from 0 to 11 for the 12 months
        final int monthIndex = month - 1;

        long totalSeconds = 0;
        long totalDays = 0;

        totalDays = getDaysBeforeYear(EPOCH_YEAR, year);

        /* current year is leap_year */

        if(year >= EPOCH_YEAR)
        {
            /* monthIndex > 1 means later than Feb */
            if((monthIndex > 1) && isLeapYear(year))
            {
                totalDays++; //add a leap day
            }
        }

        else
        //year < EPOCH_YEAR
        {
            /* monthIndex <=-1 means Feb or before */
            if((monthIndex <= 1) && isLeapYear(year))
            {
                totalDays--; //subtract a leap day
            }
        }

        totalDays += _totalDaysPriorToMonthArray[monthIndex]; //add in the days prior to this month

        totalDays += day - 1; //subtract off the current day, which hasn't completed yet

        /* don't care about leap seconds */
        totalSeconds = (totalDays * SECONDS_PER_DAY);

        totalSeconds += hour * SECONDS_PER_HOUR;
        totalSeconds += min * SECONDS_PER_MINUTE;
        totalSeconds += sec;

        return totalSeconds;

    }

    // -----------------------------------------------------------------------------------------------------------------

    private static boolean isLeapYear(final int year)
    {
        boolean isLeap = false;

        if((year % 4) == 0)
        {
            if((year % 100) == 0)
            {
                if((year % 400) == 0) //2000 is a leap year, 1900 and 2100 are not
                {
                    isLeap = true;
                }
                else
                {
                    isLeap = false;
                }
            }
            else
            {
                isLeap = true;
            }
        }
        else
        {
            isLeap = false;
        }

        return isLeap;
    } //end isLeapYear()

    // -----------------------------------------------------------------------------------------------------------------
    // -----------------------------------------------------------------------------------------------------------------
    private static int getDaysBeforeYearWithoutCache(final int epochYear, final int year)
    {

        int currentYear = 0;
        int totalDays = 0;

        /* years */
        if(epochYear <= year)
        {
            for(currentYear = epochYear; currentYear < year; currentYear++)
            {
                if(isLeapYear(currentYear))
                {
                    totalDays += 366;
                }
                else
                {
                    totalDays += 365;
                }

            }
        }
        else
        //year is before 1970 (epochYear)
        {
            for(currentYear = epochYear; currentYear > year; currentYear--)
            {
                if(isLeapYear(currentYear))
                {
                    totalDays -= 366;
                }
                else
                {
                    totalDays -= 365;
                }

            }
        }

        return totalDays;

    }

    // -----------------------------------------------------------------------------------------------------------------
    private static int getTotalDaysPriorToYearFromArray(final int year)
    {
        final int index = year - 1;
        return _totalDaysPriorToYearArray[index];

    }

    // -----------------------------------------------------------------------------------------------------------------

    private static void setTotalDaysPriorToYearToArray(final int year, final int totalDays)
    {
        final int index = year - 1;
        _totalDaysPriorToYearArray[index] = totalDays;
    }

    // -----------------------------------------------------------------------------------------------------------------

    private static void initTotalDaysPriorToYearArray(final int baseYear, final int endYear, final int epochYear)
    {

        int year = 0;
        int totalDays = 0;

        setTotalDaysPriorToYearToArray(epochYear, 0); //1970 gets initialized to 0

        //1970 or later
        for(year = epochYear + 1; year <= endYear; year++)
        {

            totalDays = getTotalDaysPriorToYearFromArray(year - 1);

            if(isLeapYear(year - 1)) //if previous year was a leap year
            {
                totalDays += 366;
            }
            else
            {
                totalDays += 365;
            }
            setTotalDaysPriorToYearToArray(year, totalDays);

        }

        //1969 or earlier
        for(year = epochYear - 1; year >= baseYear; year--)
        {

            totalDays = getTotalDaysPriorToYearFromArray(year + 1);

            if(isLeapYear(year + 1)) //if next year is a leap year
            {
                totalDays -= 366;
            }
            else
            {
                totalDays -= 365;
            }
            setTotalDaysPriorToYearToArray(year, totalDays);

        }

    } //end 

    // -----------------------------------------------------------------------------------------------------------------
    private static int getDaysBeforeYear(final int epochYear, final int year)
    {
        int daysBeforeYearRelativeToEpochDate = 0;

        if((year >= BASE_YEAR) && (year <= END_YEAR))
        {
            daysBeforeYearRelativeToEpochDate = getTotalDaysPriorToYearFromArray(year);
        }
        else
        {
            daysBeforeYearRelativeToEpochDate = getDaysBeforeYearWithoutCache(epochYear, year);
        }

        return daysBeforeYearRelativeToEpochDate;

    }
    // -----------------------------------------------------------------------------------------------------------------

}
