package ohd.hseb.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ohd.hseb.hefs.utils.tools.StreamTools;
import ohd.hseb.util.fews.OHDConstants;

/**
 * The CommandAdapter is used to invoke an external process
 * 
 * @author FewsPilot team
 */
final public class CommandAdapter
{

    private final List<String> _outputLines = new ArrayList<String>(); //will contain the external process' stderr and stdoutput lines

    /**
     * @param args arguments needed by external process
     * @throws Exception
     */
    public void run(final String[] args) throws Exception
    {
        try
        {
            invoke(args);
        }
        finally
        {
        }
    }

    /**
     * Return the list of String, which are the stdoutput and stderr of executing the system command. If no any output,
     * the list size is 0.<br>
     * Note:<br>
     * 1)This command can be used only if when calling the external program, waitForFinish is true. In another words,
     * the external program is a short one; If the external program is a long one, e.g. running FEWS, its output is not
     * saved.<br>
     * 2)after this method, {@link #_outputLines} is cleaned to avoid accumulating to the results from next command
     */
    public List<String> getCommandOutput()
    {
        final List<String> copyOutPutLines = new ArrayList<String>(_outputLines);
        _outputLines.clear();
        return copyOutPutLines;
    }

    /**
     * Run the command and wait till the process is finished.
     * 
     * @param args arguments needed by external process: args[0] is the system command, the rest elements of the array
     *            are the parameters
     * @throws Exception
     */
    public synchronized void invoke(final String[] args) throws Exception
    {
        this.invoke(args, true);
    }

    /**
     * Calls {@link #invoke(String[], boolean, boolean)} passing in the waitForFinish flag for both boolean arguments.
     */
    public synchronized void invoke(final String[] args, final boolean waitForFinish) throws Exception
    {
        invoke(args, waitForFinish, waitForFinish);
    }

    /**
     * @param args -- arguments needed by external process. Normally, args[0] is the system command and the rest array
     *            elements are the parameters. However, some commands need to be to be interpreted by a shell. For
     *            example, a command like "ls *.zip" does because the wild card "*" needs to be interpreted by a shell.
     *            Same is pipe "|". To do that, args needs to be a three element array:args[0]="/bin/sh", args[1]="-c",
     *            and args[2]=the long string representing the command and its parameters all together.
     * @param waitForFinish -- if true, wait for the external program to finish;if false, then do not wait for it to
     *            finish, e.g. running FEWS(its output is not saved in {@link #_outputLines});
     * @param storeOutput If true, then stderr and stdout are saved into {@link #_outputLines} and can be retrieved by
     *            {@link #getCommandOutput()}.
     * @throws Exception - when external process didn't succeed, throw an Exception. However, when the command is "rm"
     *             and the target file(s) do not exist, the external process will not succeed. But in this case, don't
     *             throw an Exception.
     */
    public synchronized void invoke(final String[] args, final boolean waitForFinish, final boolean storeOutput) throws Exception
    {

        final ProcessBuilder processBuilder = new ProcessBuilder(args);

        processBuilder.redirectErrorStream(true);
        //merge stdout and stderr of the external process -- this feature only available for ProcessBuilder, not available for RunTime.getRunTine().exec(..)

        final Process processRunSysCmd = processBuilder.start();//start a new process outside of JVM -- a child process

        /**
         * When calling external command from Java program, the output of the external command(the child process) is the
         * input stream to the parent process(the Java process)(and the input to the external command is the output from
         * external program) needs to be drained, otherwise if the external program has a lot of output and the buffer
         * is saturated, the Java process running external command hangs!!! Since different operating systems have
         * different buffer sizes, so the same code could behave funny(runs fine on one computer, but hangs on another
         * computer -- we have experienced that.) <br>
         * How to drain the input stream(output from external program) is tricky too: Using a BufferedReader object to
         * read the input stream by a while loop seems the way to go. However, this is only good when the external
         * program is a short one, i.e. waitForFinish is true--this is the most common cases too. If the system command
         * is not something finished quickly, e.g. running FEWS(and this is about the only reality case for us), the
         * output from the external program will not have an ending until FEWS is closed, so stdReader.readLine() in the
         * while loop below never ends till FEWS closed -- this is not something we want. So, here mutli-thread come
         * into play beautifully: when the external program is not ending soon(waitForFinish is false), spawn a separate
         * thread to drain it! (And its output is not saved into _outputLines -- imagine there are two such threads
         * running, all their outputs are saved into _outputLines. _outputLines will be messy.)
         */

        //read the external process stdout and stderr
        if(storeOutput)
        {//for quick command, no need of a child thread. The while loop below ends when the external command finishes
            final BufferedReader stdReader = new BufferedReader(new InputStreamReader(processRunSysCmd.getInputStream()));

            String stdline;

            _outputLines.clear();

            try
            {
                while((stdline = stdReader.readLine()) != null)
                {
                    _outputLines.add(stdline);//saved into _outputLines
                }
            }
            finally
            {//Should guarantee the reader is closed... put it in a finally.

                stdReader.close();
            }
        }
        else
        {//for a long wait command, e.g. running FEWS, a separate thread is needed

            /*
             * the output of external program is not saved into _outputLines because this could be con-current(another
             * external program is running and if its waitForFinish is true, then it's output is saved into _outputLines
             * -- all mixed up and we don't want it. The output is printed out to the terminal
             */
            new Thread()
            {
                @Override
                public void run()
                {

                    final BufferedReader stdReader = new BufferedReader(new InputStreamReader(processRunSysCmd.getInputStream()));

                    String stdline;

                    try
                    {
                        while((stdline = stdReader.readLine()) != null)
                        {
                            System.out.println(stdline);//print out to the terminal directly
                        }
                    }
                    catch(final IOException ioE)
                    {
                        ioE.printStackTrace();
                    }
                    //This may not be necessary when the reader is opened in a 
                    //thread (stopping the thread likely removes the open BufferedReader), but it should be cleaner.  The StreamTools method
                    //handles the try-catch needed for closing.
                    finally
                    {
                        StreamTools.closeStream(stdReader);
                    }
                }

            }.start();//this thread lives until the external program ends, like FEWS being closed
        }

        /*
         * short external program enters the block below; long external program, e.g. "fews.sh.rboff cnrfc_sa", then
         * skips this block.
         */
        if(waitForFinish)
        {//

            final int _exitCode = processRunSysCmd.waitFor();

            /*
             * when returned _exitCode is not 0, the command has failed. However, some commands' failing is not really
             * an error. So far, I know, "rm" fails if the target file(s) to be removed does not exist; "ls *.txt" fails
             * if no any *.txt files existing; "grep" fails for similar reason.
             */
            if(_exitCode != 0)
            {
                //Hank: ci and co return 1 on successful calls for some reason.
                if(args[0].contains("rm") || args[0].contains("ls") || args[0].contains("grep") || args[0].equals("ci")
                    || args[0].equals("co"))
                {//no need to throw an Exception--assuming it's not the type "/bin/sh -c command parameters"
//                    System.out.println("The command " + Arrays.toString(args) + " didn't succeed.");
                }
                else
                {//throw an Exception

                    //since _exitCode != 0, the external process definitely has produced some stderr messages, which has been saved in _outputLines in the code above
                    for(final String line: _outputLines)
                    {
                        System.out.println(line); //output to console for reminding purpose. Normally, this code should not be executed in real life
                    }

                    throw new Exception(Arrays.toString(args) + " terminated abnormally with error. _exitCode = "
                        + _exitCode);
                }
            }//close if(_exitCode != 0)

        } //close if(waitForFinish)

    }//close method

    /**
     * All the command is in one line. Sometimes this overloaded method is handy.
     * 
     * @param oneCmdLine
     * @throws Exception
     */
    public synchronized void invoke(final String oneCmdLine) throws Exception
    {
        final String[] args = OHDConstants.SPACE_PATTERN.split(oneCmdLine);

        this.invoke(args);

    }

}
