package ohd.hseb.util;

import java.util.Random;

public class MathHelper
{
    /** 5E-8: the comparison margin used to overcome the precision issue, for the Java code ported from Fortran */
    private final static double EPSILON = 5E-8;

    /**
     * Round the input parameter origValue to the number of digits after decimal point.
     */
    public static double roundToNDecimalPlaces(double numberToRound, int decimalPlacesToMaintain)
    {

        long roundingFactor = 1;
        long temporaryHolder;
        final long roundingFactorArray[] = {1, 10, 100, 1000, 10000, 100000, 1000000, 10000000, 100000000, 1000000000};

        if(decimalPlacesToMaintain < 0)
        {
            decimalPlacesToMaintain = 0;
        }
        else if(decimalPlacesToMaintain > 9)
        {
            decimalPlacesToMaintain = 9;
        }

        // I could have used pow(), but this is faster
        roundingFactor = roundingFactorArray[decimalPlacesToMaintain];

        temporaryHolder = (long)Math.floor((numberToRound * roundingFactor) + 0.5);

        numberToRound = (double)temporaryHolder / roundingFactor;

        return numberToRound;

    }

    public static long round(final double x)
    {
        if(x >= 0f)
        {
            return (long)(x + 0.5f);
        }
        long res = (long)(x - 0.5f);
        if(x - res == 0.5f)
        {
            res++;
        }
        return res;
    }

    /**
     * In placement of ">" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isGreater(final double dA, final double dB)
    {
        if(dA > (dB + EPSILON))
        {
            return true;
        }

        return false;
    }

    /**
     * In placement of "<=" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isLessOrEqual(final double dA, final double dB)
    {
        return !isGreater(dA, dB);
    }

    /**
     * In placement of ">=" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isGreaterOrEqual(final double dA, final double dB)
    {
        if(dA >= (dB - EPSILON))
        {
            return true;
        }

        return false;
    }

/**
     * In placement of "<" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isLessThan(final double dA, final double dB)
    {
        return !isGreaterOrEqual(dA, dB);
    }

    /**
     * In placement of "==" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isEqual(final double dA, final double dB)
    {
        if(Math.abs(dA - dB) < EPSILON)
        {
            return true;
        }

        return false;
    }

    /**
     * In placement of "!=" operation when comparing double values. The Java code ported from Fortran needs this method
     * in order to obtain the same results as Fortran.
     */
    public static boolean isNotEqual(final double dA, final double dB)
    {
        return !isEqual(dA, dB);
    }

    /**
     * Returns true if the parameter value is approximate to 0.0, within the range of {@value #EPSILON} value;
     * otherwise, returns false. In placement of "== 0.0".
     */
    public static boolean isEqualToZero(final double dA)
    {
        return isEqual(dA, 0.0);
    }

    /**
     * In opposite to {@link #isEqualToZero(double)}. In placement of "!= 0.0"
     */
    public static boolean notEqualToZero(final double dA)
    {
        return !isEqual(dA, 0.0);
    }

    /**
     * In placement of "> 0.0".
     */
    public static boolean isGreaterThanZero(final double dA)
    {
        return isGreater(dA, 0.0);
    }

    /**
     * In placement of "< 0.0".
     */
    public static boolean isLessThanZero(final double dA)
    {
        return isLessThan(dA, 0.0);
    }

    /**
     * A helper method to generate a random double value within the specified range.
     * 
     * @param bottomValue - the lower end of the random number range
     * @param ceilingValue - the up end of the random number range
     * @param numDigits - the number of digits in the returned double
     */
    public static double getRandomDouble(final double bottomValue, final double ceilingValue, final int numDigits)
    {
    
        final double diff = ceilingValue - bottomValue; //always positive 
    
        final Random random = new Random();
    
        double randomD = random.nextDouble(); //between 0.0 and 1.0
    
        if(bottomValue < 0.0 && ceilingValue > 0.0)
        {//the random number range is between a negative value and a positive value
            final boolean bool = random.nextBoolean();
    
            if(bool)
            {
                randomD *= bottomValue;
            }
            else
            {
                randomD *= ceilingValue;
            }
        }
        else
        {//the random number range is between two positive values or between two negative values
            randomD = bottomValue + randomD * diff;
        }
    
        return roundToNDecimalPlaces(randomD, numDigits);
    
    }

}
