/*
 * Created on May 24, 2004
 */
package ohd.hseb.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

public class XmrgReader
{

    // each value was stored in mm of precip times 100
    // Conversion factor needed to convert (mm x 100) to inches
    // need to divide by 2540.0 (25.4mm/inch x 100) to get inches
    private static final double _CONVERSION_FACTOR = 2540.0;
    private static final double _missingValue = -999.0;

    private final String _mpeQpeGridDirName = null;
    private boolean _needEndianChange = false;

    // I may be able to turn these into local variables
    private int _numBinsPerColXmrg = 0;
    private int _numBinsPerRowXmrg = 0;

    private int _swHrapRowXmrg = 0;
    private int _swHrapColXmrg = 0;

    private int _southernRowXmrg = 0;
    private int _northernRowXmrg = 0;
    private int _westernColXmrg = 0;
    private int _easternColXmrg = 0;

    private final Logger _logger;

//  -----------------------------------------------------------------------------------

    public XmrgReader(final Logger logger)
    {
        _logger = logger;
    }

//  -----------------------------------------------------------------------------------

    public XmrgGrid loadGrid(final long time, final String gridFileName)
    {

        final XmrgGrid grid = readMPEGridFile(time, gridFileName);

        return grid;
    }

//  -----------------------------------------------------------------------------------
    private DataInputStream initDataInputStream(final String fullGridFilePath)
    {
        DataInputStream dis = null;

        final boolean normalMode = true;

        try
        {

            @SuppressWarnings("resource")
            final FileInputStream fis = new FileInputStream(fullGridFilePath);
            /*
             * even though Eclipse warns fis is not closed, actually it cannot be closed because it is inside dis.
             * Closing fis causes dis to be closed
             */

            if(normalMode)
            {
                final BufferedInputStream bis = new BufferedInputStream(fis);
                dis = new DataInputStream(bis);
            }
            else
            {// "faster" mode
                final int byteCount = fis.available();
                final byte[] byteArray = new byte[byteCount];

                fis.read(byteArray);
                fis.close();

                final ByteArrayInputStream bais = new ByteArrayInputStream(byteArray);
                dis = new DataInputStream(bais);

            }
        }
        catch(final java.io.IOException e)
        {
            e.printStackTrace(_logger.getPrintWriter());
        }

        return dis;

    }

//  -----------------------------------------------------------------------------------

    private XmrgGrid readMPEGridFile(final long gridTime, final String gridFileName)
    {
        XmrgGrid grid = null;

        String fullGridFilePath;

        if(_mpeQpeGridDirName != null)
            fullGridFilePath = _mpeQpeGridDirName + "/" + gridFileName.trim();
        else
            fullGridFilePath = gridFileName.trim();

        final File xmrgFile = new File(fullGridFilePath);

        if(xmrgFile.exists())
        {
            try
            {

                final DataInputStream dis = initDataInputStream(fullGridFilePath);

                int numBytesRecordBegin;
                @SuppressWarnings("unused")
                int numBytesRecordEnd;
                //somehow, the three calling below to set numBytesRecordEnd is absolutely required even though numBytesRecordEnd value is unused

                // read in the first header of the xmrg file which contains
                // the SouthWest HRAP column, row, # of columns and # of rows
                numBytesRecordBegin = dis.readInt();

                //determine endianess
                if(numBytesRecordBegin == 16)
                {
                    _needEndianChange = false;
                }
                else
                {
                    _needEndianChange = true;
                    numBytesRecordBegin = EndianConverter.convert(numBytesRecordBegin);
                }

                _swHrapColXmrg = readInt(dis);
                _swHrapRowXmrg = readInt(dis);

                _numBinsPerRowXmrg = readInt(dis);

                if((_numBinsPerRowXmrg < 1) || (_numBinsPerRowXmrg > 1000))
                {
                    _logger.log(Logger.DEBUG, "Error: The number of HRAP Bins Per Row for " + fullGridFilePath + " is "
                        + _numBinsPerRowXmrg);
                    _logger.log(Logger.DEBUG,
                                " Note: This number should be in the range of 1 - 1000.  This MPE xmrg file is being skipped.");
                    return XmrgGrid.getInvalidGrid(gridTime);
                }

                _numBinsPerColXmrg = readInt(dis);

                if((_numBinsPerColXmrg < 1) || (_numBinsPerColXmrg > 1000))
                {
                    _logger.log(Logger.DEBUG, "Error: The number of HRAP Bins Per Col for " + fullGridFilePath + " is "
                        + _numBinsPerColXmrg);
                    _logger.log(Logger.DEBUG,
                                " Note: This number should be in the range of 1 - 1000.  This MPE xmrg file is being skipped.");
                    return (XmrgGrid.getInvalidGrid(gridTime));
                }

                numBytesRecordEnd = readInt(dis);

                // calculate the Southern and Northern most rows of the XMRG file
                _southernRowXmrg = _swHrapRowXmrg;
                _northernRowXmrg = _southernRowXmrg + _numBinsPerColXmrg - 1;

                // calculate the Western and Eastern most columns of the XMRG file
                _westernColXmrg = _swHrapColXmrg;
                _easternColXmrg = _westernColXmrg + _numBinsPerRowXmrg - 1;

                final int rowCount = _numBinsPerColXmrg;
                final int colCount = _numBinsPerRowXmrg;

                grid = new XmrgGrid(gridTime, _southernRowXmrg, _westernColXmrg, rowCount, colCount);

                // read in the second header of the xmrg file which contains
                // the userid, saved date, process flag, valid date,
                // maximum value and version number and throw them away
                numBytesRecordBegin = readInt(dis);
                numBytesRecordEnd = dis.skipBytes(numBytesRecordBegin);
                numBytesRecordEnd = readInt(dis);

                double value = _missingValue;

                for(int hrapRow = _southernRowXmrg; hrapRow <= _northernRowXmrg && grid.isValid(); hrapRow++)
                {
                    numBytesRecordBegin = readInt(dis);
                    // IF numBytesRecordBegin NOT EQUAL TO TWICE numBinsPerRowXmrg THEN BAIL
                    if(numBytesRecordBegin != (_numBinsPerRowXmrg * 2))
                    {
                        _logger.log(Logger.DEBUG, "Error: The number of bytes per record " + numBytesRecordBegin
                            + " should be twice the number of bins per row " + _numBinsPerRowXmrg);
                        grid.setIsValid(false);

                    }

                    for(int hrapCol = _westernColXmrg; hrapCol <= _easternColXmrg && grid.isValid(); hrapCol++)
                    {
                        value = readShort(dis);

                        if(value != _missingValue)
                        {
                            value /= _CONVERSION_FACTOR;
                            grid.setValue(hrapRow, hrapCol, value);
                        }
                        else
                        {
                            grid.setValue(hrapRow, hrapCol, _missingValue);
                        }
                    }

                    numBytesRecordEnd = readInt(dis);
                }

                dis.close();
            }
            catch(final EOFException ioe)
            {
                _logger.log(Logger.DEBUG, "Read End Of File" + ioe);
                grid.setIsValid(false);
            }
            catch(final IOException ioe)
            {
                _logger.log(Logger.DEBUG, "Something went wrong trying to read " + ioe);
                grid.setIsValid(false);
            }
        }
        else
        {
            grid = XmrgGrid.getInvalidGrid(gridTime);
        }

        return grid;

    } // end of readMPEGridFile method

//  -----------------------------------------------------------------------------------

// ----------------------------------------------------
    private int readInt(final DataInputStream dis) throws IOException
    {
        int value = dis.readInt();

        if(_needEndianChange)
        {
            value = EndianConverter.convert(value);
        }
        return value;
    }

// ----------------------------------------------------

    private short readShort(final DataInputStream dis) throws IOException
    {
        short value = dis.readShort();

        if(_needEndianChange)
        {
            value = EndianConverter.convert(value);
        }
        return value;
    }
    // ----------------------------------------------------

}
