package ohd.hseb.util.data;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.StreamTokenizer;
import java.util.ArrayList;

/**
 * The DataFromFile class reads a series of (ASCII) data points from a file and returns the data points as an array.
 * Initially, this class only returns arrays of double, float or String, but it could be adapted to return other data
 * types, when needed. NOTE: If a doubly dimensioned array/table is desired, consider using the DataSet and FileData
 * classes instead (of this class).
 * 
 * @author Robert J. Mills
 * @since Jan 13, 2012
 */
public class DataFromFile
{
    /*
     * Note that both _streamTokenizer and _bufferedReader are private and that no getter/setter methods have been
     * provided. This was deliberate since these instance variables are for internal use only.
     */

    private static StreamTokenizer _streamTokenizer;
    private static BufferedReader _bufferedReader;

    /*
     * Note that readArray internally uses an ArrayList Collection since Collections can grow dynamically. (Collections
     * don't require that you know how many values will ultimately be added.) However, once all the values have been
     * read in, then the total number (of values) is known, allowing the creation of a (computationally more efficient)
     * array of primitives (of the exact size) to be created.
     */
    public static double[] readArrayAsDouble(final String fileName) throws IOException
    {
        //Open the file and get ready to read the data
        prepToRead(fileName);
        _streamTokenizer.parseNumbers();

        //Read & parse the file, storing data in collection 
        final ArrayList<Double> arrayList = new ArrayList<Double>();
        while(_streamTokenizer.nextToken() != StreamTokenizer.TT_EOF)
        {
            arrayList.add(Double.valueOf(_streamTokenizer.nval));
        }
        //Close the file and release resources
        cleanUp();

        //Convert to primitives and return
        final double[] doubles = new double[arrayList.size()];
        int i = 0;
        for(final Double d: arrayList)
            doubles[i++] = d; //Uses auto-unboxing
        return doubles;
    }

    public static float[] readArrayAsFloat(final String fileName) throws IOException
    {
        //Open the file and get ready to read the data
        prepToRead(fileName);
        _streamTokenizer.parseNumbers();

        //Read & parse the file, storing data in collection 
        final ArrayList<Float> arrayList = new ArrayList<Float>();
        while(_streamTokenizer.nextToken() != StreamTokenizer.TT_EOF)
        {
            arrayList.add(Float.valueOf((float) _streamTokenizer.nval));
        }
        //Close the file and release resources
        cleanUp();

        //Convert to primitives and return
        final float[] floats = new float[arrayList.size()];
        int i = 0;
        for(final Float f: arrayList)
            floats[i++] = f; //Uses auto-unboxing
        return floats;
    }

    public static String[] readArrayAsString(final String fileName) throws IOException
    {
        //Open the file and get ready to read the data
        prepToRead(fileName);

        //Read & parse the file, storing data in collection 
        final ArrayList<String> arrayList = new ArrayList<String>();
        while(_streamTokenizer.nextToken() != StreamTokenizer.TT_EOF)
        {
            if(_streamTokenizer.ttype == StreamTokenizer.TT_NUMBER)
                arrayList.add(Double.toString(_streamTokenizer.nval));
            else
                arrayList.add(_streamTokenizer.sval);
        }
        //Close the file and release resources
        cleanUp();

        //Convert to primitives and return
        final String[] strings = new String[arrayList.size()];
        int i = 0;
        for(final String f: arrayList)
            strings[i++] = f; //Uses auto-unboxing
        return strings;
    }

    private static void prepToRead(final String fileName) throws IOException
    {
        //Prepare to open the file and read its contents.
        final File file = new File(fileName);
        final FileReader fileReader = new FileReader(file);
        _bufferedReader = new BufferedReader(fileReader);
        _streamTokenizer = new StreamTokenizer(_bufferedReader);
    }

    private static void cleanUp() throws IOException
    {
        _bufferedReader.close();
        _bufferedReader = null;
        _streamTokenizer = null;
    }
}
