/*
 * Created on Sep 12, 2006 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.util.data;

import java.util.Arrays;

/**
 * This class wraps around a table of data. It provides raw methods to access rows, columns, etc. It does no checking of
 * parameters, however, so the calling methods must know the size of the table and pass in valid rows and columns, or an
 * array exception may occur. This class uses pointer logic wherever possible in order to avoid copying arrays, and
 * provides methods that force array copies if the using class wants to.
 * 
 * @author hank
 */
public class DataTable
{
    final String CLASSNAME = "DataTable";

    private double[][] _data;

    /**
     * Basic constructor creates the data using the row and column dimensions passed in. The row dimension is the FIRST
     * dimension, column is second... i.e. _data[row][column].
     * 
     * @param numberOfRows The number of rows.
     * @param numberOfColumns The number of columns.
     */
    public DataTable(final int numberOfRows, final int numberOfColumns)
    {
        _data = new double[numberOfRows][numberOfColumns];
    }

    public double[][] getData()
    {
        return _data;
    }

    public int getNumberOfRows()
    {
        return _data.length;
    }

    public int getNumberOfColumns()
    {
        return _data[0].length;
    }

    /**
     * Sets a value within the data table at the specified row and column, without doing any checks.
     * 
     * @param row Index of row
     * @param column Index of column
     * @param value The value to set it to.
     */
    public void setValue(final int row, final int column, final double value)
    {
        _data[row][column] = value;
    }

    /**
     * Gets the value within the passed in cell.
     * 
     * @param row Index of row
     * @param column Index of column
     */
    public double getValue(final int row, final int column)
    {
        return _data[row][column];
    }

    /**
     * Sets the passed in row to be the given row array. IMPORTANT: The setting is a literal '=' operation, so the
     * passed in row is not a physical copy of the given row array, but is the actual row array. This does no checking
     * to make sure the passed in rowArray is of the right size! It should be the same size as all other rows.
     * 
     * @param row Index of rows.
     * @param rowArray array of doubles comprising the row.
     */
    public void setRow(final int row, final double[] rowArray)
    {
        _data[row] = rowArray;
    }

    /**
     * Get the row, itself, not a copy.
     */
    public double[] getRow(final int row)
    {
        return _data[row];
    }

    /**
     * Sets the passed in row as a copy of the given rowArray.
     * 
     * @param row Index of rows.
     * @param rowArray array of doubles comprising the row.
     */
    public void setRowAsCopy(final int row, final double[] rowArray)
    {
        final double[] rowValues = DataTable.copyArray(rowArray);
        setRow(row, rowValues);
    }

    /**
     * Get a copy of the row.
     */
    public double[] getCopyOfRow(final int row)
    {
        return DataTable.copyArray(_data[row]);
    }

    /**
     * Copies the passed in columnArray into the column pointed to by column.
     */
    public void setColumnAsCopy(final int column, final double[] columnArray)
    {
        int i;
        for(i = 0; i < columnArray.length; i++)
        {
            _data[i][column] = columnArray[i];
        }
    }

    /**
     * Gets a copy of the column.
     */
    public double[] getCopyOfColumn(final int column)
    {
        final double[] columnValues = new double[_data.length];
        int i;
        for(i = 0; i < columnValues.length; i++)
        {
            columnValues[i] = _data[i][column];
        }
        return columnValues;
    }

    /**
     * Sets all the values of the double array to the passed in number.
     */
    public void setAllValues(final double value)
    {
        int i;
        int j;
        for(i = 0; i < _data.length; i++)
        {
            for(j = 0; j < _data[i].length; j++)
            {
                _data[i][j] = value;
            }
        }
    }

    /**
     * Adds a row to the _data table. The row is made new, and is not initialized, meaning it picks up the default Java
     * initialization of DataSet.MISSING.
     */
    public void addRow()
    {
        final double[][] oldData = _data;
        _data = new double[oldData.length + 1][oldData[0].length];
        int i;
        for(i = 0; i < oldData.length; i++)
        {
            setRow(i, oldData[i]);
        }
        setRow(oldData.length, DataTable.createAllMissingArray(oldData[0].length));
    }

    /**
     * Adds a column to the _data table by increasing the size of each row by 1. The new value will be set to
     * DataSet.MISSING.
     */
    public void addColumn()
    {
        int i;
        double[] newRow;
        int j;
        for(i = 0; i < _data.length; i++)
        {
            newRow = new double[_data[i].length + 1];
            for(j = 0; j < newRow.length - 1; j++)
            {
                newRow[j] = _data[i][j];
            }
            newRow[j] = DataSet.MISSING;
            setRow(i, newRow);
        }
    }

    /**
     * Removes the row pointed to by index.
     */
    public void removeRow(final int index)
    {
        final double[][] newData = new double[getNumberOfRows() - 1][getNumberOfColumns()];
        int i;

        for(i = 0; i < getNumberOfRows(); i++)
        {
            if(i < index)
            {
                newData[i] = getRow(i);
            }
            if(i > index)
            {
                newData[i - 1] = getRow(i);
            }
        }

        _data = newData;
    }

    /**
     * Removes the column pointed to by index. This is less memory efficient that removeRow.
     */
    public void removeColumn(final int index)
    {
        final double[][] newData = new double[getNumberOfRows()][getNumberOfColumns() - 1];

        int i;
        int j;
        for(i = 0; i < getNumberOfRows(); i++)
        {
            for(j = 0; j < getNumberOfColumns(); j++)
            {
                if(j < index)
                {
                    newData[i][j] = getValue(i, j);
                }
                if(j > index)
                {
                    newData[i][j - 1] = getValue(i, j);
                }
            }
        }

        _data = newData;
    }

    /**
     * Copy the contents of the passed in DataTable to this DataTable. This is a copy!!! It will attempt to do a cell by
     * cell copy. If a cell in this table is found to be invalid in the other table, then that cell will be set to
     * MISSING. DataTable otherTable The table to copy from.
     */
    public void copyContentsFromDataTable(final DataTable otherTable)
    {
        int i, j;
        for(i = 0; i < _data.length; i++)
        {
            for(j = 0; j < _data[0].length; j++)
            {
                if((i < otherTable.getNumberOfRows()) && (j < otherTable.getNumberOfColumns()))
                {
                    _data[i][j] = otherTable.getValue(i, j);
                }
                else
                {
                    _data[i][j] = DataSet.MISSING;
                }
            }
        }
    }

    /**
     * Return an array of all MISSING values of the length passed in.
     */
    public static double[] createAllMissingArray(final int quantity)
    {
        final double[] returnArray = new double[quantity];
        int i;
        for(i = 0; i < quantity; i++)
        {
            returnArray[i] = DataSet.MISSING;
        }
        return returnArray;
    }

    /**
     * Creates a physical copy of the passed in array and returns it.
     * 
     * @param array
     * @return
     */
    public static double[] copyArray(final double[] array)
    {
        final double[] returnArray = new double[array.length];
        int i;
        for(i = 0; i < array.length; i++)
        {
            returnArray[i] = array[i];
        }
        return returnArray;
    }

    @Override
    public boolean equals(final Object arg0)
    {
        return Arrays.deepEquals(_data, ((DataTable)arg0)._data);
    }

    /**
     * MAIN unit test.
     * 
     * @param args
     */
    public static void main(final String[] args)
    {
        final DataTable testTable = new DataTable(3, 3);
        final double[] testArray = {-1.0, -2.0, -3.0};
        int i, j;
        for(i = 0; i < testTable.getNumberOfRows(); i++)
        {
            for(j = 0; j < testTable.getNumberOfColumns(); j++)
            {
                testTable.setValue(i, j, i + j);
                System.out.println("####>> getValue (each should be row + column)... " + i + ", " + j + " = "
                    + testTable.getValue(i, j));
            }
        }
        System.out.println("####>> setValue test....");
        MatrixMath.printMatrix(testTable.getData());

        //Add two row.
        testTable.addRow();
        testTable.addRow();
        System.out.println("####>> ADDED TWO ROWS...");
        MatrixMath.printMatrix(testTable.getData());

        //Set a row as pointer and copy.
        testTable.setRow(3, testArray);
        testTable.setRowAsCopy(4, testArray);
        System.out.println("####>> SET TWO ROWS...");
        MatrixMath.printMatrix(testTable.getData());
        testArray[0] = 100.0;
        System.out.println("####>> MANUALLY CHANGED ORIGINAL ARRAY VALUE TO 100...");
        MatrixMath.printMatrix(testTable.getData());

        //Create a second data set that is expanded.
        final DataTable testTable2 = new DataTable(10, 4);
        testTable2.copyContentsFromDataTable(testTable);
        System.out.println("####>> Created bigger test table and copied contents....");
        MatrixMath.printMatrix(testTable2.getData());

    }
}
