package ohd.hseb.util.data;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.text.NumberFormat;
import java.util.Calendar;
import java.util.TimeZone;
import java.util.Vector;

import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * The DatacardData class reads in and stores Datacard format data files. It assumes the file is written in GMT,
 * although it provides a constructor which can be used if it is NOT in GMT by simply passing in the correct time zone
 * string (e.g. "EST" or "CST", but not "CDT" -- to for day-light savings time, find the standard timezone with the same
 * offset from GMT; for example instead of "CDT" use "EST"). <br>
 * <br>
 * For average/accumulated data, the julian hour column lists the julian hour associated with the END of each time
 * period of data. So, the average flow over the period January 10, 1975 at 6 GMT to 1-10-75 at 12 GMT will be stored
 * with the julian hour corresponding to 1-10-75 at 12 GMT.<br>
 * <br>
 * In the datacard world, a day ends at 24Z, so that 1-10-1975 at 24Z is still 1-10-1975. Hence, the first period for
 * 1-10-1975 is from 1-9-1975 24Z through 1-10-1975 6Z. Thus, using the end of the period makes sense. In the real
 * world, 1-10-1975 24Z is actually 1-11-1975 00Z, so that this approach would not make sense.<br>
 * <br>
 * 
 * @author hank
 */
public class DatacardData extends DataSet
{
    public final String CLASSNAME = "DatacardData";

    //Some static variables.
    final public static int JULIAN_HOUR = 0;
    final public static int VALUE = 1;

    final static double ACCUMULATED_VALUE = -998.0; //Another possible missing value within Datacard data.
    final static int VALUES_POSITION_IN_RECORD = 20; //The spot in a record line where the values start.
    final static int[] DATE_POSITION_IN_RECORD = {12, 14, 16, 20}; //Positions in record line of pre value stuff.
    final static int BASE_SAMPLE_SIZE = 500; //the starting max sample size for the dataset.  
    final static String DEFAULT_TIMEZONE = "GMT";
    final static int DEFAULT_VALUESPERLINE = 6;
    final static String DEFAULT_VALUESFORMAT = "F12.3";

    /////////////////////////////////////////////////////////////////////////
    //Attributes
    /////////////////////////////////////////////////////////////////////////

    //All attributes are currently PUBLIC because I do not, yet, want to create the set methods
    //for them.

    //Header information -- se section VII.2 of the NWSRFS documentation.
    //Line 1
    public String _filename; //
    public String _datatype; //data type
    public String _datadim; //dimension type
    public String _dataunits; //units of the data
    public int _tsdt; //Time interval
    public String _timeseriesid; //the time series id associated with the data.
    public String _tsdesc; //a descriptor

    //Line 2
    public int _im; //The month associated with first block of data.
    public int _iy; //The year ...
    public int _lm; //The month associated with the last block of data.
    public int _ly; //The year ...
    public int _valuesperline;//The number of values per line.
    public String _valuesformat; //The format of those values.

    //Processed data.
    public Calendar _firstdate; //The calendar associated with what should be the first day with any data (even MISSING).
    public Calendar _lastdate; //The last date calendar with data.
    public int _firstjhr; //The julian hour of _firstdate.
    public int _lastjhr; //The julian hour of lastdate.
    public int _totalvaluewidth; //The total field width of the values, extracted from valuesformat.
    public int _decimaldigits; //the number of decimal places within the field.
    public int[] _positions; //The positions within the record line demarkating different values.
    public int _todayyear; //The year associated with today, in order to run processTwoDigitYear.

    //Time Zone
    public TimeZone _timezone;

    /////////////////////////////////////////////////////////////////////////
    //Constructors
    /////////////////////////////////////////////////////////////////////////
    //MAKE THIS THROW AN EXCEPTION!!!!

    /**
     * Assumes DEFAULT_TIMEZONE, which is GMT.
     */
    public DatacardData(final String filename) throws DatacardDataException
    {
        FileReader filereader;
        BufferedReader localfile;

        //First, set the time zone.
        setTimeZone(DEFAULT_TIMEZONE);

        //Look at the file info.
        final File thefile = new File(filename);

        //Check for existence and reability.
        if(!thefile.exists() || !thefile.canRead())
        {
            throw new DatacardDataException("File \"" + filename + "\" either does not exist or is not readable.");
        }

        //Try to open the data file for reading.
        try
        {
            filereader = new FileReader(thefile);
            localfile = new BufferedReader(filereader);
        }
        catch(final FileNotFoundException e1)
        {
            throw new DatacardDataException("Failed to open requested file " + filename + ".");
        }

        //Try to read in the header of the file.
        int status = 0;
        try
        {
            //Read in the header data and process.
            readInHeader(localfile);
            processHeader();
            status = 1;
            //Initialize the _data array to the right size.
            initialize(BASE_SAMPLE_SIZE, 2, false);

            //Read in the data.
            readInData(localfile);
        }
        //Turn an IOException into a DatacardDataException.
        catch(final IOException e)
        {
            String message = "I/O exception occurred while reading in the";
            if(status == 0)
            {
                message += " header";
            }
            else
            {
                message += " body";
            }
            message += " with this message:";
            throw new DatacardDataException(message + e.getMessage());
        }
        //Make sure the files are closed.
        finally
        {
            try
            {
                localfile.close();
                filereader.close();
            }
            catch(final IOException ioe)
            {
            }
        }

        //Do the time zone shift.
        doTimeZoneShift();
    }

    public DatacardData(final DatacardData base)
    {
        super(base);

        //First, set the time zone.
        setTimeZone(base._timezone.getID());

        //Line 1
        _filename = new String(base._filename);
        _datatype = new String(base._datatype);
        _datadim = new String(base._datadim);
        _dataunits = new String(base._dataunits);
        _tsdt = base._tsdt;
        _timeseriesid = new String(base._timeseriesid);
        _tsdesc = new String(base._tsdesc);

        //Line 2
        _im = base._im;
        _iy = base._iy;
        _lm = base._lm;
        _ly = base._ly;
        _valuesperline = base._valuesperline;
        _valuesformat = new String(base._valuesformat);

        //Processed data.
        _firstdate = (Calendar)(base._firstdate.clone());
        _lastdate = (Calendar)(base._lastdate.clone());
        _firstjhr = base._firstjhr;
        _lastjhr = base._lastjhr;
        _totalvaluewidth = base._totalvaluewidth;
        _decimaldigits = base._decimaldigits;
        _positions = base._positions;
        _todayyear = base._todayyear;
    }

    public DatacardData(final String filename, final String timezone) throws DatacardDataException
    {
        FileReader thefile;
        BufferedReader localfile;

        //First, set the time zone.
        setTimeZone(timezone);

        //Try to open the data file for reading.
        try
        {
            thefile = new FileReader(filename);
            localfile = new BufferedReader(thefile);
        }
        catch(final FileNotFoundException e1)
        {
            //JOptionPane.showMessageDialog(null, "Unable to open sac_sma file: " + binfilename,
            //    CLASSNAME + ": ERROR", JOptionPane.ERROR_MESSAGE);
            System.out.println("DatacardData ERROR: Failed to open requested file " + filename);
            throw new DatacardDataException("Failed to open requested file " + filename);
        }

        //Try to read in the header of the file.
        try
        {
            //Read and process header.
            readInHeader(localfile);
            processHeader();

            //Initialize the _data array to the right size.
            initialize(BASE_SAMPLE_SIZE, 2, false);

            //Read data.
            readInData(localfile);

            localfile.close();
            thefile.close();
        }
        catch(final IOException e)
        {
            String message = "Failed to read in Datacard time series file for this reason:\n";
            message += "  Unable to read in the header.";
            throw new DatacardDataException(message);
        }

        try
        {
            localfile.close();
        }
        catch(final IOException e)
        {
            e.printStackTrace();
        }

        //Do the time zone shift.
        doTimeZoneShift();
    }

    public DatacardData(final ESPData base) throws DatacardDataException
    {
        super(base);

        //First, set the time zone.
        setTimeZone(DEFAULT_TIMEZONE);
        _timezone.setID("LST");//!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! I need something to return a string corresponding to a time zone shift value!!!
        _timezone.setRawOffset(base._nlstz * 3600 * 1000);

        //Line 1
        _filename = base._timeSeriesID; //I just had to find something to put in here.
        _datatype = new String(base._dataType);
        _datadim = new String(base._dim);
        _dataunits = new String(base._tsunit);
        _tsdt = base._tsdt;
        _timeseriesid = new String(base._timeSeriesID);
        _tsdesc = new String(base._segdesc);

        //Line 2
        _im = base._im;
        _iy = base._iy;

        //Get the last date in the base DataSet.  The ESPData arrives with the julian hours
        //adjust for GMT.  However, I need to get the original time zone because, as can be
        //see in the last lines of this method, we are assuming the original time zone for 
        //the data.  Thus, I need to ADD the _nlstz value in order to undo the subtractions
        //that result in the julian hour data values.
        int lastjhr = (int)base.getValue(base.getSampleSize() - 1, ESPData.JULIAN_HOUR);
        lastjhr += base._nlstz;
        final Calendar cal = HCalendar.computeCalendarFromJulianHour(lastjhr);

        //If the hour inside of cal is hour 0, then we need to adjust back one day because 
        //it should actually be hour 24 of the previous day in the strange world of the
        //NWSRFS!  Since we only need to acquire the month and year, I should be able
        //to get away with a one day shift backwards if the hour is 0.  This is particularly
        //important when the last day of day is hour 24 of the last day of a month.  This 
        //will result in cal being the first day (hour 0) of the NEXT month.  Hence, I must
        //shift back to account for this.
        if(cal.get(Calendar.HOUR_OF_DAY) == 0)
        {
            cal.add(Calendar.HOUR, -1);
        }

        //Compute the last date components
        _lm = cal.get(Calendar.MONTH) + 1;
        _ly = cal.get(Calendar.YEAR);

        //Determine the values per line and values format based on the time step of the timeseries...
        _valuesperline = DEFAULT_VALUESPERLINE;
        _valuesformat = new String(DEFAULT_VALUESFORMAT);

        //Processed data. 
        processHeader();

        //I know that the ESPData is in the time zone specified by the shift factor base._nlstz.  To
        //account for this time zone without actually using TimeZone, I need to shift the start and
        //end julian hours by this value.  I'll do it manually here.
        _firstjhr -= base._nlstz;
        _firstdate = HCalendar.computeCalendarFromJulianHour(_firstjhr);
        _lastjhr -= base._nlstz;
        _lastdate = HCalendar.computeCalendarFromJulianHour(_lastjhr);

    }

    //This constructor builds this DatacardData using passed in info.
    public DatacardData(final DataSet base,
                        final String filename,
                        final String datatype,
                        final String datadim,
                        final String dataunits,
                        final int tsdt,
                        final String timeseriesid,
                        final String tsdesc,
                        final int im,
                        final int iy,
                        final int lm,
                        final int ly,
                        final int valuesperline,
                        final String valuesformat) throws DatacardDataException
    {
        super(base);

        //First, set the time zone.
        setTimeZone(DEFAULT_TIMEZONE);

        //Now copy the passed in values!!!
        _filename = filename;
        _datatype = datatype;
        _datadim = datadim;
        _dataunits = dataunits;
        _tsdt = tsdt;
        _timeseriesid = timeseriesid;
        _tsdesc = tsdesc;
        _im = im;
        _iy = iy;
        _lm = lm;
        _ly = ly;
        _valuesperline = valuesperline;
        _valuesformat = valuesformat;

        processHeader();
    }

    /////////////////////////////////////////////////////////////////////////
    //Input
    /////////////////////////////////////////////////////////////////////////

    /**
     * read in the header portion of the file -- which should be the first two valid data lines.
     * 
     * @param localfile The BufferedReader from which to read the header.
     * @throws IOException if reading fails for any reason.
     */
    private void readInHeader(final BufferedReader localfile) throws IOException, DatacardDataException
    {
        String aline;

        /////////////////////////////////////////////////////////////////////////////////
        //Get Line 1, and make sure I have the minimum requisite number of characters, enough
        //to get me to the last element of the line.
        aline = retrieveNextDataLine(localfile);
        if(aline.length() < 31)
        {
            throw new DatacardDataException("Invalid line found in header; expected at least 31 characters: \"" + aline
                + "\".");
        }

        //Process the line
        _filename = aline.substring(0, 11 + 1);
        //Skip 12, 13
        _datatype = aline.substring(14, 17 + 1);
        //Skip 18
        _datadim = aline.substring(19, 22 + 1);
        //Skip 23
        _dataunits = aline.substring(24, 27 + 1);
        //Skip 28
        try
        {
            _tsdt = Integer.parseInt(aline.substring(29, 30 + 1).trim());
        }
        catch(final NumberFormatException e)
        {
            _tsdt = (int)DataSet.MISSING;
            throw new DatacardDataException("Time series time step not an integer; found to be \""
                + aline.substring(29, 30 + 1).trim() + "\".");
        }
        //Skip 31, 32, 33
        if(aline.length() > 35)
        {
            _timeseriesid = aline.substring(34, Math.min(aline.length(), 45 + 1));
        }
        //Skip 46, 47, 48
        if(aline.length() > 50)
        {
            _tsdesc = aline.substring(49, Math.min(aline.length(), 68 + 1));
        }

        /////////////////////////////////////////////////////////////////////////////////
        //Get Line 2 and make sure it has the requisite number of characters -- enough to get
        //me to the last element of the line.
        aline = retrieveNextDataLine(localfile);
        if(aline.length() < 25)
        {
            throw new DatacardDataException("Invalid line found in header; expected at least 25 characters: \"" + aline
                + "\".");
        }
        try
        {
            _im = Integer.parseInt(aline.substring(0, 1 + 1).trim());
            //Skip 2,3
            _iy = Integer.parseInt(aline.substring(4, 7 + 1).trim());
            //Skip 8
            _lm = Integer.parseInt(aline.substring(9, 10 + 1).trim());
            //Skip 11, 12, 13
            _ly = Integer.parseInt(aline.substring(14, 17 + 1).trim());
            //Skip 18
            _valuesperline = Integer.parseInt(aline.substring(19, 20 + 1).trim());
        }
        catch(final NumberFormatException e)
        {
            throw new DatacardDataException("A required parameter is not an integer in this line: \"" + aline + "\".");
        }
        //Skip 21, 22, 23
        _valuesformat = aline.substring(24, Math.min(aline.length(), 31 + 1));
        _valuesformat = _valuesformat.trim(); //-> Its length does not matter, and a trimmed string is easier to deal with.

    }

    /**
     * Compute class variables using header data.
     * 
     * @return False if the _valuesformat string is invalid.
     */
    private void processHeader() throws DatacardDataException
    {
        //Compute the dates -- initialize using computeCalendarFromJulianHour
        //NOTE: If the time ever changes from 1..24 to 0..23, then I will need to change how I define both of these so that
        //  _firstjhr is from hour 0 and _lastjhr is its current value minus _tsdt!
        _firstjhr = HCalendar.computeJulianHour(_iy, _im - 1, 1, _tsdt);
        _firstdate = HCalendar.computeCalendarFromJulianHour(_firstjhr);
        _lastjhr = HCalendar.computeJulianHour(_ly + (_lm / 12), (_lm % 12), 1, 0); //This will always set it as the NEXT month at 0 GMT.
        _lastdate = HCalendar.computeCalendarFromJulianHour(_lastjhr);

        //NOTE: I add (int)(_lm/12) to the year, because if the last month is December (12), the I actually
        //      set the last julian hour to be 1/1 at 00 GMT for the NEXT year (i.e. 12/_ly at 24 GMT).

        //NEED TO PROCESS DATA VALUE FORMAT HERE (_valuesformat).
        //Will assume float here for now...
        //Get the total values width
        try
        {
            _totalvaluewidth = Integer.parseInt(_valuesformat.substring(1, _valuesformat.indexOf('.')));
            _decimaldigits = Integer.parseInt(_valuesformat.substring(_valuesformat.indexOf('.') + 1,
                                                                      _valuesformat.length()));
        }
        catch(final NumberFormatException e)
        {
            throw new DatacardDataException("Format of values in invalid; must be [field width].[decimal digits]: \""
                + _valuesformat + "\".");
        }

        //Construct the positions array for use with SegmentedLine.
        _positions = new int[_valuesperline + 1];
        int i = 0;
        for(i = 0; i <= _valuesperline; i++)
        {
            _positions[i] = VALUES_POSITION_IN_RECORD + (i) * _totalvaluewidth;
        }

        //Compute todayyear as the year of the present.  This is used in the windowing technique used in
        //processYear below.
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        final Calendar today = Calendar.getInstance(tz);
        _todayyear = today.get(Calendar.YEAR);
    }

    /**
     * Read in the data, line by line. This uses processDataLineValues and processDataLineDate to read in all the
     * values. Its only requirements are that the dates are in increasing order and that the first data is for the first
     * jhr and the last data is for the last jhr.
     * 
     * @param localfile The file tor read.
     * @return False if something appears to be invalid. A message in printed via Messenger describing the problem.
     * @throws IOException
     */
    private void readInData(final BufferedReader localfile) throws IOException, DatacardDataException
    {
        //Some variables
        Vector values;
        Double d;
        double[] asample = new double[2];
        int i;

        //Initialize variables for the looping.
        String aline = retrieveNextDataLine(localfile);
        int currentmonthjhr = (int)DataSet.MISSING;
        int firstofmonthjhr = 0;
        int endofmonthjhr = 0;
        int workingjhr = _firstjhr;

        //I want to loop from the _firstjhr to the _lastjhr, only inserting non-MISSING values.
        boolean done = false;
        while((!done) && (workingjhr <= _lastjhr))
        {
            //Is the line null?  If it is quit out -- this should be the EOF.
            if(aline == null)
            {
                done = true;
                continue;
            }

            //Get the values and the first day of the working month from the line.
            values = processDataLineValues(aline);
            firstofmonthjhr = processDataLineJulianHour(aline);

            //Look at the current month... does it need processing?  If so, update the current month
            //and set the working julian hour to be the first of the month, because I now have a new
            //month of data to process.
            if((currentmonthjhr < 0) || (currentmonthjhr != firstofmonthjhr))
            {
                //If this is true, then I'm reading in my first data line.  Make sure it corresponds
                //to _firstjhr.
                if((currentmonthjhr < 0) && (firstofmonthjhr != _firstjhr))
                {
                    throw new DatacardDataException("The first month jhr of data, " + firstofmonthjhr
                        + ", does not correspond to that specified in the header, " + _firstjhr + "; for line \""
                        + aline + "\".");
                }

                //Make sure the upcoming month is greater than this past data month.
                if(firstofmonthjhr <= currentmonthjhr)
                {
                    throw new DatacardDataException("This month of data is before the preceding month, which is not allowed; for line \""
                        + aline + "\".");
                }

                //Was the previous month complete -- if (workingjhr - _tsdt) < endofmonthjhr, then it was not.
                //workingjhr - _tsdt was the last data value actually processed from the records.
                if((workingjhr - _tsdt) < endofmonthjhr)
                {
                    throw new DatacardDataException("The PRECEDING month of data was incomplete ("
                        + (workingjhr - _tsdt) + " < " + endofmonthjhr + "); for line \"" + aline + "\".");
                }

                currentmonthjhr = firstofmonthjhr;
                workingjhr = currentmonthjhr;
                endofmonthjhr = processDataLineEndOfMonthJulianHour(aline);
            }

            //If I've got values to process...
            if((values != null) && (currentmonthjhr > 0))
            {
                //Loop through the values, adding them one at a time if not MISSING.
                for(i = 0; i < values.size(); i++)
                {
                    //I've still got a value to process within this month, but I just went off the end.
                    if(workingjhr > endofmonthjhr)
                    {
                        throw new DatacardDataException("In the line \"" + aline + "\":"
                            + "Data goes beyond end of the month.  Extra values are discarded.");
                    }

                    //Get the current working value.
                    d = (Double)values.elementAt(i);

                    //Create and add the sample, if its not missing or -998 (ACCUMULATED_VALUE).
                    if((d.doubleValue() != DataSet.MISSING) && (d.doubleValue() != ACCUMULATED_VALUE))
                    {
                        asample = new double[2];
                        asample[JULIAN_HOUR] = workingjhr;
                        asample[VALUE] = d.doubleValue();
                        addSample(asample);

                        //Check the sample size now... if I'm approaching maximum size, then increase it.
                        if(getSampleSize() == (getMaximumSampleSize() - 1))
                        {
                            changeMaximumNumberOfSamples(getMaximumSampleSize() + BASE_SAMPLE_SIZE);
                        }
                    }

                    //Goto the text time step and if I've gone to far, break out of the for loop.
                    workingjhr += _tsdt;
                }
            }

            //Get the next readable line...
            aline = retrieveNextDataLine(localfile);
        }

        //Make sure the last value I processed was the end of the last month of data.
        if((workingjhr - _tsdt) != _lastjhr)
        {
            throw new DatacardDataException("The last month of data does not correspond to that specified in the header; for line \""
                + aline + "\".");
        }
    }

    /**
     * Get the next meaningful line. A return of null marks the end of the file. A return of "" marks
     * 
     * @param localfile File to read.
     * @return Next line in the file that is not commented.
     */
    private String retrieveNextDataLine(final BufferedReader localfile)
    {
        boolean done = false;
        String aline = "";
        while(!done)
        {
            //Try to read a line from the input stream
            try
            {
                aline = localfile.readLine();

                //If aline is null, then I must be at the end of the file, though why I didn't
                //receive an IOException, I don't know.
                if(aline == null)
                {
                    return null;
                }

                //If the first character is not a '$', then I need to process it.
                if(aline.charAt(0) != '$')
                {
                    return aline;
                }
            }
            //I've reached the end of the file...
            catch(final Exception e)
            {
                done = true;

                //If aline was null, then just return null, marking the end of the file.
                if(aline == null)
                {
                    return null;
                }

                //If aline had no length, then just return null, marking the end of the file.
                if(aline.length() == 0)
                {
                    return null;
                }

                //If the first character is not a '$', then I need to process it.
                if(aline.charAt(0) != '$')
                {
                    return aline;
                }
            }
        }
        return null;
    }

    //Return a vector of Double objects containing the numbers within this line.  
    @SuppressWarnings("unchecked")
    private Vector processDataLineValues(final String aline)
    {
        if(aline == null)
        {
            return null;
        }

        //FOR NOW I AM SKIPPING TO CHARACTER 20 (starting at 0) TO REACH THE DATA IMMEDIATELY.
        //If I can ever think of a reason to check the first 20 characters, I'll insert the check here.

        //build the returned array one element at a line by breaking up the string consecutive.
        String valuestr;
        final Vector values = new Vector();

        //Segment the line by fixed width block computed in processHeader.
        final SegmentedLine segline = new SegmentedLine(aline, _positions);

        //Go through the segments, starting with the second one, one by one.
        //Count starts at 1 because the first segment is the bs before the values.
        int count;
        for(count = 1; count < segline.getNumberOfSegments(); count++)
        {
            //Get the value string
            valuestr = segline.getSegment(count);

            //Turn it into a Double and add it to the vector
            try
            {
                values.addElement(Double.valueOf(valuestr.trim()));
            }
            catch(final NumberFormatException e)
            {
                //I've failed to process this double, so I am going to assume the end of line is reached.
                return values;
            }
        }

        return values;
    }

    //Return the Calendar which specifies, for the line passed in, what the first data hour of the first day
    //of the month for that line will be.  This is used inside readInData to know when it switches months
    //and what month it switches to.
    private int processDataLineJulianHour(final String aline) throws DatacardDataException
    {
        //Segment the passed in line based on the static array DATE_POSITION_IN_RECORD.
        final SegmentedLine segline = new SegmentedLine(aline, DATE_POSITION_IN_RECORD);

        int month;
        int year;

        //Try to get the month and year of the record from the line.
        try
        {
            month = Integer.parseInt(segline.getSegment(1).trim());
            year = Integer.parseInt(segline.getSegment(2).trim());
        }
        catch(final NumberFormatException e)
        {
            throw new DatacardDataException("Unable to read/process start month and/or year from the line: " + aline);
        }

        //Compute the julian hour...
        final int datejhr = HCalendar.computeJulianHour(HCalendar.processTwoDigitYear(year, _todayyear),
                                                        month - 1,
                                                        1,
                                                        _tsdt);

        //Return the calendar corresponding to datejhr
        return datejhr;
    }

    //Get the julian hour marking the last value within that month.
    private int processDataLineEndOfMonthJulianHour(final String aline) throws DatacardDataException
    {
        //Segment the passed in line based on the static array DATE_POSITION_IN_RECORD.
        final SegmentedLine segline = new SegmentedLine(aline, DATE_POSITION_IN_RECORD);

        int month;
        int year;

        //Try to get the month and year of the record from the line.
        try
        {
            month = Integer.parseInt(segline.getSegment(1).trim());
            year = Integer.parseInt(segline.getSegment(2).trim());
        }
        catch(final NumberFormatException e)
        {
            throw new DatacardDataException("Unable to read/process end month and/or year from the line: " + aline);
        }

        //Compute the julian hour...
        int datejhr;
        if(month == 12)
        {
            datejhr = HCalendar.computeJulianHour(HCalendar.processTwoDigitYear(year, _todayyear) + 1, 0, 1, 0);
        }
        else
        {
            datejhr = HCalendar.computeJulianHour(HCalendar.processTwoDigitYear(year, _todayyear), month, 1, 0);
        }

        //Return the calendar corresponding to datejhr
        return datejhr;
    }

    //Perform a shift on the julian hour for the Time Zone of the data read in. 
    //All the data was read in ASSUMING it was in julian hours.  However, the data was most
    //likely not in julian hours.  To account for this I need to shift the JULIAN_HOUR variable
    //accordingly.
    //
    //Example: If the data is in EST.  The first value I read in had a julian hour for 1/1/1999 6h GMT.
    //  However, it was supposed to be the julian hour for 1/1/1999 6h EST, which is 1/1/1999 11h GMT.  So,
    //  my julian hour was 5 hours too few.  So, I add 5 hours to all of the samples.
    private void doTimeZoneShift()
    {
        //I need the offset in milliseconds.
        final int rawoffset = _timezone.getRawOffset();

        //Divide it by 3600*1000 to remove the millisecond and get the hours.
        final int hourshift = rawoffset / (3600 * 1000);

        //Hour shift is now the number of hours shifted to get FROM GMT TO _timezone.  However, I am
        //actually trying to go FROM _timezone TO GMT, meaning I SUBTRACT this shift value.

        //Do a shift on the data.  I multiply by -1, because I am trying to shift a julian hour thats actually in
        //the _timezone time to one that is in GMT.
        if(!shiftVariable(JULIAN_HOUR, -1 * hourshift))
        {
            return;
        }

        //Now, shift the attributes _firstjhr and _lastjhr, because they were computed assuming GMT data.
        _firstjhr -= hourshift;
        _firstdate = HCalendar.computeCalendarFromJulianHour(_firstjhr);
        _lastjhr -= hourshift;
        _lastdate = HCalendar.computeCalendarFromJulianHour(_lastjhr);

        return;
    }

    /////////////////////////////////////////////////////////////////////////
    //Output
    /////////////////////////////////////////////////////////////////////////

    //Write out the header information to a datacard file.
    private void writeDatacardOutputHeader(final BufferedWriter localfile) throws IOException
    {
        if(localfile == null)
        {
            throw new IOException();
        }

//        String tempstr;
        final NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(0);
        nf.setMinimumFractionDigits(0);
        nf.setMaximumIntegerDigits(2);
        nf.setMinimumIntegerDigits(2);

        /////////////////////////////////////////////////////////////////////////////////
        //Write Line 1
        String aline = "";
        aline += HString.formatStringToFieldWidth(12, _filename, true);
        aline += "  ";
        aline += HString.formatStringToFieldWidth(4, _datatype, true);
        aline += " ";
        aline += HString.formatStringToFieldWidth(4, _datadim, true);
        aline += " ";
        aline += HString.formatStringToFieldWidth(4, _dataunits, true);
        aline += " ";
        aline += HString.formatStringToFieldWidth(2, "" + _tsdt, true);
        aline += "   ";
        aline += HString.formatStringToFieldWidth(12, _timeseriesid, true);
        aline += "   ";
        aline += HString.formatStringToFieldWidth(20, _tsdesc, true);
        localfile.write(aline, 0, aline.length());
        localfile.newLine();

        /////////////////////////////////////////////////////////////////////////////////
        //Write line 2
        aline = "";
        aline += HString.formatStringToFieldWidth(2, "" + _im, true);
        aline += "  ";
        aline += _iy;
        aline += " ";
        aline += HString.formatStringToFieldWidth(2, "" + _lm, true);
        aline += "   ";
        aline += _ly;
        aline += " ";
        aline += HString.formatStringToFieldWidth(2, "" + _valuesperline, true);
        aline += "   ";
        aline += _valuesformat;
        localfile.write(aline, 0, aline.length());
        localfile.newLine();
        localfile.flush();
    }

    //Write the data portion of the datacard file.   
    private void writeDatacardOutputData(final BufferedWriter localfile) throws IOException
    {
        //Sort the data by the julian hour and reset the ptr.
        sortBy(JULIAN_HOUR);
        resetPtr();

        //Floating point number formatter
        final NumberFormat fnf = NumberFormat.getInstance();
        fnf.setMaximumFractionDigits(_decimaldigits);
        fnf.setMinimumFractionDigits(_decimaldigits);
        fnf.setMaximumIntegerDigits(_totalvaluewidth - _decimaldigits - 1);
        fnf.setGroupingUsed(false);

        //Integer number formatter.
        final NumberFormat znf = NumberFormat.getInstance();
        znf.setMaximumFractionDigits(0);
        znf.setMinimumFractionDigits(0);
        znf.setMaximumIntegerDigits(2);
        znf.setMinimumIntegerDigits(2);

        //Some variables.
        String aline = "";
        boolean donewithdata, donewritingfile, newmonth;
        int currentline = 1;
        int currentlineentry = 0;

        //HERE's what the jhr and dates store:
        //datadate        -- stores the date associated with the next sample in the DataSet.
        //datadatejhr     -- its julian hour.
        //currentdate     -- stores the date associated with the next printed value.
        //currentdatejhr  -- its julian hour.
        //currentfomjhr   -- the julian hour of the first of the month corresponding to the next printed value.
        //workingblockjhr -- the date of the first of the month for the current working block.
        Calendar datadate;
        int datadatejhr;
        Calendar currentdate;
        int currentdatejhr;
        int currentfomjhr = 0;
        int workingblockjhr = (int)DataSet.MISSING;

        //I need the offset in hours.  This will be the number of hours to shift to get FROM GMT TO
        //_timezone.  Hence, this is an additive factor.
        final int rawoffset = _timezone.getRawOffset();
        final int hourshift = rawoffset / (3600 * 1000);

        //NOTE ON HOUR SHIFTING:
        //The data in the data set is in GMT, but the output must follow a pattern prescribed by the
        //TimeZone _timezone.  So, I'm going to compute julian hour dates in LST (i.e. from 1/1/1900 0h LST)
        //and then adjust the julian hours that come out of the DataSet by ADDING hourshift to shift it
        //from GMT to LST.  

        //Get the first samples date and first of month jhr.
        datadatejhr = (int)getCurrentValue(JULIAN_HOUR) + hourshift; //Shift the GMT value to LST!
        datadate = HCalendar.computeCalendarFromJulianHour(datadatejhr);

        currentfomjhr = computeFirstOfMonthJHR(datadate.get(Calendar.YEAR),
                                               datadate.get(Calendar.MONTH),
                                               hourshift,
                                               datadate.get(Calendar.HOUR_OF_DAY));

        //currentfomjhr = HCalendar.computeJulianHour(datadate.get(Calendar.YEAR), datadate.get(Calendar.MONTH), 1, _tsdt); //Viewed as LST!
        currentdate = HCalendar.computeCalendarFromJulianHour(currentfomjhr);
        currentdatejhr = currentfomjhr;
        final int hourOfDayToUseForFOM = currentdate.get(Calendar.HOUR_OF_DAY);

        //System.out.println("####>> datadatejhr -- "
        //    + HCalendar.convertCalendarToString(HCalendar.computeCalendarFromJulianHour(datadatejhr)));
        //System.out.println("####>> currentfomjhr -- "
        //    + HCalendar.convertCalendarToString(HCalendar.computeCalendarFromJulianHour(currentfomjhr)));

        //This is to force hour 0 to be treated as hour 24...
        currentdate.add(Calendar.SECOND, -1);

        donewithdata = false;
        donewritingfile = false;
        newmonth = false;
        while(!donewritingfile)
        {
            //If the preceding month is not the same as the currentdate's month, then
            //this is a new month.
            //(i.e. if I reach the end of the month, then...)
            if(workingblockjhr != currentfomjhr)
            {
                newmonth = true;

                //Force the next if to printout this line.
                currentlineentry = _valuesperline;
            }

            //If I've just maxed out the line entries, then print the line...
            if(currentlineentry == _valuesperline)
            {
                //print out aline if I have one and clear it.
                if(aline.length() != 0)
                {
                    localfile.write(aline, 0, aline.length());
                    localfile.newLine();
                    localfile.flush();
                    currentline++;
                }
                aline = "";
                currentlineentry = 0;
            }

            //If this is a new month, then...
            if(newmonth)
            {
                //System.out.println("####>> Inside here!!! >> " + workingblockjhr + " != " + currentfomjhr);
                //System.out.println("####>>     datadate is " + HCalendar.convertCalendarToString(datadate, HCalendar.DEFAULT_DATETZ_FORMAT));
                //System.out.println("####>>     donewithdata = " + donewithdata);
                //Reset newmonth.
                newmonth = false;

                //If I just wrote the last month with any data from the data set, then quit the loop.
                if(donewithdata)
                {
                    donewritingfile = true;
                    continue;
                }

                workingblockjhr = this.computeFirstOfMonthJHR(datadate.get(Calendar.YEAR),
                                                              datadate.get(Calendar.MONTH),
                                                              hourshift,
                                                              hourOfDayToUseForFOM);

                //Reset precedingmonth to be the next data values's first-of-month.
                //workingblockjhr = HCalendar.computeJulianHour(datadate.get(Calendar.YEAR),
                ////                                              datadate.get(Calendar.MONTH),
                //                                              1,
                //                                              _tsdt); //Viewed as LST!  Being Initial value, I mod it!

                //Move current date stuff to workingblockjhr and I'll start from there.
                currentdate = HCalendar.computeCalendarFromJulianHour(workingblockjhr);
                currentfomjhr = workingblockjhr;
                //currentfomjhr = HCalendar.computeJulianHour(currentdate.get(Calendar.YEAR),
                //                                            currentdate.get(Calendar.MONTH),
                //                                            1,
                //                                            _tsdt); //Viewed as LST!  Being Initial value, I mod it!
                currentdatejhr = HCalendar.computeJulianHourFromCalendar(currentdate, true);
                //System.out.println("####>>     currentdate set to " + HCalendar.convertCalendarToString(datadate, HCalendar.DEFAULT_DATETZ_FORMAT));
                //System.out.println("####>>     currentfomjhr = " + currentfomjhr + "; currentdatejhr = " + currentdatejhr);

                //This is to force hour 0 to be treated as hour 24...
                currentdate.add(Calendar.SECOND, -1);
            }

            //If aline is empty, then setup aline for the next line.
            if(aline.length() == 0)
            {
                aline += HString.formatStringToFieldWidth(12, _timeseriesid, true);
                aline += HString.formatStringToFieldWidth(2, "" + (currentdate.get(Calendar.MONTH) + 1), false);
                aline += znf.format(currentdate.get(Calendar.YEAR) % 100);
                aline += HString.formatStringToFieldWidth(4, "" + currentline, false);
            }

            //If the value to print out now is the current value in the data set...
            //System.out.println("####>> Checking equality of " + currentdatejhr + " and " + datadatejhr);
            if(currentdatejhr == datadatejhr)
            {
                aline += HString.formatStringToFieldWidth(_totalvaluewidth, fnf.format(getCurrentValue(VALUE)), false);
                donewithdata = !next(); //If there is no more data, this next will not move the pointer, so the if above will fail until done.
                datadatejhr = (int)getCurrentValue(JULIAN_HOUR) + hourshift; //Shift from GMT to LST!
                datadate = HCalendar.computeCalendarFromJulianHour(datadatejhr);
                //System.out.println("####>> Just incremented datadate to " + datadatejhr);
            }
            //Otherwise, put in a missing value...
            else
            {
                aline += HString.formatStringToFieldWidth(_totalvaluewidth, fnf.format(-999.0), false);
            }

            //EMERGENCY END!
            if(currentdatejhr > HCalendar.computeJulianHourFromCalendar(_lastdate, false))
            {
                System.out.println("####>> UNEXPECTED END!!!");
                donewithdata = true;
            }

            //Increment the current date.
            currentdatejhr += _tsdt;
            //System.out.println("####>> " + currentdatejhr);
            currentdate.add(Calendar.HOUR, _tsdt);

            currentfomjhr = HCalendar.computeJulianHour(currentdate.get(Calendar.YEAR),
                                                        currentdate.get(Calendar.MONTH),
                                                        1,
                                                        hourOfDayToUseForFOM); //Viewed as LST!  Being Initial value, I mod it!
            currentlineentry++;

//            System.out.println("####>> ---------- current date jhr -- "
//                + HCalendar.convertCalendarToString(HCalendar.computeCalendarFromJulianHour(currentdatejhr)));
//            System.out.println("####>> ---------- datadatejhr -- "
//                + HCalendar.convertCalendarToString(HCalendar.computeCalendarFromJulianHour(datadatejhr)));
//            System.out.println("####>> ------------ currentfomjhr -- "
//                + HCalendar.convertCalendarToString(HCalendar.computeCalendarFromJulianHour(currentfomjhr)));
        }
    }

    private int computeFirstOfMonthJHR(final int year, final int month, final int hourShift, final int searchHourStart)
    {
        final int startOfMonthJulianHour = HCalendar.computeJulianHour(year, month, 1, 0) - hourShift;
        int currentfomjhr = HCalendar.computeJulianHour(year, month, 1, searchHourStart);
        while(currentfomjhr > startOfMonthJulianHour)
        {
            currentfomjhr -= _tsdt;
        }
        currentfomjhr += _tsdt;
        return currentfomjhr;
    }

    //Call the previous two routines to write a datacard file.
    public boolean writeDatacardOutputFile(final String filename) throws DatacardDataException
    {
        FileWriter thefile;
        BufferedWriter localfile;

        //Try to open the data file for reading.
        try
        {
            thefile = new FileWriter(filename);
            localfile = new BufferedWriter(thefile);
        }
        catch(final IOException e1)
        {
            throw new DatacardDataException("Failed to open requested file " + filename);
        }

        //Try to writeout the file contents.
        try
        {
            writeDatacardOutputHeader(localfile);
            writeDatacardOutputData(localfile);
        }
        catch(final IOException e)
        {
            throw new DatacardDataException("Failed to write out file information.");
        }
        finally
        {
            try
            {
                localfile.close();
                thefile.close();
            }
            catch(final IOException e)
            {
            }
        }

        return true;
    }

    //Call the previous two routines to write a datacard file.
    public boolean writeDatacardOutputFile() throws DatacardDataException
    {
        return writeDatacardOutputFile(_filename);
    }

    //Dump the header to standard out.
    public void dumpHeader()
    {
//        int i;

        System.out.println("========== DATACARD HEADER =========================");
        System.out.println("_filename      = \"" + _filename + "\"");
        System.out.println("_datatype      = \"" + _datatype + "\"");
        System.out.println("_datadim       = \"" + _datadim + "\"");
        System.out.println("_dataunits     = \"" + _dataunits + "\"");
        System.out.println("_tsdt          = " + _tsdt); //Time interval
        System.out.println("_timeseriesid  = \"" + _timeseriesid + "\"");
        System.out.println("_tsdesc        = \"" + _tsdesc + "\"");
        System.out.println("_im            = " + _im);
        System.out.println("_iy            = " + _iy);
        System.out.println("_lm            = " + _lm);
        System.out.println("_ly            = " + _ly);
        System.out.println("_valuesperline = " + _valuesperline);
        System.out.println("_valuesformat  = \"" + _valuesformat + "\"");
        System.out.println("====================================================");
    }

    //Dump the processed header variables.
    public void dumpProcessedHeader()
    {
        //Dump the calendar objects.
        System.out.println("========== DATACARD PROCESSED HEADER ===============");
        System.out.println("_firstjhr        = " + _firstjhr);
        System.out.println("_firstdate       = " + HCalendar.buildDateTimeTZStr(_firstdate));
        System.out.println("_lastjhr         = " + _lastjhr);
        System.out.println("_lastdate        = " + HCalendar.buildDateTimeTZStr(_lastdate));
        System.out.println("_totalvaluewidth = " + _totalvaluewidth);
        System.out.println("_decimaldigtis   = " + _decimaldigits);
        System.out.println("_todayyear       = " + _todayyear);
        System.out.println("TIMEZONE         = " + _timezone.getID());
        System.out.println("====================================================");
    }

    //Dump the data
    public void dumpData()
    {
        System.out.println("========== DATACARD DATA ===========================");
        MatrixMath.printMatrix(getData());
        System.out.println("====================================================");
    }

    //Dump the data
    public void dumpDataUsingDates()
    {
        System.out.println("========== DATACARD DATA ================================");
        MatrixMath.printMatrix(getData(), 0);
        System.out.println("====================================================");
    }

    /////////////////////////////////////////////////////////////////////////
    //TOOLS
    /////////////////////////////////////////////////////////////////////////

    /////////////////////////////////////////////////////////////////////////
    //Sets and Gets
    /////////////////////////////////////////////////////////////////////////

    public boolean setTimeZone(final String timezone)
    {
        //Internal time is a time zone shifted by 12 hours from GMT.
        if(timezone.equals("INTL"))
        {
            _timezone = TimeZone.getDefault();
            _timezone.setID("INTL");
            _timezone.setRawOffset(HCalendar.INTL_TZ_ADJ * 3600 * 1000);

            return true;
        }
        else
        {
            _timezone = TimeZone.getTimeZone(timezone);
        }
        return true;
    }

    //A test main.
    public static void main(final String args[])
    {
        if((args.length != 2) && (args.length != 3))
        {
            System.out.println("DatacardData version ob8.2 10/04/06");
            System.out.println("");
            System.out.println("Bad command line format.  Correct format:");
            System.out.println("    <program> <h,dd,dj,a,yt> <file> [<time zone string>]");
            System.out.println("where file is the file to be read and 'h' means only print");
            System.out.println("out the header, 'dd' means print out the header and data as");
            System.out.println("dates, 'dj' means print out the header and data as julian hours,");
            System.out.println("and 'a' means print out the header, data, and do 24-hour, 'yt'");
            System.out.println("means total the values for each of the years and provide the");
            System.out.println("sums.");
            System.out.println("The optional time zone string specifies what time zone to assume");
            System.out.println("the data is in.");
            System.out.println("");
            return;
        }

        //Process the second arg, if there is one.
        int outputlevel = 1;
        boolean dateprint = false;
        if(args[0].equals("h"))
        {
            outputlevel = 0;
        }
        else if(args[0].equals("dj"))
        {
            outputlevel = 1;
        }
        else if(args[0].equals("a"))
        {
            outputlevel = 2;
        }
        else if(args[0].equals("yt"))
        {
            outputlevel = 3;
        }
        else if(args[0].equals("special"))
        {
            outputlevel = 99;
        }
        else if(args[0].equals("dd"))
        {
            outputlevel = 1;
            dateprint = true;
        }

        DatacardData data;
        try
        {
            //For the "special" argument (outputlevel == 99), we are going to assume that
            //argument 3 is the name of the file we want to file in with data from the file
            //in argument 2.  Hence, we read in assuming GMT.
            if((args.length == 2) || ((args.length == 3) && (outputlevel == 99)))
            {
                System.out.println("Reading in file assuming GMT is the time zone.");
                data = new DatacardData(args[1]);
            }
            else
            {
                data = new DatacardData(args[1], args[2]);
            }

            //NOTE: This piece is hardcoded if the data 24-h.  Basically, I'm going to force it to associate
            //with each sample, the START of the day, not the END!!!
            if(data._tsdt == 24)
            {
                data.shiftVariable(DatacardData.JULIAN_HOUR, -24);
            }

            //If no averaging...
            if(outputlevel < 2)
            {
                System.out.println("##################### File Contents Read In: ####################");
                System.out.println("  If the data is 24-hour data, then the julian hour represents");
                System.out.println("  the beginning of each day.  Otherwise, it is the end of each");
                System.out.println("  time period.");
                System.out.println("#################################################################");
                if(outputlevel >= 0)
                {
                    data.dumpHeader();
                    data.dumpProcessedHeader();
                }
                if(outputlevel >= 1)
                {
                    if(dateprint)
                    {
                        data.dumpDataUsingDates();
                    }
                    else
                    {
                        data.dumpData();
                    }
                }
                System.out.println("#################################################################");
                System.out.println("");
                System.out.println("");
                return;
            }

            //If we are in the yearly averaging mode to get year sums, then...
            if(outputlevel == 3)
            {
                System.out.println("#################################################################");
                System.out.println("  Totals for each year.");
                System.out.println("#################################################################");

                //This is a subset DataSet that will be used to store data.
                DataSet subset;

                //Loop through each year.
                int year;
                for(year = data._firstdate.get(Calendar.YEAR); year <= data._lastdate.get(Calendar.YEAR); year++)
                {
                    subset = data.extractSubset(0, Calendar.YEAR, year);
                    System.out.println("" + year + "  " + subset.sum(1));
                }
                System.out.println("#################################################################");
                System.out.println("");
                System.out.println("");
                return;
            }

            //If we are running the special function, then...
            if(outputlevel == 99)
            {
                //Read in the second file.
                System.out.println("Reading in second file assuming GMT is the time zone.");
                final DatacardData data2 = new DatacardData(args[2]);
                int month;

                //Now, we start replacing the data in data2 with that in data.
                //If we run out of samples in data, then we loop back around.
                System.out.println("Using the first file to replace values in the second file.");
                data.resetPtr();
                data2.resetPtr();
                data2.setCurrentValue(1, data.getCurrentValue(1));
                while(data2.next())
                {
                    if(!data.next())
                    {
                        System.out.println("First file ran out of values at jhr " + data.getCurrentValue(0));
                        System.out.println("  Looping to beginning.");
                        data.resetPtr();
                    }

                    month = data.getCurrentValueAsDate(0).get(Calendar.MONTH);
                    if((month != 6) && (month != 7) && (month != 8))
                    {
                        data2.setCurrentValue(1, data.getCurrentValue(1));
                    }
                }

                //Output the datacard file.
                System.out.println("Writing new datacard file \"" + args[2] + ".new\".");
                data2.writeDatacardOutputFile(args[2] + ".new");
                return;
            }

            //I am doing averaging.

            //startjhr is the julian hours in GMT!  Now, I want to get the first hour of THAT day
            //in the timezone of the DatacardData!  First, get the offset in milliseconds.
            final int rawoffset = data._timezone.getRawOffset();

            //Divide it by 3600*1000 to remove the millisecond and get the hours.
            final int hourshift = rawoffset / (3600 * 1000);

            //Get the smallest julian hour.
            double startjhr = data.getSmallest(0);

            //hourshift is the number of hours to shift to go FROM GMT TO the file time zone.  Hence,
            //it is ADDED to the startjhr, since it is already in GMT.
            startjhr += hourshift;

            //If I convert the julian hour to a calendar, now, it will have the date components 
            //in local time.          
            final Calendar startdate = HCalendar.computeCalendarFromJulianHour((int)startjhr);

            //Set the hour to be 0 to get hour 0 of that day to be the start hour.
            startdate.add(Calendar.HOUR, -1 * startdate.get(Calendar.HOUR_OF_DAY));

            //Now, when I convert back, I'll get the julian hour in the ESPData time zone that
            //corresponds to hour 0 of the first day.  
            startjhr = HCalendar.computeJulianHourFromCalendar(startdate, true);

            //But, I need the julian hour in GMT for it to be the same timezone as the data I have.
            startjhr -= hourshift;

            //THESE LINES DEAL WITH DAILY AVERAGING...            
            System.out.println("");
            System.out.println("Doing the daily average assuming the start jhr and date in GMT is:");
            System.out.println("    "
                + startjhr
                + " -- "
                + HCalendar.buildDateStr(HCalendar.computeCalendarFromJulianHour((int)startjhr),
                                         HCalendar.DEFAULT_DATETZ_FORMAT));
            System.out.println("If you add the offset of the time zone given in the header, you will");
            System.out.println("see the date in the time zone of the ESPData file.  Check that this is");
            System.out.println("correct to be certain my hours are right.");
            System.out.println("");

            //Get the averaged data set.  One more thing... for this data, I want the hour to be 
            //associated with the END of it 24-h period... not the beginning.  So, I need to shift it!
            final DataSet avgdata = data.temporalAverage(0, startjhr, 24, false, false);
            avgdata.shiftVariable(JULIAN_HOUR, 24);
            final DatacardData newdata = new DatacardData(data);
            newdata._tsdt = 24;
            //newdata.setSamples(avgdata.getData());

            System.out.println("##################### Daily Averages: ###########################");
            System.out.println("  The first var is the julian hour, in GMT, associated with the");
            System.out.println("  END of the day");
            System.out.println("#################################################################");
            //newdata.dumpHeader();
            //newdata.dumpProcessedHeader();
            //newdata.dumpData();
            System.out.println("#################################################################");
            System.out.println("");

            //Output the read in data.
            System.out.println("##################### Creating Output File ####################");
            newdata.writeDatacardOutputFile("testout.txt");

        }
        catch(final DatacardDataException e)
        {
            System.out.println("Failed to read file for this reason: " + e.getMessage());
        }

    }

}
