package ohd.hseb.util.data;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.NumberFormat;
import java.util.Calendar;

import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.util.misc.HBinaryInputStream;
import ohd.hseb.util.misc.HBinaryOutputStream;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.Messenger;
import ohd.hseb.util.misc.SegmentedLine;
import ohd.hseb.util.misc.SoftwareVersionNumbers;

/**
 * This class reads in and stores the data contained within an ESPTS file. It can also write it out to a new ESPTS file.
 * The variables in the data set are <julian hour> <value>, where the julian hour is in GMT. Hence, ALL JULIAN HOUR
 * VALUES ARE CONVERTED TO GMT PRIOR TO STORAGE! It uses the _nlstz to get the GMT adjustment value to do the
 * conversion. Note that all there are, as of ob4, no get methods for the attributes (this is laziness on my part).
 * Hence, the attributes should be accessed directly, such as <object>._formatver, for example.
 * 
 * @author hank
 */
public class ESPData extends DataSet
{
    final static String CLASSNAME = "ESPData";
    final static int NUMBER_OF_FILENAME_SEGMENTS = 5; //Required number of components of the file name.
    final static int DEFAULT_GROWTH_SIZE = 100; //This is currently unused.
    final public static int JULIAN_HOUR = 0;
    final public static int VALUE = 1;

    //Data from the file name...
    public String _segmentID;
    public String _timeSeriesID;
    public String _dataType;
    public String _timeSeriesTypeExtension;
    public boolean _hsFlag;
    private boolean _hsDebug = false; // Used for printing out all values from an HS file

    //Straight From The File Header!!!
    //These attribute names will match those seen in ESPTraceEns_readTraceEnsHdr.cxx for ESPADP.
    //The only difference is that I will remove any _'s from the middle.
    public float _formatver;//formatting version number.
    public String _fsegid; //8 chars -- segment id.
    public String _ftsid; //8 chars -- ts id.
    public String _ftype; //4 chars -- data type.
    public int _tsdt; //TS time step
    public int _simflag; //simulation flag, whatever that means.
    public String _tsunit; //4 chars
    public int[] _now; //5 integers
    public int _im; //First month of data
    public int _iy; //First year of data
    public int _idarun; //Initial julian day of run, from Jan 1, 1900, in the time zone specified by _nlstz.
    public int _ldarun; //Last julian day of run, from Jan 1, 1900, in the time zone specified by _nlstz
    public int _ijdlst; //Carryover julian day, from Jan 1, 1900, in the time zone specified by _nlstz
    public int _ihlst; //Carryover julian hour, from Jan 1, 1900, in the time zone specified by _nlstz
    public int _ljdlst; //Forecast ending julian day, from Jan 1, 1900, in the time zone specified by _nlstz
    public int _lhlst; //Forecast ending julian hour, from Jan 1, 1900, in the time zone specified by _nlstz
    public int _ntraces; //Number of traces?
    public int _ncm; //Number of conditional months per trace
    public int _nlstz; //time zone number
    public int _noutds; //Thisis a daylight savings adjustment to the time zone number.  When 1, I'm in 
    //daylight savings time.
    public int _irec; //REC# trace data ???
    public String _dim; //4 chars -- unit dimensions
    public String _tscale; //4 chars -- time scale of code (inst, mean, etc.)
    public String _segdesc; //20 chars -- segment description
    public float _lat; //latitude
    public float _long; //longitude
    public String _fgrp; //8 chars -- forecast group
    public String _cgrp; //8 chars -- carryover group
    public String _rfcname; //8 chars -- rfc name
    public String _espfile; //80 chars -- trace file name
    public String _prsfstr; //80 chars -- prsf string -- if it is "PRSF", then I've got a prsf.
    public String _esptext; //80 chars -- user comments
    public int _adjcount; //adjustment counter 0-9

    //The Processed File Header Variables!!!
    public Calendar _fcstStart; //The beginning of the requested forecast time period, specified by the user of ESP.
    public Calendar _fcstEnd; //The end.
    public Calendar _runStart; //When the data actually starts...
    public Calendar _runEnd; //and ends.
    public int _numberPerMonth; //The number of trace values to read in for each month.
    public int _numberPerTrace; //The number of trace values for each trace.

    /**
     * Object specifying the file name for the ESPTS file and its parent directory.
     */
    private File _esptsFile;

    /////////////////////////////////////////////////////////////////////////
    //Constructors
    /////////////////////////////////////////////////////////////////////////

    /**
     * Empty constructor, used for building ESP time series files.
     * 
     * @param fileName The name of the ESP time series file complete with relative or absolute path.
     * @param readInFile True if it is to read in the file immediately. Use false when creating an ESP time series file.
     */
    public ESPData(final String fileName, final boolean readInFile, final boolean hsDebug) throws ESPDataException
    {
        _now = new int[5];
        _esptsFile = new File(fileName);
        setHsDebug(hsDebug);
        if(!processFileName(_esptsFile.getName()))
        {
            Messenger.writeMsg(Messenger.EVT_DATA + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT, "The file name \""
                + fileName + "\" is of improper format.\n");
            throw new ESPDataException("The file name \"" + fileName + "\" is of improper format.\n");
        }
        if(readInFile)
        {
            readInESPTSFile();
        }
    }

    public ESPData(final String fileName, final boolean readInFile) throws ESPDataException
    {
        this(fileName, readInFile, false);
    }

    /**
     * @param filename The name of the file to load complete with relative or absolute path.
     * @throws ESPDataException if the file name is unacceptable or the data is bad.
     */
    public ESPData(final String fileName) throws ESPDataException
    {
        this(fileName, true, false);
    }

    /**
     * Copy constructor.
     */
    public ESPData(final ESPData base)
    {
        super(base);
        int i;

        //Set all the attributes.//Data from the file name...
        _segmentID = new String(base._segmentID);
        _timeSeriesID = new String(base._timeSeriesID);
        _dataType = new String(base._dataType);
        _timeSeriesTypeExtension = new String(base._timeSeriesTypeExtension);
        _hsFlag = base._hsFlag;

        _formatver = base._formatver;
        _fsegid = new String(base._fsegid);
        _ftsid = new String(base._ftsid);
        _ftype = new String(base._ftype);
        _tsdt = base._tsdt;
        _simflag = base._simflag;
        _tsunit = new String(base._tsunit);
        final int[] _now = new int[5];
        for(i = 0; i < 5; i++)
        {
            _now[i] = base._now[i];
        }
        _im = base._im;
        _iy = base._iy;
        _idarun = base._idarun;
        _ldarun = base._ldarun;
        _ijdlst = base._ijdlst;
        _ihlst = base._ihlst;
        _ljdlst = base._ljdlst;
        _lhlst = base._lhlst;
        _ntraces = base._ntraces;
        _ncm = base._ncm;
        _nlstz = base._nlstz;
        _noutds = base._noutds;
        _irec = base._irec;
        _dim = new String(base._dim);
        _tscale = new String(base._tscale);
        _segdesc = new String(base._segdesc);
        _lat = base._lat;
        _long = base._long;
        _fgrp = new String(base._fgrp);
        _cgrp = new String(base._cgrp);
        _rfcname = new String(base._rfcname);
        _espfile = new String(base._espfile);
        _prsfstr = new String(base._prsfstr);
        _esptext = new String(base._esptext);
        _adjcount = base._adjcount;

        _fcstStart = (Calendar)(base._fcstStart.clone());
        _fcstEnd = (Calendar)(base._fcstEnd.clone());
        _runStart = (Calendar)(base._runStart.clone());
        _runEnd = (Calendar)(base._runEnd.clone());
        _numberPerMonth = base._numberPerMonth;
        _numberPerTrace = base._numberPerTrace;
    }

    /////////////////////////////////////////////////////////////////////////
    //FILE I/O
    /////////////////////////////////////////////////////////////////////////

    /**
     * Read in the ESPTS file specified in the _esptsFile attribute, which was passed into the constructor.
     * 
     * @throws ESPDataException If problems occur.
     */
    public void readInESPTSFile() throws ESPDataException
    {
        final String fileName = _esptsFile.getAbsolutePath();
        _now = new int[5];

        FileInputStream filein;
        HBinaryInputStream datafile;

        //Look at the file info.
        final File thefile = new File(fileName);

        //Check for existence and reability.
        if(!thefile.exists() || !thefile.canRead())
        {
            Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT, "File \""
                + fileName + "\" either does not exist or is not readable.\n");
            throw new ESPDataException("File \"" + fileName + "\" either does not exist or is not readable.");
        }

        //Try to open the data file for reading.
        try
        {
            filein = new FileInputStream(thefile);
            datafile = new HBinaryInputStream(filein);
        }
        catch(final FileNotFoundException e1)
        {
            //JOptionPane.showMessageDialog(null, "Unable to open sac_sma file: " + binfilename,
            //    CLASSNAME + ": ERROR", JOptionPane.ERROR_MESSAGE);
            Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT,
                               "Failed to open requested file " + fileName + "\n");
            throw new ESPDataException("Failed to open requested file " + fileName + ".");
        }

        //Try to read in the header of the file.
        int status = 0;
        try
        {
            //Read in the header information.
            readInHeader(datafile);
            status = 1;

            //Process the header information
            processHeaderAndInitializeDataSet();

            //Read in the data.
            readInData(datafile);

            //Close the input file.
            datafile.close();
        }
        catch(final IOException e)
        {
            String message = "Failed to read in ESP output time series file for this reason:";
            if(status == 0)
            {
                message += "  Unable to read in the header.";
            }
            else
            {
                message += "  Unable to read in the data or it does not match what the header states.";
            }
            Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT, message + "\n");

            throw new ESPDataException(message);
        }
    }

    /**
     * Check the file name for validity. The file name MUST NOT include the path. A name is bad if it has fewer than 5
     * segments, separated by '.'s. Further, the first two segments may not have more than 8 character, the 3rd 4, the
     * 4th 2, and the 5th segment must have at least 1 character.
     */
    public static boolean checkFileName(final String filename)
    {
        //Segment the name.
        final SegmentedLine segline = new SegmentedLine(filename, ".", SegmentedLine.MODE_ALLOW_EMPTY_SEGS);

        //Make sure the number of segments is acceptable.
        if(segline.getNumberOfSegments() < ESPData.NUMBER_OF_FILENAME_SEGMENTS)
        {
            return false;
        }

        //The temporary string.
        String temp;

        //Get the segment id.
        temp = segline.getSegment(0);
        if((temp.length() > 8) || (temp.length() <= 0))
        {
            return false;
        }

        //Get the time series id.
        temp = segline.getSegment(1);
        if((temp.length() > 8) || (temp.length() <= 0))
        {
            return false;
        }

        //Get the data type.
        temp = segline.getSegment(2);
        if((temp.length() > 4) || (temp.length() <= 0))
        {
            return false;
        }

        //Check the time step.
        temp = segline.getSegment(3);
        if((temp.length() > 2) || (temp.length() <= 0))
        {
            return false;
        }

        //Get the time series type extension.
        temp = segline.getSegment(4);
        if(temp.length() <= 0)
        {
            return false;
        }

        return true;
    }

    /**
     * Read in the components from the file name. This function may also serve the purpose of verifying the name of the
     * ESP file. The file name MUST NOT include the path.
     * 
     * @param filename
     * @return
     */
    private boolean processFileName(final String filename)
    {
        final SegmentedLine segline = new SegmentedLine(filename, ".", SegmentedLine.MODE_ALLOW_EMPTY_SEGS);

        //Make sure the number of segments is acceptable.
        if(segline.getNumberOfSegments() != NUMBER_OF_FILENAME_SEGMENTS)
        {
            return false;
        }

        //The temporary string.
        String temp;

        //Get the segment id.
        temp = segline.getSegment(0);
        if((temp.length() > 8) || (temp.length() <= 0))
        {
            return false;
        }
        _segmentID = temp;

        //Get the time series id.
        temp = segline.getSegment(1);
        if((temp.length() > 8) || (temp.length() <= 0))
        {
            return false;
        }
        _timeSeriesID = temp;

        //Get the data type.
        temp = segline.getSegment(2);
        if((temp.length() > 4) || (temp.length() <= 0))
        {
            return false;
        }
        _dataType = temp;

        //Check the time step.
        temp = segline.getSegment(3);
        if((temp.length() > 2) || (temp.length() <= 0))
        {
            return false;
        }

        //Get the time series type extension.
        temp = segline.getSegment(4);
        if(temp.length() <= 0)
        {
            return false;
        }
        _timeSeriesTypeExtension = temp;

        return true;
    }

    /**
     * read in the header portion of the file.
     */
    private void readInHeader(final HBinaryInputStream datafile) throws IOException
    {
        int i;

        //Read in the parameters one at a time.  All numbers are written as FLOATS, so they must be cast to ints.
        _formatver = datafile.readFloatSwap(); //0   <-- float position!
        _fsegid = HString.readBinaryString(datafile, 8); //1,2 <-- float position!
        _ftsid = HString.readBinaryString(datafile, 8); //3,4 <-- float position!
        _ftype = HString.readBinaryString(datafile, 4); //5   <-- float position!
        _tsdt = (int)datafile.readFloatSwap(); //6   <-- float position!
        _simflag = (int)datafile.readFloatSwap(); //7   <-- etc.
        _tsunit = HString.readBinaryString(datafile, 4); //8
        for(i = 0; i < 5; i++) //9,10,11,12,13
        {
            _now[i] = (int)datafile.readFloatSwap();
        }
        _im = (int)datafile.readFloatSwap(); //14
        _iy = (int)datafile.readFloatSwap(); //15
        _idarun = (int)datafile.readFloatSwap(); //16
        _ldarun = (int)datafile.readFloatSwap(); //17
        _ijdlst = (int)datafile.readFloatSwap(); //18
        _ihlst = (int)datafile.readFloatSwap(); //19
        _ljdlst = (int)datafile.readFloatSwap(); //20
        _lhlst = (int)datafile.readFloatSwap(); //21
        _ntraces = (int)datafile.readFloatSwap(); //22
        _ncm = (int)datafile.readFloatSwap(); //23
        _nlstz = (int)datafile.readFloatSwap(); //24
        _noutds = (int)datafile.readFloatSwap(); //25
        _irec = (int)datafile.readFloatSwap(); //26
        _dim = HString.readBinaryString(datafile, 4); //27
        _tscale = HString.readBinaryString(datafile, 4); //28
        _segdesc = HString.readBinaryString(datafile, 20); //29,30,31,32,33
        _lat = datafile.readFloatSwap(); //34
        _long = datafile.readFloatSwap(); //35
        _fgrp = HString.readBinaryString(datafile, 8); //36,37  
        _cgrp = HString.readBinaryString(datafile, 8); //38,39 
        _rfcname = HString.readBinaryString(datafile, 8); //40,41
        _espfile = HString.readBinaryString(datafile, 80); //42-61
        _prsfstr = HString.readBinaryString(datafile, 80); //62-81
        _esptext = HString.readBinaryString(datafile, 80); //82-101
        _adjcount = (int)datafile.readFloatSwap(); //102

        //The header contains 124 floats.  The last value corresponds to the 103rd float, since I started
        //counting at 0.  So, I need to read 104 - 124, or 21 more floats, or 84 more bytes.
        HString.readBinaryString(datafile, 84); //103-123.
    }

    /**
     * Compute class variables using header data.
     */
    public void processHeaderAndInitializeDataSet()
    {
        //NOTE: (!#!)
        //At this point, I will apply the time zone shift.  The value of _nlstz represents the
        //number of hours to add to GMT to acquire the time zone in which the data is stored.
        //Since the julian hours are in GMT, to get the true julian hour, I need to go from the
        //true time zone to GMT.  Hence, SUBTRACT _nlstz from this value!

        //Compute the calendar objects adjusted to GMT.  The values in the header may not be in
        //GMT!  
        _fcstStart = HCalendar.computeCalendarFromJulianHour((_ijdlst - 1) * 24 + _ihlst - _nlstz);
        _fcstEnd = HCalendar.computeCalendarFromJulianHour((_ljdlst - 1) * 24 + _lhlst - _nlstz);
        _runStart = HCalendar.computeCalendarFromJulianHour((_idarun - 1) * 24 + _ihlst - _nlstz);
        _runEnd = HCalendar.computeCalendarFromJulianHour((_ldarun - 1) * 24 + _lhlst - _nlstz);

        //Get the number of values to read in per 31 day chunk.  This is 31 * 24 divided by time step!
        _numberPerMonth = (int)(31.0 * 24.0 / _tsdt);

        //Get the number of values per trace as the difference in julian hours between _fcstStart and _fcstEnd
        //divided by the time step, and +1 for the end points.
        final int startjhr = HCalendar.computeJulianHourFromCalendar(_fcstStart, true);
        final int endjhr = HCalendar.computeJulianHourFromCalendar(_fcstEnd, true);
        _numberPerTrace = (int)(((double)endjhr - (double)startjhr) / _tsdt);

        //Set the hs flag to true only if the type is HS.  HS type is true if
        //_simflag == 1.
        _hsFlag = false;
        if(_simflag == 1)
        {
            _hsFlag = true;
        }

        //Now, if the _hsFlag is true, meaning this is an HS time series, then the numpertrace
        //is actually 366 * 24/_tsdt... i.e. I know I have a year of data.
        if(_hsFlag)
        {
            _numberPerTrace = (int)(366.0 * 24.0 / _tsdt);
        }

        //Initialize the data set.  This should create a bit too large of a data set,
        //but that shouldn't do any harm.
        if(isHsDebug())
        {
            initialize(((_numberPerTrace + 1) * (_ntraces + 1)), 2, false);
        }
        else
        {
            initialize(((_numberPerTrace + 1) * (_ntraces + 1)), 2, false);

        }
    }

    /**
     * Read in the data.
     */
    private void readInData(final HBinaryInputStream datafile) throws IOException
    {
        Messenger.writeMsg(Messenger.EVT_PROGRESS + Messenger.SEV_INFO + Messenger.ALWAYS_PRINT,
                           "Reading in the data...\n");

        //Setup the limiting julian hours.  The fcst* hours are for any NON-HS run.
        //The HS runs use the run* hours.  
        final int fcststartjhr = HCalendar.computeJulianHourFromCalendar(_fcstStart, true);
        final int fcstendjhr = HCalendar.computeJulianHourFromCalendar(_fcstEnd, true);
        final int runstartjhr = HCalendar.computeJulianHourFromCalendar(_runStart, true);
        final int runendjhr = HCalendar.computeJulianHourFromCalendar(_runEnd, true);

        //Some variables...
        int workingjhr; //Keeps track of the julian hour being examined in the file.
        int datajhr; //Keeps track of the julian hour as it is recorded in the DataSet.
        Calendar datadate; //The date corresponding to datajhr, without time zone adjustment.

        int i, j, k;
        final float[] values = new float[_numberPerMonth];
        double[] sample = new double[getNumberOfVariables()];
        int workingmonth;
        int ncmloop;
        int ntracesLoop;

        //Added by Hank Herr (1/2/2002)... 
        //I'm trying to see if, by switching _ncm to 12 when HS is
        //used, will the HS file be processed correctly?...  
        //The answer is yes. 
        ncmloop = _ncm;
        if(_hsFlag)
        {
            ncmloop = 12;
        }

        //Loop on the number of traces...
        if(isHsDebug())
        {
            ntracesLoop = _ntraces + 1;
        }
        else
        {
            ntracesLoop = _ntraces;
        }

        for(i = 0; i < ntracesLoop; i++)
        {

//            System.out.println("Trace #: " + i);
            //Julian hour providing the start of the month.  Any value must be AFTER this (equal to is not good).
            final int startOfMonthJulianHourInGMT = HCalendar.computeJulianHour(_fcstStart.get(Calendar.YEAR),
                                                                                _im - 1,
                                                                                1,
                                                                                0) - _nlstz;

            //Identify the julian hour of the first value in the month that can have data.  Workingjhr is the julian hour
            //for the forecast value (i.e. using the year of the forecast).  Datajhr tracks the same value as it would be
            //in the ESP trace's year.  Both are adjusted backward one step at a time until they represent the first value
            //in the month, using startOfMonthJulianHourInGMT.
            workingjhr = HCalendar.computeJulianHourFromCalendar(_fcstStart, true) + _tsdt;
            datajhr = HCalendar.computeJulianHour(_iy + i,
                                                  _fcstStart.get(Calendar.MONTH),
                                                  _fcstStart.get(Calendar.DAY_OF_MONTH),
                                                  _fcstStart.get(Calendar.HOUR_OF_DAY))
                + _tsdt;
            while(workingjhr > startOfMonthJulianHourInGMT)
            {
                workingjhr -= _tsdt;
                datajhr -= _tsdt;
            }
            workingjhr += _tsdt;
            datajhr += _tsdt;

            //Data date tracks the data julian hour, which is in GMT, in local time.
            datadate = HCalendar.computeCalendarFromJulianHour(datajhr + _nlstz);
            datadate.add(Calendar.SECOND, -1);

            //Loop on the number of months per trace.
            for(j = 0; j < ncmloop; j++)
            {
                //Get the working month -- which is equal to the data month.
                workingmonth = datadate.get(Calendar.MONTH);

                //Read in the values array.
                for(k = 0; k < values.length; k++)
                {
                    try
                    {
                        values[k] = datafile.readFloatSwap();
//                        System.out.println(values[k]);

                    }
                    catch(final IOException e)
                    {
                        System.out.println(" I/O Exception caught ");
                        values[k] = (float)DataSet.MISSING;
                    }
                }

                //Process the floats one at a time.
                for(k = 0; k < values.length; k++)
                {
                    //If (1) this is not an HS run and the current working jhr is within the forecast
                    //  start and end jhr OR (2) this is an HS run and the data jhr is within the 
                    //  run start and end jhr, then...
                    //(Note that the fcststart time does not have corresponding data: its the start
                    //time, not the first time period with data!, so I use a '>' and not '>=' )
                    if(((!_hsFlag) && (workingjhr > fcststartjhr) && (workingjhr <= fcstendjhr))
                        || ((_hsFlag) && (datajhr > runstartjhr) && (datajhr <= runendjhr)))
                    {
                        //Then we have data that we must record...

                        //This time zone shifting has actually been moved to adjust the start and end times.
                        //Hence, if they are in the correct time zone, then datajhr, which uses them, is in
                        //the correct zone.
                        sample = new double[getNumberOfVariables()];
                        sample[JULIAN_HOUR] = datajhr;
                        sample[VALUE] = values[k];
                        addSample(sample);
                    }

                    workingjhr += _tsdt;
                    datajhr += _tsdt;
                    datadate.add(Calendar.HOUR, _tsdt);

                    //This if will hold true until by incrementing the hour, I reach the next month.
                    //It should, theoretically, end when I reach hour 0 of day 1 of the next month,
                    //which is also hour 24 of the last day of this month.
                    if(datadate.get(Calendar.MONTH) != workingmonth)
                    {
                        break;
                    }
                }
            }
        }

//        System.out.println("Size of data = " + getData().length);
//        System.out.println("VAlues.length = " + values.length);
    }

    /**
     * Write out the header portion of the binary output file.
     */
    private void writeBinaryOutputHeader(final HBinaryOutputStream datafile) throws IOException
    {
        int i;

        //Read in the parameters one at a time.  All numbers are written as FLOATS, so they must be cast to ints.
        datafile.writeFloatSwap(_formatver); //0 <-- float position!
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(8, _fsegid, true), 8); //1,2
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(8, _ftsid, true), 8); //3,4
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(4, _ftype, true), 4); //5
        datafile.writeFloatSwap(_tsdt); //6
        datafile.writeFloatSwap(_simflag); //7
        HString.writeBinaryString(datafile, _tsunit, 4); //8
        for(i = 0; i < 5; i++) //9,10,11,12,13
        {
            datafile.writeFloatSwap(_now[i]);
        }
        datafile.writeFloatSwap(_im); //14
        datafile.writeFloatSwap(_iy); //15
        datafile.writeFloatSwap(_idarun); //16
        datafile.writeFloatSwap(_ldarun); //17
        datafile.writeFloatSwap(_ijdlst); //18
        datafile.writeFloatSwap(_ihlst); //19
        datafile.writeFloatSwap(_ljdlst); //20
        datafile.writeFloatSwap(_lhlst); //21
        datafile.writeFloatSwap(_ntraces); //22
        datafile.writeFloatSwap(_ncm); //23
        datafile.writeFloatSwap(_nlstz); //24
        datafile.writeFloatSwap(_noutds); //25
        datafile.writeFloatSwap(_irec); //26
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(4, _dim, true), 4); //27
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(4, _tscale, true), 4); //28
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(20, _segdesc, true), 20); //29,30,31,32,33
        datafile.writeFloatSwap(_lat); //34
        datafile.writeFloatSwap(_long); //35
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(8, _fgrp, true), 8); //36,37  
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(8, _cgrp, true), 8); //38,39 
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(8, _rfcname, true), 8); //40,41
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(80, _espfile, true), 80); //42-61
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(80, _prsfstr, true), 80); //62-81
        HString.writeBinaryString(datafile, HString.formatStringToFieldWidth(80, _esptext, true), 80); //82-101
        datafile.writeFloatSwap(_adjcount); //102

        //The header contains 124 floats.  The last value corresponds to the 103rd float, since I started
        //counting at 0.  So, I need to read 104 - 124, or 21 more floats, or 84 more bytes.
        HString.writeBinaryString(datafile, "", 84); //103-123.
    }

    /**
     * Write out the blocks of data for the output file.
     */
    private void writeBinaryOutputData(final HBinaryOutputStream datafile) throws IOException
    {
        Messenger.writeMsg(Messenger.EVT_PROGRESS + Messenger.SEV_INFO + Messenger.ALWAYS_PRINT,
                           "Writing out the data...\n");

        //Sort the data by julian hour.
        //Messenger.writeMsg(Messenger.EVT_PROGRESS + Messenger.SEV_INFO + Messenger.ALWAYS_PRINT,
        //                   "Sorting by julian hour...\n");
        //sortBy(JULIAN_HOUR);
        //I've removed the sorting because it causes a problem when forecast lengths exceed a year.
        //Thus, for this algorith to work, the data must already be appropriately sorted 
        //(i.e. by trace and within each trace)!!!

        //Setup the limiting julian hours, and shift it to the ESPTS time zone.
        //This is used to determine when we have output all of the forecast values for trace;
        //it points to the last forecasting time.
        final int fcstendjhr = HCalendar.computeJulianHourFromCalendar(_fcstEnd, true);

        //Some variables...
        int workingjhr, datajhr;
        Calendar datadate;

        int i, j, k;
//        float[] values = new float[_numberPerMonth];
//        double[] sample = new double[getNumberOfVariables()];
//        int workingmonth;
//        int numvalueswritten;
        int currentmonth;

        //Reset the data set pointer.
        resetPtr();

        //Loop on the number of traces...
        for(i = 0; i < _ntraces; i++)
        {
            //Julian hour providing the start of the month.  Any value must be AFTER this (equal to is not good).
            final int startOfMonthJulianHourInGMT = HCalendar.computeJulianHour(_fcstStart.get(Calendar.YEAR),
                                                                                _im - 1,
                                                                                1,
                                                                                0) - _nlstz;

            //Identify the julian hour of the first value in the month that can have data.  Workingjhr is the julian hour
            //for the forecast value (i.e. using the year of the forecast).  Datajhr tracks the same value as it would be
            //in the ESP trace's year.  Both are adjusted backward one step at a time until they represent the first value
            //in the month, using startOfMonthJulianHourInGMT.
            workingjhr = HCalendar.computeJulianHourFromCalendar(_fcstStart, true) + _tsdt;
            datajhr = HCalendar.computeJulianHour(_iy + i,
                                                  _fcstStart.get(Calendar.MONTH),
                                                  _fcstStart.get(Calendar.DAY_OF_MONTH),
                                                  _fcstStart.get(Calendar.HOUR_OF_DAY))
                + _tsdt;
            while(workingjhr > startOfMonthJulianHourInGMT)//MUST BE STRICT GREATER THAN BECAUSE ESPADP USES 1-24 not 0-23 hours!!!
            {
                workingjhr -= _tsdt;
                datajhr -= _tsdt;
            }
            workingjhr += _tsdt;
            datajhr += _tsdt;

            //Data date parallels the datajhr but in the time zone of the ESP data.
            datadate = HCalendar.computeCalendarFromJulianHour(datajhr + _nlstz);

            //TODO: ELIMINATE THIS LINE OF CODE IF WE EVER TRAIN ESP TO GO FROM 0..23 AND NOT 1..24!!!
            //By subtracting a SECOND from the data date, an hour of 0 will actually correspond
            //to an our of 23:59:59.  Hence, 0 is now on the previous day, which corresponds to how ESP
            //thinks... that hour 0 is actually hour 24 of the previous day.  This will not affect how I
            //search the data set, because I'm not changing the julian hour.
            datadate.add(Calendar.SECOND, -1);

            //Loop on the number of months per trace.
            for(j = 0; j < _ncm; j++)
            {
                //Initialize the number of values written.
//                numvalueswritten = 0;

                //Get the currentmonth being written -- that month assigned to workingjhr plus the current
                //conditional month.
                currentmonth = (_im - 1 + j) % 12; //0...11

                //Loop through all values of this month, increasing working hour until I'm at the first
                //value outside this month thats NOT 0 hours... because 0 is actually 24. 
                for(k = 0; k < _numberPerMonth; k++)
                {

                    //Check for the data only if the data month is in the current month.
                    if((datadate.get(Calendar.MONTH) == currentmonth) && (workingjhr <= fcstendjhr))
                    {
                        //Is the current data set value the same as the data julian hour (adjust for time zone)?
                        if(getCurrentValue(JULIAN_HOUR) == datajhr)
                        {
                            //Write out the value
                            datafile.writeFloatSwap((float)getCurrentValue(VALUE));

                            //Goto the next value -- if this fails then it just means I have no more data to
                            //print and no increment will occur.  So, all future getCurrentValue checks will fail.
                            next();
                        }
                        //Otherwise, since I sorted the data, I know that the current hour must be LARGER than
                        //the data julian hour...
                        else
                        {
                            //Print out a missing value.
                            datafile.writeFloatSwap((float)DataSet.MISSING);
                        }

                        workingjhr += _tsdt;
                        datajhr += _tsdt;
                        datadate.add(Calendar.HOUR, _tsdt);
                    }
                    //If the data month is already set for the next month, then I'm in that area
                    //where the data value corresponds to, for example, 2/30, 2/31, 4/31, etc... days
                    //that don't exist.
                    else
                    {
                        //Print out a missing value.
                        datafile.writeFloatSwap((float)DataSet.MISSING);
                    }
                }
            }
        }
    }

    /**
     * Call the writeOutHeader and writeOutData routines to put together a binary for the data contained here in.
     * 
     * @return
     */
    public boolean writeBinaryOutputFile()
    {

        FileOutputStream fileout;
        HBinaryOutputStream datafile;
        final NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(0);
        nf.setMinimumFractionDigits(0);
        nf.setMaximumIntegerDigits(2);
        nf.setMinimumIntegerDigits(2);

        //Try to open the data file for reading.
        try
        {
            fileout = new FileOutputStream(_esptsFile);
            datafile = new HBinaryOutputStream(fileout);
        }
        catch(final IOException e1)
        {
            Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT,
                               "Failed to open requested file for writing " + _esptsFile.getAbsolutePath() + "\n");
            return false;
        }

        //Try to writeout the file contents.
        try
        {
            writeBinaryOutputHeader(datafile);
            writeBinaryOutputData(datafile);
        }
        catch(final IOException e)
        {
            Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.SEV_ERROR + Messenger.ALWAYS_PRINT,
                               "Failed to write out file information.\n");
            return false;
        }

        return true;
    }

    //Dump the header to standard out.
    public void dumpHeader()
    {
        int i;

        System.out.println("========== ESP HEADER ==============================");
        System.out.println("_formatver = " + _formatver);
        System.out.println("_fsegid    = \"" + _fsegid + "\"");
        System.out.println("_ftsid     = \"" + _ftsid + "\"");
        System.out.println("_ftype     = \"" + _ftype + "\"");
        System.out.println("_tsdt      = " + _tsdt);
        System.out.println("_simflag   = " + _simflag);
        System.out.println("_tsunit    = \"" + _tsunit + "\"");
        for(i = 0; i < 5; i++) //9,10,11,12,13
        {
            System.out.println("_now[" + i + "]    = " + _now[i]);
        }
        System.out.println("_im        = " + _im);
        System.out.println("_iy        = " + _iy);
        System.out.println("_idarun    = " + _idarun + " (days from Jan 1, 1900)");
        System.out.println("_ldarun    = " + _ldarun + " (days from Jan 1, 1900)");
        System.out.println("_ijdlst    = " + _ijdlst + " (days from Jan 1, 1900)");
        System.out.println("_ihlst     = " + _ihlst + " (hour of day in ESP timezone)");
        System.out.println("_ljdlst    = " + _ljdlst + " (days from Jan 1, 1900)");
        System.out.println("_lhlst     = " + _lhlst + " (hour of day in ESP timezone)");
        System.out.println("_ntraces   = " + _ntraces);
        System.out.println("_ncm       = " + _ncm);
        System.out.println("_nlstz     = " + _nlstz + " (modifier to get from GMT to the ESP timezone of this data)");
        System.out.println("_noutds    = " + _noutds);
        System.out.println("_irec      = " + _irec);
        System.out.println("_dim       = \"" + _dim + "\"");
        System.out.println("_tscale    = \"" + _tscale + "\"");
        System.out.println("_segdesc   = \"" + _segdesc + "\"");
        System.out.println("_lat       = " + _lat);
        System.out.println("_long      = " + _long);
        System.out.println("_fgrp      = \"" + _fgrp + "\"");
        System.out.println("_cgrp      = \"" + _cgrp + "\"");
        System.out.println("_rfcname   = \"" + _rfcname + "\"");
        System.out.println("_espfile   = \"" + _espfile + "\"");
        System.out.println("_prsfstr   = \"" + _prsfstr + "\"");
        System.out.println("_esptext   = \"" + _esptext + "\"");
        System.out.println("_adjcount  = " + _adjcount);
        System.out.println("====================================================");
    }

    //Dump the processed header variables.
    public void dumpProcessedHeader()
    {
        //Dump the calendar objects.
        System.out.println("========== ESP PROCESSED HEADER ====================");
        System.out.println("_fcstStart   = " + HCalendar.buildDateTimeTZStr(_fcstStart));
        System.out.println("_fcstEnd     = " + HCalendar.buildDateTimeTZStr(_fcstEnd));
        System.out.println("_runStart    = " + HCalendar.buildDateTimeTZStr(_runStart));
        System.out.println("_runEnd      = " + HCalendar.buildDateTimeTZStr(_runEnd));
        System.out.println("_numberPerMonth = " + _numberPerMonth);
        System.out.println("_numberPerTrace = " + _numberPerTrace);
        System.out.println("_hsFlag      = " + _hsFlag);
        System.out.println("====================================================");
    }

    //Dump the data
    public void dumpData()
    {
        System.out.println("========== ESP DATA ================================");
        MatrixMath.printMatrix(getData());
        System.out.println("====================================================");
    }

    //Dump the data
    public void dumpDataUsingDates()
    {
        System.out.println("========== ESP DATA ================================");
        System.out.println("data size = " + getData().length);
        MatrixMath.printMatrix(getData(), 0);
        System.out.println("====================================================");
    }

    //Compute what the file name of this file was.
    public String computeFileName()
    {
        //Define a two digit integer formatter.
        final NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(0);
        nf.setMinimumFractionDigits(0);
        nf.setMaximumIntegerDigits(2);
        nf.setMinimumIntegerDigits(2);

        return _segmentID + "." + _timeSeriesID + "." + _dataType + "." + nf.format(_tsdt) + "."
            + _timeSeriesTypeExtension;
    }

    /////////////////////////////////////////////////////////////////////////
    //TOOLS
    /////////////////////////////////////////////////////////////////////////

    public MeasuringUnit determineMeasuringUnit()
    {
        if(_tsunit.trim().equalsIgnoreCase("CMS"))
        {
            return MeasuringUnit.cms;
        }
        if(_tsunit.trim().equalsIgnoreCase("M"))
        {
            return MeasuringUnit.meters;
        }
        if(_tsunit.trim().equalsIgnoreCase("CFS"))
        {
            return MeasuringUnit.cfs;
        }
        if(_tsunit.trim().equalsIgnoreCase("FT"))
        {
            return MeasuringUnit.feet;
        }
        if(_tsunit.trim().equalsIgnoreCase("CMSD"))
        {
            return MeasuringUnit.cmsd;
        }
        return null;
    }

    public String determineFourCharacterMeasurementString(final MeasuringUnit unit)
    {
        if(unit.getName().equalsIgnoreCase("cms"))
        {
            return "CMS ";
        }
        if(unit.getName().equalsIgnoreCase("meters"))
        {
            return "M   ";
        }
        if(unit.getName().equalsIgnoreCase("cfs"))
        {
            return "CFS ";
        }
        if(unit.getName().equalsIgnoreCase("feet"))
        {
            return "FT  ";
        }
        return "";
    }

    //Return the value of _hsFlag.
    public boolean isHistoricalSimulation()
    {
        return _hsFlag;
    }

    //ESPData is strange in that to do any temporal averaging, you need to make sure to treat all
    //years independently (as different traces) instead of merging them into one long record.  So,
    //to do temporal averaging, one must first extract a subset of the data for the forecast period
    //from the data set for the year associated with the trace, do the averaging, and then do the
    //same for the next trace, combining data sets as you go along.

    //CREATE AN ACCUMULATION THING HERE!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    //Daily Average...
    //The time associated with each value is the end of the day for that value... for ESP, hour 24
    //of that day, but for the rest of the known universe, hour 0 of the NEXT day!

    //CREATE A NOISE GENERATOR THING HERE!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

    /////////////////////////////////////////////////////////////////////////
    //Sets and Gets
    /////////////////////////////////////////////////////////////////////////

    //
    //FOR MY OWN SANITY, I HAVE NOT CREATED ALL THE GETS THAT SHOULD BE CREATED.  If someone else
    //wants to create the 30+ gets, one for each of the parameters passed in, feel free.  Otherwise, just
    //access the attributes directly.  
    //
    //THERE SHOULD BE NO REASON FOR ANYONE TO HAVE TO SET ANY OF THE VALUES.  So, you should not have
    //any of these attributes on the left hand side of an equal sign.
    //

    /**
     * Create a CS data set containing only data for the passed in trace number.
     * 
     * @param traceNumber - start at 0(i.e. 0 is the first trace)
     */
    public DataSet getCSTrace(final int traceNumber)
    {
        final int index = traceNumber * this._numberPerTrace;
        final int endIndex = index + this._numberPerTrace;

        return this.extractSubset(index, endIndex);

    }

    public File getESPTSFile()
    {
        return _esptsFile;
    }

    public void setHsDebug(final boolean _hsDebug)
    {
        this._hsDebug = _hsDebug;
    }

    public boolean isHsDebug()
    {
        return _hsDebug;
    }

    //A test main.
    public static void main(final String args[])
    {
        final int[] testlevels = {Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT,
            Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT,
            Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT, Messenger.ALWAYS_PRINT};
        Messenger.initMsg(-1, null, "stdout", testlevels);

        System.out.println("=========================================");
        System.out.println("ESPData version " + SoftwareVersionNumbers.ESPDATA_VERSION + "\n");
        System.out.println("=========================================");

        //Process the arguments.
        if((args.length != 1) && (args.length != 2) && (args.length != 3))
        {
            System.out.println("");
            System.out.println("Invalid command line.  The execution command is:");
            System.out.println("");
            System.out.println("    <program> [h,dd,dj,a,todc] <file> <hsDebug>");
            System.out.println("");
            System.out.println("where <file> is the file to read in and the options are:");
            System.out.println("");
            System.out.println("  h    -- only print out header");
            System.out.println("  dd   -- print out header and data using dates");
            System.out.println("  dj   -- print out header and data using julian hours");
            System.out.println("  a    -- print out header, data using julian hours,");
            System.out.println("          and convert to 24-hour data.");
            System.out.println("  todc -- convert ESP time series to datacard, with");
            System.out.println("          same names as <file> and \".txt\" extension.");
            System.out.println("");
            System.out.println("Conversion to 24-hour for 'a' option begins at the hour");
            System.out.println("0 of the first day of the data (in local standard time).");
            System.out.println("");
            return;
        }

        //Process the second arg, if there is one.
        int outputlevel = 1;
        boolean dateprint = false;
        if((args.length == 2) || (args.length == 3))
        {
            if(args[0].equals("h"))
            {
                outputlevel = 0;
            }
            if(args[0].equals("dj"))
            {
                outputlevel = 1;
            }
            if(args[0].equals("a"))
            {
                outputlevel = 2;
            }
            if(args[0].equals("dd"))
            {
                outputlevel = 1;
                dateprint = true;
            }
            if(args[0].equals("todc"))
            {
                outputlevel = 3;
            }
        }

        try
        {
            ESPData data;

            if(args.length == 1)
            {
                data = new ESPData(args[0], true, false);
            }
            else if(args.length == 2)
            {
                data = new ESPData(args[1], true, false);
            }
            else
            {
                data = new ESPData(args[1], true, true);
            }

            //Special case for conversion to datacard.
            if(outputlevel == 3)
            {
                try
                {

                    System.out.println("Trying to create datacard file...");
                    final DatacardData dcdata = new DatacardData(data);
                    dcdata.writeDatacardOutputFile(data.computeFileName() + ".txt");
                    System.out.println("Completed successfully.");
                }
                catch(final DatacardDataException e)
                {
                    System.out.println("Failed to create datacard file.");
                }

                //Quit at this point.
                return;
            }

            //Dump the header if output level is at least 0.
            System.out.println("##################### File Contents Read In: ####################");
            if(outputlevel >= 0)
            {
                data.dumpHeader();
                data.dumpProcessedHeader();
            }

            //Dump out data if at least 1.
            if(outputlevel >= 1)
            {
                if(dateprint)
                {
                    data.dumpDataUsingDates();
                }
                else
                {
                    data.dumpData();
                }
            }
            System.out.println("#################################################################");
            System.out.println("");
            System.out.println("");

            //Only continue on if the 'a' option is set... i.e. if the user wants to accumulate
            //to 24-hour data.
            if(outputlevel < 2)
            {
                return;
            }

            //ACCUMLATE TO 24-HOURS...

            //Get the smallest julian hour.
            double startjhr = data.getSmallest(0);

            //startjhr is the julian hours in GMT!  Now, I want to get the first hour of THAT day
            //in the timezone of the ESPData!  So, add _nsltz to get the julian hour in
            //the time zone of the ESPData file. 
            startjhr += data._nlstz;

            //If I convert the julian hour to a calendar, now, it will have the date components 
            //in local time.          
            final Calendar startdate = HCalendar.computeCalendarFromJulianHour((int)startjhr);

            //Set the hour to be 0 to get hour 0 of that day to be the start hour.
            startdate.add(Calendar.HOUR, -1 * startdate.get(Calendar.HOUR_OF_DAY));

            //Now, when I convert back, I'll get the julian hour in the ESPData time zone that
            //corresponds to hour 0 of the first day.  
            startjhr = HCalendar.computeJulianHourFromCalendar(startdate, true);

            //But, I need the julian hour in GMT for it to be the same timezone as the data I have.
            startjhr -= data._nlstz;

            //THESE LINES DEAL WITH DAILY AVERAGING...            
            System.out.println("");
            System.out.println("Doing the daily average assuming the start jhr and date in GMT is:");
            System.out.println("    "
                + startjhr
                + " -- "
                + HCalendar.buildDateStr(HCalendar.computeCalendarFromJulianHour((int)startjhr),
                                         HCalendar.DEFAULT_DATETZ_FORMAT));
            System.out.println("If you add the value of _nlstz given in the header in hours, you will");
            System.out.println("see the date in the time zone of the ESPData file.  Check that this is");
            System.out.println("correct to be certain my hours are right.");
            System.out.println("");

            //Get the averaged data set.
            final DataSet avgdata = data.temporalAverage(0, startjhr, 24, false, false);

            System.out.println("##################### Daily Averages: ###########################");
            System.out.println("  The first var is the julian hour, in GMT, associated with the");
            System.out.println("  start of the day");
            System.out.println("#################################################################");
            MatrixMath.printMatrix(avgdata.getData());
            System.out.println("#################################################################");
            System.out.println("");

            //THSE LINES CREATE AN OUTPUT FILE        
            //Output the read in data.
            //System.out.println("##################### Creating Output File ####################");
            //data._tstypeext = "TEST";
            //data.writeBinaryOutputFile();

            //THESE LINES SPIT OUT ONE TRACE        
            //Testing getTrace
            //DataSet atrace = data.getTrace(7);
            //System.out.println("##################### Here is trace number 7 ##################");
            //MatrixMath.printMatrix(atrace.getData());

            //THESE LINES SPIT OUT THE TRACES IN A DATACARD FORMAT
            //Output stuff in Datacard Format
            /*
             * try { DatacardData dcdata = new DatacardData(data);
             * System.out.println("##################### File Contents Read In: ####################");
             * System.out.println(" THIS IS THE DATACARD DATASET");
             * System.out.println("#################################################################");
             * dcdata.dumpHeader(); dcdata.dumpProcessedHeader(); dcdata.dumpData();
             * System.out.println("#################################################################");
             * dcdata.writeDatacardOutputFile(data.computeFileName() + ".txt"); } catch (DatacardDataException e) {
             * System.out.println("FAILURE!!!!!!"); System.out.println("Could not produce Datacard file!"); }
             */
        }
        catch(final ESPDataException e)
        {
            System.out.println("FAILURE!!!!!!");
        }

    }

}
