package ohd.hseb.util.data;

import java.util.Calendar;

import ohd.hseb.util.misc.HCalendar;

// ///////////////////////////////////////////////////////////////////////////////////////////////
// MatrixMath Class
// ///////////////////////////////////////////////////////////////////////////////////////////////
//
// This abstract class is a collection of static methods that perform operations on a
// double array, which is treated as a matrix. The double array coming in is first indexed
// by row and then by column; i.e. if double[][] aa = double[r][c], then aa has r rows and c
// columns, so that a.length = r and a[0].length = c. Note that the method printMatrix(...)
// is particularly valuable in dumping the contents of the DataSet object (acquired via a call
// to method getData(...)) to stdout.
//
// ///////////////////////////////////////////////////////////////////////////////////////////////
//
// Date Person Comment
// ---------- ------------ ----------------------------------------------
// 2000-06-16 Hank Herr First Version Complete
// 2002-01-02 Hank Herr Added the second printMatrix (the jhrvar one)
//
// ///////////////////////////////////////////////////////////////////////////////////////////////
//
// STATIC VARS:
//
// static final double TINY = 0.00000000000000000001;
//
// ATTRIBUTES:
//
//
// FUNCTIONS (in order of appearance):
//
// Private Tools:
// private static boolean ludcmp(double[][] a, int[] index, Double dvalue)
// private static boolean lubksb(double[][] a, int[] index, double[] b)
//
// Matrix Mathematics:
// public static double[][] matrixInverse(double[][] a)
// public static double[][] matrixMultiplication(double[][] a, double[][] b)
// public static double[][] matrixTranspose(double[][] a)
// public static double[][] matrixSum(double[][] a, double[][] b)
// public static double[][] matrixScale(double scalar, double[][] a)
// public static double vectorDotProduct(double[] a, double[] b)
//
// Other Stuff:
// private static int getNumberOfRows(double[][] a)
// private static int getNumberOfCols(double[][] a)
// static void printMatrix(double[][] a)
// static void printMatrix(double[][] a, int jhrvar)
// public static void main(String args[])
//
// ///////////////////////////////////////////////////////////////////////////////////////////////

public final class MatrixMath
{
    static final String CLASSNAME = "MatrixMath";

    //10^-20
    static final double TINY = 0.00000000000000000001;

    ///////////////////////////////////////////////////////////////////////////////
    //ROUTINES USED FOR INVERSE (defined in Numerical Recipes in C)
    ///////////////////////////////////////////////////////////////////////////////

    //The parameters index and dvalue are changed by this routine
    private static boolean ludcmp(final double[][] a, final int[] index, Double dvalue)
    {
        int i, imax, j, k;
        double big, dum, sum, temp;
        double d;

        final int n = getNumberOfRows(a);
        if(n < 0)
            return false;
        if(n != index.length)
            return false;

        final double[] vv = new double[n];
        d = 1.0;
        imax = 0;

        //This is straight from "Numerical Recipes in C", p 46.
        for(i = 0; i < n; i++)
        {
            big = 0.0;

            //Looping over the rows to get the implicit scaling information.
            for(j = 0; j < n; j++)
            {
                temp = Math.abs(a[i][j]);
                if(temp > big)
                    big = temp;
            }

            //This is the error of "Singular matrix in routine ludcmp", or 
            //no non-zero largest element.
            if(big == 0.0)
                return false;

            vv[i] = 1.0 / big;
        }

        //The loop ober columns of Crout's method
        for(j = 0; j < n; j++)
        {
            //Eq. 2.3.12, with i = j... from the book.
            for(i = 0; i < j; i++)
            {
                sum = a[i][j];
                for(k = 0; k < i; k++)
                {
                    sum -= a[i][k] * a[k][j];
                }
                a[i][j] = sum;
            }

            //Initialize for search of largest pivot element
            big = 0.0;

            //2.3.12 with i = j, and i = j + 1, ..., N of eq. 2.3.13 of the book.
            for(i = j; i < n; i++)
            {
                sum = a[i][j];
                for(k = 0; k < j; k++)
                {
                    sum -= a[i][k] * a[k][j];
                }
                a[i][j] = sum;

                dum = vv[i] * Math.abs(sum);
                if(dum >= big)
                {
                    big = dum;
                    imax = i;
                }
            }

            //Do we need to interchange rows
            if(j != imax)
            {
                for(k = 0; k < n; k++)
                {
                    dum = a[imax][k];
                    a[imax][k] = a[j][k];
                    a[j][k] = dum;
                }
                d = -1 * d;

                //interchange the scale factor
                vv[imax] = vv[j];
            }

            index[j] = imax;

            //This may or may not be desirable... look into it.
            //if (a[j][j] == 0.0)
            //    a[j][j] = TINY;

            if(j != n - 1)
            {
                dum = 1.0 / a[j][j];
                for(i = j + 1; i < n; i++)
                    a[i][j] *= dum;
            }

        }

        dvalue = Double.valueOf(d);

        return true;
    }

    //The back substitution routine.  a and index are unchanged, b is the output value.
    private static boolean lubksb(final double[][] a, final int[] index, final double[] b)
    {
        int i, ii = 0, ip, j;
        double sum;

        final int n = getNumberOfRows(a);
        if(n < 0)
            return false;
        if(n != index.length)
            return false;

        //When ii is set to a positive value, it will become the index of the first
        //nonvanishing element of b.  We now do the forward substitution, eq. 2.3.6.
        //The only new wrinkle is to unscramble the permutation as we go.
        for(i = 0; i < n; i++)
        {
            ip = index[i];
            sum = b[ip];
            b[ip] = b[i];
            if(ii == 0)
            {
                for(j = ii; j <= i - 1; j++)
                {
                    sum -= a[i][j] * b[j];
                }
            }
            //A non-zero element was encountered, so from now on we will have to do the
            //sums in the loop above.
            else if(sum != 0)
                ii = i;
            b[i] = sum;
        }

        //Now we do the back substitution, eq. 2.3.7.
        for(i = n - 1; i >= 0; i--)
        {
            sum = b[i];
            for(j = i + 1; j < n; j++)
            {
                sum -= a[i][j] * b[j];
            }

            //Store the component in the solution vector.
            b[i] = sum / a[i][i];
        }

        return true;
    }

    ///////////////////////////////////////////////////////////////////////////////
    // Basic Matrix Functions
    ///////////////////////////////////////////////////////////////////////////////

    //Returns the inverse of the matrix passed in. 
    //Method is pulled from "Numerical Recipes in C"   
    public static double[][] matrixInverse(final double[][] a)
    {
        //copy a, first, to make sure it is not editted.
        final double[][] cpya = MatrixMath.matrixScale(1.0, a);

        final int n = getNumberOfRows(cpya);
        if(n < 0)
            return null;
        if(n != getNumberOfCols(cpya))
            return null;

        final Double d = null;
        final double[] col = new double[n];
        final double[][] y = new double[n][n];
        int i, j;
        final int[] index = new int[n];

        ludcmp(cpya, index, d);

        for(j = 0; j < n; j++)
        {
            for(i = 0; i < n; i++)
                col[i] = 0.0;

            col[j] = 1.0;
            lubksb(cpya, index, col);
            for(i = 0; i < n; i++)
                y[i][j] = col[i];
        }
        return y;
    }

    //Return the result of multiplying two matrices together.  
    public static double[][] matrixMultiplication(final double[][] a, double[][] b)
    {
        int i, j;
        int cols, rows;
        double res[][];

        //Verify that the number of columns of a equals the rows of b.
        cols = getNumberOfCols(a);
        if(cols <= 0)
            return null;

        rows = getNumberOfRows(b);
        if(rows <= 0)
            return null;
        if(cols != rows)
            return null;

        //Define res by getting its rows and columns first!
        cols = getNumberOfCols(b);
        if(cols <= 0)
            return null;

        rows = getNumberOfRows(a);
        if(rows <= 0)
            return null;

        //Setout the memory for the result
        res = new double[rows][cols];

        //Transpose b.
        b = matrixTranspose(b);

        //Now do the loop.
        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                res[i][j] = vectorDotProduct(a[i], b[j]);
            }
        }

        return res;
    }

    //The first bracket is for rows, the second is for columns
    public static double[][] matrixTranspose(final double[][] a)
    {
        int i, j;
        int cols, rows;
        double res[][];

        //Store the number of rows
        rows = getNumberOfRows(a);
        if(rows <= 0)
            return null;

        //Store the number of columns and define res.
        cols = getNumberOfCols(a);
        if(cols <= 0)
            return null;

        //Set aside the memory for res
        res = new double[cols][rows];

        //This loop then copies element by element from a to res,
        //switching the rows for columns.
        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                res[j][i] = a[i][j];
            }
        }

        return res;
    }

    //Return the result of multiplying two matrices together.  
    public static double[][] matrixSum(final double[][] a, final double[][] b)
    {
        int i, j;
        int cols, rows;
        double res[][];

        //Verify that the number of columns of a equals the rows of b.
        cols = getNumberOfCols(a);
        if(cols <= 0)
            return null;
        if(cols != getNumberOfCols(b))
            return null;

        rows = getNumberOfRows(a);
        if(rows <= 0)
            return null;
        if(rows != getNumberOfRows(b))
            return null;

        //Define results
        res = new double[rows][cols];

        //Add the elements of the matrix one-by-one.
        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                res[i][j] = a[i][j] + b[i][j];
            }
        }

        //Return the results
        return res;
    }

    //Return the result of multiplying two matrices together.  
    public static double[][] matrixScale(final double scalar, final double[][] a)
    {
        int i, j;
        int cols, rows;
        double res[][];

        //Verify that the number of columns of a equals the rows of b.
        cols = getNumberOfCols(a);
        if(cols <= 0)
            return null;
        rows = getNumberOfRows(a);
        if(rows <= 0)
            return null;

        //Define results
        res = new double[rows][cols];

        //Add the elements of the matrix one-by-one.
        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                res[i][j] = scalar * a[i][j];
            }
        }

        //Return the results
        return res;
    }

    //Give the dot-product of two vectors.
    public static double vectorDotProduct(final double[] a, final double[] b)
    {
        int i = 0;
        int rows;
        double total = 0;

        //Check that vectors a and b are valid.
        if((a == null) || (b == null))
            return DataSet.MISSING;
        if((a.length <= 0) || (b.length <= 0) || (a.length != b.length))
            return DataSet.MISSING;

        rows = a.length;

        //compute the dot product of the two vectors.
        for(i = 0; i < rows; i++)
        {
            total += a[i] * b[i];
        }

        return total;
    }

    ///////////////////////////////////////////////////////////////////////////////
    // TOOL USED IN MatrixMath
    ///////////////////////////////////////////////////////////////////////////////

    //Returns the number of rows of a matrix, checking to make sure it is valid.
    private static int getNumberOfRows(final double[][] a)
    {
        if(a == null)
            return -1;

        if(a.length <= 0)
            return -1;

        return a.length;
    }

    //Returns the number of columns of a matrix, making sure it is valid and
    //that all rows have the same number of columns.
    private static int getNumberOfCols(final double[][] a)
    {
        final int rows = getNumberOfRows(a);
        if(rows <= 0)
            return -1;

        int i;
        final int prev = a[0].length;
        for(i = 1; i < rows; i++)
        {
            if(a[i] == null)
                return -1;

            if(a[i].length <= 0)
                return -1;

            if((prev != -1) && (a[i].length != prev))
                return -1;
        }

        return prev;
    }

    public static void printMatrix(final int[][] a)
    {
        System.out.println("---------------------------");
        System.out.println("HERE IS THE MATRIX:");
        System.out.println("---------------------------");

        if((a == null) || (a.length == 0))
        {
            System.out.println("EMPTY!");
            System.out.println("---------------------------");
            return;
        }

        final int rows = a.length;
        final int cols = a[0].length;
        int i, j;

        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                System.out.print("" + a[i][j] + "    ");
            }
            System.out.println("");
        }
        System.out.println("---------------------------");

    }

    //This is a tool to test this class.  It just prints a matrix.    
    public static void printMatrix(final double[][] a)
    {
        System.out.println("---------------------------");
        System.out.println("HERE IS THE MATRIX:");
        System.out.println("---------------------------");

        if((a == null) || (a.length == 0))
        {
            System.out.println("EMPTY!");
            System.out.println("---------------------------");
            return;
        }

        final int rows = a.length;
        final int cols = a[0].length;
        int i, j;

        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                System.out.print("" + a[i][j] + "    ");
            }
            System.out.println("");
        }
        System.out.println("---------------------------");
    }

    //This is a tool to test this class.  It just prints a matrix.    
    public static void printMatrix(final float[][] a)
    {
        System.out.println("---------------------------");
        System.out.println("HERE IS THE MATRIX:");
        System.out.println("---------------------------");

        if((a == null) || (a.length == 0))
        {
            System.out.println("EMPTY!");
            System.out.println("---------------------------");
            return;
        }

        final int rows = a.length;
        final int cols = a[0].length;
        int i, j;

        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                System.out.print("" + a[i][j] + "    ");
            }
            System.out.println("");
        }
        System.out.println("---------------------------");
    }

    //This is a tool to test this class.  It just prints a matrix.
    //This version will dump the matrix, changing the jhrvar data to 
    //a date.    
    public static void printMatrix(final double[][] a, final int jhrvar)
    {
        System.out.println("---------------------------");
        System.out.println("HERE IS THE MATRIX:");
        System.out.println("---------------------------");

        if((a == null) || (a.length == 0))
        {
            System.out.println("EMPTY!");
            System.out.println("---------------------------");
            return;
        }

        final int rows = a.length;
        final int cols = a[0].length;
        int i, j;

        Calendar date; //this will store the date to print.
        for(i = 0; i < rows; i++)
        {
            for(j = 0; j < cols; j++)
            {
                //If this is the jhrvar column, then print it as a date.
                if(j == jhrvar)
                {
                    date = HCalendar.computeCalendarFromJulianHour((int)a[i][j]);
                    System.out.print("" + HCalendar.buildDateTimeStr(date) + "    ");
                }
                //Otherwise, its just a number.
                else
                {
                    System.out.print("" + a[i][j] + "    ");
                }
            }
            System.out.println("");
        }
        System.out.println("---------------------------");
//        System.out.println("COUNTER = " + counter);
    }

    /**
     * Return the intersection point between two points. The point is returned as 2-d array.
     * 
     * @param line1 double[] of four pieces: x1, y1, x2, y2, in that order.
     * @param line2 double[] of four pieces: x1, y1, x2, y2, in that order.
     * @retuen 2-d double[] with x coordinate at index 0, y at 1.
     */
    public static double[] computeIntersectionPoint(final double[] line1, final double[] line2)
    {
        final double A1 = line1[1] - line1[3];
        final double B1 = line1[0] - line1[2];
        final double C1 = A1 * line1[0] + B1 * line1[1];

        final double A2 = line2[1] - line2[3];
        final double B2 = line2[0] - line2[2];
        final double C2 = A1 * line2[0] + B1 * line2[1];

        final double det = A1 * B2 - A2 * B1;
        final double[] results = new double[2];
        if(det == 0)
        {
            return null;
        }
        else
        {
            results[0] = (B2 * C1 - B1 * C2) / det;
            results[1] = (A1 * C2 - A2 * C1) / det;
        }

        return results;
    }

    public static void main(final String args[])
    {
        final double a[][] = new double[2][3];
        final double b[][] = new double[3][3];
        final double c[][] = new double[2][2];

        a[0][0] = 2;
        a[0][1] = 1;
        a[0][2] = 3;
        a[1][0] = 0;
        a[1][1] = 3;
        a[1][2] = 1;

        b[0][0] = 1;
        b[0][1] = 1;
        b[0][2] = 3;
        b[1][0] = 3;
        b[1][1] = 3;
        b[1][2] = 1;
        b[2][0] = 0;
        b[2][1] = 2;
        b[2][2] = 2;

        c[0][0] = 2;
        c[0][1] = 1;
        c[1][0] = 0;
        c[1][1] = 3;

        //a = matrixInverse(c);

        //MatrixMath.printMatrix(a);
        final double cvf[] = {25, 10, 0, 0, 0, 0, 0, 0, 0, 0, 0};
        final double gamma[] = {10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8};
        double G[][];
        double Ginv[][];
        double subgamma[][];
        double phi[][];

        int i, j, k;
        for(k = 0; k < 11; k++)
        {
            subgamma = new double[k + 1][1];
            G = new double[k + 1][k + 1];
            Ginv = new double[k + 1][k + 1];
            phi = new double[k + 1][1];

            System.out.println("Constructing G and subgamma...");
            for(i = 0; i < k + 1; i++)
            {
                for(j = 0; j < k + 1; j++)
                {
                    G[i][j] = cvf[Math.abs(i - j)];
                }
                subgamma[i][0] = gamma[i];
            }

            System.out.println("computing Ginv and phi...");
            Ginv = matrixInverse(G);
            phi = matrixMultiplication(Ginv, subgamma);

            System.out.println("HERE IS THE PHI MATRIX FOR h = " + (k + 1));
            MatrixMath.printMatrix(phi);
        }

    }

}
