package ohd.hseb.util.fews.ensmodels;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;

import ohd.hseb.measurement.Measurement;
import ohd.hseb.measurement.MeasuringUnit;
import ohd.hseb.measurement.RegularTimeSeries;
import ohd.hseb.util.Logger;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.data.ESPData;
import ohd.hseb.util.data.ESPDataException;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.FewsAdapterDAO;
import ohd.hseb.util.fews.FewsRegularTimeSeries;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.OHDUtilities;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HString;

/**
 * An {@link ArrayList} of {@link FewsRegularTimeSeries}, with a {@link #_memberTemplate}. It provides tools to apply
 * the member template to the members and populate a result map for use with ModelDriver execute method calls.
 * 
 * @author hank
 */
public class FewsEnsemble extends ArrayList<FewsRegularTimeSeries>
{
    private static final long serialVersionUID = 1L;

    private FewsRegularTimeSeries _memberTemplate = null;
    private String _ensembleId = null;
    public final static String ENSPOST_ENSEMBLE_ID = "Ensemble Post-Processor";

    /**
     * For the EPG and maybe others, the initial state time (or time 0) of the forecast to generate must be carried
     * around with the ensemble. This is also used in generateForecastedEnsemble if no start time is given to the
     * method.
     */
    private Calendar _forecastInitialStateTime = null;

    /**
     * The times associated with each member
     */
    private HashMap<Integer, long[]> _ensembleMemberIndexToTimesMap;

    /**
     * The values for an ensemble member.
     */
    private HashMap<Integer, double[]> _ensembleMemberIndexToValuesMap;

    /**
     * The values for a time step. Time series index refers to an index in time (i.e. the array or index of a value
     * within a RegularTimeSeries list of values), not a member index.
     */
    private HashMap<Integer, double[]> _timeSeriesIndexToValuesMap;

    private Logger _logger;

    public FewsEnsemble()
    {
        super();
        _logger = new Diagnostics();
    }

    /**
     * Copy constructor
     * 
     * @param fewsEnsemble - FewsEnsemble.
     */
    public FewsEnsemble(final FewsEnsemble fewsEnsemble)
    {
        this(fewsEnsemble.getEnsembleId(), new ArrayList<RegularTimeSeries>());

        for(final FewsRegularTimeSeries ts: fewsEnsemble)
        {
            this.add(new FewsRegularTimeSeries(ts));
        }
    }

    /**
     * Calls the other constructor, assuming null for locationId, parameterId, and qualifierId.
     * 
     * @param ensembleId The ensemble id in FEWS.
     * @param inputTSList List of RegularTimeSeries (assumed to be FewsRegularTimeSeries).
     */
    public FewsEnsemble(final String ensembleId, final List<RegularTimeSeries> inputTSList)
    {
        this(null, null, null, ensembleId, inputTSList);
    }

    /**
     * Grabs all members from the inputTSList that have the locationId, parameterId, qualifierId,and ensembleId
     * specified and places them into this list (THEY ARE NOT COPIES!!!) based on their memberIndex. Only use this
     * constructor if the inputTSList has known memberIndex values. Also, this should only be needed if you have a large
     * time series values with many different time series in it.
     * 
     * @param locationId The location id in FEWS.
     * @param parameterId The time series type (i.e. parameterId in FEWS).
     * @param qualifierId The qualifier id in FEWS.
     * @param ensembleId The ensemble id in FEWS.
     * @param inputTSList List of RegularTimeSeries (assumed to be FewsRegularTimeSeries).
     */
    @SuppressWarnings("deprecation")
    public FewsEnsemble(final String locationId,
                        final String parameterId,
                        final String qualifierId,
                        String ensembleId,
                        final List<RegularTimeSeries> inputTSList)
    {
        super();

        //Set the ensembleId and extract matching time series from inputTSList.
        int i;
        setEnsembleId(ensembleId);
        for(i = 0; i < inputTSList.size(); i++)
        {
            if(locationId != null)
            {
                if((inputTSList.get(i).getLocationId() == null)
                    || (!inputTSList.get(i).getLocationId().equalsIgnoreCase(locationId)))
                {
                    continue;
                }
            }
            if(parameterId != null)
            {
                if((inputTSList.get(i).getTimeSeriesType() == null)
                    || (!inputTSList.get(i).getTimeSeriesType().equalsIgnoreCase(parameterId)))
                {
                    continue;
                }
            }

            //If the ensemble id is specified for checking...
            if(ensembleId != null)
            {
                //If it has zero-length set it to NO_ENSEMBLE_ID (main).
                if(ensembleId.length() == 0)
                {
                    ensembleId = RegularTimeSeries.NO_ENSEMBLE_ID;
                }

                if(!inputTSList.get(i).getEnsembleId().equalsIgnoreCase(ensembleId))
                {
                    continue;
                }
            }

            if(qualifierId != null)
            {
                if((inputTSList.get(i).getQualifierIds().get(0) == null)
                    || (!inputTSList.get(i).getQualifierIds().get(0).equalsIgnoreCase(qualifierId)))
                {
                    continue;
                }
            }
            this.add((FewsRegularTimeSeries)inputTSList.get(i));
        }
        synchronizeListOrderToMemberIndex();
    }

    /**
     * Read in the ensemble from espData. Synchronizes the member index so that it matches the list order after reading
     * from ESPData.
     * 
     * @param ensembleId
     * @param espData
     */
    public FewsEnsemble(final String ensembleId, final ESPData espData)
    {
        super();
        readFromESPData(ensembleId, espData);
    }

    public List<RegularTimeSeries> convertToRegularTimeSeries()
    {
        final List<RegularTimeSeries> results = new ArrayList<RegularTimeSeries>();
        results.addAll(this);
        return results;
    }

    /**
     * @throws FewsEnsembleException If the members are not consistent. This checks start time, end time, location id,
     *             ensemble id, time step (intervalHours), time series type, and accumulative/instaneous flag.
     */
    @SuppressWarnings("deprecation")
    public void validateConsistencyOfEnsembleMembers() throws FewsEnsembleException
    {
        int i;
        for(i = 1; i < this.size(); i++)
        {
            if(get(i).getStartTime() != get(i - 1).getStartTime())
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": start times are different.");
            }
            if(get(i).getEndTime() != get(i - 1).getEndTime())
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": end times are different.");
            }
            if(!get(i).getLocationId().equalsIgnoreCase(get(i - 1).getLocationId()))
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": location ids are different.");
            }
            if(!get(i).getQualifierIds().get(0).equalsIgnoreCase(get(i - 1).getQualifierIds().get(0)))
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": qualifier ids are different: " + get(i).getQualifierIds().get(0)+ " and "
                    + get(i - 1).getQualifierIds().get(0));
            }

            //Ensemble id requires a null check first.
            if(!get(i).getEnsembleId().equalsIgnoreCase(get(i - 1).getEnsembleId()))
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": ensemble ids are different.");
            }

            if(get(i).getIntervalInHours() != get(i - 1).getIntervalInHours())
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": time steps are different.");
            }
            if(!get(i).getTimeSeriesType().equalsIgnoreCase(get(i - 1).getTimeSeriesType()))
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": time series data type is different(" + get(i).getTimeSeriesType() + " and "
                    + get(i - 1).getTimeSeriesType() + ").");
            }
            if(get(i).getType() != get(i - 1).getType())
            {
                throw new FewsEnsembleException("Ensemble member " + i + " is not consistent with " + (i - 1)
                    + ": type of time series (acc or inst) is different.");
            }
        }
    }

    /**
     * Converts all members to the units in the member template. Sets the ensembleId to be the id within this
     * FewsEnsemble object. Sets the time series type, location id, long name, source org, source system, and file
     * description to those values in the member template. Synchronizes the member index to match the list order.<br>
     * <br>
     * Use this after building the ensemble and before outputting it. DO NOT USE IT if the ensembleMemberIndex within
     * each member is correct, but the list has not been synchronized to match the member indices. If you do, then the
     * indices will be changed to match the list order, possibly making the indices incorrect.
     */
    public void applyTemplateToAllMembers()
    {
        if(_memberTemplate == null)
        {
            return;
        }

        //Convert the measurements, first.
        convertToUnit(_memberTemplate.getMeasuringUnit());

        //Now set the rest.
        for(int i = 0; i < this.size(); i++)
        {
            //Use the _ensembleId as the ensemble id for each time series.  Do not use the template.
            get(i).setEnsembleId(_ensembleId);
            get(i).setTimeSeriesType(_memberTemplate.getTimeSeriesType().trim());

            get(i).setType(_memberTemplate.getType());
            get(i).setLocationId(_memberTemplate.getLocationId().trim());
            get(i).setLongName(_memberTemplate.getLongName().trim());
            get(i).setSourceOrganization(_memberTemplate.getSourceOrganization().trim());
            get(i).setSourceSystem(_memberTemplate.getSourceSystem().trim());
            get(i).setFileDescription(_memberTemplate.getFileDescription().trim());
        }

        //I know this synchronizing could be done in the for loop above, but I want to do it
        //via this method so that I synchronize consistently.
        //TODO Do not call this automatically!  Force them to do so later.
        //synchronizeMemberIndexToListOrder();
        synchronizeListOrderToMemberIndex();
    }

    /**
     * @param targetUnit The unit to which to convert all member time series.
     */
    public void convertToUnit(final MeasuringUnit targetUnit)
    {
        for(int i = 0; i < this.size(); i++)
        {
            get(i).convert(targetUnit);
        }
    }

    /**
     * @param template The member template to use; this calls setMemberTemplate(template) and
     *            applyTemplateToAllMembers() prior to populating the resultMap (a map of time series type ->
     *            RegularTimeSeries). See applyTemplateToAllMembers() for details on how this is used. Be sure that the
     *            default ensembleId is set within the member template prior to calling this as appropriate for the
     *            FewsAdapter model being called.
     * @param params Model parameters, specifying the ensmeble id and possible other parameters that affect output
     *            list/map.
     * @param resultMap The map to populate.
     */
    @SuppressWarnings("deprecation")
    public void populateResultMap(final FewsRegularTimeSeries template,
                                  final EnsembleUtilParameters params,
                                  final Map<String, RegularTimeSeries> resultMap) throws Exception
    {
        //Set the ensembleId from the override in the parameters, if found.
        if((params != null) && (params.isParamExisting(EnsembleUtilParameters.OUTPUT_ENSEMBLE_ID)))
        {
            setEnsembleId(params.getOutputEnsembleId());
        }
        setMemberTemplate(template);
        applyTemplateToAllMembers();
        for(int i = 0; i < this.size(); i++)
        {
            //I don't believe it matters what the key value is.  The key is no longer used.
            resultMap.put(this.get(i).getLocationId() + this.get(i).getTimeSeriesType() + this.get(i).getQualifierIds().get(0)
                + this.get(i).getEnsembleId() + this.get(i).getEnsembleMemberIndex()
                + +this.get(i).getIntervalInHours(), this.get(i));
        }
    }

    /**
     * Synchronizes the list ordering to be the member indices of the individual time series.
     */
    public void synchronizeListOrderToMemberIndex()
    {
        final Comparator<FewsRegularTimeSeries> comp = new FewsEnsembleMemberIndexComparator<FewsRegularTimeSeries>();
        Collections.sort(this, comp);
    }

    /**
     * Synchronizes the member indices to match the list order.
     */
    public void synchronizeMemberIndexToListOrder()
    {
        for(int i = 0; i < size(); i++)
        {
            get(i).setEnsembleMemberIndex(i);
        }
    }

    /**
     * Prepends the passed in observed value, assumed to be in the same units as this, to the members. Useful for
     * inserting latest observed value before all other values, which can be useful when computing statistics.
     */
    public void prependValueToMembers(final double value)
    {
        for(int i = 0; i < this.size(); i++)
        {
            get(i).prependTimeSeries(get(i).getStartTime() - get(i).getIntervalInHours() * HCalendar.MILLIS_IN_HR,
                                     value);
        }
    }

    /**
     * Get sub time series.
     */
    public FewsEnsemble generateSubSeries(final long startTime, final long endTime)
    {
        final FewsEnsemble ens = new FewsEnsemble();
        for(int i = 0; i < size(); i++)
        {
            final RegularTimeSeries regSeries = get(i).getSubTimeSeries(startTime, endTime);
            final FewsRegularTimeSeries fewsSeries = new FewsRegularTimeSeries(get(i), regSeries);
            fewsSeries.setStartTime(startTime);
            fewsSeries.setEndTime(endTime);
            ens.add(fewsSeries);
        }
        ens.setEnsembleId(_ensembleId);
        return ens;
    }

    public void determineForecastInitialStateTimeFromData()
    {
        _forecastInitialStateTime = HCalendar.computeCalendarFromMilliseconds(get(0).getStartTime()
            - get(0).getIntervalInHours() * HCalendar.MILLIS_IN_HR);
    }

    /**
     * @return Long.MIN_VALUE if there are no members. Otherwise, it returns the _forecastBasisTime plus one time series
     *         time step (interval).
     */
    private long computeFirstForecastTime()
    {
        if(size() <= 0)
        {
            return Long.MIN_VALUE;
        }
        return _forecastInitialStateTime.getTimeInMillis() + get(0).getIntervalInHours() * HCalendar.MILLIS_IN_HR;
    }

    /**
     * @return The index that points to the first forecast value, or that value one time interval beyond the forecast
     *         basis time. Returns -1 if the forecast basis time is not defined, or if there are no ensemble members.
     * @throws FewsEnsembleException
     */
    public int computeIndexOfFirstForecastValue() throws FewsEnsembleException
    {
        if(getForecastInitialStateTime() == null)
        {
            return -1;
        }

        final long firstForecastValueTime = computeFirstForecastTime();
        if(firstForecastValueTime == Long.MIN_VALUE)
        {
            return -1;
        }
        for(int i = 0; i < size(); i++)
        {
            if(get(0).getMeasurementTimeByIndex(i) == firstForecastValueTime)
            {
                return i;
            }
            if(get(0).getMeasurementTimeByIndex(i) > firstForecastValueTime)
            {
                break;
            }
        }
        throw new FewsEnsembleException("Forecast basis time is defined, but cannot find the first forecast value "
            + "that should exist in the ensemble.");
    }

    /**
     * Assumes null for the argument to the other version of the method.
     */
    public FewsEnsemble generateForecastedEnsemble() throws FewsEnsembleException
    {
        return generateForecastedEnsemble(null);
    }

    /**
     * Generate an ensemble of time series for which each member contains only forecast values; i.e. those values
     * starting from the index returned by computeIndexOfFirstForecastValue and going to the end of the series.
     * 
     * @return modelRunStartTime The start time to assume for the first forecast value to include in the forecast
     *         ensemble. If null, it is assumed that the start time should be calculated based on the index of the first
     *         forecast value herein.
     * @return FewsEnsemble where each time series contains only those values that are forecasted values. Any value
     *         equal to or before the passed in time is discarded. It will return this if the entire series is part of
     *         the forecast ensemble or if _forecastBasisTime is null (its default value).
     * @throws FewsEnsembleException if the ensemble does not appear to be valid. That means that the first forecast
     *             value could not be found, because the basis time and forecast times do not appear to correspond
     *             correctly.
     */
    public FewsEnsemble generateForecastedEnsemble(final Calendar timeOfFirstValueInForecastedEnsemble) throws FewsEnsembleException
    {
        long firstValueTimeInMillis;
        if(timeOfFirstValueInForecastedEnsemble == null)
        {
            final int forecastStartIndex = this.computeIndexOfFirstForecastValue();
            if(forecastStartIndex <= 0)
            {
                return this;
            }
            firstValueTimeInMillis = this.computeFirstForecastTime();
        }
        else
        {
            firstValueTimeInMillis = timeOfFirstValueInForecastedEnsemble.getTimeInMillis();
        }
        final FewsEnsemble subSeries = this.generateSubSeries(firstValueTimeInMillis, this.getEndTime());
        return subSeries;
    }

    /**
     * Sets the ensembleId and calls readFromESPData(data).
     * 
     * @param ensembleId The ensemble id to assign.
     * @param data The ESPData to read.
     */
    public void readFromESPData(final String ensembleId, final ESPData data)
    {
        setEnsembleId(ensembleId);
        readFromESPData(data);
    }

    /**
     * Load the ensembles for the FewsEnsemble instance from the ESPData object passed in. Remember to call
     * setMemberTemplate(...) and applyTemplateToAllMembers after calling this to ensure that the members have proper
     * headers. Though this method will attempt to initialize the various fields in each time series, it will likely not
     * be the desired values in the end; hence the need for a template.
     * 
     * @param data ESPData containing data and info for this FewsEnsemble.
     */
    public void readFromESPData(final ESPData data)
    {

        this.clear();
        this._ensembleMemberIndexToTimesMap = null;
        this._ensembleMemberIndexToValuesMap = null;
        this._timeSeriesIndexToValuesMap = null;

        final long desiredWindowStartTime = data._fcstStart.getTimeInMillis() + data._tsdt * HCalendar.MILLIS_IN_HR;//data._fcstStart.getTimeInMillis() is T0(carryover date)
        final long desiredWindowEndTime = data._fcstEnd.getTimeInMillis();
        final int intervalInHours = data._tsdt;
        final MeasuringUnit unit = data.determineMeasuringUnit();

        //get a String like "1977-04-12 12:00:00 GMT", then extract month, dayOfMonth, hour info

        int i, j;

        //For each trace, construct a FewsRegularTimeSeries and do it.
        if(data._hsFlag == false)
        {
            for(i = 0; i < data._ntraces; i++)
            {//for CS data, one trace will be one TS

                final FewsRegularTimeSeries ts = new FewsRegularTimeSeries(desiredWindowStartTime,
                                                                           desiredWindowEndTime,
                                                                           intervalInHours,
                                                                           unit);

                ts.setTimeSeriesType(data._dataType);
                ts.setEnsembleMemberIndex(data._iy + i);
                ts.setEnsembleId(_ensembleId);
                //ts.setCreationDate(HCalendar.convertCalendarToString(data._fcstStart, "CCYY-MM-DD"));
                //ts.setCreationTime(HCalendar.convertCalendarToString(data._fcstStart, "hh:mm:ss"));
                ts.setLocationId(data._segmentID);

                ts.setLongName(data._segdesc);
                ts.setSourceOrganization("OHD");
                ts.setSourceSystem("CHPS");
                ts.setFileDescription("ESPTS");

                final DataSet traceDataSet = data.getCSTrace(i);

                for(j = 0; j < ts.getMeasurementCount(); j++)
                {

                    final double value = traceDataSet.getValue(j, ESPData.VALUE);
                    final Measurement measurement = new Measurement(value, unit);

                    ts.setMeasurementByIndex(measurement, j);

                }//close for(j = 0; j < ts.getMeasurementCount(); j++)

                add(ts);
            } //close for(i = 0; i < data._ntraces; i++)

        } //close if(data._hsFlag == false)
        else
        {//HS data will be one super long time series, which includes all traces
            final FewsRegularTimeSeries ts = new FewsRegularTimeSeries(data._runStart.getTimeInMillis(),
                                                                       data._runEnd.getTimeInMillis(),
                                                                       intervalInHours,
                                                                       unit);

            ts.setTimeSeriesType(data._dataType);
            ts.setEnsembleMemberIndex(data._iy + 0);
            ts.setEnsembleId(_ensembleId);
            //ts.setCreationDate(HCalendar.convertCalendarToString(data._fcstStart, "CCYY-MM-DD"));
            //ts.setCreationTime(HCalendar.convertCalendarToString(data._fcstStart, "hh:mm:ss"));
            ts.setLocationId(data._segmentID);

            ts.setLongName(data._segdesc);
            ts.setSourceOrganization("OHD");
            ts.setSourceSystem("CHPS");
            ts.setFileDescription("ESPTS");

            final int endIndex = data._ntraces * data._numberPerTrace;
            final DataSet traceDataSet = data.extractSubset(0, endIndex);

            for(j = 0; j < ts.getMeasurementCount(); j++)
            {

                final double value = traceDataSet.getValue(j, ESPData.VALUE);
                final Measurement measurement = new Measurement(value, unit);

                ts.setMeasurementByIndex(measurement, j);

            }//close for(j = 0; j < ts.getMeasurementCount(); j++)

            add(ts);
        }
    }//close method

    /**
     * Sets up the ESPData header information for that passed in.
     * 
     * @param data The ESPData object whose header will be setup based on this object. The start time, end time,
     *            measuring unit, and time step are acquired from the first element of this FewsEnsemble, so make sure
     *            that element exists before calling.
     * @param firstYear The first year within this ensemble.
     */
    private void specifyHeaderParameters(final ESPData data,
                                         final int firstYear,
                                         final TimeZone targetTimeZone) throws ESPDataException
    {
        //Determine the creation date calendar to specify _now parameters of ESPData.
        Calendar creationDate = null;
        if(size() > 0)
        {
            final SimpleDateFormat dateFmt = new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);
            dateFmt.setTimeZone(OHDConstants.GMT_TIMEZONE);
            try
            {
                final Date date = dateFmt.parse(get(0).getCreationDate() + " " + get(0).getCreationTime());
                creationDate = HCalendar.computeCalendarFromDate(date);
            }
            catch(final ParseException e)
            {
                throw new ESPDataException("Cannot parse creation date '" + get(0).getCreationDate() + " "
                    + get(0).getCreationTime() + "'.");
            }
        }

        //Determine other dates.
        final Calendar startDate = HCalendar.computeCalendarFromMilliseconds(this.get(0).getStartTime());
        startDate.add(Calendar.HOUR, -1 * get(0).getIntervalInHours());
        final Calendar endDate = HCalendar.computeCalendarFromMilliseconds(this.get(0).getEndTime());

        final SimpleDateFormat simpleDateFormat = new SimpleDateFormat(OHDConstants.DATE_TIME_FORMAT_STR);
        simpleDateFormat.setTimeZone(OHDConstants.GMT_TIMEZONE);

        //System.out.println(simpleDateFormat.format(startDate.getTime()));

        final Calendar runStartDate = (Calendar)startDate.clone();
        runStartDate.set(Calendar.YEAR, firstYear);
        // Fixed for Fogbugz 570 and 632, added by RHC
        final GregorianCalendar gregorianCalendarStartDate =
                                                           (GregorianCalendar)GregorianCalendar.getInstance(TimeZone.getTimeZone("GMT"));
        gregorianCalendarStartDate.setTime(startDate.getTime());
        final GregorianCalendar gregorianCalendarEndDate =
                                                         (GregorianCalendar)GregorianCalendar.getInstance(TimeZone.getTimeZone("GMT"));
        gregorianCalendarEndDate.setTime(endDate.getTime());
        final GregorianCalendar gregorianCalendarRunStartDate =
                                                              (GregorianCalendar)GregorianCalendar.getInstance(TimeZone.getTimeZone("GMT"));
        gregorianCalendarRunStartDate.setTime(runStartDate.getTime());

        /*
         * Fixed for Fogbugz case 570, added by RHC If T0 is 1st day of a year, start date and run start date will be
         * yyyy-12-31 then decrement run start date year by 1
         */
        if(gregorianCalendarStartDate.get(Calendar.MONTH) == 11
            && gregorianCalendarStartDate.get(Calendar.DAY_OF_MONTH) == 31
            && gregorianCalendarRunStartDate.get(Calendar.MONTH) == 11
            && gregorianCalendarRunStartDate.get(Calendar.DAY_OF_MONTH) == 31 && data._hsFlag == false)
        {
            runStartDate.set(Calendar.YEAR, firstYear - 1);
        }

        /*
         * Fixed for Fogbugz 632, added by RHC if start date is Feb. 29th and run start date is not a leap year, then
         * set the run start date to Feb. 28th instead default to March 1st
         */
        if(gregorianCalendarStartDate.isLeapYear(startDate.get(Calendar.YEAR))
            && gregorianCalendarStartDate.get(Calendar.MONTH) == 1
            && gregorianCalendarStartDate.get(Calendar.DAY_OF_MONTH) == 29
            && !gregorianCalendarRunStartDate.isLeapYear(runStartDate.get(Calendar.YEAR)))
        {
            runStartDate.set(Calendar.MONTH, 1);
            runStartDate.set(Calendar.DAY_OF_MONTH, 28);
        }
        /*
         * runStartDate.set(Calendar.MONTH, 9); runStartDate.set(Calendar.DAY_OF_MONTH, 1);
         * System.out.println(simpleDateFormat.format(runStartDate.getTime()));
         */
        Calendar tempCal = (Calendar)runStartDate.clone();
        tempCal.set(Calendar.YEAR, firstYear + size() - 1); //-1 accounts for both end points being traces
        final long runEndMillis = tempCal.getTime().getTime()
            + (endDate.getTime().getTime() - startDate.getTime().getTime());
        final Calendar runEndDate = HCalendar.computeCalendarFromMilliseconds(runEndMillis);

        //The end date used for computing the conditional month must be extended by one day if the forecast
        //year is a leap year, because for non-leap year traces, it will require an extra day.
        //============ This determines if the start - end period includes a leap day.
        final Calendar workingCal = (Calendar)endDate.clone();
        int checkYear = Integer.MIN_VALUE;

        //The number of conditional months computation must account for the time zone of the binary file AND
        //for the fact that hour 0 of day 2 is hour 24 of day 1 in the ESPADP/NWSRFS world.
        final Calendar startDateInESPTZ = (Calendar)startDate.clone();
        startDateInESPTZ.add(Calendar.MILLISECOND, targetTimeZone.getRawOffset());
        final Calendar endDateInESPTZ = (Calendar)endDate.clone();
        endDateInESPTZ.add(Calendar.MILLISECOND, targetTimeZone.getRawOffset());
        final Calendar workingCalInESPTZ = (Calendar)workingCal.clone();
        workingCalInESPTZ.add(Calendar.MILLISECOND, targetTimeZone.getRawOffset());
        if(HCalendar.isLeapYear(startDateInESPTZ.get(Calendar.YEAR)))
        {
            checkYear = startDateInESPTZ.get(Calendar.YEAR);
        }
        else if(HCalendar.isLeapYear(endDateInESPTZ.get(Calendar.YEAR)))
        {
            checkYear = endDateInESPTZ.get(Calendar.YEAR);
        }
        if(checkYear != Integer.MIN_VALUE)
        {
            final Calendar checkCal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
            checkCal.set(Calendar.YEAR, checkYear);
            checkCal.set(Calendar.MONTH, 1);
            checkCal.set(Calendar.DAY_OF_MONTH, 29);
            if(startDateInESPTZ.before(checkCal) && (endDateInESPTZ.after(checkCal)))
            {
                workingCalInESPTZ.add(Calendar.DAY_OF_YEAR, 1);
            }
        }
        workingCalInESPTZ.add(Calendar.SECOND, -1);//This will make it so that the ncm assumes 1-24 hours and not 0-23.
        final int numberOfConditionalMonths = workingCalInESPTZ.get(Calendar.MONTH)
            - startDateInESPTZ.get(Calendar.MONTH) + 1
            + (workingCalInESPTZ.get(Calendar.YEAR) - startDateInESPTZ.get(Calendar.YEAR)) * 12;

        //Specify the header parameters.
        data._formatver = 1.0f;
        data._fsegid = HString.formatStringToFieldWidth(8, data._segmentID, true);
        data._ftsid = HString.formatStringToFieldWidth(8, data._timeSeriesID, true);
        data._ftype = HString.formatStringToFieldWidth(4, data._dataType, true);
        data._tsdt = get(0).getIntervalInHours();
        data._simflag = 3;
        data._tsunit = HString.formatStringToFieldWidth(4, get(0).getMeasuringUnit().getName(), true);

        /**
         * WORK if use this data._tsunit = get(0).getMeasuringUnit().getName();
         */

        //Now parameters are the creation date.
        data._now[0] = creationDate.get(Calendar.MONTH) + 1;
        data._now[1] = creationDate.get(Calendar.DAY_OF_MONTH);
        data._now[2] = creationDate.get(Calendar.YEAR);
        data._now[3] = creationDate.get(Calendar.HOUR_OF_DAY) * 100 + creationDate.get(Calendar.MINUTE);
        data._now[4] = creationDate.get(Calendar.SECOND) * 100;

        data._im = runStartDate.get(Calendar.MONTH) + 1;
        data._iy = runStartDate.get(Calendar.YEAR);

        data._nlstz = OHDUtilities.getTimeZoneRawOffsetInHours(targetTimeZone);

        //Carryover is the hour of the day for T0 in GMT.
        tempCal = (Calendar)startDate.clone();
        tempCal.add(Calendar.HOUR, data._nlstz);
        final int carryOverHour = tempCal.get(Calendar.HOUR_OF_DAY);

        //Hour of day for end time.
        tempCal = (Calendar)endDate.clone();
        tempCal.add(Calendar.HOUR, data._nlstz);
        final int endDateHourOfDay = tempCal.get(Calendar.HOUR_OF_DAY);

        //TODO: I'm not sure the algorithms for determining the julian days below are accounting for the possibility that
        //the time zone shift could back up the day by one.  However, I don't think any RFC will encounter this problem 
        //because they forecast at 12Z, and I don't have time to look into it.

        //Date items.
        data._idarun = HCalendar.computeJulianDay(runStartDate.get(Calendar.YEAR),
                                                  runStartDate.get(Calendar.DAY_OF_YEAR));
        data._ldarun = HCalendar.computeJulianDay(runEndDate.get(Calendar.YEAR), runEndDate.get(Calendar.DAY_OF_YEAR));
        data._ijdlst = HCalendar.computeJulianDay(startDate.get(Calendar.YEAR), startDate.get(Calendar.DAY_OF_YEAR));
        data._ihlst = carryOverHour;
        data._ljdlst = HCalendar.computeJulianDay(endDate.get(Calendar.YEAR), endDate.get(Calendar.DAY_OF_YEAR));
        data._lhlst = endDateHourOfDay;

        //If 0's, then set to 24 appropriately.
        if(data._ihlst <= 0)
        {
            data._ijdlst--;
            data._idarun--;
            data._ihlst += 24;
        }
        if(data._lhlst <= 0)
        {
            data._ljdlst--;
            data._ldarun--;
            data._lhlst += 24;
        }

        data._ntraces = size();
        data._ncm = numberOfConditionalMonths;
        data._noutds = 0;
        data._irec = 2;
        data._dim = "L3/T";
        data._tscale = "INST";

        // Manually add data._dim and data._tscale based on data unit, data.tsunit
        if(data._tsunit.equalsIgnoreCase("CMSD"))
        {
            data._dim = "L3";
            data._tscale = "ACCM";
        }
        // Manually add data._dim and data._tscale based on data type, data.ftype
        if(data._ftype.equalsIgnoreCase("RAIM"))
        {
            data._dim = "L";
            data._tscale = "ACCM";
        }
        if(data._ftype.equalsIgnoreCase("SSTG") || data._ftype.equalsIgnoreCase("STG")
            || data._ftype.equalsIgnoreCase("SPEL") || data._ftype.equalsIgnoreCase("SWE"))
        {
            data._dim = "L";
        }

        //The segment description is never empty by setting to, in order of
        //preference, longName, stationName, or locationId. 
        if((this.get(0).getLongName() != null) && (!this.get(0).getLongName().isEmpty()))
        {
            data._segdesc = this.get(0).getLongName();
        }
        else if((this.get(0).getName() != null) && (!this.get(0).getName().isEmpty()))
        {
            data._segdesc = this.get(0).getName();
        }
        else
        {
            data._segdesc = this.get(0).getLocationId();
        }

        data._lat = -999.0f;
        data._long = -999.0f;

        //If using 1.8 for pi xml version in Module config, the inputs.xml will
        //include latitude and logitude parameters within the header 

        if((get(0).getLatitude() != null) && (get(0).getLongitude() != null))
        {
            data._lat = get(0).getLatitude();
            data._long = get(0).getLongitude();
        }
        else
        {
            if(_logger != null)
            {
                _logger.log(Logger.WARNING,
                            "No coordinates were provided for location <" + this.get(0).getLocationId()
                                + ">. Make sure that the pi version in the general section of the module configuration file is 1.8 or higher.");
            }

        }

        data._fgrp = "IGNORE  ";
        data._cgrp = "IGNORE  ";
        data._rfcname = "IGNORE  ";
        data._prsfstr = "    ";
        data._esptext = "";
        data._espfile = data.getESPTSFile().getName();
        data._prsfstr = "";
        data._adjcount = 0;

        data.processHeaderAndInitializeDataSet();
    }

    /**
     * @return [location id].[location id].[time series type].[interval]. All components of an ESP binary file name, but
     *         the extension.
     */
    public String generateESPBinaryFileNameWithoutExtension()
    {
        return generateESPBinaryFileNameWithoutExtension(null);
    }

    /**
     * @return [location id].[override or location id].[time series type].[interval]. All components of an ESP binary
     *         file name, but the extension.
     */
    public String generateESPBinaryFileNameWithoutExtension(final String segmentIdOverride)
    {
        if(segmentIdOverride != null)
        {
            return segmentIdOverride + "." + get(0).getLocationId() + "." + get(0).getTimeSeriesType() + "."
                + HString.formatIntegerToFieldWidth(2, get(0).getIntervalInHours(), true);
        }
        return get(0).getLocationId() + "." + get(0).getLocationId() + "." + get(0).getTimeSeriesType() + "."
            + HString.formatIntegerToFieldWidth(2, get(0).getIntervalInHours(), true);
    }

    /**
     * @return Full ESP binary file name, with extension either being an ensemble post-processor one or CS.
     */
    public String generateESPBinaryFileName()
    {
        //Identify the extension to use based on the ensemble id.
        String extension = determineESPBinaryFileExtension(_ensembleId);
        if(extension == null)
        {
            if(_ensembleId.toUpperCase().indexOf("HISTORIC") >= 0)
            {
                extension = "HS";
            }
            else
            {
                extension = "CS";
            }
        }

        return generateESPBinaryFileNameWithoutExtension() + "." + extension;
    }

    /**
     * @param ensembleId The ensemble id of the ensemble for which the extension must be determined. Will return null if
     *            it is not an ensemble generated by EnsPostModelDriver.
     * @return The String to use as the extension on the binary file name.
     */
    public static String determineESPBinaryFileExtension(final String ensembleId)
    {
        //The algorithm herein must match that in EnsPostModelParameters.determineOutputEnsembleId().
        if(ensembleId.startsWith(ENSPOST_ENSEMBLE_ID))
        {
            final String extension = ensembleId.substring(ENSPOST_ENSEMBLE_ID.length()).trim();
            return extension;
        }
        else
        {
            return null;
        }
    }

    /**
     * Assumes last year with data is 2010 (water year?). The first year to use is determined from this when calling the
     * other version of this method.
     * 
     * @param fileName See other version.
     * @return See other version.
     * @throws ESPDataException
     */
    public ESPData generateESPData(final TimeZone targetTimeZone) throws ESPDataException
    {
        final int firstYear = 2010 - size();
        return generateESPData(null, firstYear, targetTimeZone);
    }

    /**
     * Calls the other version with a first year of 2010-size() or the first ensemble member index, if that index is
     * after 1900. Binary format cannot be before 1900, so it assumes if the first member index is after 1900, then the
     * member indices specify the year. Furthermore, it checks that the time series member indices are properly set, so
     * that they do not skip any years.
     * 
     * @param fileName See other version.
     * @return See other version.
     * @throws ESPDataException See generateESPData. Also if the member indices indicate that they are years and some of
     *             the years are missing (i.e. the member indices skip numbers), then an error is thrown.
     */
    public ESPData generateESPData(final String fileName, final TimeZone targetTimeZone) throws ESPDataException
    {
        int firstYear = Calendar.getInstance().get(Calendar.YEAR) - size();

        this.synchronizeListOrderToMemberIndex();
        if(get(0).getEnsembleMemberIndex() >= 1900)
        {
            firstYear = get(0).getEnsembleMemberIndex();

            for(int i = 1; i < size(); i++)
            {
                if(get(i).getEnsembleMemberIndex() != get(i - 1).getEnsembleMemberIndex() + 1)
                {
                    throw new ESPDataException("Time series member indices start after 1900, which "
                        + "means they must correspond to years, but some years are missing.");
                }
            }
        }

        //Issue #396 (espadp is displaying the year of the esp time series files
        //created by the espadp adapter incorrectly)
        final long startTime = this.get(0).getStartTime();

        final Calendar datadate = HCalendar.computeCalendarFromMilliseconds(startTime);

        final int checkMonth = datadate.get(Calendar.MONTH) + 1;

        if(checkMonth > 9)
        {
            firstYear = firstYear - 1;
        }

        return generateESPData(fileName, firstYear, targetTimeZone);
    }

    /**
     * Generate an ESPData object from this FewsEnsemble. This assumes 2010 will be the last year within the ESPTS file.
     * It also still has the same problem of only allowing for about 90 ensemble members.
     * 
     * @param fileName File name for the ESPTS file that corresponds to this. This name specifies the segment id, time
     *            series id, data type, and time step that will be inserted into the header of the ESPData object. The
     *            rest of the info is determined based on the time series timings and using filler.
     * @param firstYearToUse The first year to assign to an ensemble.
     * @return ESPData object.
     * @throws ESPDataException if problem occurs during construction (usually a bogus file name).
     */
    public ESPData generateESPData(String fileName,
                                   final int firstYearToUse,
                                   final TimeZone targetTimeZone) throws ESPDataException
    {
        int i, j;
        double[] asample;
        Calendar currentTraceTime;

        if(fileName == null)
        {
            fileName = generateESPBinaryFileName();
        }

        final ESPData data = new ESPData(fileName, false);
        specifyHeaderParameters(data, firstYearToUse, targetTimeZone);
        data.processHeaderAndInitializeDataSet();

        for(i = 0; i < size(); i++)
        {
            //Get the trace start calendar.
            currentTraceTime = HCalendar.computeCalendarFromMilliseconds(get(i).getMeasurementTimeByIndex(0));
            currentTraceTime.set(Calendar.YEAR, firstYearToUse + i);
            for(j = 0; j < get(i).getMeasurementCount(); j++)
            {
                asample = new double[2];
                asample[0] = HCalendar.computeJulianHourFromCalendar(currentTraceTime, true);
                asample[1] = get(i).getMeasurementValueByIndex(j, get(i).getMeasuringUnit());
                data.addSample(asample);

                //Increment the trace time.
                currentTraceTime.add(Calendar.HOUR, get(i).getIntervalInHours());
            }
        }
        return data;
    }

    public void setForecastInitialStateTime(final Calendar stateTime)
    {
        _forecastInitialStateTime = stateTime;
    }

    public Calendar getForecastInitialStateTime()
    {
        return _forecastInitialStateTime;
    }

    public void setMemberTemplate(final FewsRegularTimeSeries memberTemplate)
    {
        _memberTemplate = memberTemplate;
    }

    public FewsRegularTimeSeries getMemberTemplate()
    {
        return _memberTemplate;
    }

    /**
     * Sets the ensemble id and relays the ensemble id to each member.
     * 
     * @param id The ensemble id.
     */
    public void setEnsembleId(final String id)
    {
        _ensembleId = id;
        for(int i = 0; i < this.size(); i++)
        {
            this.get(i).setEnsembleId(id);
        }
    }

    public String getEnsembleId()
    {
        return _ensembleId;
    }

    public long getStartTime()
    {
        if(size() > 0)
        {
            return get(0).getStartTime();
        }
        return Long.MIN_VALUE;
    }

    public int getStartTimeIndex()
    {
        if(size() > 0)
        {
            return 0;
        }
        return -1;
    }

    public long getEndTime()
    {
        if(size() > 0)
        {
            return get(0).getEndTime();
        }
        return Long.MIN_VALUE;
    }

    /**
     * @return The last index for which data exists.
     */
    public int getEndTimeIndex()
    {
        if(size() > 0)
        {
            return get(0).getMeasurementCount() - 1;
        }
        return -1;
    }

    public long getMillisecondForTimeIndex(final int index)
    {
        if(size() > 0)
        {
            return get(0).getMeasurementTimeByIndex(index);
        }
        return Long.MIN_VALUE;
    }

    /**
     * @param index The index for which values are needed.
     * @return Array of values if found or null if the index is invalid. It is possible, though not likely, that the
     *         returned array could contain get(i).getMissingValue() values.
     */
    public double[] getEnsembleValuesForIndex(final int index)
    {
        if((index < 0) || (index > get(0).getMeasurementCount()))
        {
            return null;
        }
        if(_timeSeriesIndexToValuesMap == null)
        {
            _timeSeriesIndexToValuesMap = new HashMap<Integer, double[]>();
        }
        if(_timeSeriesIndexToValuesMap.get(index) == null)
        {
            final double[] values = new double[this.size()];
            Measurement working;
            for(int i = 0; i < size(); i++)
            {
                working = get(i).getMeasurementByIndex(index);
                if(working != null)
                {
                    values[i] = working.getValue();
                }
                else
                {
                    values[i] = get(i).getMissingValue();
                }
            }
            return values;
        }
        return _timeSeriesIndexToValuesMap.get(index);
    }

    /**
     * Get the values across the ensemble for a particular time.
     * 
     * @param time The time to check in milliseconds.
     * @param exacthMatchingTime If false, then the time need not exactly match. In this case, the array, if the times
     *            don't exactly match, will contain the first values with a time AFTER the desired time. This follows
     *            the same logic of the aggregator which associates the time at the end of each aggregation period with
     *            that period. So, this routine will return the values associated with the period encompassing the
     *            passed in time.
     * @return Array of double representing the values found for that time. Null is returned if the time is not exactly
     *         matched or is outside the measuring time range, or if there are no ensemble members.
     */
    public double[] getEnsembleValuesForTime(final long time, final boolean exactMatchingTime)
    {
        if(size() == 0)
        {
            return null;
        }
        if((time < get(0).getStartTime()) || (time > get(0).getEndTime()))
        {
            return null;
        }

        //Find the index that corresponds to this time by brute force.
        int index = 0;
        while(index < get(0).getMeasurementCount())
        {
            //Check 1 -- exact match.
            //Check 2 -- first time AFTER the desired time, and the caller does not need
            //an exact matching time.
            if((get(0).getMeasurementTimeByIndex(index) == time)
                || ((!exactMatchingTime) && (get(0).getMeasurementTimeByIndex(index) > time)))
            {
                return getEnsembleValuesForIndex(index);
            }
            //If we get here, then we the know the caller does not want an exact matching time.
            //We also know that there is no exact matching time, because we've already passed
            //the desired time, so return null.
            if(get(0).getMeasurementTimeByIndex(index) > time)
            {
                return null;
            }
            index++;
        }
        return null;
    }

    /**
     * @param memberIndex The ensemble member for which values are to be gotten.
     * @return Array of millsecond longs.
     */
    public long[] getMemberTimes(final int memberIndex)
    {
        if(_ensembleMemberIndexToTimesMap == null)
        {
            _ensembleMemberIndexToTimesMap = new HashMap<Integer, long[]>();
        }
        if(_ensembleMemberIndexToTimesMap.get(memberIndex) == null)
        {
            final long[] times = FewsEnsemble.buildTimesFromFewsRegularTimeSeries(get(memberIndex));
            _ensembleMemberIndexToTimesMap.put(memberIndex, times);
        }
        return _ensembleMemberIndexToTimesMap.get(memberIndex);
    }

    /**
     * @param memberIndex The ensemble member for which values are to be gotten.
     * @return Array of values ready for charting.
     */
    public double[] getMemberValues(final int memberIndex)
    {
        if(_ensembleMemberIndexToValuesMap == null)
        {
            _ensembleMemberIndexToValuesMap = new HashMap<Integer, double[]>();
        }
        if(_ensembleMemberIndexToValuesMap.get(memberIndex) == null)
        {
            final double[] ySeries = FewsEnsemble.buildValuesFromFewsRegularTimeSeries(get(memberIndex));
            _ensembleMemberIndexToValuesMap.put(memberIndex, ySeries);
        }
        return _ensembleMemberIndexToValuesMap.get(memberIndex);
    }

    /**
     * Converts the passed in list to a FewsEnsemble, ASSUMING every member is actually a FewsRegularTimeSeries!!!
     * 
     * @param ts List of RegularTimeSeries where each element is a FewsRegularTimeSeries
     * @return FewsEnsemble
     */
    public static FewsEnsemble convertToFewsEnsemble(final List<RegularTimeSeries> ts)
    {
        int i;
        final FewsEnsemble ensemble = new FewsEnsemble();
        for(i = 0; i < ts.size(); i++)
        {
            ensemble.add((FewsRegularTimeSeries)ts.get(i));
        }
        ensemble.setEnsembleId(((FewsRegularTimeSeries)ts.get(0)).getEnsembleId());
        ensemble.determineForecastInitialStateTimeFromData();
        return ensemble;
    }

    /**
     * Converts the passed in FewsEnsemble to a List of RegularTimeSeries, ASSUMING every member is actually a
     * FewsRegularTimeSeries!!!
     * 
     * @param ensemble - FewsEnsemble where each element is a FewsRegularTimeSeries
     * @return ArrayList of RegularTimeSeries
     */
    public static List<RegularTimeSeries> convertToList(final FewsEnsemble ensemble)
    {
        int i;
        final List<RegularTimeSeries> list = new ArrayList<RegularTimeSeries>();
        for(i = 0; i < ensemble.size(); i++)
        {
            list.add(ensemble.get(i));
        }
        return list;
    }

    public static long[] buildTimesFromFewsRegularTimeSeries(final RegularTimeSeries ts)
    {
        final long[] times = new long[ts.getMeasurementCount()];
        int j;
        for(j = 0; j < ts.getMeasurementCount(); j++)
        {
            times[j] = ts.getMeasurementTimeByIndex(j);
        }
        return times;
    }

    public static double[] buildValuesFromFewsRegularTimeSeries(final RegularTimeSeries ts)
    {
        final double[] values = new double[ts.getMeasurementCount()];
        int j;
        for(j = 0; j < ts.getMeasurementCount(); j++)
        {
            values[j] = ts.getMeasurementByIndex(j).getValue();
        }
        return values;
    }

    public static void main(final String[] args)
    {
        if(args.length != 3)
        {
            System.out.println("Incorrect usage.  Correct usage is: <executable> <proper binary file name> <xml file name> <target time zone hr shift>");
            System.out.println("Time zone is the number of hours shifted off of GMT with no decimal place (-5, -6, etc).");
            System.exit(1);
        }

        System.out.println("Reading in binary file " + args[0] + "...");
        ESPData binaryData = null;
        try
        {
            binaryData = new ESPData(args[0]);
        }
        catch(final ESPDataException e)
        {
            System.out.println("Unable to read in binary file: " + e.getMessage());
            e.printStackTrace();
        }

        //This must correspond to how ENS_POST works
        String ensembleId = "Conditional Simulation";
        if(binaryData.isHistoricalSimulation())
        {
            ensembleId = "Historical Simulation";
        }
        else if(!binaryData._timeSeriesTypeExtension.equalsIgnoreCase("CS"))
        {
            ensembleId = ENSPOST_ENSEMBLE_ID + " " + binaryData._timeSeriesTypeExtension;
        }

        System.out.println("Assigning ensembleId of " + ensembleId);

        System.out.println("Creating a FewsEnsemble object....");
        final FewsEnsemble ensemble = new FewsEnsemble(ensembleId, binaryData);
        final Map<String, RegularTimeSeries> map = new HashMap<String, RegularTimeSeries>();

        System.out.println("Writing out XML file " + args[1]);
        System.out.println("Target time zone is " + args[2]);
        final Logger logger = new Diagnostics();

        try
        {
            final String tzString = "GMT" + args[2];

            ensemble.populateResultMap(ensemble.get(0), null, map);
            FewsAdapterDAO.writeTimeSeriesToFewsXML(ensemble, args[1], logger, TimeZone.getTimeZone(tzString));
        }
        catch(final Exception e)
        {
            System.out.println("Unable to generate XML file: " + e.getMessage());
            e.printStackTrace();
        }
    }
}
