package ohd.hseb.util.fews.ensmodels;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.MiscTools;
import ohd.hseb.util.Logger;

/**
 * This class extracts environment variables to export from the passed in Properties object. It then applies them to a
 * process environment via the exportEnvironmentVariables method. It also provides a get method to allow a caller to get
 * an apps-defaults token (which encompasses environment variables) or environment variables just as an affected process
 * would see it.
 * 
 * @author hank
 */
public class ProcessEnvironmentExporter
{

    public final static String ENVIRONMENT_TOKEN_PREFIX = "env.";

    private final Properties _props;

    private final HashMap<String, String> _map;

    /**
     * Takes a list of properties and extracts the lines for which the key is of the form "env.***". *** is assumed to
     * specify an environment variable, while the key's value specifies what the environment variable is to equal. Call
     * exportEnvironmentVariables after construction to export the environment variable into a process' environment.
     * 
     * @param props A Properties listing that contains keys of the format "env.***".
     */
    public ProcessEnvironmentExporter(final Properties props)
    {
        _props = props;
        _map = new HashMap<String, String>();
        final Iterator iter = _props.keySet().iterator();
        while(iter.hasNext())
        {
            final String key = (String)iter.next();
            if(key.startsWith(ENVIRONMENT_TOKEN_PREFIX))
            {
                final String envVar = key.substring(ENVIRONMENT_TOKEN_PREFIX.length(), key.length());
                _map.put(envVar, HString.removeBeginAndEndQuotes(_props.getProperty(key)));
            }
        }
    }

    public void addEnvironmentVariable(final String envVar, final String value)
    {
        _map.put(envVar, value);
    }

    /**
     * @param env Map that is assumed to be an environment() acquired from a ProcessBuilder object.
     */
    public void exportEnvironmentVariables(final Map<String, String> env, final Logger logger)
    {
        final Iterator<String> iter = _map.keySet().iterator();

        while(iter.hasNext())
        {
            final String key = iter.next();
            final String value = _map.get(key);
            logger.log(Logger.DEBUG, "Exporting " + key + "=" + value);
            env.put(key, value);
        }
    }

    public String getAppsDefaultsValue(final String envVar)
    {
        if(_map.get(envVar) != null)
        {
            return _map.get(envVar);
        }
        return MiscTools.getAppsDefaults(envVar, "");
    }

    public String getAppsDefaultsValue(final String envVar, final String defaultReturnValue)
    {
        if(_map.get(envVar) != null)
        {
            return _map.get(envVar);
        }
        return MiscTools.getAppsDefaults(envVar, defaultReturnValue);
    }

    public String getEnvironmentVariableValue(final String envVar)
    {
        if(_map.get(envVar) != null)
        {
            return _map.get(envVar);
        }
        if(System.getenv(envVar) != null)
        {
            return System.getenv(envVar);
        }
        return "";
    }

    public String getEnvironmentVariableValue(final String envVar, final String defaultReturnValue)
    {
        if(_map.get(envVar) != null)
        {
            return _map.get(envVar);
        }
        if(System.getenv(envVar) != null)
        {
            return System.getenv(envVar);
        }
        return defaultReturnValue;
    }
}
