/*
 * Created on Mar 5, 2008 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.util.fews.ensmodels.aggregator;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import ohd.hseb.measurement.Measurement;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.fews.FEWS_RTS_TYPE;
import ohd.hseb.util.fews.FewsRegularTimeSeries;
import ohd.hseb.util.misc.HCalendar;

/**
 * Aggregate a FewsRegularTimeSeries. The resulting time series will have the same meta info, but the start time, end
 * time, and interval will reflect the aggregation. Also, the type will become accumulated, regardless of what it
 * started with, and the time of each measurement indicates the END of the time period over which aggregation occurred.
 * 
 * @author hank
 */
public class FewsRegularTimeSeriesAggregator
{
    public static int TIME_WINDOW_LB_INDEX = 0;
    public static int TIME_WINDOW_UB_INDEX = 1;

    public static final int MISSING = -999;

    public static String DEFAULT_AGGREGATION_PACKAGE = "ohd.hseb.ensmodels.aggregator.computers";
    public static String DEFAULT_AGGREGATION_CLASS_SUFFIX = "Computer";
    public static String AGGREGATION_TYPE_AVERAGE = "Average";
    public static String AGGREGATION_TYPE_SUM = "Sum";
    public static String AGGREGATION_TYPE_MINIMUM = "Minimum";
    public static String AGGREGATION_TYPE_MAXIMUM = "Maximum";

    private final FewsRegularTimeSeries _timeSeries;
    private int _lastWorkingIndex = -1;
    private boolean _accumulatedTimeSeries = false;

    /**
     * Provides efficient list access to this information. The indices of this list must be consistent with those of
     * _timeSeries.
     */
    private List<long[]> _affectedTimeWindowByMeasurementIndex;

    public FewsRegularTimeSeriesAggregator(FewsRegularTimeSeries timeSeries)
    {
        _timeSeries = timeSeries;

        if(timeSeries.getType() == FEWS_RTS_TYPE.ACCUMULATIVE)
        {
            _accumulatedTimeSeries = true;
        }

        buildMeasurementToAffectedTimeWindowMap();
    }

    /**
     * Determines what time window is associated with each measurement. For instantaneous data, the time window starts
     * from the midpoint between the measurement and the previous measurement and ends at the midpoint between the
     * measurement and the next measurement. For accumulated data, the time window starts from the previous measurement
     * and ends at the time of the measurement.<br>
     * <br>
     * Note that for the first value of the time series, the previous measurement is considered to have a time equal to
     * the start time of the time series minus one interval (i.e. 6 hours before the start time if the time series has a
     * 6 hour time step). <br>
     * <br>
     * For the last value in the time series, the upper bound of its affected time window is always the end time of the
     * time series, for either instant or accumulated data.
     */
    private void buildMeasurementToAffectedTimeWindowMap()
    {
        _affectedTimeWindowByMeasurementIndex = new ArrayList<long[]>();
        int i;

        //Initialize the measurement trackers.  The previous measurement will point to the time
        //of the first measurement minus the time series interval.
        long previousMeasurementTime = _timeSeries.getStartTime() - _timeSeries.getIntervalInHours()
            * HCalendar.MILLIS_IN_HR;
//        Measurement previousMeasurement = 
        new Measurement(DataSet.MISSING, _timeSeries.getMeasuringUnit());
        long currentMeasurementTime = -1;
//        Measurement currentMeasurement = null;
        long nextMeasurementTime = _timeSeries.getStartTime();
//        Measurement nextMeasurement = 
        _timeSeries.getMeasurementByIndex(0);

        for(i = 0; i < _timeSeries.getMeasurementCount(); i++)
        {
            //Set the current measurement to be the next and recompute the next.
//            currentMeasurement = nextMeasurement;
            currentMeasurementTime = nextMeasurementTime;
//            nextMeasurement = null;
            if(i < _timeSeries.getMeasurementCount() - 1)
            {
//                nextMeasurement = 
                _timeSeries.getMeasurementByIndex(i + 1);
            }
            nextMeasurementTime = currentMeasurementTime + _timeSeries.getIntervalInHours() * HCalendar.MILLIS_IN_HR;

            //Determine lower bound.
            long[] timeWindow = new long[2];
            if(!_accumulatedTimeSeries)
            {
                timeWindow[0] = ((currentMeasurementTime + previousMeasurementTime) / 2);
            }
            else
            {
                timeWindow[0] = previousMeasurementTime;
            }

            //Compute the upper bound.
            if(!_accumulatedTimeSeries)
            {
                timeWindow[1] = ((currentMeasurementTime + nextMeasurementTime) / 2);
            }
            else
            {
                timeWindow[1] = currentMeasurementTime;
            }

            _affectedTimeWindowByMeasurementIndex.add(timeWindow);
//            previousMeasurement = currentMeasurement;
            previousMeasurementTime = currentMeasurementTime;
        }
    }

    /**
     * Aggregates time series between start time and end time to acquire a single number.
     * 
     * @param startTime The start time of the aggregation; must be at or after start time of time series.
     * @param endTime The end time of the aggregation; must be at or before end time of time series.
     * @param aggregationFullOrPartialClassName The aggregation to perform.
     * @return Value which is the single aggregation value for the time series.
     * @throws AggregationException if a parameter is invalid.
     */
    public double aggregateTimeSeries(long startTime, long endTime, String aggregationFullOrPartialClassName) throws AggregationException
    {
        //Load the aggregation comupter, first.
        AggregationComputer computer = AggregationComputerFactory.loadAggregationComputer(this,
                                                                                          aggregationFullOrPartialClassName);

        if(endTime < startTime)
        {
            throw new AggregationException("Aggregation end time is before start time.");
        }
        if(startTime < _timeSeries.getStartTime())
        {
            throw new AggregationException("Aggregation start time is before start of time series.");
        }
        if(endTime > _timeSeries.getEndTime())
        {
            throw new AggregationException("Aggregation end time is after end of time series.");
        }

        List<Integer> affectedIndices = this.findMeasurementIndicesAffectingTimeWindow(0, startTime, endTime);
        double value = computer.computeAggregatedValue(startTime, endTime, affectedIndices);

        return value;
    }

    /**
     * @param rawTimeSeries - Original unaggregated Timeseries
     * @param rawAggregatedTimeSeries - Aggregated Timeseries
     * @return - Disaggregated Timeseries
     * @throws AggregationException
     */
    public FewsRegularTimeSeries disaggregateTimeSeries(FewsRegularTimeSeries rawTimeSeries,
                                                        FewsRegularTimeSeries rawAggregatedTimeSeries,
                                                        boolean CPPAggregateMethod) throws AggregationException
    {
        FewsRegularTimeSeries aggregatedTimeSeries = _timeSeries;
        FewsRegularTimeSeries newTraces = new FewsRegularTimeSeries(rawTimeSeries);

//        int fromScale = 
        aggregatedTimeSeries.getIntervalInHours();
        int toScale = rawTimeSeries.getIntervalInHours();
//        int fromdivto = fromScale / toScale;
        long startDate, endDate;
        double origDailyAvg = MISSING;
        double adjDailyAvg = MISSING;
        double prevOrigAvg = MISSING;
        double prevAdjAvg = MISSING;
        double working = MISSING;
        long CPPAggregateShiftValue = 0;

        startDate = rawTimeSeries.getStartTime();
        endDate = rawTimeSeries.getEndTime();

        Calendar cal = Calendar.getInstance();

        if(CPPAggregateMethod)
        {
            CPPAggregateShiftValue = newTraces.getIntervalInMillis();
        }

        long currentAggregatedDate = aggregatedTimeSeries.getStartTime() - aggregatedTimeSeries.getIntervalInMillis();
//        long currentAggregatedDate = aggregatedTimeSeries.getStartTime();

        for(long currentDate = startDate; currentDate <= endDate; currentDate += (HCalendar.MILLIS_IN_HR * toScale))
        {
            if(((currentDate == currentAggregatedDate + CPPAggregateShiftValue) || (currentDate == startDate))
                && (currentDate != endDate))
            {
                prevOrigAvg = origDailyAvg;
                prevAdjAvg = adjDailyAvg;

                currentAggregatedDate += aggregatedTimeSeries.getIntervalInMillis();

                Measurement aggMeasurement = aggregatedTimeSeries.getMeasurementByTime(currentAggregatedDate);
                Measurement rawMeasurement = rawAggregatedTimeSeries.getMeasurementByTime(currentAggregatedDate);

                if(aggMeasurement == null)
                {
                    adjDailyAvg = MISSING;
                }
                else
                {
                    adjDailyAvg = aggMeasurement.getValue();
                }

                if(rawMeasurement == null)
                {
                    origDailyAvg = MISSING;
                }
                else
                {
                    origDailyAvg = rawMeasurement.getValue();
                }
            }

            if((adjDailyAvg == MISSING) || (origDailyAvg == MISSING))
            {
                working = MISSING;
            }
            else
            {
                working = (float)rawTimeSeries.getMeasurementByTime(currentDate).getValue();
            }

            if(working != MISSING)
            {
                if((prevOrigAvg != MISSING) && (prevAdjAvg != MISSING)) // right on the 24 hour marker
                {
                    working = working * (0.5 * (adjDailyAvg / origDailyAvg) + 0.5 * (prevAdjAvg / prevOrigAvg));
                }
                else
                // within the aggregation period
                {
                    working = working * (adjDailyAvg / origDailyAvg);
                }
            }

            newTraces.setMeasurementByTime(working, currentDate);
            cal.setTimeInMillis(currentDate);

//            System.out.println ( "adjDailyAvg/OrigDailyAvg prevAdj/PrevOrig working: " + adjDailyAvg + "/" + origDailyAvg + " " + prevAdjAvg+"/"+prevOrigAvg + "   " + working );

            prevOrigAvg = MISSING;
            prevAdjAvg = MISSING;
        }
        return newTraces;
    }

    /**
     * Returns a regular time series that is an aggregate of the time series passed in.
     * 
     * @param startTime The start time of the aggregation.
     * @param aggregateIntervalInHours The width of the aggregation.
     * @param aggregationFullOrPartialClassName The full class name of the AggregationComputer implementer to use, or
     *            just the part of the name after the package and before "Computer", if it is is within the standard
     *            package (see AggregationComputerFactory.DEFAULT_AGGREGATION_COMPUTER_PACKAGE). For example, Average
     *            and Sum are valid names.
     * @return RegularTimeSeries providing the aggregated time series.
     */
    public FewsRegularTimeSeries aggregateTimeSeries(long startTime,
                                                     int aggregateIntervalInHours,
                                                     String aggregationFullOrPartialClassName) throws AggregationException
    {
        //Load the aggregation comupter, first.
        AggregationComputer computer = AggregationComputerFactory.loadAggregationComputer(this,
                                                                                          aggregationFullOrPartialClassName);

        //Check time step.
        if(_timeSeries.getIntervalInHours() > aggregateIntervalInHours)
        {
            throw new AggregationException("Aggregation time step is less than the original time series time step; cannot aggregate.");
        }

        //Determine the end time.  This is the last time for which an aggregate will be computed.
        long aggregateEndTime = startTime;
        long aggregateIntervalInMillis = aggregateIntervalInHours * HCalendar.MILLIS_IN_HR;
        while(aggregateEndTime + aggregateIntervalInMillis < _timeSeries.getEndTime())
        {
            aggregateEndTime += aggregateIntervalInMillis;
        }

        //This piece will cause it to behave as ESPADP!!!!
        //Backup if the last time series measurement's affected time window does not include
        //the end time.  That means we do not have enough data to compute the last time.
        if(aggregateEndTime + aggregateIntervalInHours * HCalendar.MILLIS_IN_HR > _affectedTimeWindowByMeasurementIndex.get(_timeSeries.getMeasurementCount() - 1)[TIME_WINDOW_UB_INDEX])
        {
            aggregateEndTime -= aggregateIntervalInHours * HCalendar.MILLIS_IN_HR;
        }

        //Check to see if we have a valid window.
        if(aggregateEndTime < startTime)
        {
            throw new AggregationException("Either the time window or underlying time series does not allow for aggregation.");
        }

        //Create a result time series to hold the aggregated time series.  Add aggregateIntervalInMillis
        //in order to ensure that the end time of the aggregation time windows is used to denote that time window's value.
        FewsRegularTimeSeries resultTS = new FewsRegularTimeSeries(startTime + aggregateIntervalInMillis,
                                                                   aggregateEndTime + aggregateIntervalInMillis,
                                                                   aggregateIntervalInHours,
                                                                   _timeSeries.getMeasuringUnit());

        //Work through the window, calling the computer's computeAggregateValue as we go.
        long currentWindowStartTime = startTime;
        long currentWindowEndTime = startTime + aggregateIntervalInMillis;
        this._lastWorkingIndex = 0;
        double value;
//        double counter = 0;
        while(currentWindowStartTime <= aggregateEndTime)
        {
            List<Integer> affectedIndices = this.findMeasurementIndicesAffectingTimeWindow(_lastWorkingIndex,
                                                                                           currentWindowStartTime,
                                                                                           currentWindowEndTime);

            value = computer.computeAggregatedValue(currentWindowStartTime, currentWindowEndTime, affectedIndices);

            resultTS.setMeasurementByTime(new Measurement(value, resultTS.getMeasuringUnit()), currentWindowEndTime);

            currentWindowStartTime = currentWindowEndTime;
            currentWindowEndTime += aggregateIntervalInMillis;
//            System.out.println(counter++);
        }

        //Copy header info from the original time series.
        //TODO I need to be able to convert parameter ids based on aggregation
        //and original id.  I also need to change the type to be ACCUMULATED_TYPE.
        //For now, I can do neither as I have no cross reference telling me what
        //parameterId to use for an accumulated time series.
        //resultTS.setType(FewsRegularTimeSeries.ACCUMULATED_TYPE);
        resultTS.setType(_timeSeries.getType());
        resultTS.setTimeSeriesType(_timeSeries.getTimeSeriesType());
        resultTS.setLocationId(_timeSeries.getLocationId().trim());
        resultTS.setLongName(_timeSeries.getLongName().trim());
        resultTS.setSourceOrganization(_timeSeries.getSourceOrganization().trim());
        resultTS.setSourceSystem(_timeSeries.getSourceSystem().trim());
        resultTS.setFileDescription(_timeSeries.getFileDescription().trim());
        resultTS.setEnsembleId(_timeSeries.getEnsembleId());
        resultTS.setEnsembleMemberIndex(_timeSeries.getEnsembleMemberIndex());
        return resultTS;
    }

    /**
     * Provide a list of all measurement indices that have an affected time window overlapping with the given time
     * window. <br>
     * <br>
     * This method updates _lastWorkingIndex to be the index of the last measurement found that affects the time window
     * in question. If the time windows to check are non-overlapping, just pass in _lastWorkingIndex into the next call
     * of the method as the starting index.
     * 
     * @param startIndex Typically, _lastWorkingIndex or 0.
     * @param timeLB The inclusive lower bound on the time window.
     * @param timeUB The inclusive upper bound on the time window.
     * @return List of Integers specifying indices of measurements affecting the time window.
     */
    private List<Integer> findMeasurementIndicesAffectingTimeWindow(int startIndex, long timeLB, long timeUB)
    {
        List<Integer> results = new ArrayList<Integer>();

        int i;
        for(i = startIndex; i < _affectedTimeWindowByMeasurementIndex.size(); i++)
        {
            //If the window of interest is AFTER the current affected time window,
            //then I need to move forward to the next measurement
            if(timeLB > this.getAffectedTimeWindowUpperBound(i))
            {
                continue;
            }

            //If the window of interest is before the current affected time window, 
            //then I have gone too far.  So break out, set the workingIndex to be
            //the previous index, and return the results.
            if(timeUB < this.getAffectedTimeWindowLowerBound(i))
            {
                break;
            }
            results.add(Integer.valueOf(i));
        }

        //The last working index points to the index of the last measurement affecting this
        //time window.
        if((i == _affectedTimeWindowByMeasurementIndex.size()) && (timeUB == getAffectedTimeWindowLowerBound(i - 1)))
        {
            results.remove(results.size() - 1);
            //XXX
//            _lastWorkingIndex = -1; // Commented out as a workaround for hefs_ens_post_cp's aggregation of hs data from 24 hours 0z to 24 hours 12z
        }
        else
        {
            _lastWorkingIndex = Math.max(0, i - 2);
        }

        return results;
    }

    /**
     * It looks at the time window given, and compares it to the time window affected by the measurement at index
     * measurementIndicesAffectingWindow[indexOfInterest]. It determines if time window given overlaps the affected time
     * window, and if it does not, returns null. Note that touching does not count as an overlap; in other words if the
     * affected time window's upper bound is equal to timeLB, then null will be returned because they only touch; they
     * do not overlap! Otherwise, it returns a time window that specifies the overlapping region of the two.
     * 
     * @param timeLB The lower bound of the time window given.
     * @param timeUB The upper bound.
     * @param measurementIndicesAffectingWindow Array of indices as returned by the
     *            findMeasurementIndicesAffectingTimeWindow method and passed into the computation methods.
     * @param indexOfInterest The points to the measurement index to examine within measurementIndicesAffectingWindow.
     * @return Either null if no overlapping window exists (i.e. the measurement does not affect the time window given);
     *         or a long[2], where 0 is the lower bound and 1 is the upper bound. Note that if the overlap occurs only
     *         at any end point, this will return null.
     */
    public long[] determineOverlappingWindow(long timeLB,
                                             long timeUB,
                                             List<Integer> measurementIndicesAffectingWindow,
                                             int indexOfInterest)
    {
        long affectedTimeWindowLB = getAffectedTimeWindowLowerBound(measurementIndicesAffectingWindow.get(indexOfInterest));
        long affectedTimeWindowUB = getAffectedTimeWindowUpperBound(measurementIndicesAffectingWindow.get(indexOfInterest));

        //The lower bound used is either that specified by the affectedTimeWindowLB, or timeLB,
        //whichever is larger (later).  It is always timeLB if this is the first index to use.
        long overlapLB = Math.max(affectedTimeWindowLB, timeLB);
        if(indexOfInterest == 0)
        {
            overlapLB = timeLB;
        }

        //The upper bound used is either that specified by the affectedTimeWindowUB, or timeUB,
        //whichever is smaller (earlier).  It is always timeUB if this is the last index to use.
        long overlapUB = Math.min(affectedTimeWindowUB, timeUB);
        if(indexOfInterest == measurementIndicesAffectingWindow.size() - 1)
        {
            overlapUB = timeUB;
        }

        //Return null if the overlapLB is too big or the overlapUB is too small; either indicates
        //that the time windows do not overlap.
        if((overlapLB >= timeUB) || (overlapUB <= timeLB))
        {
            return null;
        }

        long[] results = {overlapLB, overlapUB};
        return results;
    }

    public long getAffectedTimeWindowLowerBound(int measurementIndex)
    {
        return _affectedTimeWindowByMeasurementIndex.get(measurementIndex)[TIME_WINDOW_LB_INDEX];
    }

    public long getAffectedTimeWindowUpperBound(int measurementIndex)
    {
        return _affectedTimeWindowByMeasurementIndex.get(measurementIndex)[TIME_WINDOW_UB_INDEX];
    }

    public FewsRegularTimeSeries getTimeSeries()
    {
        return _timeSeries;
    }

}
