package ohd.hseb.util.fews.ohdutilities;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import ohd.hseb.util.Logger;
import ohd.hseb.util.fews.GroupOfParameters;
import ohd.hseb.util.fews.OHDConstants;
import ohd.hseb.util.fews.ParameterType;
import ohd.hseb.util.fews.ParameterType.Row;
import ohd.hseb.util.fews.ParameterType.Table;
import ohd.hseb.util.fews.Parameters;

public class UtilityParameters extends Parameters
{

    /**
     * This class assumes the _logger is set using setLogger() for writing out diagnostics
     */
    public UtilityParameters()
    {
        //empty
    }

    public double getDoubleDataParameter(final String group, final String tag) throws Exception
    {
        return this.getParamWithValidation(group, tag, ParameterType.doubleType).getDblValue();
    }

    public Double getDoubleDataParameter(final String tag) throws Exception
    {
        return this.getParamWithValidation(OHDConstants.DEFAULT_STRING, tag, ParameterType.doubleType).getDblValue();
    }

    /**
     * Check the parameter exists or not in the corresponding GroupOfParameters and if it is the correct parameter type.
     * If not, throws an Exception; if passing the two checks, returns the parameter.
     */
    private ParameterType getParamWithValidation(final String groupId, final String paramName, final int paramType) throws Exception
    {
        final Map<String, ParameterType> defaultParamMap = super._groupParametersMap.get(groupId).getParameterMap();

        if(defaultParamMap.containsKey(paramName) == false)
        {
            throw new Exception("The parameter " + paramName + " does not exist in the '" + groupId + "' parameter.");
        }

        final ParameterType param = defaultParamMap.get(paramName);

        if(param.getType() != paramType)
        {
            throw new Exception("The parameter " + paramName + " in " + groupId + " is not the type "
                + ParameterType.typeName[paramType]);
        }

        return param;
    }

    /**
     * Return a double array as the parameter based on the group Id and the parameter tag.
     * 
     * @throws Exception
     */
    public double[] getDoubleArrayParameter(final String groupId, final String paramTag) throws Exception
    {

        final Table table = this.getParamWithValidation(groupId, paramTag, ParameterType.tableType).getTable();

        double[] newArray = null;
        int tableSize = 0;

        // convert the table result to double[]
        if(table.getRow() != null && (tableSize = table.getRow().size()) > 0)
        {
            newArray = new double[tableSize];
            for(int i = 0; i < tableSize; i++)
            {
                final Row row = table.getRow().get(i);
                newArray[i] = Double.parseDouble(row.getA());
            }
        }
        return newArray;
    }

    /**
     * Returns an integer parameter based on the parameter tag in the default GroupOfParameter.
     * 
     * @throws Exception
     */
    public Integer getIntDataParameter(final String tag) throws Exception
    {
        return this.getParamWithValidation(OHDConstants.DEFAULT_STRING, tag, ParameterType.integerType).getIntValue();
    }

    /**
     * Returns an integer parameter based on group name and the parameter tag.
     * 
     * @throws Exception
     */
    public int getIntDataParameter(final String group, final String tag) throws Exception
    {
        return this.getParamWithValidation(group, tag, ParameterType.integerType).getIntValue();
    }

    /**
     * Returns a String parameter based on group name and the parameter tag.
     * 
     * @throws Exception
     */
    public String getStringDataParameter(final String group, final String tag) throws Exception
    {
        return this.getParamWithValidation(group, tag, ParameterType.stringType).getStringValue();
    }

    /**
     * Returns a String parameter based on the parameter tag in the default GroupOfParameter.
     * 
     * @throws Exception
     */
    public String getStringDataParameter(final String tag) throws Exception
    {
        return this.getParamWithValidation(OHDConstants.DEFAULT_STRING, tag, ParameterType.stringType).getStringValue();
    }

    /**
     * Returns a boolean parameter based on group name and the parameter tag.
     * 
     * @throws Exception
     */
    public boolean getBooleanDataParameter(final String group, final String tag) throws Exception
    {
        return this.getParamWithValidation(group, tag, ParameterType.booleanType).getBoolValue();
    }

    /**
     * Returns a boolean parameter based on the parameter tag in the default GroupOfParameter.
     * 
     * @throws Exception
     */
    public Boolean getBooleanDataParameter(final String tag) throws Exception
    {
        return this.getParamWithValidation(OHDConstants.DEFAULT_STRING, tag, ParameterType.booleanType).getBoolValue();
    }

    public Table getTableDataParameter(final String group, final String tag) throws Exception
    {
        return this.getParamWithValidation(group, tag, ParameterType.tableType).getTable();
    }

    public Table getTableDataParameter(final String tag) throws Exception
    {
        return this.getParamWithValidation(OHDConstants.DEFAULT_STRING, tag, ParameterType.tableType).getTable();
    }

    /**
     * Gets all parameters for a giving group.
     * 
     * @param groupId The id of the group of parameters to retrieve.
     * @return a ModelGroupOfParameters object containing all parameters values for the giving group id.
     */

    public GroupOfParameters getParameters(final String groupId)
    {
        return _groupParametersMap.get(groupId);
    }

    /**
     * Check if the parameter exist or not in the groupId' GroupOfParameter. But, do not check the parameter type.
     */
    public boolean isParamExistingInGroup(final String groupId, final String tag)
    {

        if(_groupParametersMap.get(groupId).getParameterMap().containsKey(tag))
        {
            return true;
        }

        return false;

    }

    /**
     * Check if the parameter existing or not in the default GroupOfParameter. But, do not check the parameter type.
     */
    public boolean isParamExisting(final String tag)
    {
        return isParamExistingInGroup(OHDConstants.DEFAULT_STRING, tag);
    }

    public Map<String, GroupOfParameters> getParamsMap()
    {
        return Collections.unmodifiableMap(_groupParametersMap);
    }

    @Override
    public List<GroupOfParameters> getParamsList()
    {
        return Collections.unmodifiableList(new ArrayList<GroupOfParameters>(_groupParametersMap.values()));
    }

    /**
     * Insert individual objects of type {@link GroupOfParameters} into _groupParametersMap with the key of its Id. Then
     * call {@link #extractValuesFromMap()}.
     */
    @Override
    public void setParametersFromList(final List<GroupOfParameters> paramsList) throws Exception
    {
        // Check if valid period already exist in map, we need to add/replace the new values to these default otherwise it will overwrite all existing values.
        // need to iterate in all keys.
        if(paramsList != null && paramsList.size() > 0)
        {
            for(final GroupOfParameters groupParams: paramsList)
            {
                _groupParametersMap.put(groupParams.getId(), groupParams);
            }
        }

        extractValuesFromMap(); //retrieve all the values from _paramsMap to subclass instance variables if the method has been overwriten
    }

    /**
     * This methods should avoid being used is only for backward compatibility it is maintained. use
     * {@link #setParametersFromList(List)} instead
     * 
     * @param paramsMap
     */
    @Deprecated
    public void setParametersFromMap(final Map<String, GroupOfParameters> paramsMap) throws Exception
    {
        // Check if valid period already exist in map, we need to add/replace the new values to these default otherwise it will overwrite all existing values.
        // need to iterate in all keys.
        if(_groupParametersMap != null && _groupParametersMap.size() > 0)
        {
            for(final String groupId: paramsMap.keySet())
            {
                if(_groupParametersMap.containsKey(groupId))
                {
                    _groupParametersMap.get(groupId).getParameterMap().putAll(paramsMap.get(groupId).getParameterMap());
                }
                else
                {
                    _groupParametersMap.put(groupId, paramsMap.get(groupId));
                }
            }
        }
        else
        {
            _groupParametersMap.putAll(paramsMap);
        }

        extractValuesFromMap(); //retrieve all the values from _paramsMap to subclass instance variables if the method has been overwriten
    }

    /**
     * overloaded method for simplicity: param as <doubleData> in params.xml
     * 
     * @param tag
     * @param value - double
     */
    public void insertDoubleParameter(final String groupId, final String tag, final double value)
    {
        this.insertParameter(groupId, tag, Double.valueOf(value));
    }

    public void insertDoubleArrayParameter(final String groupId, final String tag, final double[] values)
    {
        final Table table = new Table();
        final Row columnTypes = new Row();
        Row row = null;
        for(int i = 0; i < values.length; i++)
        {
            row = new Row();
            row.setA(Double.toString(values[i]));
            table.setRow(row);
        }
        columnTypes.setA("double");
        table.setColumnTypes(columnTypes);

        this.insertParameter(groupId, tag, table);
    }

    /**
     * overloaded method for simplicity: param as <intData> in params.xml
     * 
     * @param tag
     * @param value - int
     */
    public void insertIntegerParameter(final String groupId, final String tag, final int value)
    {
        this.insertParameter(groupId, tag, Integer.valueOf(value));
    }

    /**
     * overloaded method for simplicity: param as <boolValue> in params.xml
     * 
     * @param tag
     * @param value - boolean
     */
    public void insertBooleanParameter(final String groupId, final String tag, final boolean value)
    {
        this.insertParameter(groupId, tag, Boolean.valueOf(value));
    }

    public void insertParameter(final String groupId, final String tag, final Object value)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        if(value instanceof Double)
        {
            param.setDblValue((Double)value);
        }
        else if(value instanceof Integer)
        {
            param.setIntValue((Integer)value);
        }
        else if(value instanceof Boolean)
        {
            param.setBoolValue((Boolean)value);
        }
        else if(value instanceof String)
        {
            param.setStringValue((String)value);
        }
        else if(value instanceof Table)
        {
            param.setTable((Table)value);
        }
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = _groupParametersMap.get(groupId);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(groupId);
            group.setParameter(param);
            _groupParametersMap.put(groupId, group);
        }
    }

    public void insertParameter(final String tag, final Object value)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        if(value instanceof Double)
        {
            param.setDblValue((Double)value);
        }
        else if(value instanceof Integer)
        {
            param.setIntValue((Integer)value);
        }
        else if(value instanceof Boolean)
        {
            param.setBoolValue((Boolean)value);
        }
        else if(value instanceof String)
        {
            param.setStringValue((String)value);
        }
        else if(value instanceof Table)
        {
            param.setTable((Table)value);
        }
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = _groupParametersMap.get(OHDConstants.DEFAULT_STRING);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(OHDConstants.DEFAULT_STRING);
            group.setParameter(param);
            _groupParametersMap.put(OHDConstants.DEFAULT_STRING, group);
        }
    }

    /**
     * Insert/Update a parameter based in the Group Id
     * 
     * @param groupId
     * @param tag
     * @param value
     */
    public void insertParameters(final String groupId, final GroupOfParameters modelGroup)
    {
        _groupParametersMap.put(groupId, modelGroup);

    }

    /**
     * overloaded method for simplicity: param as
     * <table>
     * in params.xml
     * 
     * @param tag
     * @param value - Table
     */
    public void insertTableParameter(final String groupId, final String tag, final Table value)
    {
        GroupOfParameters group = null;
        final ParameterType param = new ParameterType();
        param.setId(tag);
        param.setTable(value);
        //check if the group exist, if so add the parameter to the this group, otherwise we need to create the group
        group = _groupParametersMap.get(groupId);
        if(group != null)
        {
            group.setParameter(param);
        }
        else
        {
            group = new GroupOfParameters();
            group.setId(groupId);
            group.setParameter(param);
            _groupParametersMap.put(groupId, group);
        }
    }

    /**
     * Delete the group of parameters based in a group id
     * 
     * @param groupId the group of parameters to be deleted.
     */
    public void deleteParameters(final String groupId)
    {
        if(_groupParametersMap.containsKey(groupId))
        {
            _groupParametersMap.remove(groupId);
        }
    }

    /**
     * For Java models(snow17, sacsma, sacsmaHT and UHG), this method can be implemented to extract values from
     * _paramsMap to instance variables to avoid frequently access {@link #_validPeriodParametersMap}. For legacy
     * models, this method remains empty and do nothing.
     * <p>
     * This method is called in {@link #setParametersFromMap(Map)}.
     */
    @Override
    protected void extractValuesFromMap() throws Exception
    {
        //empty
    }

    @Override
    protected void setValuesToMap() throws Exception
    {
    }

    //In alphabetic order, each line is "parameter_name= value"
    @Override
    public String toString()
    {
        final StringBuilder resultStr = new StringBuilder();

        resultStr.append("Parameters:").append(OHDConstants.NEW_LINE);
        resultStr.append("Version = " + super.getVersion()).append(OHDConstants.NEW_LINE);

        if(this.getParamsMap().isEmpty())
        {
            System.out.println("Empty map ... wrong in some point");
        }

        for(final Entry<String, GroupOfParameters> entry: _groupParametersMap.entrySet())
        {
            resultStr.append("Key = " + entry.getKey()).append(OHDConstants.NEW_LINE);
            resultStr.append("GroupOfParameters =").append(OHDConstants.NEW_LINE);
            resultStr.append(entry.getValue().toString()).append(OHDConstants.NEW_LINE);
        }

        return resultStr.toString();
    }

    /**
     * Similar to equals method the only difference is that is does not check the whole object only look for the default
     * parameter. The Default parameters is the one with Group Id = "Default" and does not have a valid period (null in
     * this case).
     */
    public boolean equalsDefaultParams(final Object other)
    {
        boolean returnVal = false;

        if(other == this)
        {
            returnVal = true;
        }
        else if(other == null)
        {
            returnVal = false;
        }
        else if(!getClass().equals(other.getClass()))
        {
            returnVal = false;
        }
        else if(other instanceof Parameters)
        {
            final UtilityParameters other2 = (UtilityParameters)other;
            returnVal = testEqualsWithNull(this.getVersion(), other2.getVersion());

            final GroupOfParameters params = this.getParamsMap().get(OHDConstants.DEFAULT_STRING);
            final GroupOfParameters paramsMods = other2.getParamsMap().get(OHDConstants.DEFAULT_STRING);

            if(params.equals(paramsMods))
            {
                returnVal = true;
            }
            else
            {
                _logger.log(Logger.DEBUG, "parameters are not equal");
                returnVal = false;
            }

        }
        _logger.log(Logger.DEBUG, "parameters are equal");
        return returnVal;
    }

    @Override
    public boolean equals(final Object other)
    {
        boolean returnVal = false;

        if(other == this)
        {
            returnVal = true;
        }
        else if(other == null)
        {
            returnVal = false;
        }
        else if(!getClass().equals(other.getClass()))
        {
            returnVal = false;
        }
        else if(other instanceof UtilityParameters)
        {
            final UtilityParameters other2 = (UtilityParameters)other;
            returnVal = testEqualsWithNull(this.getVersion(), other2.getVersion());

            // first check the size() is equal
            if(this.getParamsMap().size() != other2.getParamsMap().size())
            {
                _logger.log(Logger.ERROR, "Error: the two Parameters objects have different number of parameters");
                return false;
            }

            for(final String groupId: this.getParamsMap().keySet())
            {
                final GroupOfParameters params = this.getParamsMap().get(groupId);
                final GroupOfParameters paramsMods = other2.getParamsMap().get(groupId);

                if(params.equals(paramsMods))
                {
                    continue;
                }
                else
                {
                    _logger.log(Logger.DEBUG,
                                "params are not equal:" + this.getParamsMap().size() + " " + params.getId() + "\n\n "
                                    + paramsMods.getId());

                    System.out.println("**********************");
                    System.out.println(params.toString());
                    System.out.println("**********************");
                    System.out.println(paramsMods.toString());
                    System.out.println("**********************");
                    returnVal = false;
                }

            }

        }
        _logger.log(Logger.DEBUG, "params are equal");
        return returnVal;

    }

    private static boolean testEqualsWithNull(final Object a, final Object b)
    {
        return (a == null && b == null) || (a != null && b != null) && (a.equals(b));
    }

    /**
     * This methods is not use.
     */
    @Override
    public int getNumberOfValidPeriods()
    {
        return 0;
    }

    /**
     * This methods is not use.
     */
    @Override
    public void setNumberOfValidPeriods(final int validPeriodNumber)
    {
    }

    @Override
    public int hashCode()
    {
        assert false : "hashCode not designed";
        return 0; // any arbitrary constant will do 
    }

    /**
     * Returns true when does not containing any entry of area ID versus GroupOfParameters.
     */
    public boolean isEmpty()
    {
        return _groupParametersMap.isEmpty();
    }

}
