package ohd.hseb.util.io;

import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;

/**
 * This subclass of DataInputStream supplies five extra functions. Each is the same as the the routine in
 * DataInputStream that reads in a primitive numerical data types (short, int, long, float, or double), except that it
 * will reverse the bytes if the environment variable HOSTOS says that this is a linux machine. This is required because
 * Java ALWAYS assumes big-endian! For examples of how this is used, see ESPData. <br>
 * <br>
 * UPDATING A ROUTINE TO SWITCH FROM A DataInputStream TO A HBinaryInputStream:<br>
 * <br>
 * First, you need to change all references to "DataInputStream" into "HBinaryInputStream". Next, all read commands for
 * float, double, short, int, or long are changed to the corresponding *Swap routine in here. So readFloat becomes
 * readFloatSwap, readInt becomes readIntSwap, etc. <br>
 * <br>
 * 
 * @author hank
 */
public class EndianConvertingInputStream extends DataInputStream
{
    final public static String CLASSNAME = "HBinaryInputStream";

    //Static vars
    final public static int FLOAT_SIZE = 4;
    final public static int DOUBLE_SIZE = 8;
    final public static int SHORT_SIZE = 2;
    final public static int INT_SIZE = 4;
    final public static int LONG_SIZE = 8;

    //Flags to pass in to the setEndianFlag routine.  
    final public static boolean LITTLE_ENDIAN_DATA_FLAG = true;
    final public static boolean BIG_ENDIAN_DATA_FLAG = false;

    ///////////////////////////////////////////////////////////////////////////
    //Attributes
    ///////////////////////////////////////////////////////////////////////////
    boolean _reversebytesflag; //If this is true, then reverse the bytes before processing.

    ///////////////////////////////////////////////////////////////////////////
    //Constructors
    ///////////////////////////////////////////////////////////////////////////

    /**
     * This is the default constructor and will just call super followed by something to set the reversebytesflag if
     * necessary.
     * 
     * @param in InputStream to serve as base for this object.
     */
    public EndianConvertingInputStream(final InputStream in)
    {
        //Call the constructor for the super class.
        super(in);

        //Set the flag to determine if endians are to be swapped.
        //Here, I should read an environment variable or an apps-defaults token in order
        //to determine if I am on a linux system.  The default value is...
        _reversebytesflag = false; //True is for linux runs

        //Get the system property sun.cpu.endian.
        final String endian = System.getProperty("sun.cpu.endian");
        if((endian == null) || (endian.length() == 0))
        {
            //System getProperty failed to find sun.cpu.endian Java property.  Assuming big endian.
            return;
        }

        //Check for "little" (LINUX_SETTING).  If so, turn on the reversal flag.
        if(endian.equalsIgnoreCase("little"))
        {
            _reversebytesflag = true;
        }
        else
        {
        }
    }

    /**
     * This is the default constructor and will just call super and set the _reversebytesflag to the passed in flag.
     * Note that the flag should be one of the static flag variables above, for sanity sake.
     * 
     * @param in
     * @param flag
     */
    public EndianConvertingInputStream(final InputStream in, final boolean flag)
    {
        //Call the constructor for the super class.
        super(in);

        //Set the flag to determine if endians are to be swapped.
        _reversebytesflag = flag;
    }

    ///////////////////////////////////////////////////////////////////////////
    //Tools
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Reads in a float; uses readByteArraySwap to read bytes for swapping.
     * 
     * @return
     * @throws IOException
     */
    public float readFloatSwap() throws IOException
    {
        float value = 0.0f;
        if(_reversebytesflag)
        {
            final int intBits = this.readInt();
            final int reversedIntBytes = Integer.reverseBytes(intBits);
            value = Float.intBitsToFloat(reversedIntBytes);
        }
        else
        {
            value = this.readFloat();
        }
        return value;
    }

    /**
     * Reads in a double; uses readByteArraySwap to read bytes for swapping.
     */
    public double readDoubleSwap() throws IOException
    {
        //Now, read in the bytes of the number and reverse them if necessary.
        final byte[] bytearray = readByteArraySwap(DOUBLE_SIZE);

        //Now I need to get my four bytes into a float.  So, I put the byte array into a 
        //ByteArrayInputStream...
        final ByteArrayInputStream bytestream = new ByteArrayInputStream(bytearray, 0, DOUBLE_SIZE);

        //cast it into a DataInputStream...
        final DataInputStream binfile = new DataInputStream(bytestream);

        //and read the float.
        final double value = binfile.readDouble();

        return value;
    }

    /**
     * Reads in an int; uses readByteArraySwap to read bytes for swapping.
     */
    public int readIntSwap() throws IOException
    {
        //Now, read in the bytes of the number and reverse them if necessary.
        final byte[] bytearray = readByteArraySwap(INT_SIZE);

        //Now I need to get my four bytes into a float.  So, I put the byte array into a 
        //ByteArrayInputStream...
        final ByteArrayInputStream bytestream = new ByteArrayInputStream(bytearray, 0, INT_SIZE);

        //cast it into a DataInputStream...
        final DataInputStream binfile = new DataInputStream(bytestream);

        //and read the float.
        final int value = binfile.readInt();

        return value;
    }

    /**
     * Reads in a long; uses readByteArraySwap to read bytes for swapping.
     */
    public long readLongSwap() throws IOException
    {
        //Now, read in the bytes of the number and reverse them if necessary.
        final byte[] bytearray = readByteArraySwap(LONG_SIZE);

        //Now I need to get my four bytes into a float.  So, I put the byte array into a 
        //ByteArrayInputStream...
        final ByteArrayInputStream bytestream = new ByteArrayInputStream(bytearray, 0, LONG_SIZE);

        //cast it into a DataInputStream...
        final DataInputStream binfile = new DataInputStream(bytestream);

        //and read the float.
        final long value = binfile.readLong();

        return value;
    }

    /**
     * Reads in a short; uses readByteArraySwap to read bytes for swapping.
     */
    public short readShortSwap() throws IOException
    {
        //Now, read in the bytes of the number and reverse them if necessary.
        final byte[] bytearray = readByteArraySwap(SHORT_SIZE);

        //Now I need to get my four bytes into a float.  So, I put the byte array into a 
        //ByteArrayInputStream...
        final ByteArrayInputStream bytestream = new ByteArrayInputStream(bytearray, 0, SHORT_SIZE);

        //cast it into a DataInputStream...
        final DataInputStream binfile = new DataInputStream(bytestream);

        //and read the float.
        final short value = binfile.readShort();

        return value;
    }

    /**
     * Reverse the bytes in the passed in array; do the swap!
     * 
     * @param bytearray Array of bytes.
     * @return Reversed array of bytes.
     */
    public byte[] reverseBytes(final byte[] bytearray)
    {
        int i = 0;
        int j = bytearray.length - 1;
        byte tmpbyte;

        //Loop through until i is no longer less than j.
        for(i = 0; i < j; i++)
        {
            //swap the bytes in the i and j position.
            tmpbyte = bytearray[i];
            bytearray[i] = bytearray[j];
            bytearray[j] = tmpbyte;

            //decrement j.
            j--;
        }

        return bytearray;
    }

    /**
     * Read in the appropriate byte array, swapping endians if necessary.
     * 
     * @param size Nubmer of bytes to read
     * @return The byte array, swapped if needed.
     * @throws IOException if DataInputStream read throws an exception using this in constructor.
     */
    public byte[] readByteArraySwap(final int size) throws IOException
    {
        //I'm not sure how to actually get the binary data from the file itself. 
        //However, I do know how to use DataInputStream.  So, I'll define a 
        //DataInputStream as a copy of this.
        final DataInputStream binfile = new DataInputStream(this);

        //Now, read in the four bytes making up the byte array.
        byte[] bytearray = new byte[size];
        binfile.read(bytearray, 0, size);

        //And call the reverse bytes routine if necessary.
        if(_reversebytesflag)
        {
            //Add some stuff here to swap the endians.
            bytearray = reverseBytes(bytearray);
        }

        return bytearray;
    }

    public byte[] readByteArraySwapNew(final int size) throws IOException
    {
        //I'm not sure how to actually get the binary data from the file itself. 
        //However, I do know how to use DataInputStream.  So, I'll define a 
        //DataInputStream as a copy of this.

        //DataInputStream binfile = new DataInputStream(this);

        //Now, read in the four bytes making up the byte array.
        byte[] bytearray = new byte[size];

        //binfile.read(bytearray, 0, size);
        this.in.read(bytearray, 0, size);

        //And call the reverse bytes routine if necessary.
        if(_reversebytesflag)
        {
            //Add some stuff here to swap the endians.
            if(size == 4) //4-byte - int and float
            {
                final int intBits = ByteBuffer.wrap(bytearray).getInt();
                final int reversedIntBytes = Integer.reverseBytes(intBits);
                bytearray = ByteBuffer.allocate(size).putInt(reversedIntBytes).array();
            }
            else
            //8-byte - long and double
            {
                final long longBits = ByteBuffer.wrap(bytearray).getLong();
                final long reversedLongBytes = Long.reverseBytes(longBits);
                bytearray = ByteBuffer.allocate(size).putLong(reversedLongBytes).array();
            }

            // bytearray = reverseBytes(bytearray);
        }

        return bytearray;
    }

    ///////////////////////////////////////////////////////////////////////////
    //Sets and Gets
    ///////////////////////////////////////////////////////////////////////////

    /**
     * Endian flag should be true if the file to read in requires swapping (i.e. is in little endian) or false
     * otherwise. Static vars LITTLE_ENDIAN_DATA_FLAG and BIG_ENDIAN_DATA_FLAG can be used. Wrapper on
     * setReverseBytesFlag.
     * 
     * @param flag The new flag value.
     */
    public void setEndianFlag(final boolean flag)
    {
        setReverseBytesFlag(flag);
    }

    /**
     * Sets the reverse bytes flag.
     * 
     * @param flag The new flag value.
     */
    public void setReverseBytesFlag(final boolean flag)
    {
        _reversebytesflag = flag;
    }

    public boolean getEndianFlag()
    {
        return getReverseBytesFlag();
    }

    public boolean getReverseBytesFlag()
    {
        return _reversebytesflag;
    }

}
