/*
 * Created on Sep 22, 2005 To change the template for this generated file go to Window&gt;Preferences&gt;Java&gt;Code
 * Generation&gt;Code and Comments
 */
package ohd.hseb.util.misc;

import java.util.HashMap;

import ohd.hseb.db.Database;

/**
 * This class is designed to manage Database connections by associating them with a Thread. This means that there can be
 * one Database connection per Thread.
 * 
 * @author hank
 */
public class ArchiveDatabaseController
{
    final static String CLASSNAME = "ArchiveDatabaseController";

    private final HashMap _identifierToConnectionMap;
    private final HashMap _identifierToURLMap;
    private String _initialURL;

    public ArchiveDatabaseController()
    {
        _identifierToConnectionMap = new HashMap();
        _identifierToURLMap = new HashMap();
    }

    /**
     * @param identifier A good choice would be Thread.currentThread().getName().
     */
    public void createDatabaseConnection(final String identifier)
    {
        createDatabaseConnection(identifier, _initialURL);
    }

    /**
     * Creates a new Database connection with a given id and using the url.
     * 
     * @param id Identifier uniquely identifying this connection.
     * @param url Url specifying the database.
     */
    @SuppressWarnings("unchecked")
    public void createDatabaseConnection(final String identifier, final String url)
    {
        Messenger.writeMsg(Messenger.EVT_DATA + Messenger.TIER_2 + Messenger.SEV_DEBUG,
                           "Creating database connection for thread " + identifier + " with url " + url + ".\n");

        //Make sure Thread is Unique.
        if(_identifierToConnectionMap.get(identifier) != null)
        {
            return;
        }

        final Database db = new Database();
        db.connect(url);
        _identifierToConnectionMap.put(identifier, db);
        _identifierToURLMap.put(identifier, url);
        if(_initialURL == null)
        {
            _initialURL = url;
        }
    }

    @SuppressWarnings("unchecked")
    public void reconnect(final String identifier)
    {
        if(_identifierToConnectionMap.get(identifier) != null)
        {
            return;
        }
        final String url = (String)_identifierToURLMap.get(identifier);

        final Database db = new Database();
        db.connect(url);
        _identifierToConnectionMap.put(identifier, db);
    }

    /**
     * Removes the connection at the given index, but keeps a memory of the url for reconnect purposes.
     * 
     * @param index
     */
    public void removeDatabaseConnection(final String identifier)
    {
        final Database db = (Database)_identifierToConnectionMap.get(identifier);
        if(db != null)
        {
            _identifierToConnectionMap.remove(identifier);
        }
        Messenger.writeMsg(Messenger.EVT_DATA + Messenger.TIER_2 + Messenger.SEV_DEBUG,
                           "Disconnected and removed database connection for " + identifier + ".\n");
    }

    /**
     * Removes the connection at the given index and its URL, making it impossible to reconnect.
     * 
     * @param index
     */
    public void removeDatabaseConnectionFull(final String identifier)
    {
        final Database db = (Database)_identifierToConnectionMap.get(identifier);
        if(db != null)
        {
            _identifierToConnectionMap.remove(identifier);
            _identifierToURLMap.remove(identifier);
        }
        Messenger.writeMsg(Messenger.EVT_DATA + Messenger.TIER_2 + Messenger.SEV_DEBUG,
                           "Disconnected and removed database connection and url for " + identifier + ".\n");
    }

    /**
     * Removes all connections.
     */
    public void removeAllConnections()
    {
        _identifierToConnectionMap.clear();
    }

    /**
     * Retrieves the connection with the given id. If the connection does not exist, it returns
     * 
     * @param id
     * @return Database object. Null if none found with the id.
     */
    public Database getConnection(final String identifier)
    {
        Messenger.writeMsg(Messenger.EVT_DATA + Messenger.TIER_3 + Messenger.SEV_DEBUG, "Getting connection for "
            + identifier + ".\n");
        if(_identifierToConnectionMap.get(identifier) == null)
        {
            createDatabaseConnection(identifier);
        }
        return (Database)_identifierToConnectionMap.get(identifier);
    }

    /**
     * Retrieves the connection with the given id, creating a new one if it does not already exist.
     * 
     * @param id
     * @param url
     * @return Database object with connection; null should never be returned.
     */
    public Database getConnection(final String identifier, final String url)
    {
        Messenger.writeMsg(Messenger.EVT_DATA + Messenger.TIER_3 + Messenger.SEV_DEBUG, "Getting connection for "
            + identifier + ".\n");
        if(_identifierToConnectionMap.get(identifier) == null)
        {
            createDatabaseConnection(identifier, url);
        }
        return (Database)_identifierToConnectionMap.get(identifier);
    }

    /**
     * Retrieves the url used for the connection with the given id.
     * 
     * @param id
     * @return Database object. Null if none found with the id.
     */
    public String getUrl(final String identifier)
    {
        return (String)_identifierToURLMap.get(identifier);
    }
}
