package ohd.hseb.util.misc;

import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.TimeZone;

import ohd.hseb.hefs.utils.datetime.DateTools;
import ohd.hseb.hefs.utils.tools.StringTools;
import ohd.hseb.util.data.DataSet;
import ohd.hseb.util.data.DatacardData;

/**
 * A class that contains static functions for use with Calendar and Date objects. Primarily, it is used to convert
 * Calendar objects to and from julian hours. Also, it can be used to translate String objects of specific format to and
 * from Calendar objects. It is important to realize that all of these methods assume GMT by default, and that julian
 * hours, herein, is assumed to be computed relative to hour 0 or January 1, 1900.<br>
 * <br>
 * Lastly, the computeCalendarRelativeToPresent method takes a String of the form,<br>
 * <br>
 * <+ or -> [number units] [number units] ...<br>
 * <br>
 * where the spaces are purely optional. The number MUST be an integer and the units must be one of the Strings in the
 * VALID_UNITS constant array below. If the passed in adjustment factor String is valid, the returned Calendar will be
 * set to the current time adjusted appropriately by the adjustment String. If it is not valid, the null will be
 * returned.<br>
 * <br>
 * <br>
 * 
 * @author hank To change the template for this generated type comment go to Window&gt;Preferences&gt;Java&gt;Code
 *         Generation&gt;Code and Comments
 */
public abstract class HCalendar
{
    final static String CLASSNAME = "HCalendar";

    //Character codes for the date format string.  See the default strings, below, for examples.
    public final static String CENTURY = "CC";
    public final static String YEAR = "YY";
    public final static String MONTH = "MM";
    public final static String DAY = "DD";
    public final static String HOUR = "hh";
    public final static String MINUTE = "mm";
    public final static String SECOND = "ss";
    public final static String TIMEZONE = "TZC";
    public final static String MONTH_AB = "MON";

    //The constant adjustment for intl time (i.e. what we add to GMT to get intl.)
    public final static int INTL_TZ_ADJ = -12;

    //Default date formats.
    public final static String DEFAULT_DATE_FORMAT = "CCYY-MM-DD hh:mm:ss";
    public final static String DEFAULT_DATETZ_FORMAT = "CCYY-MM-DD hh:mm:ss TZC";
    public final static String DEFAULT_ESPADP_FORMAT = "MMDDCCYY:hh";
    public final static String DEFAULT_SHORTTERM_FORMAT = "MMDDYYHH";
    public final static String DEFAULT_DATEONLY_FORMAT = "CCYY-MM-DD";
    public final static String DEFAULT_DHMTZ_FORMAT = "YYMMDDhh TZC";

    //Y2K year fix.
    public final static int Y2K_UB_WINDOW = 10; //The upperbound on the Y2K window -- i.e. if two digit year is 
    //larger than this, we ASSUME its 1900.  Otherwise, use 2000.

    public final static long MILLIS_IN_DAY = 24 * 60 * 60 * 1000;
    public final static long MILLIS_IN_HR = 60 * 60 * 1000;
    public final static long MILLIS_IN_MIN = 60 * 1000;
    public final static long SECONDS_IN_HR = 3600;

    //Adjustment factor strings and indices.
    public final static String[] VALID_UNITS = {"weeks", "week", "wk", "days", "day", "dy", "hours", "hour", "hr",
        "minutes", "minute", "seconds", "second"};
    public final static long[] UNITS_TO_MILLIS = {7 * 24 * MILLIS_IN_HR, 7 * 24 * MILLIS_IN_HR, 7 * 24 * MILLIS_IN_HR,
        24 * MILLIS_IN_HR, 24 * MILLIS_IN_HR, 24 * MILLIS_IN_HR, 1 * MILLIS_IN_HR, 1 * MILLIS_IN_HR, 1 * MILLIS_IN_HR,
        60 * MILLIS_IN_MIN, 60 * MILLIS_IN_MIN, 1000, 1000};
    public final static String[] BASIC_UNITS = {"weeks", "days", "hours"}; //Stripped down list above. Each

    //corresponds to a VALID_UNIT entry.

    public static String[] MONTH_ABBREVIATIONS = {"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
        "Nov", "Dec"};

    /**
     * Compute a Calendar object based on the julian hours (since 1/1/1900 GMT, Midnight) The calendar returned is in
     * GMT, and it **assumes** the jhour was computed in GMT!
     * 
     * @param jhour
     * @return
     */
    public static Calendar computeCalendarFromJulianHour(final int jhour)
    {
        //Get a calendar instance, using the GMT time zone.
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        final Calendar date = Calendar.getInstance(tz);

        //Set the calendar instance to Jan 1, 1900, 0 hours.
        date.set(1900, 0, 1, 0, 0, 0);
        date.set(Calendar.MILLISECOND, 0);

        //Add the hours
        date.add(Calendar.HOUR, jhour);

        return date;
    }

    /**
     * Compute a Calendar object based on the julian days (since 1/1/1900 GMT, Midnight) The calendar returned is in
     * GMT, and it assigns an hour of 0.
     * 
     * @param jday
     * @return
     */
    public static Calendar computeCalendarFromJulianDay(final int jday)
    {
        //Get a calendar instance, using the GMT time zone.
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        final Calendar date = Calendar.getInstance(tz);

        //Set the calendar instance to Jan 1, 1900, 0 hours.
        date.set(1900, 0, 1, 0, 0, 0);
        date.set(Calendar.MILLISECOND, 0);

        //Add the hours after subtracting 1 to make sure that jday 1 is 1/1/1900.
        date.add(Calendar.HOUR, (jday - 1) * 24);

        return date;
    }

    /**
     * Compute a Calendar from the passed in date object. It assumes the date is in GMT.
     * 
     * @param d The {@link Date} to convert.
     * @return A {@link Calendar} converted from the provided {@link Date} in GMT.
     */
    public static Calendar computeCalendarFromDate(final Date d)
    {
        //Get a calendar instance, using the GMT time zone.
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        return computeCalendarFromDate(d, tz);
    }

    /**
     * @param d The {@link Date} to convert.
     * @param tz The time zone of the desired {@link Calendar}.
     * @return A {@link Calendar} converted from the provided {@link Date} in the specified {@link TimeZone}.
     */
    public static Calendar computeCalendarFromDate(final Date d, final TimeZone tz)
    {
        final Calendar date = Calendar.getInstance(tz);
        date.setTime(d);
        return date;
    }

    /**
     * Calls {@link #computeCalendarFromDate(Date)}.
     * 
     * @param millis Java milliseconds to convert.
     * @return A {@link Calendar} in milliseconds with a GMT time zone.
     */
    public static Calendar computeCalendarFromMilliseconds(final long millis)
    {
        return HCalendar.computeCalendarFromDate(new Date(millis));
    }

    /**
     * Calls {@link #computeCalendarFromDate(Date, TimeZone)}.
     * 
     * @param millis Java milliseconds to convert.
     * @param tz The {@link TimeZone} of the desired {@link Calendar}.
     * @return {@link Calendar} corresponding to the provided milliseconds in the specified {@link TimeZone}.
     */
    public static Calendar computeCalendarFromMilliseconds(final long millis, final TimeZone tz)
    {
        return HCalendar.computeCalendarFromDate(new Date(millis), tz);
    }

    /**
     * @param factor Adjustment factor for relative dating.
     * @return {@link Calendar} corresponding to the present day adjusted by the provided factor in GMT time zone.
     */
    public static Calendar computeCalendarRelativeToPresent(final String factor)
    {
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        final Calendar date = Calendar.getInstance(tz);
        return computeCalendarRelativeToPresent(date, factor);
    }

    /**
     * This routine can be used to convert a String of known format into a date, based on the system date. The passed in
     * String is simply an adjustment factor string, used to adjust relative to the current time:<br>
     * <br>
     * <+ or -> quantity unit quantity unit ...<br>
     * <br>
     * where <+ or -> is either '+' or '-', and:<br>
     * <br>
     * - quantity is a positive number,<br>
     * - unit is either weeks, days, or hrs (not case sensitive)<br>
     * <br>
     * Every quantity and unit may be separated by spaces, tabs, or newlines.<br>
     * <br>
     * The returned Calendar is the current system time adjusted by the adjustment factor given in the String. A return
     * of null implies and invalid factor string.<br>
     * 
     * @param factor
     * @return
     */
    public static Calendar computeCalendarRelativeToPresent(final Calendar presentTime, String factor)
    {
        //Grab the system time.
        final Calendar date = (Calendar)presentTime.clone();

        //First, trim the factor to remove any leading spaces.  If the String
        //has 0-length, then just return the system date.
        factor = factor.trim();
        if(factor.length() == 0)
            return date;

        //Grab the first character, which must be a + or -.  If it is not, return
        //a null Calendar.  Remove the char from factor and trim it again to remove 
        //any spaces from the beginning.
        final char sign = factor.charAt(0);
        int mult = 0;
        if(sign == '+')
            mult = 1;
        else if(sign == '-')
            mult = -1;
        else
            return null;
        factor = factor.substring(1, factor.length()).trim();

        //Now, I need to do a while loop until the factor string has 0-length.
        int index;
        String num, unit;
        while(factor.length() > 0)
        {
            //Find the first non-numeric character in the remaining portion of the
            //factor and pickup the number that precedes it.  If there is no number,
            //then return null (error).  Remove the number from the string.
            index = HString.findFirstNonNumericCharacter(factor);
            if(index < 0)
                return null;
            num = factor.substring(0, index).trim();
            factor = factor.substring(index, factor.length()).trim();

            //Pickup the first numeric character in the remaining string and 
            //do as above.  The string that is picked up is the <unit>.
            index = HString.findFirstNumericCharacter(factor);
            if(index < 0)
                index = factor.length();
            unit = factor.substring(0, index).trim();
            factor = factor.substring(index, factor.length()).trim();

            //Add the number * units to the picked up date.
            if(HCalendar.computeAdjustFactor(num, unit) == DataSet.MISSING)
                return null; //an error!!!
            date.add(Calendar.MILLISECOND, mult * (int)HCalendar.computeAdjustFactor(num, unit));
        }

        //Return the Calendar object.
        return date;
    }

    /**
     * Returns the time interval implied by the passed in string, after converting it to hours. It uses the
     * computeAdjustFactor static method in this class.
     * 
     * @param str String specifying the units.
     * @return The number of hours.
     */
    public static long computeIntervalValueInMillis(final String str)
    {
        //Call computeAdjustFactor, depending on if there is 
        //any non-numeric character in the string.
        long temp;
        final int pos = HString.findFirstNonNumericCharacter(str);
        if((pos < 0) || (pos >= str.length()))
        {
            temp = HCalendar.computeAdjustFactor(str, "");
        }
        else
        {
            temp = HCalendar.computeAdjustFactor(str.substring(0, pos), str.substring(pos, str.length()));
        }

        return temp;
    }

    /**
     * This routine can be used to calculate a number of hours given by an adjustment factor specified by the passed in
     * num and unit. The num and unit are:<br>
     * <br>
     * num is a positive number,<br>
     * unit>is a VALID_UNIT above.<br>
     * <br>
     * If unit is either null or "", then it is assumed to be "hours". The returned Integer is the number of hours
     * corresponding to the adjustment factor.<br>
     * <br>
     * 
     * @param num
     * @param unit
     * @return
     */
    public static long computeAdjustFactor(final String num, final String unit)
    {
        int number;
        long unitinhrs = 1;
        int i;

        //Convert the number to an integer.  If it fails, return null.
        try
        {
            number = Integer.parseInt(num.trim());
        }
        catch(final NumberFormatException nfe)
        {
            return (long)DataSet.MISSING;
        }

        //If the unit is null or empty, then return number.  Default
        //unit is hours.
        if((unit == null) || (unit.length() == 0))
        {
            return number;
        }

        //Check the <unit> for either "weeks", "days", or "hours".  If it is
        //none of those, return null.  Otherwise, record either 7*24, 24, or
        //1 as the base unit (these number are the number of hours to add for
        //that unit).
        for(i = 0; i < HCalendar.VALID_UNITS.length; i++)
        {
            if(unit.trim().equalsIgnoreCase(HCalendar.VALID_UNITS[i]))
            {
                unitinhrs = HCalendar.UNITS_TO_MILLIS[i];
                break;
            }
        }
        if(i == HCalendar.VALID_UNITS.length)
            return (long)DataSet.MISSING;

        return number * unitinhrs;
    }

    /**
     * Given a Calendar, comput the julian hour. If gmt is true, then we are to ASSUME the calendar is already in GMT,
     * regardless of its true TimeZone. This allows callers to play around with a Calendar without actually paying any
     * attention to its TimeZone, but just let it be GMT.
     * 
     * @param date The date
     * @param gmt Whether to assume GMT or use time zone from date.
     * @return Julian hours from 1/1/1900 00:00:00 GMT.
     */
    public static int computeJulianHourFromCalendar(final Calendar date, final boolean gmt)
    {
        final Calendar working = (Calendar)date.clone();

        //Get the offset from GMT in milliseconds, if necessary.
        //This is the number of milliseconds to add to date
        //to convert it **FROM** GMT to this time zone -- usually <0 in the USA.
        long gmtoffset;
        if(!gmt)
        {
            final TimeZone tz = working.getTimeZone();

            //The raw offset appears to account for daylight savings, so I removed the commented
            //out code below.
            gmtoffset = tz.getRawOffset();
        }
        else
        {
            gmtoffset = 0;
        }

        //Add the gmtoffset to the date -- the -1 is there because the offset is to convert GMT to this
        //timezone, but we need to do it the other way around.
        working.add(Calendar.MILLISECOND, (int)(-1 * gmtoffset));

        //Get the month, day, year, and hour.
        final int year = working.get(Calendar.YEAR);
        final int dayofyear = working.get(Calendar.DAY_OF_YEAR); //do I need +1 ?
        final int hour = working.get(Calendar.HOUR_OF_DAY);

        //Compute the julian hour -- from code written by David Street
        return HCalendar.computeJulianHour(year, dayofyear, hour);
    }

    /**
     * Compute the julian hour based on the year (1900...), dayofyear (1..366), and hourofday (0..23) This is all in GMT
     * (Z-time).
     * 
     * @param year
     * @param dayofyear
     * @param hourofday
     * @return
     */
    public static int computeJulianHour(final int year, final int dayofyear, final int hourofday)
    {
        int yearsfrom1900, daysfrom1900, yjj; //I don't know what yjj is doing

        daysfrom1900 = 0;
        if(year >= 1900)
        {
            yearsfrom1900 = year - 1900;
            if(yearsfrom1900 > 0)
            {
                daysfrom1900 = 365 * yearsfrom1900 + (yearsfrom1900 - 1) / 4;
                if(yearsfrom1900 > 200)
                {
                    yjj = yearsfrom1900 - 101;
                    daysfrom1900 = daysfrom1900 - yjj / 100 + yjj / 400;
                }
            }
            daysfrom1900 += dayofyear - 1; //-1 because day of year starts at 1, not 0.
        }

        return 24 * daysfrom1900 + hourofday;
    }

    /**
     * Compute the julian hour based on the year (1900...), dayofyear (1..366), and hourofday (0..23) This is all in GMT
     * (Z-time).
     * 
     * @param year
     * @param dayofyear
     * @param hourofday
     * @return
     */
    public static int computeJulianDay(final int year, final int dayofyear)
    {
        int currentYear = 1900;
        int daysFrom1900 = 0;

        while(currentYear < year)
        {
            daysFrom1900 += 365;
            if(HCalendar.isLeapYear(currentYear))
            {
                daysFrom1900 += 1;
            }
            currentYear++;
        }
        daysFrom1900 += dayofyear;
        return daysFrom1900;
    }

    /**
     * Compute the julian hour based on the year, month (0..11), day (1..31), and hour of day (0..23). <br>
     * NOTE: MONTH STARTS AT 0!!! Assumes GMT!<br>
     * 
     * @param year
     * @param month
     * @param dayofmonth
     * @param hourofday
     * @return
     */
    public static int computeJulianHour(final int year, final int month, final int dayofmonth, final int hourofday)
    {
        int dayofyear;

        //I'm going to compute the day of the year, and pass that into the computeJulianHour
        //that uses the day of the year.

        //Month is stored as 0..11, so this should work. 
        dayofyear = month * 30;

        //Account for months that are not 30 days...
        if(month >= 1) //if the month is Feb or later, add 1 for day 31 of Jan
            dayofyear++;
        if(month >= 2) //if the month is Mar or later...
        {
            //If its a leap year, then subtract 1 from dayofyear.
            if(isLeapYear(year))
                dayofyear--;

            //Otherwise, subtract 2
            else
                dayofyear -= 2;
        }
        if(month >= 3) //if the month is Apr or later, add 1 for day 31 of Mar
            dayofyear++;
        if(month >= 5) //if the month is June or later, add 1 for day 31 of May
            dayofyear++;
        if(month >= 7) //if the month is Aug or later, add 1 for day 31 of July
            dayofyear++;
        if(month >= 8) //if the month is Sept or later, add 1 for day 31 of Aug
            dayofyear++;
        if(month >= 10) //if the month is Nov or later, add 1 for day 31 of Oct
            dayofyear++;

        dayofyear += dayofmonth;

        return computeJulianHour(year, dayofyear, hourofday);
    }

    /**
     * Converts a format for this class to a {@link SimpleDateFormat} format.
     * 
     * @param format Format to convert.
     * @return Format valid with {@link SimpleDateFormat}.
     */
    public static String convertToSimpleDateFormat(final String format)
    {
        final StringBuffer formatBuffer = new StringBuffer(format);
        StringTools.replace(formatBuffer, CENTURY, "yy");
        StringTools.replace(formatBuffer, YEAR, "yy");
        StringTools.replace(formatBuffer, DAY, "dd");
        StringTools.replace(formatBuffer, HOUR, "HH");
        StringTools.replace(formatBuffer, TIMEZONE, "zzz");
        StringTools.replace(formatBuffer, MONTH_AB, "MMM");
        return formatBuffer.toString();
    }

    /**
     * Calls {@link #computeCalendarFromMilliseconds(long)} to get a {@link Calendar}, and then converts it via
     * {@link #buildDateStr(Calendar, String)} using the provided format.
     * 
     * @param milliseconds Milliseconds for which to build date string.
     * @param format Format of the string to use; see constants in this class. MUST BE A VALID FORMAT FOR THIS CLASS! Do
     *            not use SimpleDateFormat formats.
     * @return Date string with the specified format assuming GMT time zone.
     */
    public static String buildDateStr(final long milliseconds, final String format)
    {
        return DateTools.getThreadSafeSimpleDateFormat(convertToSimpleDateFormat(format), DateTools.GMT_TIME_ZONE)
                        .format(milliseconds);
    }

    /**
     * Calls {@link #buildDateStr(long, String)} with format {@link #DEFAULT_DATEONLY_FORMAT}.
     */
    public static String buildDateStr(final long millis)
    {
        return buildDateStr(millis, DEFAULT_DATEONLY_FORMAT);
    }

    /**
     * Calls {@link #buildDateStr(long, String)} with format {@link #DEFAULT_DATE_FORMAT}.
     */
    public static String buildDateTimeStr(final long millis)
    {
        return buildDateStr(millis, DEFAULT_DATE_FORMAT);
    }

    /**
     * Calls {@link #buildDateStr(long, String)} with format {@link #DEFAULT_DATETZ_FORMAT}.
     */
    public static String buildDateTimeTZStr(final long millis)
    {
        return buildDateStr(millis, DEFAULT_DATETZ_FORMAT);
    }

    /**
     * @param millis Java milliseconds to convert.
     * @param format The {@link HCalendar} compliant format.
     * @param tz The {@link TimeZone} to apply.
     * @return A {@link String} converted from a {@link Calendar} created via
     *         {@link #computeCalendarFromMilliseconds(long, TimeZone)}.
     */
    public static String buildDateStr(final long millis, final String format, final TimeZone tz)
    {
        return buildDateStr(computeCalendarFromMilliseconds(millis, tz), format);
    }

    /**
     * @param milliseconds
     * @return The number of hours based on the milliseconds. The result is truncated (i.e., cast) into an integer.
     */
    public static int convertMillisecondsToHours(final long milliseconds)
    {
        return (int)(milliseconds / MILLIS_IN_HR);
    }

    /**
     * Convert a {@link Calendar} to a {@link String} based on the format string. For valid strings within format, see
     * {@link #convertStringToCalendar(String, String)} below. Valid components are stored as constants.
     * 
     * @param date Date to build a string from.
     * @param format Format of the string. MUST BE A VALID FORMAT FOR THIS CLASS! Do not use SimpleDateFormat formats.
     * @return Properly formatted string.
     */
    public static String buildDateStr(final Calendar date, final String format)
    {
        return DateTools.getThreadSafeSimpleDateFormat(convertToSimpleDateFormat(format), DateTools.GMT_TIME_ZONE)
                        .format(date.getTime());
    }

    /**
     * Calls {@link #buildDateStr(Calendar, String)} with format {@link #DEFAULT_DATEONLY_FORMAT}.
     */
    public static String buildDateStr(final Calendar cal)
    {
        return buildDateStr(cal, DEFAULT_DATEONLY_FORMAT);
    }

    /**
     * Calls {@link #buildDateStr(Calendar, String)} with format {@link #DEFAULT_DATE_FORMAT}.
     */
    public static String buildDateTimeStr(final Calendar cal)
    {
        return buildDateStr(cal, DEFAULT_DATE_FORMAT);
    }

    /**
     * Calls {@link #buildDateStr(Calendar, String)} with format {@link #DEFAULT_DATETZ_FORMAT}.
     */
    public static String buildDateTimeTZStr(final Calendar date)
    {
        return buildDateStr(date, HCalendar.DEFAULT_DATETZ_FORMAT);
    }

    /**
     * Convert a string to a calendar based on a format. The format is a sequence of characters describing the format of
     * the date string. For example: CCYY-MM-DD is a valid string, saying the first 4 digits is the year, 6,7 is the
     * month and 9,10 is the day. Valid character codes: CC, YY, MM, DD, hh, mm, ss, TZC.<br>
     * Default time zone is GMT!!! Unless the string passed in overrides it.<br>
     * Returns null if one of the numbers in the date string is not valid.
     * 
     * @param str
     * @param format
     * @return
     */
    public static Calendar convertStringToCalendar(final String str, final String format)
    {
        int century = 0, year = 0, month = 0, day = 0, hour = 0, minute = 0, second = 0;
        TimeZone tz = TimeZone.getTimeZone("GMT");

        //Get the current system calendar date.
        final Calendar date = Calendar.getInstance();

        //One big try based on string to int converting:
        try
        {
            //Look for each of the components.
            century = parseDateComponent(str, format, CENTURY, (date.get(Calendar.YEAR) / 100));
            year = parseDateComponent(str, format, YEAR, (date.get(Calendar.YEAR) % 100));
            month = parseDateComponent(str, format, MONTH, date.get(Calendar.MONTH) + 1);
            day = parseDateComponent(str, format, DAY, date.get(Calendar.DAY_OF_MONTH));
            hour = parseDateComponent(str, format, HOUR, date.get(Calendar.HOUR_OF_DAY));
            minute = parseDateComponent(str, format, MINUTE, date.get(Calendar.MINUTE));
            second = parseDateComponent(str, format, SECOND, date.get(Calendar.SECOND));

            //This handles the time zone ID -- the trim will remove any blanks from around the
            //time zone, so that a two character time zone ID can be accepted.
            tz = TimeZone.getTimeZone(HString.getKeyFromString(str, format, TIMEZONE).trim());
        }
        //Catch an number format errors
        catch(final NumberFormatException e)
        {
            return null;
        }

        //NOTE: Calendar has built in mechanisms to handle months, days, hours, etc being out of range.
        //  Namely, it loops the current date appropriately... i.e. month 1 day 33 becomes month 2 day 2.

        //Change the date to match the date specified by string and return it.
        date.setTimeZone(tz);
        date.set(Calendar.YEAR, century * 100 + year);
        date.set(Calendar.MONTH, month - 1);
        date.set(Calendar.DAY_OF_MONTH, day);
        date.set(Calendar.HOUR_OF_DAY, hour);
        date.set(Calendar.MINUTE, minute);
        date.set(Calendar.SECOND, second);
        date.set(Calendar.MILLISECOND, 0);

        //DEBUG PRINT:
        //System.out.println(">>>>DATE: " + date.get(Calendar.YEAR) + "-" + (date.get(Calendar.MONTH) + 1) + "-" +
        //date.get(Calendar.DAY_OF_MONTH) + " " + date.get(Calendar.HOUR_OF_DAY) + ":" + date.get(Calendar.MINUTE) +
        //":" + date.get(Calendar.SECOND));

        return date;
    }

    /**
     * A wrapper function on HString.getKeyFromString which uses it for extracting a date component. If the component
     * specified in the key is not present in the format, then this function will return defaultreturn. Otherwise, it
     * return the characters from str in the same position as the key is in format, parsed into a integer.<br>
     * THIS FUNCTION MAY RESULT IN A NumberFormatException!!!<br>
     * 
     * @param str
     * @param format
     * @param key
     * @param defaultreturn
     * @return
     * @throws NumberFormatException
     */
    public static int parseDateComponent(final String str,
                                         final String format,
                                         final String key,
                                         final int defaultreturn) throws NumberFormatException
    {
        String temp;

        temp = HString.getKeyFromString(str, format, key);

        if(temp.length() == 0)
        {
            return defaultreturn;
        }

        return Integer.parseInt(temp);
    }

    /**
     * Compute the number of days in a month, given the year and the month (0..11).
     * 
     * @param year
     * @param month
     * @return
     */
    public static int computeDaysInMonth(final int year, final int month)
    {
        final int tmon = month + 1;
        if((tmon == 9) || (tmon == 4) || (tmon == 6) || (tmon == 11))
            return 30;
        if(tmon == 2)
        {
            if(isLeapYear(year))
                return 29;
            else
                return 28;
        }
        return 31;
    }

    /**
     * Is the passed in year a leap year?
     * 
     * @param year
     * @return
     */
    public static boolean isLeapYear(final int year)
    {
        //If the year is divisible by 400, then it is a leap year
        if((year % 400) == 0)
        {
            return true;
        }

        //If the year is divisible by 4, but NOT by 100, then it is a leap year
        if((((year - 1900) % 4) == 0) && (((year - 1900) % 100) != 0))
        {
            return true;
        }

        //Its not a leap year...
        return false;
    }

    /**
     * @param lower Lower bound of date range
     * @param upper Upper bound of date range
     * @return True if a leap day is within the bounds, including the end points.
     */
    public static boolean isLeapDayWithinInterval(final Calendar lower, final Calendar upper)
    {
        int year = lower.get(Calendar.YEAR);
        if(isLeapYear(year))
        {
            if((lower.get(Calendar.MONTH) == 1) && (lower.get(Calendar.DAY_OF_MONTH) == 29))
            {
                return true;
            }
            final Calendar test = Calendar.getInstance(lower.getTimeZone());
            test.set(year, 1, 29, 0, 0, 0);
            if(lower.before(test) && (upper.after(test)))
            {
                return true;
            }
            return false;
        }

        year = upper.get(Calendar.YEAR);
        if(isLeapYear(year))
        {
            //At this point, I know that lower is not a leap year.  Hence, it must be before the leap year and 
            //checking is done relative to 00:00:00 on 2/29.
            final Calendar test = Calendar.getInstance(upper.getTimeZone());
            test.set(year, 1, 29, 0, 0, 0);
            if(lower.before(test) && (upper.after(test)))
            {
                return true;
            }
            return false;
        }
        return false;
    }

    /**
     * Return the four digit year corresponding to the data card 2 digit year. This uses the windowing technique
     * employed by NWSRFS (a window around current year, specified by Y2K_UB_WINDOW)! Move this to {@link DatacardData}.
     * 
     * @param year2d
     * @param todayYear
     * @return Full year using the Y2K_UB_WINDOW solution.
     */
    public static int processTwoDigitYear(final int year2d, final int todayYear)
    {
        //Compare 2000 + year2d with _todayyear + Y2K_UB_WINDOW.  If it is larger than it, then 
        //I assume 1900.  Otherwise, assume 2000.
        if((2000 + year2d) > (todayYear + Y2K_UB_WINDOW))
            return 1900 + year2d;

        return 2000 + year2d;
    }

    /**
     * This returns a Calendar representing the first of the month for the passed in calendar.
     * 
     * @param cal
     * @return
     */
    public static Calendar computeFirstOfMonth(final Calendar cal)
    {
        cal.set(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH), 1, 0, 0, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal;
    }

    /**
     * @return {@link Calendar} specifying that for the first millisecond for the given year.
     */
    public static Calendar computeFirstOfYear(final int year)
    {
        final Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        cal.set(year, 1, 1, 0, 0, 0);
        cal.set(Calendar.MILLISECOND, 0);
        return cal;
    }

    /**
     * @return {@link Calendar} specifying that for the last millisecond for the given year.
     */
    public static Calendar computeLastOfYear(final int year)
    {
        final Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        cal.set(year, 12, 31, 23, 59, 58);
        cal.set(Calendar.MILLISECOND, 999);
        return cal;
    }

    /**
     * This returns a Calendar representing the last of the month for the passed in calendar.
     * 
     * @param cal
     * @return
     */
    public static Calendar computeLastOfMonth(final Calendar cal)
    {
        cal.set(cal.get(Calendar.YEAR),
                cal.get(Calendar.MONTH),
                HCalendar.computeDaysInMonth(cal.get(Calendar.YEAR), cal.get(Calendar.MONTH)),
                23,
                59,
                59);
        cal.set(Calendar.MILLISECOND, 999);
        return cal;
    }

    /**
     * Calls {@link #processDate(Calendar, String, boolean)} setting the {@link Calendar} based on the provided date
     * string.
     * 
     * @param datestr Specifies the date.
     * @param endTimeFlag true if this an end-time, meaning that, if the hh:mm:ss are not set, then they will be set to
     *            23:59:59. Otherwise, they are set to 00:00:00.
     * @return {@link Calendar} containing the date.
     */
    public static Calendar processDate(final String datestr, final boolean endTimeFlag)
    {
        final TimeZone tz = TimeZone.getTimeZone("GMT");
        final Calendar date = Calendar.getInstance(tz);
        return processDate(date, datestr, endTimeFlag);
    }

    /**
     * This processes a specified date string and returns an appropriate Calendar. If null is returned, then the datestr
     * could not be converted into a date. This method will call convertStringToCalendar using one of these formats:
     * DEFAULT_DATETZ_FORMAT, DEFAULT_ESPADP_FORMAT, DEFAULT_DATEONLY_FORMAT. It makes the choice based on the trimmed
     * string length. Furthermore, if the first character is '*', it will assume the date is relative and will compute a
     * Calendar relative to the current system time accordingly.
     * 
     * @param datestr
     * @param endTimeFlag true if this an end-time, meaning that, if the hh:mm:ss are not set, then they will be set to
     *            23:59:59. Otherwise, they are set to 00:00:00.
     * @return {@link Calendar} containing the date.
     */
    public static Calendar processDate(final Calendar presentTime, String datestr, final boolean endTimeFlag)
    {
        //First, trim the passed in datestr.
        datestr.trim();

        //Next, is the first char a *?
        if(datestr.charAt(0) == '*')
        {
            //The date is calculated relative to the current time in GMT.
            if(datestr.length() > 1)
                datestr = datestr.substring(1, datestr.length());
            else
                                     //If this else is triggered, then datestr was passed in as "*".
                                     datestr = "";
            return computeCalendarRelativeToPresent(presentTime, datestr);
        }

        //Otherwise, I assume it is an absolute date.  The date may be in several
        //formats.  The format is assume is based on the length of the string.
        if(datestr.length() == DEFAULT_DATETZ_FORMAT.length())
        {
            return convertStringToCalendar(datestr, DEFAULT_DATETZ_FORMAT);
        }
        if(datestr.length() == DEFAULT_ESPADP_FORMAT.length())
        {
            return convertStringToCalendar(datestr, DEFAULT_ESPADP_FORMAT);
        }
        if(datestr.length() == DEFAULT_DATEONLY_FORMAT.length())
        {
            final Calendar result = convertStringToCalendar(datestr, DEFAULT_DATEONLY_FORMAT);
            if(result != null)
            {
                if(endTimeFlag)
                {
                    result.set(Calendar.HOUR_OF_DAY, 23);
                    result.set(Calendar.MINUTE, 59);
                    result.set(Calendar.SECOND, 59);
                }
                else
                {
                    result.set(Calendar.HOUR_OF_DAY, 0);
                    result.set(Calendar.MINUTE, 0);
                    result.set(Calendar.SECOND, 0);
                }
            }
            return result;
        }

        //This is the default format I assume.  
        return convertStringToCalendar(datestr, DEFAULT_DATE_FORMAT);
    }

    /**
     * This processes a specified date string and returns an appropriate Calendar. If null is returned, then the datestr
     * could not be converted into a date.
     * 
     * @param datestr The String to process. It is processed by calling the other processDate method, assuming the date
     *            is NOT an end time (meaning the time components are specified as 0 if not in the string).
     * @return Calendar containing the date.
     */
    public static Calendar processDate(final String datestr)
    {
        return processDate(datestr, false);
    }

    /**
     * The start of the given year in milliseconds.
     * 
     * @param year the given year
     * @return millisecond time of the start of the year
     */
    public static long yearInMillis(final int year)
    {
        final Calendar date = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        date.set(year, 0, 1, 0, 0, 0);
        return date.getTimeInMillis();
    }

    /**
     * @param monIndex Jan = 0, Dec = 11.
     * @return Three letter abbreviation for a month.
     */
    public static String getMonthAbbreviation(final int monIndex)
    {
        return MONTH_ABBREVIATIONS[monIndex];
    }

    /**
     * @param monthAbbreviation Jan, Feb, etc.
     * @return Index of the month; 0 = Jan, 11 = Dec.
     */
    public static int getMonthIndex(final String monthAbbreviation)
    {
        return Arrays.asList(MONTH_ABBREVIATIONS).indexOf(monthAbbreviation);
    }

    //The main...
    public static void main(final String args[])
    {
        final Calendar cal = Calendar.getInstance();
        System.out.println("####>> " + HCalendar.buildDateStr(cal));

        //Performance test
        final HStopWatch timer = new HStopWatch();
        for(long millis = 0; millis < 1000000; millis++)
        {
            cal.setTimeInMillis(millis);
            final String str = HCalendar.buildDateTimeTZStr(cal);
            if((millis == 0L) || (millis == 1000L) || (millis == 100000L))
            {
                System.out.println("####>> date str for " + millis + ": " + str);
            }
        }
        System.out.println("####>> TIME -- " + timer.getElapsedMillis());

        //Simple date format .. much faster!
        final SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss z");
        timer.reset();
        for(long millis = 0; millis < 1000000; millis++)
        {
            cal.setTimeInMillis(millis);
            final String str = sdf.format(cal.getTime());
            if((millis == 0L) || (millis == 1000L) || (millis == 100000L))
            {
                System.out.println("####>> date str for " + millis + ": " + str);
            }
        }
        System.out.println("####>> TIME -- " + timer.getElapsedMillis());

//        //  ==== THIS IS FOR RETURNING THE DATE CORRESPONDING TO A JULIAN DAY/HOUR ====
//        if(args.length != 2)
//        {
//            System.out.println("");
//            System.out.println("Incorrect usage.  The usage is: ");
//            System.out.println("    jrun HCalendar <h, d, r> <julian number>");
//            System.out.println("where h or d tells it to take the number as a julian <h>our or <d>ay.");
//            System.out.println("and r says that the String following specifies a date relative to today");
//            System.out.println("that you wish to have translated.");
//            System.out.println("");
//            return;
//        }
//    
//        int jhr;
//        Calendar date;
//        if(args[0].equals("h"))
//        {
//            System.out.println("  <user specified a julian hour>  ");
//            jhr = Integer.parseInt(args[1]);
//            date = HCalendar.computeCalendarFromJulianHour(jhr);
//        }
//        else if(args[0].equals("d"))
//        {
//            System.out.println("  <user specified a julian day>  ");
//            jhr = 24 * (Integer.parseInt(args[1]) - 1);
//            date = HCalendar.computeCalendarFromJulianHour(jhr);
//        }
//        else if(args[0].equals("m"))
//        {
//            System.out.println("  <user specified millis>  ");
//            date = HCalendar.computeCalendarFromMilliseconds(Long.parseLong(args[1]));
//        }
//        else
//        {
//            System.out.println("  <user specified a relative date using \"" + args[1] + "\">  ");
//            date = HCalendar.computeCalendarRelativeToPresent(args[1]);
//            if(date == null)
//            {
//                System.out.println("The relative string is invalid!!!");
//                return;
//            }
//        }
//    
//        System.out.println("THE DATE IS:");
//        System.out.println("" + HCalendar.buildDateStr(date, HCalendar.DEFAULT_DATETZ_FORMAT));
//        System.out.println("");
    }
}
