package ohd.hseb.util.misc;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.text.CollationKey;
import java.text.Collator;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import ohd.hseb.util.data.DataSet;

/**
 * This class provides static methods for String related functionality. The method getKeyFromString(...) reads in a
 * String of a specified format and extracts the value of the key String from within it. This used by HCalendar to
 * acquire the date fields from a specifically formatted date string. The method replaceSubstring(...) puts key values
 * into a String of specified format. The methods readBinarString(...) and writeBinaryString(...) can be used to
 * read/write Strings from Fortran written files where the Strings are disguised as floating point numbers (for example,
 * see ESPData).
 * 
 * @author hank
 */
public abstract class HString
{

    public final String RESERVED_HTMLTAGGED = new String("@HtMl");

    /**
     * Given a string format, a string following that format, and a key, return the characters from string which are in
     * the same positions that key is in within the format string. Example: <br>
     * <br>
     * string: 1999-01-10 -- format: ccyy-mm-dd -- key: yy<br>
     * <br>
     * This would result in a return value of "99".<br>
     * <br>
     * 
     * @param str
     * @param format
     * @param key
     * @return String corresponding to key in str.
     */
    public static String getKeyFromString(final String str, final String format, final String key)
    {
        int index;

        //Get the index of the characters of key within format.
        index = format.indexOf(key);

        //If index is invalid, return a string of length 0.
        if((index < 0) || (index >= format.length()))
        {
            return "";
        }

        //If the first index is after the end of the passed in string, return
        //nothing.
        if(index >= str.length())
        {
            return "";
        }

        //Get the substring starting at index with the length of key.  If that puts it past
        //the end of the string, then just stop at the end.
        final String substr = str.substring(index, Math.min(index + key.length(), str.length()));

        //Return it.
        return substr;
    }

    /**
     * Finds key within str, removes it, and inserts value in its place. If the key isn't found, it returns the entire
     * string, with nothing replaced.
     * 
     * @param str
     * @param key
     * @param value
     * @return new String
     */
    public static String replaceSubstring(final String str, final String key, final String value)
    {

        //Get the index.
        final int index = str.indexOf(key);
        if((index < 0) || (index > str.length()))
        {
            return str;
        }

        //Copy the pieces of str around the key to a new string.
        final String working = str.substring(0, index) + str.substring(index + key.length(), str.length());

        //Insert the value into the spot that used to contain the key.
        final StringBuffer buff = new StringBuffer(working);
        buff.insert(index, value);

        //return the buffer
        return buff.toString();
    }

    /**
     * Return the index of the first non-numeric character in the passed in string. MISSING is returned if all
     * characters are numeric.
     * 
     * @param word
     * @return index of first non-numeric character
     */
    public static int findFirstNonNumericCharacter(final String word)
    {
        return findFirstNonNumericCharacter(word, false);
    }

    /**
     * Return the index of the first non-numeric character in the passed in string. MISSING is returned if all
     * characters are numeric.
     * 
     * @param word
     * @param includeDecimalPoint True if decimal points ('.') are considered numeric characters.
     * @return index of first non-numeric character
     */
    public static int findFirstNonNumericCharacter(final String word, final boolean includeDecimalPoint)
    {
        int i = 0;
        for(i = 0; i < word.length(); i++)
        {
            //If it is not true that the character is a digit or a decimal point (if included), then...
            if(!((Character.isDigit(word.charAt(i))) || ((word.charAt(i) == '.') && includeDecimalPoint)))
                break;
        }

        if(i == word.length())
            return (int)DataSet.MISSING;
        return i;
    }

    /**
     * Return the index of the first non-numeric character in the passed in string.
     * 
     * @param word
     * @return index of first numeric character.
     */
    public static int findFirstNumericCharacter(final String word)
    {
        int i = 0;
        for(i = 0; i < word.length(); i++)
        {
            if(Character.isDigit(word.charAt(i)))
                break;
        }

        if(i == word.length())
            return (int)DataSet.MISSING;
        return i;
    }

    /**
     * Alphabetize the passed in Vector of Strings and return the ordered Vector. I choose to use collation keys,
     * because Java in a Nutshell says this may be a more efficient way to compare strings.
     * 
     * @param strings
     * @return Ordered vector.
     */
    @SuppressWarnings("unchecked")
    public static Vector alphabetize(final Vector strings)
    {
        //The size must be larger than 1 for this to have any meaning.
        if(strings.size() <= 1)
            return strings;

        //The string collator
        final Collator coll = Collator.getInstance();

        //Some variables.
        int i, j;
        String str;
        CollationKey onekey;

        //Get the collation keys and copy the existing vector at the same time.
        final CollationKey[] collkeys = new CollationKey[strings.size()];
        final Vector sorted = new Vector();
        for(i = 0; i < strings.size(); i++)
        {
            collkeys[i] = coll.getCollationKey((String)(strings.elementAt(i)));
            sorted.addElement(strings.elementAt(i));
        }

        //I now have an array of collation keys that can be used to compare and sort.

        //I'm going to do a very inefficient sort, which should usually work fine for small vectors,
        //that will find the smallest key and then add the corresponding key to the vector.
        for(j = 0; j < (sorted.size() - 1); j++)
        {
            for(i = j + 1; i < sorted.size(); i++)
            {
                //The one in position i is less than that in position j, so swap them.
                if(collkeys[i].compareTo(collkeys[j]) < 0)
                {
                    //sway the strings.
                    str = (String)(sorted.elementAt(i));
                    sorted.setElementAt(sorted.elementAt(j), i);
                    sorted.setElementAt(str, j);

                    //Swap the collation keys.
                    onekey = collkeys[i];
                    collkeys[i] = collkeys[j];
                    collkeys[j] = onekey;
                }
            }
        }

        return sorted;
    }

    /**
     * Format the string to be displayed in a field of the width specified.
     * 
     * @param width The width of the field.
     * @param word The word to be displayed in the field.
     * @param leftjustified If the word should be left (true) or right (false) justified in the field.
     * @param truncateIfTooLong If true, then if the work is longer than the field, the left most characters are removed
     *            (higher order digits... may not be a good idea) to force it to fit.
     * @return The word displayed in the appropriate field.
     */
    public static String formatStringToFieldWidth(final int width,
                                                  final String word,
                                                  final boolean leftjustified,
                                                  final boolean truncateIfTooLong)
    {
        int i;
        String field = "";

        //Add the word here if its left justified.
        if(leftjustified)
            field += word;

        //Put in the filler spaces.    
        for(i = 0; i < width - word.length(); i++)
            field += " ";

        if(!leftjustified)
            field += word;

        //If the length of field is larger than width, then I truncate the higher digits
        if(field.length() > width)
            return field.substring(field.length() - width, field.length());

        return field;
    }

    /**
     * Format a string to be of a specific field width by filling in spaces either after if leftjustified is true, or
     * before otherwise. If the word is already larger than width, then this will remove the larger digits.
     * 
     * @param width
     * @param word
     * @param leftjustified
     * @return Formatted String.
     */
    public static String formatStringToFieldWidth(final int width, final String word, final boolean leftjustified)
    {
        return formatStringToFieldWidth(width, word, leftjustified, true);
    }

    /**
     * @param width Field width
     * @param number The int to print
     * @param fillInWithZeros True to precede the int with 0s, or false to leave empty.
     * @return Left-justified number string, possibly filled with zeros before the number.
     */
    public static String formatIntegerToFieldWidth(final int width, final int number, final boolean fillInWithZeros)
    {
        String field = "" + number;
        char fillerCharacter = ' ';
        if(fillInWithZeros)
        {
            fillerCharacter = '0';
        }
        while(field.length() < width)
        {
            field = fillerCharacter + field;
        }
        return field;
    }

    /**
     * This method will word wrap a String, adding \n to set the String so that it does not exceed the desired field
     * width. If a line MUST exceed the maxLineLength (because a word is too long), then it will be forced onto its own
     * line.
     * 
     * @param word The word to wrap
     * @param maxLineLength The largest allowable length for any line.
     * @return
     */
    public static String wordWrap(final String word, final int maxLineLength)
    {
        String newWord = "";
        int currentIndex = 0;
        int lastIndex = 0;

        while(currentIndex <= word.length())
        {
            //Go maxLineLength many characters up in the String.
            final int nextIndex = currentIndex + maxLineLength;
            final int newNewLineIndex = word.indexOf('\n', currentIndex + 1);
            if(newNewLineIndex < 0)
            {
                currentIndex = nextIndex;
            }
            else
            {
                currentIndex = Math.min(nextIndex, newNewLineIndex);
            }

            //If we are off the end, then add the end of the string to 
            //newWord and break out.
            if(currentIndex >= word.length())
            {
                newWord += word.substring(lastIndex, Math.min(word.length(), currentIndex));
                break;
            }

            //Back up from currentIndex until I find a character that is neither a
            //number nor a letter.  The first such character marks the line break
            //point.
            while(currentIndex >= lastIndex)
            {
                if((!Character.isDigit(word.charAt(currentIndex))) && (!Character.isLetter(word.charAt(currentIndex))))
                {
                    break;
                }
                currentIndex--;
            }
            //If we have no non-digit and letter character while backing up, then...
            if(currentIndex == lastIndex - 1)
            {
                //Move forward until we find a letter that is not a digit and not
                //a letter.  Don't go past the end of the word.
                while((currentIndex < word.length()) && (Character.isDigit(word.charAt(currentIndex)))
                    && (Character.isLetter(word.charAt(currentIndex))))
                {
                    currentIndex++;
                }
            }

            //If currentIndex equals word.length(), then we need to add the end
            //of word to the new word.
            if(currentIndex == word.length())
            {
                newWord += word.substring(lastIndex, word.length());
                break;
            }

            //currentIndex points to the place where copying ends.
            newWord += word.substring(lastIndex, currentIndex + 1).trim() + "\n";
            lastIndex = currentIndex + 1;
        }
        return newWord;
    }

    /**
     * Read in a string from the DataInputStream in binary format.
     * 
     * @param datafile
     * @param numbytes
     * @return String read from file.
     * @throws IOException
     */
    public static String readBinaryString(final DataInputStream datafile, final int numbytes) throws IOException
    {
        final byte[] b = new byte[numbytes + 1];

        datafile.read(b, 0, numbytes);

        int stringlen = 0;
        for(stringlen = 0; stringlen < numbytes; stringlen++)
        {
            if(b[stringlen] == 0)
                break;
        }

        return new String(b, 0, stringlen);
    }

    /**
     * Write out a string to the DataOutputStream in binary format.
     * 
     * @param datafile
     * @param word
     * @param numbytes
     * @throws IOException
     */
    public static void writeBinaryString(final DataOutputStream datafile, final String word, final int numbytes) throws IOException
    {
        if(word == null)
            return;

        final byte[] b = word.getBytes();
        datafile.write(b, 0, b.length);

        int i;
        for(i = b.length; i < numbytes; i++)
        {
            datafile.writeByte(0);
        }
    }

    /**
     * Add the element to the list if necessary. Return false if it was not necessary to add the string. Return true
     * otherwise. "Necessary" means it is not already in the list.
     * 
     * @param v
     * @param word
     * @return False if the string is already in the Vector.
     */
    @SuppressWarnings("unchecked")
    public static boolean addStringToVector(final Vector v, final String word)
    {
        if(word == null)
            return false;

        //See if the element is already in the list.
        int i;
        for(i = 0; i < v.size(); i++)
        {
            //If I find the word, then return false, meaning I did not need to add the word.
            if(word.equals(v.elementAt(i)))
            {
                return false;
            }
        }
        v.addElement(word);
        return true;
    }

    /**
     * Sorts an array of strings in place -- this does an inefficient linear sort.
     * 
     * @param thestrings
     * @param increasing
     * @return Sorted array.
     */
    public static String[] sortArrayOfStrings(final String[] thestrings, final boolean increasing)
    {
        final Collator collator = Collator.getInstance();

        int i;
        boolean changed;
        String tempstr;

        changed = true;
        while(changed)
        {
            //Initialize changed to false.
            changed = false;

            for(i = 1; i < thestrings.length - 1; i++)
            {

                //The first line swaps them if we are sorting into increasing order
                //The second line swaps them if we are sorting into decreasing order
                if(((increasing) && (collator.compare(thestrings[i], thestrings[i - 1]) < 0))
                    || ((!increasing) && (collator.compare(thestrings[i], thestrings[i - 1]) > 0)))
                {
                    tempstr = thestrings[i - 1];
                    thestrings[i - 1] = thestrings[i];
                    thestrings[i] = tempstr;
                    changed = true;
                }
            }
        }
        return thestrings;
    }

    /**
     * Inverts an array of strings and returns the inverted array.
     * 
     * @param thestrings
     * @return Inverted array.
     */
    public static String[] invertArrayOfStrings(final String[] thestrings)
    {
        int i, j;
        String tempstr;

        i = 0;
        j = thestrings.length - 1;
        while((i < thestrings.length) && (j >= 0) && (i < j))
        {
            tempstr = thestrings[j];
            thestrings[j] = thestrings[i];
            thestrings[i] = tempstr;
            i++;
            j--;
        }

        return thestrings;
    }

    /**
     * Remove the quotes around a string, but only if both are present.
     * 
     * @param quotedstring
     * @return String with removed quotes.
     */
    public static String removeBeginAndEndQuotes(final String quotedstring)
    {
        //Trim it first...
        quotedstring.trim();

        //If I have nothing left in the string after trimming, return the passed
        //in string.
        if(quotedstring.length() == 0)
            return quotedstring;

        //Remove the first and last ", if both are present.
        if((quotedstring.charAt(0) == '\"') && (quotedstring.charAt(quotedstring.length() - 1) == '\"'))
        {
            return quotedstring.substring(1, quotedstring.length() - 1);
        }
        return quotedstring;
    }

    /**
     * Remove trailing characters from the passed in string.
     * 
     * @param string The string to modify.
     * @param ch The trailing character to remove.
     * @return The new string.
     */
    public static String removeTrailingCharacter(final String string, final char ch)
    {
        int firstNonChAtEnd;
        if(string == null)
        {
            return null;
        }
        for(firstNonChAtEnd = string.length() - 1; firstNonChAtEnd >= 0; firstNonChAtEnd--)
        {
            if(string.charAt(firstNonChAtEnd) != ch)
            {
                break;
            }
        }
        return string.substring(0, firstNonChAtEnd + 1);
    }

    /**
     * Return true if the string is in quotes.
     * 
     * @param word
     * @return boolean
     */
    public static boolean isQuoted(final String word)
    {
        //Trim the word.
        if(word == null)
        {
            return false;
        }
        final String newword = word.trim();
        if(newword.length() == 0)
        {
            return false;
        }

        //Is the first character a quote?  If no, return false.
        if(word.charAt(0) != '\"')
            return false;

        //What about the last character.
        if(word.charAt(word.length() - 1) != '\"')
            return false;

        //Both first and last character are in quotes.
        return true;
    }

    /**
     * Compute a user-set delimited String from a List of Strings.
     * 
     * @param words A List of Strings
     * @param delimiter The String to place between each array element in the returned String.
     */
    public static String buildStringFromList(final List words, final String delimiter)
    {
        final StringBuffer result = new StringBuffer();
        int i;

        for(i = 0; i < words.size(); i++)
        {
            result.append(words.get(i).toString());
            if(i < words.size() - 1)
            {
                result.append(delimiter);
            }
        }

        return result.toString();
    }

    /**
     * Compute a user-set delimited String from a List of Doubles.
     * 
     * @param words A List of Doubles
     * @param delimiter The String to place between each array element in the returned String.
     */
    public static String buildStringFromListOfDoubles(final List<Double> words, final String delimiter)
    {
        if(words == null)
        {
            return "";
        }
        String result = "";
        int i;

        for(i = 0; i < words.size(); i++)
        {
            result += "" + ((Double)(words.get(i))).doubleValue();
            if(i < words.size() - 1)
            {
                result += delimiter;
            }
        }

        return result;
    }

    /**
     * Compute a user-set delimited String from an Array of Strings.
     * 
     * @param words An array of Strings
     * @param delimiter The String to place between each array element in the returned String.
     */
    public static String buildStringFromArray(final String[] words, final String delimiter)
    {
        if(words == null)
        {
            return null;
        }
        final List<String> arrayList = new ArrayList<String>();
        int i;
        for(i = 0; i < words.length; i++)
        {
            arrayList.add(words[i]);
        }

        return buildStringFromList(arrayList, delimiter);
    }

    /**
     * Compute a user-set delimited String from an Array of doubles.
     * 
     * @param words An array of doubles
     * @param delimiter The String to place between each array element in the returned String.
     */

    public static String buildStringFromArray(final double[] words, final String delimiter)
    {
        final List<String> arrayList = new ArrayList<String>();
        int i;
        for(i = 0; i < words.length; i++)
        {
            arrayList.add("" + words[i]);
        }

        return buildStringFromList(arrayList, delimiter);
    }

    public static List<String> buildListFromString(final String listStr, final String delimiter)
    {
        final SegmentedLine segLine = new SegmentedLine(listStr, delimiter, SegmentedLine.MODE_NO_EMPTY_SEGS);
        return segLine.getSegments();
    }

}
