package ohd.hseb.util.misc;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;

/**
 * The Messenger class implements the APEX messenger API. The commands initMsg(...) and writeMsg(...) are replicas of
 * their C++ counterpart located in util - util_api and documented heavily in the header files of util -
 * inc/messenger_inc.<br>
 * <br>
 * Used to generate messages for printing to screen, file, or someplace else. <br>
 * <br>
 * To initialize a global messenger, call the initMsg static method. To write a message to the global messenger, call
 * the writeMsg static method.
 * 
 * @author hank
 */
public class Messenger
{
    final static String CLASSNAME = "Messenger";
    //These are the constants as given directly in the msgwrappers.h file.

    //Modulus amounts used to acquire the values of the parameters.
    public final static int EVT_MOD_LEVEL = 128; // 2^7
    public final static int EVT_MOD_SEV = 512; // 2^9
    public final static int EVT_MOD_PS = 2048; // 2^11
    public final static int EVT_MOD_DQ = 8192; // 2^13
    public final static int EVT_MOD_EVT = 2097152; // 2^21

    //Default parameter specifier.
    public final static int USE_DEFAULT_PARAM = -999;

    /*
     * *Key values of the trace level. If a message is printed with a trace level*of ALWAYS_PRINT, the message will
     * always be printed, unless the user specifically*overrides the required priority and sets it to be 0. The trace
     * level varies from*0 through 127.*BITS 0-7
     */
    public final static int ALWAYS_PRINT = 1;
    public final static int TIER_1 = 20;
    public final static int TIER_2 = 40;
    public final static int TIER_3 = 60;
    public final static int TIER_4 = 80;
    public final static int TIER_5 = 100;
    public final static int TIER_MAX = 127;

    /*
     * *Available Severity Levels*BITS 8, 9
     */
    public final static int MSG_NUMBER_SEV = 4;
    public final static int SEV_DEBUG = 0; /* 0 */
    public final static int SEV_INFO = 128; /* 1 */
    public final static int SEV_WARNING = 256; /* 2 */
    public final static int SEV_ERROR = 384; /* 3 */
    public final static String[] MSG_DEF_SEV = {"DEBUG  ", "INFO   ", "WARNING", "ERROR  "};

    /*
     * *Available Problem Statuses*BITS 10, 11
     */
    public final static int MSG_NUMBER_PS = 4;
    public final static int PS_UNKNOWN = 0; /* 0 */
    public final static int PS_NONE = 512; /* 1 */
    public final static int PS_CORRECT = 1024; /* 2 */
    public final static int PS_NON_CORRECT = 1536; /* 3 */

    /*
     * *Available Data Quality Levels*BITS 12, 13
     */
    public final static int MSG_NUMBER_DQ = 4;
    public final static int DQ_UNKNOWN = 0; /* 0 */
    public final static int DQ_GOOD = 2048; /* 1 */
    public final static int DQ_QUESTIONABLE = 4096; /* 2 */
    public final static int DQ_BAD = 6144; /* 3 */

    /*
     * *Default events settings.*BITS 14-21
     */
    public final static int MSG_NUMBER_EVENTS = 11; /* The default number of events */
    public final static int EVT_UNKNOWN = 0; /* 0 */
    public final static int EVT_LOGGING = 8192; /* 1 */
    public final static int EVT_START_RTN = 16384; /* 2 */
    public final static int EVT_FINISH_RTN = 24576; /* 3 */
    public final static int EVT_PROGRESS = 32768; /* 4 */
    public final static int EVT_PARAMETERS = 40960; /* 5 */
    public final static int EVT_SOURCE_IO = 49152; /* 6 */
    public final static int EVT_DATA = 57344; /* 7 */
    public final static int EVT_MISC = 65536; /* 8 */
    public final static int EVT_REGRESSION = 73728; /* 9 */
    public final static int EVT_GUI = 81920; /* 10 */
    public final static String[] MSG_DEF_EVENTS = {"", "LOGGING   ", "START RTN ", "FINISH RTN", "PROGRESS  ",
        "PARAMETERS", "SOURCE I/O", "DATA      ", "MISC      ", "REGRESSION", "GUI       "};
    public final static int[] MSG_DEF_LEVELS = {1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1};

    //The global static messenger object.
    public static Messenger GLOBAL_MSG = null;

    /////////////////////////////////////////////////////////////////////////////
    //Attributes
    /////////////////////////////////////////////////////////////////////////////

    //Attributes to store output parameters.
    int _numevents;
    int[] _reqdlevels;
    String _filename;
    String[] _eventnames;

    //Output file to where data is to go.  
    PrintStream _outputfile;

    /////////////////////////////////////////////////////////////////////////////
    //Static methods that can be called just as the C/C++ methods are called,
    //but with the "Messenger." prefix.
    /////////////////////////////////////////////////////////////////////////////

    /**
     * Initiliazes the GLOBAL_MSG object for messaging.<br>
     * <br>
     * Define your own events and pass them into the constructor. Also takes filename and the levels required to result
     * in a print for each event (0 means no printing, ALWAYS_PRINT means only ALWAYS_PRINT messages are printed, TIER_1
     * means ALWAYS_PRINT and TIER_1 messages are printed, etc...). <br>
     * 
     * @param numevents
     * @param eventnames
     * @param filename
     * @param reqdlevels
     */
    public static void initMsg(int numevents, String[] eventnames, String filename, int[] reqdlevels)
    {
        //Check events for default.
        if(numevents < 0)
        {
            numevents = Messenger.MSG_NUMBER_EVENTS;
            eventnames = Messenger.MSG_DEF_EVENTS;
        }

        //Theoretically, this should never happen unless numevents is USE_DEFUALT_PARAM.
        //But, just in case, here it is.
        if(eventnames == null)
        {
            eventnames = Messenger.MSG_DEF_EVENTS;
        }

        //Check filename for default.  Use default if it is NULL or 0-length.
        if(filename == null)
        {
            filename = "stdout";
        }
        else if(filename.length() == 0)
        {
            filename = "stdout";
        }

        //Check reqdlevels for default.
        if(reqdlevels == null)
        {
            reqdlevels = Messenger.MSG_DEF_LEVELS;
        }
        //Otherwise, loop through the passed in levels.  If an individual element
        //is -999, then try to pull the default level from MSG_DEF_LEVELS.
        else
        {
            int i;
            for(i = 0; i < numevents; i++)
            {
                //If the passed in level is -999, then...
                if(reqdlevels[i] < 0)
                {
                    //Use the MSG_DEF_LEVELS for that element if we are not off of
                    //the end of the array.
                    if(i < Messenger.MSG_NUMBER_EVENTS)
                    {
                        reqdlevels[i] = Messenger.MSG_DEF_LEVELS[i];
                    }
                }
            }
        }

        //At this point, I will diverge from the C++ code...
        //All of the input parameters are now valid.  Just set the parameters in here.
        Messenger.GLOBAL_MSG = new Messenger(numevents, eventnames, filename, reqdlevels);
    }

    /**
     * The bitcode has the following components:<br>
     * <br>
     * 1-7 Level (ALWAYS_PRINT, TIER_1, TIER_2,...)<br>
     * 8,9 Severity (SEV_WARNING, ...)<br>
     * 10,11 Problem status (PS_NONE, PS_UNKNOWN,...)<br>
     * 11,12 data quality (DQ_GOOD, ...)<br>
     * 13-21 event (EVT_DATA, ...)<br>
     * <br>
     * Pass in the bitcode as a sum of static constants in this class and then specify the message. A \n should be
     * included if a carriage return is to be included when it is printed to a file or screen.<br>
     * <br>
     * If Messenger.GLOBAL_MSG is null, this will output nothing!
     * 
     * @param bitcode Sum of constants that define the error bit code.
     * @param message
     */
    public synchronized static void writeMsg(int bitcode, String message)
    {
        //Acquire the parameters of the writeMessage call.
        int level = bitcode % Messenger.EVT_MOD_LEVEL;
        bitcode -= level;
        int sev = ((bitcode % Messenger.EVT_MOD_SEV) / Messenger.EVT_MOD_LEVEL);
        bitcode -= sev;
        int ps = ((bitcode % Messenger.EVT_MOD_PS) / Messenger.EVT_MOD_SEV);
        bitcode -= ps;
        int dq = ((bitcode % Messenger.EVT_MOD_DQ) / Messenger.EVT_MOD_PS);
        bitcode -= dq;
        int evt = ((bitcode % Messenger.EVT_MOD_EVT) / Messenger.EVT_MOD_DQ);

        //Send to the global messenger.
        if(Messenger.GLOBAL_MSG == null)
        {
            //System.out.println(message);
        }
        else
        {
            Messenger.GLOBAL_MSG.writeMessage(evt, sev, level, ps, dq, message);
        }
    }

    /**
     * Overrides the default required levels with new ones.
     * 
     * @param reqdlevels
     */
    public static void overrideRequiredLevels(int[] reqdlevels)
    {
        if(reqdlevels != null)
        {
            Messenger.GLOBAL_MSG.setReqdLevels(reqdlevels);
        }
    }

    /////////////////////////////////////////////////////////////////////////////
    //CONSTRUCTORS AND DESTRUCTORS
    /////////////////////////////////////////////////////////////////////////////
    public Messenger(int numevents, String[] eventnames, String filename, int[] reqdlevels)
    {
        //If numevents is invalid, then turn off the messenger by setting numevents
        //to MISSING.
        if(numevents <= 0)
        {
            _numevents = -1;
            return;
        }

        //Now, I want to allocate the space for the various storage devices and set pointers
        //to the two passed in array.
        _numevents = numevents;
        _reqdlevels = reqdlevels;
        _filename = filename;
        _eventnames = eventnames;

        //Call the file opening routine.
        openFile();
    }

    //The destructor... more or less.
    @Override
    protected void finalize()
    {
        //Close the open file... if it is open.
        if((_outputfile != System.out) && (_outputfile != System.err))
        {
            _outputfile.close();
        }
    }

    /////////////////////////////////////////////////////////////////////////////
    //TOOLS
    /////////////////////////////////////////////////////////////////////////////
    public void openFile()
    {
//        int i, j;

        //If the filename is NULL or 0-length, then set the _outputfile to System.out.
        if(_filename == null)
        {
            _outputfile = System.out;
            return;
        }
        else if(_filename.length() == 0)
        {
            _outputfile = System.out;
            return;
        }

        //If the file to open is "stdout", then just set the _outputfile to System.out.
        else if(_filename.equalsIgnoreCase("stdout"))
        {
            _outputfile = System.out;
            return;
        }

        //Same for stderr, except use System.err.
        else if(_filename.equalsIgnoreCase("stderr"))
        {
            _outputfile = System.err;
            return;
        }

        //Otherwise, open the file as a FileOutputStream and pass it to the PrintStream
        //constructor.
        try
        {
            //Create a File object for the file.
            File file = new File(_filename);

            //Check for writability.  If not, use System.out.
            if((file.exists()) && (!file.canWrite()))
            {
                _outputfile = System.out;
            }
            //Otherwise, try to open it in a FileOutputStream and construct _outputfile
            //from it.
            else
            {
                //To be consistent with what is done in C/C++, I want to open the file
                //for appending.  Hence, I need to pass in the file name (I can't use
                //the file object, since there is no append option constructor) and then
                //pass in true.
                FileOutputStream fileoutstr = new FileOutputStream(_filename, true);
                _outputfile = new PrintStream(fileoutstr);
            }
        }
        //If I have an IO exception, then just use System.out.
        catch(IOException ioe)
        {
            _outputfile = System.out;
        }
    }

    /**
     * This writes out a message. The format and structure exactly follow the C/C++ version given in Messenger.cxx (util
     * -- util_api).
     * 
     * @param msgevent Event -- EVT_DATA, for example
     * @param severity SEV_WARNING, SEV_ERROR, SEV_INFO, or SEV_DEBUG
     * @param level TIER_1, ALWAYS_PRINT, etc (0-127).
     * @param problemstatus PS_NONE, PS_CORRECT, PS_...
     * @param dataqual DQ_GOOD, DQ_...
     * @param message Message to print.
     */
    public void writeMessage(int msgevent, int severity, int level, int problemstatus, int dataqual, String message)
    {
        //Check for valid messenger settings.
        if((_numevents > 0) && //If numevents is usable
            ((msgevent >= 0) && (msgevent < _numevents)) && //and msgevent is valid
            ((severity >= 0) && (severity < Messenger.MSG_NUMBER_SEV)) && //and severity is valid
            (level <= _reqdlevels[msgevent])) //and it is severe enough...
        {
            //and then print to the file...
            _outputfile.print(_eventnames[msgevent] + " " + Messenger.MSG_DEF_SEV[severity] + " >> " + message);
        }
    }

    /////////////////////////////////////////////////////////////////////////////
    //SET AND GETS
    /////////////////////////////////////////////////////////////////////////////

    public void setReqdLevels(int[] reqdlevels)
    {
        if(reqdlevels != null)
        {
            _reqdlevels = reqdlevels;
        }
    }

    /////////////////////////////////////////////////////////////////////////////
    //A TEST MAIN
    /////////////////////////////////////////////////////////////////////////////
    public static void main(String args[])
    {
        //Use the defaults to create the global messenger, except use the file name passed
        //in!.
        int[] testlevels = {1, 1, 1, 1, 1, 1, Messenger.TIER_2, 1, 1, 1, 1};
        Messenger.initMsg(-1, null, args[0], testlevels);

        //Try to print some messages.
        Messenger.writeMsg(Messenger.EVT_SOURCE_IO + Messenger.TIER_2, " THIS IS A START!!!\n");
    }

}
