package ohd.hseb.util.misc;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Enumeration;
import java.util.List;
import java.util.Vector;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import ohd.hseb.util.AppsDefaults;
import ohd.hseb.util.data.DataSet;

/**
 * This is a collection of miscellaneous static methods. The most useful is getAppsDefaults(...) and the List - array
 * converters (listFromArray, vectorFromArray, stringArrayFromList).
 * 
 * @author hank
 */
public abstract class MiscTools
{
    public static String CLASSNAME = "MiscTools";

    //Y2K year fix.
    public final static int Y2K_UB_WINDOW = 10; //The upperbound on the Y2K window -- i.e. if two digit year is 

    //larger than this, we ASSUME its 1900.  Otherwise, use 2000.

    ////////////////////////////////////////////////////
    //OTHER FUNCTIONS
    ////////////////////////////////////////////////////

    /**
     * The java get_apps_defaults function. This is just a wrapper on the get_apps_defaults system function, so that
     * function MUST BE AVAILABLE via the current user path!!!<br>
     * <br>
     * Possible Return Values:<br>
     * null -- The command "get_apps_defaults" failed to execute or was not found.<br>
     * empty string -- No apps-defaults token by that name was found.<br>
     * nonempty string -- The token value.<br>
     * <br>
     * 
     * @param token
     */
    public static String getAppsDefaults(final String token)
    {
        if(token == null)
            return null;

        final AppsDefaults apps_defaults = new AppsDefaults();
        String value = apps_defaults.getToken(token);
        if(value == null)
        {
            value = "";
        }
        return value;
    }

    /**
     * Another version of getAppsDefaults that allows the caller to specify the path that get_apps_defaults is in.
     * 
     * @param commandpath
     * @param token
     * @return See getAppsDefaults(String token).
     */
    public static String getAppsDefaults(final String commandpath, final String token)
    {
        final Runtime commandprompt = Runtime.getRuntime();

        try
        {
            //Perform the command
            final Process therun = commandprompt.exec(commandpath + "/get_apps_defaults " + token);

            //Get the standard out for the command
            final InputStream stdout = therun.getInputStream();

            //Some variables...
            String tokenvalue = "";
            byte currentbyte;

            //Read in the first byte
            currentbyte = (byte)stdout.read();

            //Starting with the first byte, copy standard out one character at a time
            //and stop when I reach \n or currentbyte is negative (i.e. eof has been reached).
            while((currentbyte >= 0) && ((char)currentbyte != '\n'))
            {
                tokenvalue = tokenvalue + (char)currentbyte;
                currentbyte = (byte)stdout.read();
            }

            //Return the token value.   
            return tokenvalue;

        }
        catch(final IOException e)
        {
        }

        return null;
    }

    /**
     * Create the directory with the name passed in. It will do a recursive mkdir.
     */
    @SuppressWarnings("unchecked")
    public static void mkdirRecursive(final String dirName) throws IOException
    {
        final File directory = new File(dirName);
        final List<File> recursiveDirectoriesToBuild = new ArrayList<File>();
        File workingDir = directory;

        //Collect the portion of the directory tree that cannot be built.
        while((workingDir != null) && (!workingDir.exists()))
        {
            recursiveDirectoriesToBuild.add(workingDir);
            workingDir = workingDir.getParentFile();
        }

        int i;

        //Try to make each directory, in reverse List order.
        for(i = recursiveDirectoriesToBuild.size() - 1; i >= 0; i--)
        {
            if(!((File)recursiveDirectoriesToBuild.get(i)).mkdir())
            {
                throw new IOException("Unable to build directory "
                    + ((File)recursiveDirectoriesToBuild.get(i)).getAbsolutePath());
            }
        }
    }

    /**
     * Filter the contents of a directory, returning a list of "acceptable" files. The check for acceptability is done
     * by calling the isFileNameAcceptable function within the DirectoryFilterer passed in. Possible return values:<br>
     * <br>
     * 1. null -- implies it could not open the directory for reading.<br>
     * 2. 0-length vector -- implies no files were found that were acceptable or the dirname actually corresponds to a
     * file.<br>
     * 3. vector of positive length -- the list of files within the directory that are acceptable.<br>
     * <br>
     * The DirectoryFilterer passed in is merely a class that implements the DirectoryFilterer interface.<br>
     * <br>
     * 
     * @param dirname
     * @param filter
     * @return Vector of file names.
     * @throws IOException
     */
    @SuppressWarnings("unchecked")
    public static Vector filterDirectory(final String dirname, final DirectoryFilterer filter) throws IOException
    {
        //Open up the directory as afile.
        final File afile = new File(dirname);

        //If the directory does not exist...
        if(!afile.exists())
        {
            return null;
        }
        //If I can't read it...
        if(!afile.canRead())
        {
            return null;
        }

        //Get the list of files within the directory.
        final String[] files = afile.list();
        final Vector v = new Vector();

        //For each element in the array of files...
        int i;
        for(i = 0; i < files.length; i++)
        {
            //If filter passes the file, add it to the vector
            if(filter.isFileNameAcceptable(files[i]))
            {
                v.addElement(files[i]);
            }
        }

        //Return the vector of strings.
        return v;
    }

    /**
     * This method meshes the two vectors of strings into only one vector that contains unique elements of both.<br>
     * <br>
     * CASE IS IGNORED!!!<br>
     * <br>
     * 
     * @param v1
     * @param v2
     * @return new vector
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public static Vector meshVectorsOfStrings(final Vector v1, final Vector v2)
    {
        //Return null if both are null.
        if((v1 == null) && (v2 == null))
            return null;

        //If either is null but the other isn't, return it.
        if((v1 == null) && (v2 != null))
            return new Vector(v2);
        if((v1 != null) && (v2 == null))
            return new Vector(v1);

        //The final Vector.
        final Vector result = new Vector();

        //Start with v1...
        int i, j;
        String current;
        for(i = 0; i < v1.size(); i++)
        {
            //Get the working string.
            current = new String((String)(v1.elementAt(i)));

            //Search the result vector for a repeat.
            for(j = 0; j < result.size(); j++)
            {
                if(current.equalsIgnoreCase((String)(result.elementAt(j))))
                    break;
            }

            //If j is equal to result.size(), then current was not found, so
            //add it to the end.
            if(j == result.size())
                result.addElement(current);
        }

        //Go through v2...
        for(i = 0; i < v2.size(); i++)
        {
            //Get the working string.
            current = new String((String)(v2.elementAt(i)));

            //Search the result vector for a repeat.
            for(j = 0; j < result.size(); j++)
            {
                if(current.equalsIgnoreCase((String)(result.elementAt(j))))
                    break;
            }

            //If j is equal to result.size(), then current was not found, so
            //add it to the end.
            if(j == result.size())
                result.addElement(current);
        }

        return result;
    }

    /**
     * This static method returns a List that contains the objects in the passed in array.
     * 
     * @param data
     * @return ArrayList
     */
    public static ArrayList<Object> listFromArray(final Object[] data)
    {
        final ArrayList<Object> result = new ArrayList<Object>(data.length);
        int i;
        for(i = 0; i < data.length; i++)
        {
            result.add(data[i]);
        }
        return result;
    }

    /**
     * Search the passed in list, using the passed in comparator, for an object that matches the passed in object.
     * 
     * @param list The list to search.
     * @param object The object to find.
     * @param comparator The comparator to use; it looks for a return value of 0.
     * @return index or MISSING if not found.
     */
    @SuppressWarnings("unchecked")
    public static int findObjectInList(final List list, final Object object, final Comparator comparator)
    {
        int i;
        for(i = 0; i < list.size(); i++)
        {
            if(comparator.compare(object, list.get(i)) == 0)
            {
                return i;
            }
        }
        return (int)DataSet.MISSING;
    }

    /**
     * This static method returns a List that contains the objects in the passed in array.
     * 
     * @param data
     * @return Vector
     */
    @SuppressWarnings("unchecked")
    public static Vector vectorFromArray(final Object[] data)
    {
        final Vector result = new Vector(data.length);
        int i;
        for(i = 0; i < data.length; i++)
        {
            result.add(data[i]);
        }
        return result;
    }

    /**
     * The static method converts a List of Strings to a String array.
     * 
     * @param data
     * @return The array
     */
    public static String[] stringArrayFromList(final List data)
    {
        int i;
        final String[] results = new String[data.size()];
        for(i = 0; i < data.size(); i++)
            results[i] = (String)(data.get(i));
        return results;
    }

    /**
     * Sets all of the array values equal to that passed in.
     */
    public static double[][][] setAllArrayValues(final double[][][] array, final double value)
    {
        int i, j, k;
        for(i = 0; i < array.length; i++)
        {
            for(j = 0; j < array[i].length; j++)
            {

                for(k = 0; k < array[i][j].length; k++)
                {
                    array[i][j][k] = value;
                }
            }
        }
        return array;
    }

    /**
     * Performs a garbage collect, and prints out stats pertaining to it.
     */
    public static void performGarbageCollect()
    {
        final Runtime runtime = Runtime.getRuntime();
        Messenger.writeMsg(Messenger.EVT_MISC + Messenger.TIER_2 + Messenger.SEV_INFO,
                           "Garbage collecting destroyed objects (total memory available: " + runtime.totalMemory()
                               + ", free memory: " + runtime.freeMemory() + ", max memory: " + runtime.maxMemory()
                               + " [bytes]).\n");
        System.gc();
        Messenger.writeMsg(Messenger.EVT_MISC + Messenger.TIER_2 + Messenger.SEV_INFO,
                           "Garbage collecting completed (total memory available: " + runtime.totalMemory()
                               + ", free memory: " + runtime.freeMemory() + ", max memory: " + runtime.maxMemory()
                               + " [bytes]).\n");
    }

    /**
     * Generically call a wait for the number of milliseconds passed in.
     * 
     * @param wiatTime milliseconds
     */
    public static void waitForThisTime(final long milliseconds)
    {
        final Thread thread = new Thread();
        try
        {
            synchronized(thread)
            {
                thread.wait(milliseconds);
            }
        }
        catch(final InterruptedException e)
        {
        }
    }

    /**
     * @param pckgname The name of the package to search.
     * @return Array of Class objects representing all found in that pacakge.
     * @throws ClassNotFoundException If the package is invalid.
     */
    public static Class[] getClasses(final String pckgname) throws ClassNotFoundException
    {
        final List<Class> classes = getClassesForPackage(pckgname);
        final Class[] results = new Class[classes.size()];
        for(int i = 0; i < results.length; i++)
        {
            results[i] = classes.get(i);
        }
        return results;
    }

    /**
     * Via {@link ClassLoader#getResources(String)}, it acquires a list of resources that include that package. It then
     * scans the resources for jar files, looks in the jar file for .class files in that package, and returns those that
     * do not include '$' in their names ($ classes are embedded classes).<br>
     * <br>
     * This code was copied from somewhere on-line, but I can't recall where.
     * 
     * @param pckgname Name of package for which to search for classes.
     * @return List of classes found in that package. It will not return embedded or nested classes; only those with
     *         independent .class files.
     * @throws ClassNotFoundException
     */
    public static List<Class> getClassesForPackage(final String pckgname) throws ClassNotFoundException
    {

        final List<Class> classes = new ArrayList<Class>();
        final ArrayList<File> directories = new ArrayList<File>();
        try
        {
            final ClassLoader cld = Thread.currentThread().getContextClassLoader();
            if(cld == null)
            {
                throw new ClassNotFoundException("Can't get class loader.");
            }

            // Ask for all resources for the path
            final Enumeration<URL> resources = cld.getResources(pckgname.replace('.', '/'));
            while(resources.hasMoreElements())
            {
                final URL res = resources.nextElement();

                //If the resource is a jar file, then open it up and look inside.
                if(res.getProtocol().equalsIgnoreCase("jar"))
                {
                    final JarURLConnection conn = (JarURLConnection)res.openConnection();
                    final JarFile jar = conn.getJarFile();
                    for(final JarEntry e: Collections.list(jar.entries()))
                    {

                        if(e.getName().startsWith(pckgname.replace('.', '/')) && e.getName().endsWith(".class")
                            && !e.getName().contains("$"))
                        {
                            final String className = e.getName()
                                                      .replace("/", ".")
                                                      .substring(0, e.getName().length() - 6);
                            try
                            {
                                final Class clas = Class.forName(className);
                                classes.add(clas);
                            }
                            catch(final Throwable t)
                            {
                                //Ignore any Class that cannot be acquired via forName.  This likely means
                                //a dependent class cannot be loaded and it should not be used.  For example,
                                //if a JUnit test class is included, TestCase may not be in the class path,
                                //so do not use the test class.
                            }
                        }
                    }
                }
                //Otherwise, add the resource directly.
                else
                {
                    directories.add(new File(URLDecoder.decode(res.getPath(), "UTF-8")));
                }
            }
        }
        catch(final NullPointerException x)
        {
            throw new ClassNotFoundException(pckgname + " does not appear to be "
                + "a valid package (Null pointer exception)");
        }
        catch(final UnsupportedEncodingException encex)
        {
            throw new ClassNotFoundException(pckgname + " does not appear to be "
                + "a valid package (Unsupported encoding)");
        }
        catch(final IOException ioex)
        {
            throw new ClassNotFoundException("IOException was thrown when trying " + "to get all resources for "
                + pckgname);
        }

        //Now loop through the list of directories adding any found .class files.
        for(final File directory: directories)
        {
            if(directory.exists())
            {
                final String[] files = directory.list();
                for(final String file: files)
                {
                    if(file.endsWith(".class"))
                    {
                        classes.add(Class.forName(pckgname + '.' + file.substring(0, file.length() - 6)));
                    }
                }
            }
            else
            {
                throw new ClassNotFoundException(pckgname + " (" + directory.getPath()
                    + ") does not appear to be a valid package");
            }
        }
        return classes;
    }

    /**
     * This calls {@link #getClassesForPackage(String)}.
     * 
     * @param thePackage The package to look in.
     * @param theInterface The interface to look for.
     * @return Classes withint he given package that implement the given interface.
     */
    public static List<Class> getClassessOfInterface(final String thePackage, final Class theInterface)
    {
        final List<Class> classList = new ArrayList<Class>();
        try
        {
            for(final Class discovered: getClassesForPackage(thePackage))
            {
                if(Arrays.asList(discovered.getInterfaces()).contains(theInterface))
                {
                    classList.add(discovered);
                }
            }
        }
        catch(final ClassNotFoundException ex)
        {
        }

        return classList;
    }

    //The main...
    public static void main(final String args[])
    {

        //This is just a main used for testing ideas.

        final String str1 = "HG,HT,HR,HT,HT";
        final SegmentedLine segline1 = new SegmentedLine(str1);
        final String str2 = "QG,QT,HR,QG,RR,RT";
        final SegmentedLine segline2 = new SegmentedLine(str2);

        final Vector result = MiscTools.meshVectorsOfStrings(segline1.getSegments(), segline2.getSegments());
        final SegmentedLine segline3 = new SegmentedLine();
        segline3.setSegments(result);

        System.out.println("####>> The old... " + str1 + "; " + str2);
        System.out.println("####>> The new... " + segline3.reconstructLine(",", true));

        return;
    }

}
