package ohd.hseb.util.misc;

import java.util.Iterator;
import java.util.Vector;

/**
 * This class takes a line of text that contains words separated by known characters and extracts all of the words. The
 * characters that are treated as separators are passed into the constructor. It also allows the user to specify if
 * consecutive separators should be treated as one separator, or mark empty words.<br>
 * <br>
 * Within the constructors, the mode specifies whether or not to treat two consecutive separators as defining an "empty"
 * token, {@link #MODE_ALLOW_EMPTY_SEGS}, or treat two consecutive separators as one separator
 * {@link #MODE_NO_EMPTY_SEGS}. The default in the constructors that do not contain a passed in mode parameter is to
 * allow empty segments.<br>
 * <br>
 * The method that segments a line is {@link #segmentLine(String)}.<br>
 * <br>
 * It is also possible to segment the line based on column indices within the line. The constructor that accomplishes
 * this is {@link #SegmentedLine(String, int[])}. The method that is used to segment the line
 * {@link #segmentLineByColumns(String)}.<br>
 * <br>
 * The method {@link #reconstructLine(String, boolean)} and {@link #reconstructLine(String, boolean, boolean)} can be
 * used to build a {@link String} corresponding to the segments of the current {@link SegmentedLine} object.<br>
 * <br>
 * To navigate the segments found, use the methods {@link #getNumberOfSegments()} within a for loop and
 * {@link #getSegment(int)} to acquire each segment. Or you can use the provided {@link Iterator}.
 */

public class SegmentedLine implements Iterable<String>
{

    public final static String DEFAULT_SEPARATORS = " ,;?!:'\"";
    public final static boolean MODE_ALLOW_EMPTY_SEGS = true;
    public final static boolean MODE_NO_EMPTY_SEGS = false;

    /**
     * Records the found segments.
     */
    private Vector<String> _segments;

    private String _separators;
    private boolean _mode;

    /**
     * Stores column positions for line tokenizing.
     */
    private int[] _positions;

    /**
     * Constructor to build an empty instance.
     */
    public SegmentedLine()
    {
        initialize(null, DEFAULT_SEPARATORS, MODE_ALLOW_EMPTY_SEGS, null);
    }

    /**
     * Constructor that initializes the separators to use and the mode, but the segments are empty until
     * {@link #segmentLine(String)} is called.
     * 
     * @param separators Collection of characters each of which is a separator.
     * @param mode Either {@link #MODE_ALLOW_EMPTY_SEGS} or {@link #MODE_NO_EMPTY_SEGS}.
     */
    public SegmentedLine(final String separators, final boolean mode)
    {
        initialize(null, separators, mode, null);
    }

    /**
     * Segment the passed in line assuming {@link #DEFAULT_SEPARATORS} and {@link #MODE_ALLOW_EMPTY_SEGS}.
     * 
     * @param line
     */
    public SegmentedLine(final String line)
    {
        initialize(line, DEFAULT_SEPARATORS, MODE_ALLOW_EMPTY_SEGS, null);
    }

    /**
     * Segment the passed in line using the specified separators and {@link #MODE_ALLOW_EMPTY_SEGS}.
     * 
     * @param line
     * @param separators Collection of characters each of which is a separator.
     */
    public SegmentedLine(final String line, final String separators)
    {
        initialize(line, separators, MODE_ALLOW_EMPTY_SEGS, null);
    }

    /**
     * Segment the passed in line using the specified separators and mode.
     * 
     * @param line
     * @param separators Collection of characters each of which is a separator.
     * @param mode Either {@link #MODE_ALLOW_EMPTY_SEGS} or {@link #MODE_NO_EMPTY_SEGS}.
     */
    public SegmentedLine(final String line, final String separators, final boolean mode)
    {
        initialize(line, separators, mode, null);
    }

    /**
     * Segment the line based on column indices of characters with the line. The positions array specifies the indices
     * as the first index of each new segment. For each position which is beyond the length of the String, a segment of
     * "" is assumed. Otherwise it is just the substring from the indexed character to the character immediately before
     * the next indexed character.
     * 
     * @param line Line to segment.
     * @param positions The column positions. Must all be non-negative numbers and in increasing order!
     */
    public SegmentedLine(final String line, final int[] positions)
    {
        initialize(line, DEFAULT_SEPARATORS, MODE_ALLOW_EMPTY_SEGS, positions);
    }

    /**
     * This does basic initialization for {@link SegmentedLine}. If positions is NOT null, then the line will be
     * segmented based on positions (see the constructor above). Otherwise it will be segmented based on the separators.
     * 
     * @param line
     * @param separators Collection of characters each of which is a separator.
     * @param mode Either {@link #MODE_ALLOW_EMPTY_SEGS} or {@link #MODE_NO_EMPTY_SEGS}.
     * @param positions The postions; all non-negative and in increasing order.
     */
    public void initialize(final String line, String separators, final boolean mode, final int[] positions)
    {
        //I cannot allow a null separators string!
        if(separators == null)
            separators = DEFAULT_SEPARATORS;
        _separators = new String(separators);

        //Initialize other attributes based on what is passed in.
        _mode = mode;
        _positions = positions;

        //Go no further if line is null.
        if(line == null)
            return;

        //Segment the line based on whether or not positions is null.
        if(positions == null)
            segmentLine(line);
        else
            segmentLineByColumns(line);
    }

    /**
     * Segment the passed in line using the attributes of the class, as defined in the constructors.
     * 
     * @param line
     * @return Returns the number of segments found.
     */
    @SuppressWarnings({"unchecked", "rawtypes"})
    public int segmentLine(String line)
    {
        int i;
        int previous = 0; //The index of the first character to copy into the next segment

        //If line is null, then make it an empty string, so I don't get null ptr problems.
        if(line == null)
            line = "";

        //Initialize variables
        _segments = new Vector();
        int numsegments = 0;

        //Go through each character of line.
        for(i = 0; i < line.length(); i++)
        {
            //if the character is a separator or we are at the last character of the line, then...
            if((_separators.indexOf(line.charAt(i)) >= 0) || (i == line.length() - 1))
            {
                //if there is nothing between the previous separator and this separator,
                //or if this is the last character in the line.
                if(i == previous)
                {
                    //If I don't have a seperator and I'm at the end of the line, then add 
                    //the one character string.
                    if((i == line.length() - 1) && (_separators.indexOf(line.charAt(i)) < 0))
                    {
                        _segments.addElement(line.substring(previous, i + 1));
                        numsegments++;
                    }

                    //else, If I have a separator and I'm allowing empty segments, then...
                    else if(_mode)
                    {
                        //Add a segment between the two separators
                        numsegments++;
                        _segments.addElement("");

                        //If this is also the end of the line, then add a segment between the last
                        //separator and the end of the line.
                        if(i == line.length() - 1)
                        {
                            numsegments++;
                        }
                    }

                    //Increment previous without copying anything
                    previous++;
                }
                //otherwise, we have something to copy...
                else
                {
                    //If this is the end of the line and its not a separator, then include last char
                    if((i == line.length() - 1) && (_separators.indexOf(line.charAt(i)) < 0))
                    {
                        _segments.addElement(line.substring(previous, i + 1));
                        numsegments++;
                    }

                    //Otherwise
                    else
                    {
                        _segments.addElement(line.substring(previous, i));
                        previous = i + 1;
                        numsegments++;

                        //If its the end of the line and I'm allowing empty segments, then I have to 
                        //add an empty segment
                        if((_mode) && (i == line.length() - 1))
                        {
                            numsegments++;
                            _segments.addElement("");
                        }
                    }
                }
            }
        }

        return numsegments;

        //_segments now contains all of the segments of the line    
    }

    /**
     * Segment the line based on indices of characters in the given string specified by the _positions attribute.
     * 
     * @param line
     * @return Returns the number of segments found.
     */
    @SuppressWarnings({"rawtypes", "unchecked"})
    public int segmentLineByColumns(final String line)
    {
        int numsegments = 0;
        _segments = new Vector();
        int i;

        //Check for null.
        if((line == null) || (_positions == null))
            return numsegments;

        //Make sure the positions are valid by seeing if they are consecutive and within the 
        //size of the string.
        for(i = 0; i < _positions.length; i++)
        {
            //The positions must be positive or zero.
            if(_positions[i] < 0)
                return numsegments;

            //if i > 0, the position must be larger than the previous position.
            if(i > 0)
            {
                if(_positions[i] <= _positions[i - 1])
                    return numsegments;
            }
        }

        //The positions are valid.  

        //Initialize some variables. 
        int prevpos = 0;
        int currentpos;

        //Startpt tells me to start at 0, if the first element in the pos array is NOT 0, otherwise
        //start at 1.  
        int startpt = 0;
        if(_positions[0] == 0)
            startpt = 1;
        for(i = startpt; i < _positions.length; i++)
        {
            currentpos = _positions[i];

            //If I am off the end of the string completely, just add "".
            if(prevpos >= line.length())
                _segments.addElement("");
            //Otherwise add the substring.
            else
                _segments.addElement(line.substring(prevpos, Math.min(currentpos, line.length())));

            prevpos = currentpos;
        }

        //Set and return the number of segments.
        numsegments = _segments.size();
        return numsegments;
    }

    /**
     * Reconstruct the segmented line with one specific separator between segements. If quoteelements is true, then each
     * element should be in separate quotes after reconstruction.
     * 
     * @param separator The separator(s) used in the reconstructed line.
     * @param quoteelements If true, the line will have each segment in double quotes.
     * @return
     */
    public String reconstructLine(final String separator, final boolean quoteelements)
    {
        return reconstructLine(separator, quoteelements, false);
    }

    /**
     * Reconstruct the segmented line with one specific separator between segements. If quoteelements is true, then each
     * element should be in separate quotes after reconstruction.
     * 
     * @param separator The separator(s) used in the reconstructed line.
     * @param quoteelements If true, the line will have each segment in quotes.
     * @param singlequotes If true, the line will use single quotes. Otherwise it will use double quotes.
     * @return
     */
    public String reconstructLine(final String separator, final boolean quoteelements, final boolean singlequotes)
    {
        String masterstr = "";
        int i;

        //Set the quotation mark.
        char quote = '"';
        if(singlequotes)
        {
            quote = '\'';
        }

        //Build the string from the segments.  
        for(i = 0; i < getNumberOfSegments(); i++)
        {
            //If I am to quote the elements, then do so here...
            if(quoteelements)
                masterstr += quote;

            masterstr += (_segments.elementAt(i));

            //...and do so here
            if(quoteelements)
                masterstr += quote;

            //Don't add the separator if this is the last segment to add.
            if(i < getNumberOfSegments() - 1)
                masterstr += separator;
        }

        //Return the master string.
        return masterstr;
    }

    /**
     * Removes any repeat segments. This will result in a group of segments each of which is unique.
     * 
     * @param ignorecase If true, case will be ignored in comparing segments.
     */
    public void removeRepeats(final boolean ignorecase)
    {
        //I need two loops going.
        int i, j;

        //The first loop starts at the end and works its way forward up to
        //segment number 1.
        for(i = getNumberOfSegments() - 1; i > 0; i--)
        {
            //The second loops from the beginning and goes up to the value
            //of i minus 1.
            for(j = 0; j < i; j++)
            {
                //If the two segments are equal, remove the one pointed to by i
                //And quit from this internal loop.
                if(((ignorecase) && (getSegment(i).equalsIgnoreCase(getSegment(j))))
                    || ((!ignorecase) && (getSegment(i).equals(getSegment(j)))))
                {
                    removeSegment(i);
                    break;
                }
            }
        }
    }

    /**
     * Removes the segment at the specified index. If the index is invalid, nothing happens.
     * 
     * @param index
     */
    public void removeSegment(final int index)
    {
        if((index < 0) || (index >= getNumberOfSegments()))
            return;

        _segments.remove(index);
    }

    public void setSeparators(final String separators)
    {
        if(separators == null)
            _separators = new String("");
        else
            _separators = new String(separators);
    }

    public void setPositions(final int[] positions)
    {
        if(positions == null)
            return;
        if(positions.length == 0)
            return;

        _positions = positions;
    }

    public int getNumberOfSegments()
    {
        return _segments.size();
    }

    public String getSeparators()
    {
        return _separators;
    }

    /**
     * The tool normally used to acquire a segment.
     * 
     * @param num The index of the segment to acquire.
     * @return The segment returned as a String.
     */
    public String getSegment(final int num)
    {
        if((num < 0) || (num >= getNumberOfSegments()))
            return null;

        return _segments.elementAt(num);
    }

    public Vector<String> getSegments()
    {
        return _segments;
    }

    @SuppressWarnings("unchecked")
    public void setSegments(final Vector segments)
    {
        if(segments == null)
            return;
        _segments = segments;
    }

    @Override
    public Iterator<String> iterator()
    {
        if(_segments == null)
        {
            throw new IllegalStateException("Cannot acquire iterator for a SegmentedList for which the segments have not been created.");
        }
        return _segments.iterator();
    }

    ////////////////////////////////////////////////////
    //TEST MAIN
    ////////////////////////////////////////////////////
    public static void main(final String args[])
    {
        //SegmentedLine segline = new SegmentedLine(args[0], "|", false);
        final int[] pos = {20, 27, 34};
        final SegmentedLine segline = new SegmentedLine(args[0], pos);

        System.out.println("Number of segments... " + segline.getNumberOfSegments());
        System.out.println("Second segment... " + segline.getSegment(1));
        int counter = 0;
        for(final String item: segline)
        {
            System.out.println("####>> " + counter + " = " + item);
            counter++;
        }
    }
}
