package ohd.hseb.hefs.mefp.adapter;

import java.io.File;
import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.TimeZone;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.utils.adapter.DirectoryPropertyVariable;
import ohd.hseb.hefs.utils.adapter.HEFSModelAdapter;
import ohd.hseb.hefs.utils.adapter.SimplePropertyVariable;
import ohd.hseb.hefs.utils.gui.about.AboutFile;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tsarrays.LaggedEnsemble;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.xml.XMLTools;
import ohd.hseb.hefs.utils.xml.vars.XMLFile;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.SegmentedLine;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * This model generates a lagged ensemble based on the forecast time and time series in an archive. This model requires
 * the following property to be specified in the run information:<br>
 * <br>
 * cfsv2OperationalArchiveDirectory: name of the directory that contains the building archive of CFSv2 exported
 * pi-timeseries files.<br>
 * firstFileSearchWindowHours: A window width for how far before the forecast time to look to find the first ensemble
 * member.<br>
 * <br>
 * Other information it uses is gleaned from the provided CFSv2 time series and run-time information.<br>
 * <br>
 * The ensemble generated by this will have exactly 270 days worth of values in it (1080 values) and 15 ensemble
 * members, some of which may be all empty if data is not available. The forecast time will be the current T0. There
 * will be no values returned for T0, itself. If a returned ensemble member is all missing, then the corresponding
 * archived datafile was not found.
 * 
 * @author hank.herr
 */
@AboutFile("version/hefsplugins_config.xml")
public class CFSv2LaggedEnsembleModelAdapter extends HEFSModelAdapter
{
    private static final Logger LOG = LogManager.getLogger(CFSv2LaggedEnsembleModelAdapter.class);
    private static final int NUMBER_OF_LEAD_DAYS = 270;
    private static final String ARCHIVE_DIR_PROPERTY = "cfsv2OperationalArchiveDirectory";
    private static final String FIRST_FILE_SEARCH_WINDOW = "firstFileSearchWindowHours";
    private static final String FILE_PURGE_WINDOW = "filePurgeWindowDays";

    private final SimpleDateFormat _dateFormatter = new SimpleDateFormat("yyyyMMddHH");

    /**
     * Populated by {@link #findFirstTimeSeries(TimeSeriesArray)}, which is called within
     * {@link #findFirstTSAndDetermineListOfLaggedDates(TimeSeriesArray)}. The later method should throw an exception if
     * no appropriate first time series was found.
     */
    private TimeSeriesArray _firstTimeSeries = null;

    /**
     * When building a lagged ensemble for a specific basis time series, this is the start time for that lagged
     * ensemble.
     */
    private long _workingStartTime = -1L;

    /**
     * When building a lagged ensemble for a specific basis time series, this is the end time for that lagged ensemble.
     */
    private long _workingEndTime = -1L;

    /**
     * Sets up {@link #_dateFormatter} and initializes the two valid run-info properties.
     */
    public CFSv2LaggedEnsembleModelAdapter()
    {
        _dateFormatter.setTimeZone(TimeZone.getTimeZone("GMT"));

        this.addPropertyVariables(new DirectoryPropertyVariable(new XMLFile(ARCHIVE_DIR_PROPERTY, false),
                                                                true,
                                                                true,
                                                                true,
                                                                false),
                                  new SimplePropertyVariable(new XMLInteger(FIRST_FILE_SEARCH_WINDOW, 48, 1, null),
                                                             false),
                                  new SimplePropertyVariable(new XMLInteger(FILE_PURGE_WINDOW, -1, null, null), false));
    }

    /**
     * @return The archive directory name.
     */
    public File getArchiveDir()
    {
        return ((SimplePropertyVariable<File>)getPropertyVariable(ARCHIVE_DIR_PROPERTY)).getValue();
    }

    /**
     * @return How far back to go to find the first window.
     */
    public Integer getFirstFileSearchWindow()
    {
        return ((SimplePropertyVariable<Integer>)getPropertyVariable(FIRST_FILE_SEARCH_WINDOW)).getValue();
    }

    /**
     * @return Width of purging window in DAYS!
     */
    public Integer getFilePurgeWindow()
    {
        return ((SimplePropertyVariable<Integer>)getPropertyVariable(FILE_PURGE_WINDOW)).getValue();
    }

    @Override
    protected void extractRunInfo(final RunInfo runInformation) throws Exception
    {
        super.extractRunInfo(runInformation);
    }

    /**
     * @return The file found for the given time in millis or null if neither an XML or FI file exists. This called
     *         {@link #buildArchivedFile(TimeSeriesArray, long, boolean)} to build the file names to check for.
     */
    private File returnExistingFile(final TimeSeriesArray basisTimeSeries, final long millis)
    {
        File file = buildArchivedFile(basisTimeSeries, millis, false);
        if(file.exists())
        {
            return file;
        }
        file = buildArchivedFile(basisTimeSeries, millis, true);
        if(file.exists())
        {
            return file;
        }
        return null;
    }

    /**
     * @return Most recent {@link TimeSeriesArray} found within the window defined by
     *         {@link #getFirstFileSearchWindow()} that is a complete TS relative to {@link #_workingStartTime} and
     *         {@link #_workingEndTime}. Or, if any values are missing, it is the longest partial file, with most recent
     *         breaking ties. If null is returned, then either no time series were found within the window or all time
     *         series are completely missing.
     */
    private void findFirstTimeSeries(final TimeSeriesArray basisTimeSeries)
    {
        LOG.debug("Looking for the first time series to use in the lagged ensemble...");
        //Initialize _firstTimeSeries to null.  If this is still null after all is done, see the javadoc for meaning.
        _firstTimeSeries = null;

        final LinkedHashMap<Long, TimeSeriesArray> timeToTSMap = Maps.newLinkedHashMap();
        final HashMap<TimeSeriesArray, File> tsToFileMap = Maps.newHashMap(); //Map needed for log messaging.

        //First loop looks at all files wtihin the first file search window.  The first file it finds that is complete
        //is used.
        int hours = 0;
        for(hours = 0; hours <= getFirstFileSearchWindow(); hours += 6)
        {
            final long workingTime = getForecastTime() - hours * HCalendar.MILLIS_IN_HR;
            final File firstFile = returnExistingFile(basisTimeSeries, workingTime);
            if(firstFile != null)
            {
                try
                {
                    final TimeSeriesArrays ts = TimeSeriesArraysTools.readFromFile(firstFile);
                    if(ts.size() != 1)
                    {
                        throw new Exception("Expected one time series in file, but found " + ts.size() + ".");
                    }
                    if(checkForCompleteness(ts.get(0)))
                    {
                        LOG.debug("Complete first time series relative to forecast window found in file "
                            + firstFile.getAbsolutePath());
                        _firstTimeSeries = TimeSeriesArrayTools.copyTimeSeries(ts.get(0));
                        return;
                    }
                    LOG.debug("File " + firstFile.getAbsolutePath() + " is partial; looking for complete file first.");
                    timeToTSMap.put(workingTime, ts.get(0));
                    tsToFileMap.put(ts.get(0), firstFile);
                }
                catch(final Exception e)
                {
                    LOG.debug("Unable to read time series from file " + firstFile.getAbsolutePath()
                        + "; skipping file: " + e.getMessage());
                }
            }
        }

        LOG.debug("No file within " + getFirstFileSearchWindow()
            + " hours of T0 can be found that are complete.  Looking for the longest complete partial file.");

        //Check the map in key order (its a LinkedHashMap).  Find the first time series with the hightest cound of non-missing
        //values.  If non have a count greater than 0, then the initial null value will never be changed and null will be returned, indicating
        //all time series are all missing.
        TimeSeriesArray tsWithHighestCount = null;
        int highestCount = 0;
        for(final TimeSeriesArray ts: timeToTSMap.values())
        {
            final int count = TimeSeriesArrayTools.countNumberOfNonMissingValues(ts);
            if(count > highestCount)
            {
                highestCount = count;
                tsWithHighestCount = ts;
            }
        }

        //Copy is used to ensure that the time series use DefaultTimeSeriesHeader.
        if(tsWithHighestCount != null)
        {
            LOG.debug("Most complete time series relative to forecast window found in file "
                + tsToFileMap.get(tsWithHighestCount) + ".");
            _firstTimeSeries = TimeSeriesArrayTools.copyTimeSeries(tsWithHighestCount);
        }

        //If _firstTimeSeries is null, an exception is thrown later that will handle the log messaging.
    }

    /**
     * @param basisTimeSeries The time series providing the basis for the lagged ensemble.
     * @return A list of the dates for which an ensemble member should be sought. Each date is returned as a string of
     *         the format yyyyMMddHH, which can be used directly in a file name to find the corresponding archived file.
     *         The first data is found using the {@link #_firstFileSearchWindowHours} combined with basis time series
     *         provided and the forecast time from the run info file.
     */
    private List<Date> findFirstTSAndDetermineListOfLaggedDates(final TimeSeriesArray basisTimeSeries) throws Exception
    {
        final List<Date> laggedDates = new ArrayList<Date>();

        //Based on fortran comments:
//        c     ensemble        .acfsv2
//        c      member          file
//        c     --------  ---------------------
//        c        1      prate.01.daily.acfsv2 <-- OR first file found within 24 hours of a desired T0
//        c
//        c        2      today -  5 days @ 18Z
//        c        3      today -  5 days @ 12Z
//        c        4      today -  5 days @ 06Z
//        c        5      today -  5 days @ 00Z
//        c
//        c        6      today - 10 days @ 18Z
//        c        7      today - 10 days @ 12Z
//        c        8      today - 10 days @ 06Z
//        c        9      today - 10 days @ 00Z
//        c
//        c       10      today - 15 days @ 18Z
//        c       11      today - 15 days @ 12Z
//        c       12      today - 15 days @ 06Z
//        c       13      today - 15 days @ 00Z
//        c
//        c       14      today - 20 days @ 18Z
//        c       15      today - 20 days @ 12Z
//        c       16      today - 20 days @ 06Z

        //Find the first acceptable time series.  See the method for the time series that will be returned.
        //XXX should this be broken out.
        findFirstTimeSeries(basisTimeSeries);
        if(_firstTimeSeries == null)
        {
            throw new Exception("No acceptable first time series file was found in the archive directory with a time within "
                + getFirstFileSearchWindow()
                + " hours of "
                + HCalendar.buildDateTimeTZStr(getForecastTime())
                + " for location "
                + basisTimeSeries.getHeader().getLocationId()
                + " and parameter "
                + basisTimeSeries.getHeader().getParameterId() + ".");
        }

        //Now for the rest, the times are hardcoded based on the number of days back from the desired T0.
        final Calendar working = HCalendar.computeCalendarFromMilliseconds(getForecastTime());
        working.add(Calendar.DAY_OF_YEAR, -5);
        working.set(Calendar.HOUR_OF_DAY, 18);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 12);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 6);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 0);
        laggedDates.add(working.getTime());

        working.add(Calendar.DAY_OF_YEAR, -5);
        working.set(Calendar.HOUR_OF_DAY, 18);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 12);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 6);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 0);
        laggedDates.add(working.getTime());

        working.add(Calendar.DAY_OF_YEAR, -5);
        working.set(Calendar.HOUR_OF_DAY, 18);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 12);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 6);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 0);
        laggedDates.add(working.getTime());

        working.add(Calendar.DAY_OF_YEAR, -5);
        working.set(Calendar.HOUR_OF_DAY, 18);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 12);
        laggedDates.add(working.getTime());
        working.set(Calendar.HOUR_OF_DAY, 6);
        laggedDates.add(working.getTime());

        return laggedDates;
    }

    /**
     * @return A time series of all {@link Float#NaN} from startTime to endTime, with header parameters set to those in
     *         the base. If the base time series is partial, this will return base. Otherwise, if empty, it will return
     *         a new time series of all missing values.
     */
    private TimeSeriesArray extendTimeSeriesWithMissings(final TimeSeriesArray base,
                                                         final long startTime,
                                                         final long endTime)
    {
        final TimeSeriesArray allMissingTS = buildEmptyTimeSeries(base, base.getHeader().getForecastTime());
        TimeSeriesArrayTools.fillNaNs(allMissingTS, startTime, endTime);
        if(base.isEmpty())
        {
            return allMissingTS;
        }
        TimeSeriesArrayTools.extendFromOtherTimeSeries(base, allMissingTS);
        return base;
    }

    /**
     * @param forecastTime The forecastTime of the time series to create.
     * @return An empty time series (i.e., no values in it at all, not event {@link Float#NaN} with the given
     *         forecastTime and all other header parameters set to those in the base.
     */
    private TimeSeriesArray buildEmptyTimeSeries(final TimeSeriesArray base, final long forecastTime)
    {
        final TimeSeriesArray missingTS = TimeSeriesArrayTools.copyTimeSeries(base);
        ((DefaultTimeSeriesHeader)missingTS.getHeader()).setForecastTime(forecastTime);
        missingTS.clear();

        return missingTS;
    }

    /**
     * Calls {@link #buildArchivedFile(String, String, long)} pulling the location id and parameter id from the time
     * series.
     */
    private File buildArchivedFile(final TimeSeriesArray ts, final long laggedDate, final boolean fastInfoset)
    {
        return buildArchivedFileName(ts.getHeader().getLocationId(),
                                     ts.getHeader().getParameterId(),
                                     laggedDate,
                                     fastInfoset);
    }

    /**
     * @return The name of the archived file with the given location id and lagged date.
     */
    private File buildArchivedFileName(final String locationId,
                                       final String parameterId,
                                       final long laggedDate,
                                       final boolean fastInfoset)
    {
        final File tsFile = FileTools.newFile(getArchiveDir(), locationId, locationId + "." + parameterId + "."
            + _dateFormatter.format(laggedDate) + "." + XMLTools.getXMLFileExtension(fastInfoset));
        return tsFile;
    }

    /**
     * Calls {@link #removeOldFiles(String, long, long)}, providing it the location id from the given time series, the
     * basis time equal to {@link #getForecastTime()} and a window equal to {@link #getFilePurgeWindow()} in
     * milliseconds.
     * 
     * @param ts
     */
    private void removeOldFiles(final TimeSeriesArray ts)
    {
        LOG.info("Removing old files for location " + ts.getHeader().getLocationId() + ".");
        removeOldFiles(ts.getHeader().getLocationId(), getForecastTime(), getFilePurgeWindow() * 24
            * HCalendar.MILLIS_IN_HR);
        LOG.info("Successfully removed old files.");
    }

    /**
     * @param locationId The location for which to purge old files.
     * @param basisTime The time used for comparison.
     * @param window The width of the window in milliseconds such that if a file's date is before that window relative
     *            to the basis time, it is removed.
     */
    private void removeOldFiles(final String locationId, final long basisTime, final long window)
    {
        LOG.debug("Removing old files in directory for location " + locationId + " relative to basis time "
            + HCalendar.buildDateStr(basisTime) + " with a window of " + (window / (24 * HCalendar.MILLIS_IN_HR))
            + " days.");
        int removeCount = 0;
        for(final File dataFile: FileTools.newFile(getArchiveDir(), locationId)
                                          .listFiles(FileTools.makeExtensionFilter(Lists.newArrayList("xml"))))
        {
            try
            {
                final long fileTime = getDateFromFile(dataFile);
                if(fileTime < basisTime - window)
                {
                    LOG.debug("Removing file " + dataFile.getAbsolutePath());
                    if(!dataFile.delete())
                    {
                        LOG.warn("Unable to remove file " + dataFile.getAbsolutePath());
                    }
                    else
                    {
                        removeCount++;
                    }
                }
                else if(fileTime > basisTime)
                {
                    LOG.warn("In directory " + dataFile.getParent() + ", date parsed from file '" + dataFile.getName()
                        + "' was " + HCalendar.buildDateStr(fileTime) + " which appears to be after the target date "
                        + HCalendar.buildDateStr(basisTime) + "; file will be skipped.");
                }
            }
            catch(final ParseException e)
            {
                LOG.warn("In directory " + dataFile.getParent() + ", unable to parse XML file name '"
                    + dataFile.getName() + "' to determine date; file will be skipped: " + e.getMessage());
            }
        }
        LOG.debug("Number of files removed for " + locationId + ": " + removeCount);
    }

    private long getDateFromFile(final File file) throws ParseException
    {
        return getDateFromFileName(file.getName());
    }

    private long getDateFromFileName(final String fileName) throws ParseException
    {
        final SegmentedLine segLine = new SegmentedLine(fileName, ".", SegmentedLine.MODE_ALLOW_EMPTY_SEGS);
        if(segLine.getNumberOfSegments() != 4)
        {
            throw new ParseException("File with name " + fileName
                + " in improperly formatted and does not appear to be a CFSv2 time series file.", 0);
        }
        final String dateStr = segLine.getSegment(2);
        final long millis = _dateFormatter.parse(dateStr).getTime();
        return millis;
    }

    /**
     * Note that for {@link TimeSeriesArrays}, which is created based on the output of this method, all time series MUST
     * have the same header. To ensure that the XML read time series and empty time series are consistent,
     * {@link TimeSeriesArrayTools#copyTimeSeries(TimeSeriesArray)} is called.
     * 
     * @param basisTimeSeries The time series to use as a basis, which may be all empty since its data will not be used.
     * @return A {@link List} of {@link TimeSeriesArray} instances from which the lagged ensemble must be built. Each
     *         item in the list corresponds to one of the dates returned by {@link #determineListOfLaggedDates()}. If an
     *         expected file is not found or not readable, an empty time series will be added to the list witht the same
     *         header info as t0Member.
     * @throws Exception
     */
    private List<TimeSeriesArray> buildListOfTimeSeriesForLagging(final TimeSeriesArray basisTimeSeries) throws Exception
    {
        //Find the first time series and determine the list of lagged dates beyond that.  Calling this method will
        //throw an exception if no first ts can be found.
        final List<Date> laggedDates = findFirstTSAndDetermineListOfLaggedDates(basisTimeSeries);

        //Create the results array and put the first time series in it.
        final List<TimeSeriesArray> results = Lists.newArrayList();
        results.add(_firstTimeSeries);

        //For each lagged date after the first...
        for(int i = 0; i < laggedDates.size(); i++)
        {
            final Date laggedDate = laggedDates.get(i);

            //If the file does not exist, a time series will nothing in it is added to the list.
            final File tsFile = returnExistingFile(basisTimeSeries, laggedDate.getTime());
            if(tsFile == null)
            {
                LOG.warn("Cannot find or read either the .xml archive file "
                    + buildArchivedFile(basisTimeSeries, laggedDate.getTime(), false).getAbsolutePath()
                    + " or the .fi file; a time series of all missing will be added to the ensemble.");

                final TimeSeriesArray missingTS = buildEmptyTimeSeries(basisTimeSeries, laggedDate.getTime());
                results.add(missingTS);
                continue; //Go to the next for loop item.
            }

            LOG.debug("For member " + i + " corresponding to lagged forecast date " + laggedDate
                + ", the file to read is " + tsFile.getAbsolutePath());

            //Otherwise, read the file and add the time series, but there must be only one!
            try
            {
                final TimeSeriesArrays readTS = TimeSeriesArraysTools.readFromFile(tsFile);
                if(readTS.size() != 1)
                {
                    throw new Exception("File must contain only 1 time series, but contains " + readTS.size() + ".");
                }
                ((DefaultTimeSeriesHeader)readTS.get(0).getHeader()).setForecastTime(laggedDate.getTime());
                results.add(TimeSeriesArrayTools.copyTimeSeries(readTS.get(0))); //Copy is done to ensure header is DefaultTimeSeriesHeader.

                LOG.debug("Processed file for lagged date " + laggedDate + ".");
            }
            catch(final Exception e)
            {
                throw new Exception("Unable to read time series " + tsFile.getAbsolutePath() + ": " + e.getMessage());
            }
        }
        return results;
    }

    /**
     * I spell out the checks below in separate if-s because I think it improves readability.
     * 
     * @param ts Time series to check.
     * @return False if the time series is not complete (empty or any value is missing).
     */
    private boolean checkForCompleteness(final TimeSeriesArray ts)
    {
        if(ts.isEmpty())
        {
            return false;
        }
        if(TimeSeriesArrayTools.containsMissingValuesWithinWindow(ts, _workingStartTime, _workingEndTime))
        {
            return false;
        }
        return true;
    }

    @Override
    protected String getModelName()
    {
        return "CFSv2 Lagged Ensemble";
    }

    /**
     * @param basisTimeSeries The time series to use as a basis for the lagged ensemble.
     * @return The lagged ensemble sliced to the appropriate time window based on {@link #_forecastTimeT0} and
     *         {@link #NUMBER_OF_LEAD_DAYS}.
     * @throws Exception
     */
    private List<TimeSeriesArray> buildLaggedEnsembleForOneLocation(final TimeSeriesArray basisTimeSeries) throws Exception
    {
        _workingStartTime = getForecastTime() + basisTimeSeries.getHeader().getTimeStep().getStepMillis();
        _workingEndTime = getForecastTime() + NUMBER_OF_LEAD_DAYS * 24 * HCalendar.MILLIS_IN_HR;

        //Get the base ensemble list.
        List<TimeSeriesArray> baseEnsemble = null;
        try
        {
            baseEnsemble = buildListOfTimeSeriesForLagging(basisTimeSeries);
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Error building ensemble component list for "
                + basisTimeSeries.getHeader().getLocationId() + ": " + e.getMessage());
        }

        //Trim all members of the ensemble and set all forecast times based on the starting time series using 
        //LaggedEnsemble.
        final LaggedEnsemble ensemble = new LaggedEnsemble(baseEnsemble,
                                                           getForecastTime(),
                                                           _workingStartTime,
                                                           _workingEndTime);
        ensemble.assignEnsembleId("CFSv2");
        ensemble.assignEnsembleMemberIndices(1);

        //Check for completeness of each member, and replace by missing if any are not.
        //TODO LIMIN: Is this the expected behavior?  Here, if it finds any missings, even at the end, it will
        //force the member to be unused by making it all missing.  This is likely only a problem in testing.
        for(int i = 0; i < ensemble.size(); i++)
        {
            if(!checkForCompleteness(ensemble.get(i)))
            {
                LOG.warn("At least one value in lagged ensemble member " + i
                    + " is missing (turn on debug to see the corresponding lagged forecast date);"
                    + " the entire member will be set to missing so that it is not used.");
                ensemble.set(i, extendTimeSeriesWithMissings(ensemble.get(i), _workingStartTime, _workingEndTime));

            }
        }
        return ensemble;
    }

    @Override
    protected TimeSeriesArrays executeModel(final TimeSeriesArrays inputTS) throws Exception
    {
        LOG.info("There are " + inputTS.size() + " time series provided.  " + "The time series values will be ignored.");
        LOG.info("Instead, for each provided time series location and parameter id, "
            + "a lagged ensemble will be constructed based on the archive directory contents.");
        LOG.info("Time series with repeat locationIds will be skipped.  ");
        LOG.info("The time step of the provided time series must match the time step of the archive "
            + "time series, even though the data values are not used.");

        //Build the lagged ensemble.
        final List<TimeSeriesArray> results = new ArrayList<TimeSeriesArray>();
        for(final TimeSeriesArray ts: TimeSeriesArraysTools.convertTimeSeriesArraysToList(inputTS))
        {
            LOG.info("Building lagged ensemble for " + ts.getHeader().getLocationId() + " with T0 "
                + DateTime.getDateStringFromLong(getForecastTime(), TimeZone.getTimeZone("GMT")));
            final List<TimeSeriesArray> laggedEnsemble = buildLaggedEnsembleForOneLocation(ts);
            LOG.info("Lagged ensemble for " + ts.getHeader().getLocationId() + " includes " + laggedEnsemble.size()
                + " members.");
            results.addAll(laggedEnsemble);
        }

        if(getFilePurgeWindow() > 0)
        {
            LOG.info("File purge window has been specified.  Removing old files for all locations for which lagged ensembles were just created...");
            LOG.info("To see details of old file removal, set the run file property printDebugInfo to 1 (or more) and run the module in debug mode through CHPS.");
            for(final TimeSeriesArray ts: TimeSeriesArraysTools.convertTimeSeriesArraysToList(inputTS))
            {
                removeOldFiles(ts);
            }
        }

        return TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(DefaultTimeSeriesHeader.class, results);
    }

    public static void main(final String[] args) throws IOException
    {
        HEFSModelAdapter.runAdapter(args, CFSv2LaggedEnsembleModelAdapter.class.getName());
    }
}
