package ohd.hseb.hefs.mefp.adapter;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.TreeSet;

import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.models.MEFPEnsembleGeneratorModel;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.precipitation.PrecipitationEnsembleGenerationModel;
import ohd.hseb.hefs.mefp.models.temperature.TemperatureEnsembleGenerationModel;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.sources.rfcfcst.RFCForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventRestrictor;
import ohd.hseb.hefs.pe.sources.ForecastSourceTools;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.Dyad;
import ohd.hseb.hefs.utils.Triad;
import ohd.hseb.hefs.utils.adapter.HEFSModelAdapter;
import ohd.hseb.hefs.utils.adapter.MappedPropertyVariable;
import ohd.hseb.hefs.utils.adapter.SimplePropertyVariable;
import ohd.hseb.hefs.utils.gui.about.AboutFile;
import ohd.hseb.hefs.utils.log4j.LoggingTools;
import ohd.hseb.hefs.utils.tools.FileTools;
import ohd.hseb.hefs.utils.tools.NumberTools;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.SamplingTools;
import ohd.hseb.hefs.utils.tools.ThreadedRandomTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.vars.XMLBoolean;
import ohd.hseb.hefs.utils.xml.vars.XMLDouble;
import ohd.hseb.hefs.utils.xml.vars.XMLEnum;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLLong;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HStopWatch;
import ohd.hseb.util.misc.HString;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * Adapter for executing the MEFP ensemble generation models, {@link PrecipitationEnsembleGenerationModel} and
 * {@link TemperatureEnsembleGenerationModel}. <br>
 * <br>
 * Next the input time series are processed with an assumption that time series qualifier ids match one forecast source
 * prefixes returned by {@link MEFPForecastSource#getSourceId()} for the included sources. If a matching source is found
 * for a time series, that time series is added for the source. If no matching source is found, a warning message is
 * generated and the time series is skipped. The time series for each source are kept in a
 * {@link LocationForecastSourceInformation} list. Some basic checks are performed on the sources, including CFSv2
 * having the appropriate number of members and temperature models having two data types.<br>
 * <br>
 * The model is applied in a standard way. For precipitation, one ensemble is generated, but for temperature two are
 * generated: one for tfmn and one for tfmx. Each model is updated with information from various attributes and
 * executed. The ensembles generated are all gathered into a single list to be output by the superclass
 * {@link HEFSModelAdapter}.
 * 
 * @author hankherr
 */
@AboutFile("version/hefsplugins_config.xml")
public class MEFPEnsembleGeneratorModelAdapter extends HEFSModelAdapter
{
    private static final Logger LOG = LogManager.getLogger(MEFPEnsembleGeneratorModelAdapter.class);

//    private final static String DEFAULT_LOCATION_FOR_FORECAST_DAYS = "@@DEFAULT@@";

    //Constants representing possible run-information property keys.
    protected final static String NUMBER_OF_FORECAST_DAYS_SUFFIX = "NumberOfForecastDays";
    protected final static String EPT_OPTION_SUFFIX = "UseEPT";
    protected final static String EXCLUDE_BASE_EVENTS = "ExcludeBaseEvents";
    protected final static String EXCLUDE_MODULATION_EVENTS = "ExcludeModulationEvents";
    protected final static String EXCLUDE_EVENTS_WITH_DUR_LESS_THAN = "ExcludeEventsWithDurLessThan";
    protected final static String EXCLUDE_EVENTS_WITH_DUR_MORE_THAN = "ExcludeEventsWithDurMoreThan";
    protected final static String ONE_EVENT_ONLY = "OneEventOnly";
    protected final static String COND_COEFF_VAR_MAX = "condCoeffVarMax";
// Redmine #72689
//    protected final static String EPT_STRATIFIED_SAMPLING = "eptUseStratifiedSampling";
//    protected final static String USE_STRATIFIED_RANDOM_SAMPLING = "useStratifiedRandomSampling";
    protected final static String TESTING = "testing";
    protected final static String INITIAL_ENSEMBLE_YEAR = "initialEnsembleYear";
    protected final static String LAST_ENSEMBLE_YEAR = "lastEnsembleYear";
    protected final static String HINDCAST_MODE = "hindcasting";
    protected final static String PARAMETER_DIR = "parameterDir";
    public final static String BEHAVIOR_IF_EVENT_MISSING = "behaviorIfEventMissing";
    protected final static String MEMBER_INDEXING_YEAR = "memberIndexingYear";
    protected final static String EPT_THRESHOLD_TO_INCLUDE_SAMPLING_LOG_MESSAGES = "eptThresholdToIncludeSamplingLogMessages";
    // Redmine #72689
    protected final static String SAMPLING_TECHNIQUE = "samplingTechnique";

    private final List<MEFPForecastSource> _allowedSources = new ArrayList<>();

    private final HashMap<Dyad<String, Boolean>, HashMap<MEFPForecastSource, LocationForecastSourceInformation>> _locationToSourceToInfoMap = new LinkedHashMap<>();

    private final HashMap<Triad<MEFPForecastSource, String, Integer>, TreeSet<Long>> _sourceToMissingDataTimes = new HashMap<>();

    /**
     * Populates the {@link #_locationAndDataTypeToSourceInformationMap} attribute based on the given time series and
     * {@link #_includedSources}.
     * 
     * @param timeSeries
     * @throws Exception
     */
    private void populateSourceInformation(final TimeSeriesArrays timeSeries) throws Exception
    {
        _locationToSourceToInfoMap.clear();
        for(int i = 0; i < timeSeries.size(); i++)
        {
            final TimeSeriesArray ts = timeSeries.get(i);
            final LocationAndDataTypeIdentifier identifier = LocationAndDataTypeIdentifier.get(ts);

            //Get the list of sourceInfos() for the location and data type.
            HashMap<MEFPForecastSource, LocationForecastSourceInformation> sourceInfos = _locationToSourceToInfoMap.get(buildLocationAndDataTypeFlagDyad(identifier));

            //If no source infos exist, then create a new list with entries for all included sources.  Each entry will be any empty map 
            //(null return values) to start.
            if((sourceInfos == null) || (sourceInfos.isEmpty()))
            {
                sourceInfos = Maps.newLinkedHashMap();
                _locationToSourceToInfoMap.put(buildLocationAndDataTypeFlagDyad(identifier), sourceInfos);
                for(int sourceIndex = 0; sourceIndex < _allowedSources.size(); sourceIndex++)
                {
                    final LocationForecastSourceInformation sourceInfo = new LocationForecastSourceInformation(identifier.isPrecipitationDataType(),
                                                                                                               _allowedSources.get(sourceIndex));

                    sourceInfos.put(sourceInfo.getSource(), sourceInfo);
                }
            }

            //If we are in hindcasting mode, no more work needs to be done for this ts: just add an entry to the map to track which 
            //locations to run for and move on to the next.
            if(getHindcastMode())
            {
                continue;
            }

            //Loop through the sourceInfos to find the one whose forecast source matches the qualifier id of the time series.  If there is no
            //qualifier id, the RFC forecast source is assumed.
            LocationForecastSourceInformation sourceInfo = null;
            if(ts.getHeader().getQualifierCount() == 0)
            {
                sourceInfo = sourceInfos.get(new RFCForecastSource());
                if(sourceInfo == null)
                {
                    LOG.warn("Time series found with no qualifier id for location " + ts.getHeader().getLocationId()
                        + " and parameter " + ts.getHeader().getParameterId()
                        + ", meaning it is for the RFC forecast source, but the RFC forecast source is not included.");
                }
            }
            else
            {
                //Variable is used for messaging purposes later.
                final List<MEFPForecastSource> allowedSourcesMinusClimatology = new ArrayList<>();

                //This is the loop: For each source, it checks if any time series qualifier id matches the source id.  If so,
                //the time series is assigned to that source (first come first serve, order dictated by standard order).
                //The string check is case insensitive.
                for(final MEFPForecastSource source: _allowedSources)
                {
                    if(source.isClimatologySource())
                    {
                        continue;
                    }

                    allowedSourcesMinusClimatology.add(source);
                    for(int qualIdIndex = 0; qualIdIndex < ts.getHeader().getQualifierCount(); qualIdIndex++)
                    {
                        if(source.getSourceId().equalsIgnoreCase(ts.getHeader().getQualifierId(qualIdIndex)))
                        {
                            sourceInfo = sourceInfos.get(source);
                            break;
                        }
                    }
                    if(sourceInfo != null)
                    {
                        break;
                    }
                }
                if(sourceInfo == null)
                {
                    LOG.warn("Time series for " + ts.getHeader().getLocationId() + " and parameter "
                        + ts.getHeader().getParameterId() + " found with qualifier ids "
                        + Arrays.toString(TimeSeriesArrayTools.getQualifierIds(ts))
                        + " none of which match a valid MEFP forecast source ("
                        + HString.buildStringFromList(allowedSourcesMinusClimatology, ",")
                        + "); time series will be ignored.");
                    continue;
                }
            }
            //If a sourceInfo is identified in the if-else above, use it.  Otherwise, the time series is ignored.
            if(sourceInfo != null)
            {
                sourceInfo.addTimeSeries(ts);
            }
        }

        //Validate the map if we are not in hindcast mode.  Hindcast validation occurs later.
        if(!getHindcastMode())
        {
            validateSourceInformation();
            generateMissingDataTimeMessages();
        }
    }

    /**
     * Adds an entry to {@link #_sourceToMissingDataTimes}.<br>
     * <br>
     * THIS METHOD IS WRITTEN ASSUMING THAT THE TIMES ARE PROCESSED IN INCREASING ORDER. YOU MUST NEVER PASS IN A TIME
     * THAT IS SMALLER THAN ANY OF THE PREVIOUSLY PROCESSED TIMES.
     * 
     * @param source {@link MEFPForecastSource} for which to add a missing time.
     * @param time The time that is missing.
     * @param timeStepHours The time step of the data being examined.
     * @param parameterId The data type
     */
    private void addSourceMissingDataTime(final MEFPForecastSource source,
                                          final long time,
                                          final int timeStepHours,
                                          final String parameterId)
    {
        final Triad<MEFPForecastSource, String, Integer> key = new Triad<MEFPForecastSource, String, Integer>(source,
                                                                                                              parameterId,
                                                                                                              timeStepHours);
        TreeSet<Long> set = _sourceToMissingDataTimes.get(key);
        if(set == null)
        {
            set = new TreeSet<Long>();
            _sourceToMissingDataTimes.put(key, set);
        }
        set.add(time);
    }

    /**
     * Generates log messages based on {@link #_sourceToMissingDataTimes}.
     */
    private void generateMissingDataTimeMessages()
    {
        for(final Triad<MEFPForecastSource, String, Integer> key: _sourceToMissingDataTimes.keySet())
        {
            String message = "For source " + key.getFirst().getSourceId() + " and data type " + key.getSecond()
                + ", missing data was found in the input time series and " + getBehaviorIfEventMissing()
                + " will be invoked for at least one location at the following times: ";

            final TreeSet<Long> times = _sourceToMissingDataTimes.get(key);
            String item = "";
            long itemLastTime = Long.MIN_VALUE;
            boolean firstItem = true;
            boolean multiIntervalItem = false;

            //For each time in the set of sorted times, do...
            final Iterator<Long> iter = times.iterator();
            while(iter.hasNext())
            {
                final Long time = iter.next();

                //First time through loop, initialize item and itemLastTime
                boolean processItem = true;
                if(item.isEmpty())
                {
                    item = HCalendar.buildDateTimeStr(time);
                    itemLastTime = time;
                    processItem = false;
                }

                //If the time last associated with the working item is one step before the current time,
                //then extend the item and mark this as a multi-interval item.
                else if(itemLastTime + key.getThird() * HCalendar.MILLIS_IN_HR == time)
                {
                    itemLastTime = time;
                    multiIntervalItem = true;
                    processItem = false;
                }

                //Otherwise, we are at the end of one interval item, so generate output and reset.
                if(processItem || !iter.hasNext())
                {
                    //First, if this is a multiIntervalItem, then print out the end point as the current time.
                    if(multiIntervalItem)
                    {
                        item += " - " + HCalendar.buildDateTimeStr(time);
                    }

                    //The item is now ready for output.  First, add a separator if needed and then add it to the message.
                    if(!firstItem)
                    {
                        message += "; ";
                    }
                    message += item;

                    //Prep the next item in the list.  Set the subElement and clear the last time.
                    item = HCalendar.buildDateTimeStr(time);
                    itemLastTime = time;
                    multiIntervalItem = false;
                    firstItem = false;
                }
            }

            message += ".";
            LOG.warn(message);

        }
    }

    /**
     * Calls link {@link LocationForecastSourceInformation#validate()} for each
     * {@link LocationForecastSourceInformation} within {@link #_locationAndDataTypeToSourceInformationMap} values. Note
     * that only those sources that are used (number of days > 0) are validated.
     * 
     * @throws Exception If any validation fails.
     */
    private void validateSourceInformation() throws Exception
    {
        //For each locationAndDataType in the location to source info map...
        for(final Dyad<String, Boolean> locationAndDataType: _locationToSourceToInfoMap.keySet())
        {
            //Get the source infos.
            final HashMap<MEFPForecastSource, LocationForecastSourceInformation> sourceInfos = _locationToSourceToInfoMap.get(locationAndDataType);

            //For each source info, validate the source information but only if the source is to be used.
            try
            {
                for(final LocationForecastSourceInformation sourceInfo: sourceInfos.values())
                {
                    try
                    {
                        if(getNumberOfForecastDays(locationAndDataType.getFirst(), sourceInfo.getSource()) > 0)
                        {
                            sourceInfo.validate();
                        }
                    }
                    catch(final IllegalArgumentException e)
                    {
                        throw new Exception("INTERNAL ERROR: We should not have been able to get this exception here: "
                            + e.getMessage());
                    }
                }
            }
            catch(final Exception e)
            {
//                e.printStackTrace();
                throw new Exception("For location " + locationAndDataType.getFirst() + " and "
                    + HEFSTools.determineDataTypeString(locationAndDataType.getSecond()) + ": " + e.getMessage());
            }
        }
    }

    /**
     * This uses {@link MEFPTools#determineParameterFileName(String, boolean)} to determine the name of the parameter
     * file. The day for which to load parameters is determined based on the {@link #_forecastTime}.
     * 
     * @param locationId
     * @param precipitation
     * @return The parameters for the given location id and precipitation flag.
     * @throws Exception
     */
    private MEFPFullModelParameters loadParameters(final String locationId, final boolean precipitation) throws Exception
    {
        //Determine the day of the year if the forecast time was given.
        final List<Integer> dayOfYearList = new ArrayList<Integer>();
        final Calendar forecastTimeCal = HCalendar.computeCalendarFromMilliseconds(getForecastTime());
        final int dayOfYear = HEFSTools.computeDayOfYearWithLeapDayBeingMar1(forecastTimeCal);
        dayOfYearList.add(dayOfYear);

        //Initialize the parameters for reading and read them.
        final String parameterFileName = MEFPTools.determineParameterFileName(locationId, precipitation);
        final File parameterFile = FileTools.newFile(getParameterDir(), parameterFileName);
        LOG.debug("Parameter file: " + parameterFile.getAbsolutePath());

        //Initialize the parameters to include all sources.  If sources are included in the parameter file, 
        //the copy of the passed in list used by MEFPFullModelParameters will be updated.
        final MEFPFullModelParameters parameters = new MEFPFullModelParameters(_allowedSources);

        //This needs to mimick the defaults setup in the setupForModelRun method.  If not specified, EPT is used for
        //any model that allows for it.
        for(final MEFPForecastSource source: getIncludedSources(locationId))
        {
            boolean loadEvents = false;
            //Events are never loaded for temperature runs.
            if(precipitation)
            {
                if(getEPTOption(source) != null)
                {
                    //EPT option is specified.  It dictates if events must be loaded.
                    loadEvents = getEPTOption(source) && source.canEPTModelBeUsedForSource();
                }
                else
                {
                    //No EPT option... use the source default.
                    loadEvents = source.canEPTModelBeUsedForSource(); //Default to if source uses EPT.
                }
            }
            parameters.addOperationalRunScenario(source, loadEvents);
        }

        //Reading mode is based on if this is a hindcasting run.
        parameters.readParametersTarArchive(parameterFile, dayOfYearList, false, !getHindcastMode());
        return parameters;
    }

    /**
     * @param identifier
     * @return A {@link Dyad} that can be used as the key to {@link #_locationAndDataTypeToSourceInformationMap}.
     */
    private Dyad<String, Boolean> buildLocationAndDataTypeFlagDyad(final LocationAndDataTypeIdentifier identifier)
    {
        return new Dyad<String, Boolean>(identifier.getLocationId(), identifier.isPrecipitationDataType());
    }

    /**
     * Sets miscellaneous data type specific parameters for the provided model.
     * 
     * @param model The model in which parameters are being set.
     * @param source The source for which parameters are being set.
     * @param parameters The loaded estimated model parameters.
     * @param dataType The data type of the data for which parameters were loaded.
     */
    private void setupModelForRun(final MEFPEnsembleGeneratorModel model,
                                  final MEFPForecastSource source,
                                  final MEFPFullModelParameters parameters,
                                  final ParameterId dataType)
    {
        //Set parameters for the model.
        model.setSourceNumberOfForecastDays(source,
                                            getNumberOfForecastDays(parameters.getIdentifier().getLocationId(), source));

        //PRECIPITATION...
        if(dataType.isPrecipitation())
        {
            //Set the EPT option
            if(getEPTOption(source) != null)
            {
                ((PrecipitationEnsembleGenerationModel)model).setUseEPT(source, getEPTOption(source));
            }

            //EPT option defaults to that used in parameters.  If the source does not allow for it, then do not set the
            //useEPT flag, which cases it to default to false within PrecipitationEnsembleGenerationModel.
            else
            {
                if(source.canEPTModelBeUsedForSource())
                {
                    ((PrecipitationEnsembleGenerationModel)model).setUseEPT(source, true); //Default to using EPT if available.
                    LOG.info("For source "
                        + source.getName()
                        + " no EPT option was specified with run properties; using the option for which parameters were estimated: "
                        + ((PrecipitationEnsembleGenerationModel)model).useEPT(source) + ".");
                }

            }

            //Models source event values must be prepped in hindcast mode or when using EPT.
            if(getHindcastMode() || ((PrecipitationEnsembleGenerationModel)model).useEPT(source))
            {
                model.setCurrentSourceCanonicalEventValues(source, parameters.getSourceModelParameters(source)
                                                                             .getPrecipitationSourceEventValues());
            }
        }
        //TEMPERATURE...
        else
        {
            ((TemperatureEnsembleGenerationModel)model).setTmaxData(dataType.isMax());

            //Models source event values must be prepped in hindcast mode.
            if(getHindcastMode())
            {
                if(dataType.isMax())
                {
                    model.setCurrentSourceCanonicalEventValues(source, parameters.getSourceModelParameters(source)
                                                                                 .getTMAXSourceEventValues());
                }
                else
                {
                    model.setCurrentSourceCanonicalEventValues(source, parameters.getSourceModelParameters(source)
                                                                                 .getTMINSourceEventValues());
                }
            }
        }
    }

    /**
     * @param providedForecastTimeSeries The forecast time series provided to the adapter via inputs.xml. Note that when
     *            in hindcast mode, the canonical events are loaded directly from the parameter file (handled later in
     *            the ensemble generation model), or, if not available, the time series is loaded from the reforecast
     *            files. Regardless, this provided time series will not be used in calculations later.
     * @param model The model in which parameters are being set.
     * @param source The source for which parameters are being set.
     * @param parameters The loaded estimated model parameters.
     * @param dataType The data type of the data for which parameters were loaded.
     * @throws Exception From
     *             {@link #loadHindcastTimeSeries(MEFPForecastSource, LocationAndDataTypeIdentifier, ParameterId, long)}
     *             .
     */
    private final void addSourceTimeSeries(final TimeSeriesArrays providedForecastTimeSeries,
                                           final MEFPEnsembleGeneratorModel model,
                                           final MEFPForecastSource source,
                                           final MEFPFullModelParameters parameters,
                                           final ParameterId dataType) throws Exception
    {
        //By default, the forecast time series is gotten via sourceInfos.  However, if in hindcast mode and source canonical event 
        //values are not available for the desired forecast time, then load the hindcast time series from MEFPPE run area files.
        //Note that if a time series for the given forecast time is not found, an exception will be thrown.
        if((getHindcastMode()) && (!model.areHindcastCanonicalEventsAlreadyComputed(source, getForecastTime())))
        {
            throw new Exception("Canonical event values for the " + source.getSourceId()
                + " forecast source and forecast time " + HCalendar.buildDateTimeTZStr(getForecastTime())
                + " were not found for location " + parameters.getIdentifier().buildStringToDisplayInTree()
                + "; cannot generate hindcast.");
        }
        else if(getHindcastMode())
        {
            LOG.debug("Canonical event values for the " + source.getSourceId() + " forecast source and forecast time "
                + HCalendar.buildDateTimeTZStr(getForecastTime())
                + " were found in the parameters and will be used (no time series need to be loaded).");
        }

        //Add the source time series.
        if((providedForecastTimeSeries != null) && (!providedForecastTimeSeries.isEmpty()))
        {
            model.addSourceForecastTimeSeries(source, providedForecastTimeSeries);
        }

    }

    /**
     * Ensures that for a given location, the maximum number of forecast days over all sources is at least 1.
     * 
     * @throws Exception If all sources have a number of forecast days of 0.
     */
    private void setAndValidateMaximumNumberOfForecastDays(final String locationId) throws Exception
    {
        int maxNumberOfLeadDays = -1;
        for(final MEFPForecastSource source: _allowedSources)
        {
            final int numberOfDays = getNumberOfForecastDays(locationId, source);
            if(numberOfDays >= maxNumberOfLeadDays)
            {
                maxNumberOfLeadDays = numberOfDays;
            }
        }
        if(maxNumberOfLeadDays < 0)
        {
            throw new Exception("No sources are included in the run-info properties.");
        }
        if(maxNumberOfLeadDays == 0)
        {
            throw new Exception("The maximum number of lead days specified for any source is zero; no forecast can be generated.");
        }
    }

    /**
     * @param locationId The location id to look for.
     * @return Based on {@link #getNumberOfForecastDays(String, MEFPForecastSource)} and {@link #_allowedSources}, it
     *         returns the sources found for the given location. Any source for which the number of days is 0 or more
     *         and any source in {@link #_allowedSources} will be returned. This does NOT reflect those sources used in
     *         MEFP; for that the number of forecast days must be strictly greater than 0.
     */
    private List<MEFPForecastSource> getIncludedSources(final String locationId) throws IllegalArgumentException
    {
        final List<MEFPForecastSource> locationSources = Lists.newArrayList();
        for(final MEFPForecastSource source: _allowedSources)
        {
            final int numberOfDays = getNumberOfForecastDays(locationId, source);
            if(numberOfDays >= 0)
            {
                locationSources.add(source);
            }
        }

        //Sort the location sources appropriately and return.
//        MEFPTools.sortMEFPForecastSources(locationSources);
        return locationSources;
    }

    /**
     * @return The number of forecast days for the location and source. The default is returned if the location is not
     *         found. If the default is also not found, then -1 is returned indicating that the run info does no even
     *         allow it.
     */
    private int getNumberOfForecastDays(final String locationId, final MEFPForecastSource source)
    {
        //TODO This method is called a lot to determine both the number of forecast days and whether or not a source is include (#days > 0)
        //Is there something that can be done to clean that up???  See System.err commented out line below -- use it to see how often it shows up.

        //This property variable will always exist, so just get it.
        final MappedPropertyVariable<Integer> var = (MappedPropertyVariable)this.getPropertyVariable(getSourcePropertyPrefix(source)
            + NUMBER_OF_FORECAST_DAYS_SUFFIX);

        //Attempt to get the number of days for the given locationId.  By definition, it will return null or an XMLInteger.
        //If not found, try to find the default value, which if not found is an error.  
        Integer numberOfDays = null;
        if(var != null)
        {
            numberOfDays = var.getValue(locationId);
            if(numberOfDays == null)
            {
                numberOfDays = var.getValue(null);
            }
        }
        if(numberOfDays == null)
        {
            LOG.debug("Number of forecast days is not defined for source " + source.getSourceId() + " and location "
                + locationId + ", and there is no default defined.");
            return -1;
        }

        //This system.err call shows up a lot!
//        System.err.println("####>> NUMBER OF DAYS FOUND FOR " + locationId + " and source " + source.getName() + " is "
//            + numberOfDays);
        return numberOfDays;
    }

    public Boolean getEPTOption(final MEFPForecastSource source)
    {
        final SimplePropertyVariable<Boolean> var = (SimplePropertyVariable<Boolean>)getPropertyVariable(getSourcePropertyPrefix(source)
            + EPT_OPTION_SUFFIX);
        return var.getValue();
    }

    public Boolean getExcludeBaseEvents(final MEFPForecastSource source)
    {
        final SimplePropertyVariable<Boolean> var = (SimplePropertyVariable<Boolean>)getPropertyVariable(getSourcePropertyPrefix(source)
            + EXCLUDE_BASE_EVENTS);
        return var.getValue();
    }

    public Boolean getExcludeModulationEvents(final MEFPForecastSource source)
    {
        final SimplePropertyVariable<Boolean> var = (SimplePropertyVariable<Boolean>)getPropertyVariable(getSourcePropertyPrefix(source)
            + EXCLUDE_MODULATION_EVENTS);
        //Apparently this can happen, though I think only in unit testing, not in practice.
        if(var == null)
        {
            return true; //Default value for excluding events.  
        }
        return var.getValue();
    }

    public Integer getExcludeEventsWithDurLessThan(final MEFPForecastSource source)
    {
        final SimplePropertyVariable<Integer> var = (SimplePropertyVariable<Integer>)getPropertyVariable(getSourcePropertyPrefix(source)
            + EXCLUDE_EVENTS_WITH_DUR_LESS_THAN);
        return var.getValue();
    }

    public Integer getExcludeEventsWithDurMoreThan(final MEFPForecastSource source)
    {
        final SimplePropertyVariable<Integer> var = (SimplePropertyVariable<Integer>)getPropertyVariable(getSourcePropertyPrefix(source)
            + EXCLUDE_EVENTS_WITH_DUR_MORE_THAN);
        return var.getValue();
    }

    /**
     * Tracks the number of times the modulation events warning has been created. It must be printed only once per
     * adapter run. Can this be made a static variable within
     * {@link #buildCanonicalEventRestrictor(MEFPFullModelParameters)} where it is used?
     */
    private boolean _modulationEventsWarningPrintedOnce = false;

    /**
     * @return A {@link CanonicalEventRestrictor} for use in running a model given the specific set of parameters.
     *         Restrictions are performed based on the exclude run-info properties and the one event only property.
     */
    private CanonicalEventRestrictor buildCanonicalEventRestrictor(final MEFPFullModelParameters parameters)
    {
        //Message designed to let users know if they specify for modulation events to be used but none are available.
        boolean excludeModEvents = true;
        for(final MEFPForecastSource source: parameters.getOrderedForecastSources())
        {
            excludeModEvents = excludeModEvents && getExcludeModulationEvents(source);
        }
        if(!excludeModEvents && parameters.getAlgorithmModelParameters().getModulationCanonicalEvents().isEmpty())
        {
            LOG.info("For location, "
                + parameters.getIdentifier().buildStringToDisplayInTree()
                + ", for at least one forecast source, modulation events are to be included, but there are no modulation events defined in the parameter file (see run-info properties *"
                + EXCLUDE_MODULATION_EVENTS + ")");
            if(!_modulationEventsWarningPrintedOnce)
            {
                _modulationEventsWarningPrintedOnce = true;
                LOG.warn("For at least one location, modulation events were indicated to be included but were not defined in the parameter file (see the INFO level messages to identify the locations).");
            }
        }

        //Return the constrictor.
        return new CanonicalEventRestrictor()
        {
            @Override
            public boolean useEvent(final MEFPForecastSource source, final CanonicalEvent event)
            {
                //Never use an event for which parameters are not computed.
                if(!parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().contains(event))
                {
                    return false;
                }

                final Boolean excludeBaseEvents = getExcludeBaseEvents(source);
                final Boolean excludeModEvents = getExcludeModulationEvents(source);
                final Integer eventDurLB = getExcludeEventsWithDurLessThan(source); //lb is inclusive
                final Integer eventDurUB = getExcludeEventsWithDurMoreThan(source); //ub is inclusive
                final CanonicalEvent evt = getOneEventOnly(source);

//                System.err.println("####>> ---- " + excludeBaseEvents + ", " + excludeModEvents + ", " + eventDurLB
//                    + ", " + eventDurUB + ", " + evt);

                //If event is not null, then we are only using that specific event!
                if(evt != null)
                {
                    final boolean checkValue = evt.equals(event);
                    if(!checkValue)
                    {
                        LOG.debug("Event " + event
                            + " rejected because it does not match the one event for execution which has start "
                            + evt.getStartLeadPeriod() + " and end " + evt.getEndLeadPeriod()
                            + "; see run-info property " + getSourcePropertyPrefix(source) + ONE_EVENT_ONLY);
                    }
                    return checkValue;
                }

                //Check the exclusions.
                if(excludeBaseEvents != null)
                {
                    if(parameters.getAlgorithmModelParameters().getBaseCanonicalEvents().contains(event)
                        && excludeBaseEvents)
                    {
                        LOG.debug("Event " + event + " rejected due to being a base event; see run-info property "
                            + getSourcePropertyPrefix(source) + EXCLUDE_BASE_EVENTS);
                        return false;
                    }
                }
                if(excludeModEvents != null)
                {
                    if(parameters.getAlgorithmModelParameters().getModulationCanonicalEvents().contains(event)
                        && excludeModEvents)
                    {
                        LOG.debug("Event " + event
                            + " rejected due to being a modulation event; see run-info property "
                            + getSourcePropertyPrefix(source) + EXCLUDE_MODULATION_EVENTS);
                        return false;
                    }
                }
                if(eventDurLB != null)
                {
                    if(event.getDuration() < eventDurLB)
                    {
                        LOG.debug("Event " + event + " rejected due to duration, " + event.getDuration()
                            + ", being less than " + eventDurLB + "; see run-info property "
                            + getSourcePropertyPrefix(source) + EXCLUDE_EVENTS_WITH_DUR_LESS_THAN);
                        return false;
                    }
                }
                if(eventDurUB != null)
                {
                    if(event.getDuration() > eventDurUB)
                    {
                        LOG.debug("Event " + event + " rejected due to duration, " + event.getDuration()
                            + ", being more than " + eventDurUB + "; see run-info property "
                            + getSourcePropertyPrefix(source) + EXCLUDE_EVENTS_WITH_DUR_MORE_THAN);
                        return false;
                    }
                }

                //Default is to use.
                return true;
            }
        };
    }

    /**
     * @param source
     * @return A {@link CanonicalEvent} with a the start and end time set based on the property value. The number and
     *         number of members are set to -1, but they are not used in identifying a canonical event, so it shouldn't
     *         matter.
     * @throws Exception If the value is not a property '[int][separators][int]' format.
     */
    public CanonicalEvent getOneEventOnly(final MEFPForecastSource source)
    {
        //Formatting has been checked when the property was originally set!
        final SimplePropertyVariable<String> var = (SimplePropertyVariable<String>)getPropertyVariable(getSourcePropertyPrefix(source)
            + ONE_EVENT_ONLY);
        if(var.getValue() == null)
        {
            return null;
        }
        final SegmentedLine segLine = new SegmentedLine(var.getValue(),
                                                        SegmentedLine.DEFAULT_SEPARATORS,
                                                        SegmentedLine.MODE_NO_EMPTY_SEGS);
        final Integer start = Integer.parseInt(segLine.getSegment(0).trim());
        final Integer end = Integer.parseInt(segLine.getSegment(1).trim());
        return new CanonicalEvent(-1, start, end, -1);

    }

    public Double getEPTThresholdToIncludeSamplingLogMessages()
    {
        return ((SimplePropertyVariable<Double>)getPropertyVariable(EPT_THRESHOLD_TO_INCLUDE_SAMPLING_LOG_MESSAGES)).getValue();
        
    }
    
// Redmine #72689
//    public Boolean getEPTStratifiedSampling()
//    {
//        return ((SimplePropertyVariable<Boolean>)getPropertyVariable(EPT_STRATIFIED_SAMPLING)).getValue();
//    }
//
//    public Boolean getUseStratifiedRandomSampling()
//    {
//        return ((SimplePropertyVariable<Boolean>)getPropertyVariable(USE_STRATIFIED_RANDOM_SAMPLING)).getValue();
//    }

    // Redmine #72689
    public SamplingTools.SamplingTechnique getSamplingTechnique()
    {
    	return ((SimplePropertyVariable<SamplingTools.SamplingTechnique>)getPropertyVariable(SAMPLING_TECHNIQUE)).getValue();
    }

    // Redmine #72689
    public void setSamplingTechnique(final SamplingTools.SamplingTechnique sampleTechnique)
    {
        ((SimplePropertyVariable<SamplingTools.SamplingTechnique>)getPropertyVariable(SAMPLING_TECHNIQUE)).setValue(sampleTechnique);
    }
    
    public Double getCondCoeffVarMax()
    {
        return ((SimplePropertyVariable<Double>)getPropertyVariable(COND_COEFF_VAR_MAX)).getValue();
    }

    public Long getTesting()
    {
        return ((SimplePropertyVariable<Long>)getPropertyVariable(TESTING)).getValue();
    }

    public int getInitialEnsembleYear()
    {
        return ((SimplePropertyVariable<Integer>)getPropertyVariable(INITIAL_ENSEMBLE_YEAR)).getValue();
    }

    public int getLastEnsembleYear()
    {
        return ((SimplePropertyVariable<Integer>)getPropertyVariable(LAST_ENSEMBLE_YEAR)).getValue();
    }

    public boolean getHindcastMode()
    {
        return ((SimplePropertyVariable<Boolean>)getPropertyVariable(HINDCAST_MODE)).getValue();
    }

    public void setHindcastMode(final boolean b)
    {
        ((SimplePropertyVariable<Boolean>)getPropertyVariable(HINDCAST_MODE)).setValue(b);
    }

    public String getParameterDir()
    {
        return ((SimplePropertyVariable<String>)getPropertyVariable(PARAMETER_DIR)).getValue();
    }

    public MEFPEnsembleGeneratorModel.BehaviorIfEventMissing getBehaviorIfEventMissing()
    {
        return ((SimplePropertyVariable<MEFPEnsembleGeneratorModel.BehaviorIfEventMissing>)getPropertyVariable(BEHAVIOR_IF_EVENT_MISSING)).getValue();
    }

    public void setBehaviorIfEventMissing(final MEFPEnsembleGeneratorModel.BehaviorIfEventMissing behavior)
    {
        ((SimplePropertyVariable<MEFPEnsembleGeneratorModel.BehaviorIfEventMissing>)getPropertyVariable(BEHAVIOR_IF_EVENT_MISSING)).setValue(behavior);
    }

    public Calendar getMemberIndexingYear()
    {
        return ((SimplePropertyVariable<Calendar>)getPropertyVariable(MEMBER_INDEXING_YEAR)).getValue();
    }

    public void setMemberIndexingYear(final Calendar memberIndexingYear)
    {
        ((SimplePropertyVariable<Calendar>)getPropertyVariable(MEMBER_INDEXING_YEAR)).setValue(memberIndexingYear);
    }

    private String getSourcePropertyPrefix(final MEFPForecastSource source)
    {
        if(source.isClimatologySource())
        {
            return "climatology";
        }
        return source.getSourceId().toLowerCase();
    }

    /**
     * Method is used to load allowable forecast sources based on a plug-in file. If not found, default sources are
     * used.
     */
    @Override
    protected void loadModuleDataSetsBeforeExtractingRunTimeInformation(final RunInfo runInfo) throws Exception
    {
        //Does nothing, but just in case...
        super.loadModuleDataSetsBeforeExtractingRunTimeInformation(runInfo);

        final File definedSourcesFile = FileTools.newFile(runInfo.getWorkDir(),
                                                          MEFPParameterEstimatorRunInfo.FORECAST_SOURCES_DEFINITION_FILENAME);
        if(definedSourcesFile.exists())
        {
            try
            {
                ForecastSourceTools.readSourcesFromXML(null, definedSourcesFile, _allowedSources);
                LOG.info("Sources read in from " + definedSourcesFile.getAbsolutePath() + " were as follows: "
                    + HString.buildStringFromList(_allowedSources, ", "));
            }
            catch(final Exception e)
            {
                throw new Exception("Unable to parse defined forecast sources: " + e.getMessage(), e);
            }
        }
        else
        {
            LOG.info("Exported module data set file defining allowable forecast sources, "
                + MEFPParameterEstimatorRunInfo.FORECAST_SOURCES_DEFINITION_FILENAME
                + ", was not found in the work directory; using default sources.");
            _allowedSources.addAll(MEFPTools.instantiateDefaultMEFPForecastSources());
        }
    }

    @SuppressWarnings("unchecked")
    @Override
    protected void extractRunInfo(final RunInfo runInformation) throws Exception
    {
        //The property variables are constructed here because the runInformation work directory is used as a default 
        //for the PARAMETER_DIR property.

        //For each source, add a number of forecast days and an ept option property variable.
        for(final MEFPForecastSource source: _allowedSources)
        {
            //Number of forecast days is a mapped variable where the location id is specified after the '.'.
            addPropertyVariables(new MappedPropertyVariable(getSourcePropertyPrefix(source)
                + NUMBER_OF_FORECAST_DAYS_SUFFIX, new XMLString("notUsed"), //Location id, or default if none
                                                            new XMLInteger((Integer)null, 0, null) //Number of days >= 0
            ));

            //The EPT option variable is a boolean variable but it must be checked against the valid choices for the sources
            //when finalizing reading.
            addPropertyVariables(new SimplePropertyVariable(new XMLBoolean(getSourcePropertyPrefix(source)
                + EPT_OPTION_SUFFIX, null)
            {

                @Override
                public void finalizeReading() throws XMLReaderException
                {
                    try
                    {
                        if(!source.canEPTModelBeUsedForSource())
                        {
                            LOG.warn("EPT option cannot be set for the " + source.getName()
                                + " source, since it must be off; ignoring the run-information property.");
                        }
                        else if(!get())
                        {
                            LOG.warn("The run property "
                                + getXMLTagName()
                                + " is set to false, indicating that the IPT model (not EPT) will be used; that model has been deprecated and may be removed in a future release!");
                        }
                    }
                    catch(final IllegalArgumentException e)
                    {
                        LOG.warn("Forecast source " + source.getName()
                            + " is invalid; ignoring the run-information property.");
                    }
                }
            },
                                                            false));

            //Event controls
            addPropertyVariables(new SimplePropertyVariable(new XMLBoolean(getSourcePropertyPrefix(source)
                                     + EXCLUDE_BASE_EVENTS, false), false),
                                 new SimplePropertyVariable(new XMLBoolean(getSourcePropertyPrefix(source)
                                     + EXCLUDE_MODULATION_EVENTS, false), false), //Changed first true to false so that default is to include the events
                                 new SimplePropertyVariable(new XMLInteger(getSourcePropertyPrefix(source)
                                     + EXCLUDE_EVENTS_WITH_DUR_LESS_THAN, null, 0, null), false),
                                 new SimplePropertyVariable(new XMLInteger(getSourcePropertyPrefix(source)
                                     + EXCLUDE_EVENTS_WITH_DUR_MORE_THAN, null, 0, null), false));

            //This property must include a start and end.
            addPropertyVariables(new SimplePropertyVariable(new XMLString(getSourcePropertyPrefix(source)
                + ONE_EVENT_ONLY, null), false)
            {

                @Override
                public void setValue(final Object value)
                {
                    final SegmentedLine segLine = new SegmentedLine((String)value,
                                                                    SegmentedLine.DEFAULT_SEPARATORS,
                                                                    SegmentedLine.MODE_NO_EMPTY_SEGS);

                    //Must have two segments.
                    if(segLine.getNumberOfSegments() != 2)
                    {
                        throw new IllegalArgumentException("The value of the property "
                            + getSourcePropertyPrefix(source) + ONE_EVENT_ONLY
                            + " must contain two integer components, but '" + value + "' does not.");
                    }

                    //Each segment must be an integer.
                    try
                    {
                        Integer.parseInt(segLine.getSegment(0).trim());
                    }
                    catch(final NumberFormatException e)
                    {
                        throw new IllegalArgumentException("The value of the property "
                            + getSourcePropertyPrefix(source) + ONE_EVENT_ONLY
                            + " must contain two integer components, but the first component of '" + value
                            + "' is not an integer.");
                    }
                    try
                    {
                        Integer.parseInt(segLine.getSegment(1).trim());
                    }
                    catch(final NumberFormatException e)
                    {
                        throw new IllegalArgumentException("The value of the property "
                            + getSourcePropertyPrefix(source) + ONE_EVENT_ONLY
                            + " must contain two integer components, but the second component of '" + value
                            + "' is not an integer.");
                    }
                }
            });
        }

        addPropertyVariables(new SimplePropertyVariable(new XMLEnum(MEFPEnsembleGeneratorModel.BehaviorIfEventMissing.class,
                                                                    BEHAVIOR_IF_EVENT_MISSING,
                                                                    MEFPEnsembleGeneratorModel.BehaviorIfEventMissing.fillNextSource),
                                                        false));
        addPropertyVariables(new SimplePropertyVariable(new MemberIndexYearXMLVariable(MEMBER_INDEXING_YEAR,
                                                                                       TimeSeriesEnsemble.STANDARD_HISTORICAL_WATER_YEAR),
                                                        false));
        // Redmine #72689
        addPropertyVariables(new SimplePropertyVariable(new XMLEnum(SamplingTools.SamplingTechnique.class,
                													SAMPLING_TECHNIQUE,
                													SamplingTools.SamplingTechnique.RANDOM),
        												false));
        
        //Here are the basics.  
        addPropertyVariables(new SimplePropertyVariable(new XMLDouble(COND_COEFF_VAR_MAX, null, 0.000001d, null), false),
        		             // Redmine #72689
                             //new SimplePropertyVariable(new XMLBoolean(EPT_STRATIFIED_SAMPLING, false), false),
                             new SimplePropertyVariable(new XMLDouble(EPT_THRESHOLD_TO_INCLUDE_SAMPLING_LOG_MESSAGES, null, 0.0d, 1.0d), false),
                             // Redmine #72689
                             //new SimplePropertyVariable(new XMLBoolean(USE_STRATIFIED_RANDOM_SAMPLING, false), false),
                             new SimplePropertyVariable(new XMLLong(TESTING, null), false),
                             new SimplePropertyVariable(new XMLInteger(INITIAL_ENSEMBLE_YEAR, null), true),
                             new SimplePropertyVariable(new XMLInteger(LAST_ENSEMBLE_YEAR, null), true),
                             new SimplePropertyVariable(new XMLBoolean(HINDCAST_MODE, false), false),
                             new SimplePropertyVariable(new XMLString(PARAMETER_DIR, runInformation.getWorkDir()),
                                                        false));
        
        

        //Extract the run information.
        super.extractRunInfo(runInformation);
        
        //Deprecated/removed properties check --------------
        if(runInformation.getProperties().getProperty("useResampledClimatology") != null)
        {
            LOG.warn("The run file property 'useResampledClimatology' is no longer used as of 1.2.1 (raw climatology is no longer allowed) and will be ignored.");
        }

        //Deprecated/removed properties check --------------
        if(runInformation.getProperties().getProperty("useStratifiedRandomSampling") != null)
        {
        	throw new Exception("The run file property 'useStratifiedRandomSampling' is no longer used (replace it with '"+SAMPLING_TECHNIQUE+"' with value " +SamplingTools.SamplingTechnique.STRATIFIED_RANDOM+")");
        }
        
        //Deprecated/removed properties check --------------
        if(runInformation.getProperties().getProperty("eptUseStratifiedSampling") != null)
        {
        	throw new Exception("The run file property 'eptUseStratifiedSampling' is no longer used (replace it with '"+SAMPLING_TECHNIQUE+"' with value " +SamplingTools.SamplingTechnique.RANDOM+")");
        }
        
        //Check the forecast time to see if it is not 12Z.
        if(HCalendar.computeCalendarFromMilliseconds(getForecastTime()).get(Calendar.HOUR_OF_DAY) != 12)
        {
            LOG.warn("The MEFP Ensemble Generator is designed to execute with a 12Z forecast time (T0).  The specified forecast time "
                + HCalendar.buildDateTimeTZStr(getForecastTime())
                + " is not a 12Z time, so MEFP output may not be valid.");
        }
    }

    @Override
    protected String getModelName()
    {
        return "MEFPEnsembleGeneratorModelAdapter";
    }

    /**
     * Executes the model for the given parameters and list of sources for the location. When this is called, the
     * following attributes MUST be appropriately populated for the location for which parameters are loaded:
     * {@link #_lidSourceNumberOfDaysTable} and {@link #_locationToSourceToInfoMap}. The
     * {@link #_lidSourceNumberOfDaysTable} will be used to determine which sources are included in the run and the
     * number of forecast lead days for each source. The {@link #_locationToSourceToInfoMap} will be used to specify the
     * time series.
     * 
     * @param parameters Parameters to use.
     * @return The time series generated by the model that is executed.
     * @throws Exception For various reasons, and exception may be thrown.
     */
    private TimeSeriesArrays runModel(final MEFPFullModelParameters parameters) throws Exception
    {
        //Make sure we have a maximum number of forecast days that exceeds 0.
        try
        {
            setAndValidateMaximumNumberOfForecastDays(parameters.getIdentifier().getLocationId());
        }
        catch(final Exception e)
        {
            throw new Exception("Validation of number of forecast days failed for "
                + parameters.getIdentifier().buildStringToDisplayInTree() + ": " + e.getMessage(), e);
        }

        //Get the included sources.  This should not be able to throw an illegal argument exception at this point.
        final List<MEFPForecastSource> includedSourcesForLocation = getIncludedSources(parameters.getIdentifier()
                                                                                                 .getLocationId());

        final boolean precipitation = parameters.getIdentifier().isPrecipitationDataType();

        final Dyad<String, Boolean> locationAndDataType = buildLocationAndDataTypeFlagDyad(parameters.getIdentifier());
        final TimeSeriesArrays results = new TimeSeriesArrays(DefaultTimeSeriesHeader.class,
                                                              (getLastEnsembleYear() - getInitialEnsembleYear()));

        //Instantiate the model to use and set its parameters.
        final MEFPEnsembleGeneratorModel model = MEFPTools.determineEnsembleGenerationModel(precipitation);
        model.setModelParameters(parameters);
        model.setHindcastMode(getHindcastMode());

        //Set model flags, possibly based on data type.
        if(precipitation)
        {
            if(getCondCoeffVarMax() != null)
            {
                ((PrecipitationEnsembleGenerationModel)model).setCondCoeffVarMax(getCondCoeffVarMax());
            }
            // Redmine #72689
            //model.setStratifiedSampling(getEPTStratifiedSampling());
            model.setStratifiedSampling(getSamplingTechnique());
            ((PrecipitationEnsembleGenerationModel)model).setEPTThresholdToIncludeSamplingLogMessages(getEPTThresholdToIncludeSamplingLogMessages());
        }
        else
        {
            // Redmine #72689
            //model.setStratifiedSampling(getUseStratifiedRandomSampling());
            model.setStratifiedSampling(getSamplingTechnique());
        }

        model.setCanonicalEventRestrictor(buildCanonicalEventRestrictor(parameters));
        model.setBehavoirIfEventMissing(getBehaviorIfEventMissing());
        model.setMemberIndexingCal(getMemberIndexingYear());

        //For each data type for which we need to run the model...
        for(final ParameterId dataType: model.getProcessedDataTypes())
        {
            LOG.debug("Generating forecast ensemble for data type " + dataType + "...");
            //Setup the model information based on the location source information.
            final HashMap<MEFPForecastSource, LocationForecastSourceInformation> sourceInfos = _locationToSourceToInfoMap.get(locationAndDataType);
            model.clearSourceMaps();
            for(final MEFPForecastSource source: includedSourcesForLocation)
            {
                //Skip the source if the number of days is 0.
                if(getNumberOfForecastDays(parameters.getIdentifier().getLocationId(), source) <= 0)
                {
                    continue;
                }

                // Confirm that the source which is included (i.e., number of fcst days > 0) has parameters for it.
                
                if(parameters.getSourceModelParameters(source) == null)
                {
                    throw new Exception("For location " + parameters.getIdentifier().buildStringToDisplayInTree()
                        + ", the source " + source.getName() + " has a positive number of forecast days, "
                        + getNumberOfForecastDays(parameters.getIdentifier().getLocationId(), source)
                        + ", but no parameters available for that source within the parameter file.");
                }
                
                //Set model parameters.
                setupModelForRun(model, source, parameters, dataType);

                //By default, the forecast time series is gotten via sourceInfos.  However, if in hindcast mode and source canonical event 
                //values are not available for the desired forecast time, then load the hindcast time series from MEFPPE run area files.
                //If this is climatology, this is not a concern.
                if(!source.isClimatologySource())
                {
                    final TimeSeriesArrays providedForecastTimeSeries = sourceInfos.get(source).get(dataType);
                    addSourceTimeSeries(providedForecastTimeSeries, model, source, parameters, dataType);
                }
            }

            //Generate the ensemble and add it to the results.
            final TimeSeriesEnsemble ens = model.generateEnsemble(getForecastTime(), getLastEnsembleYear()
                - getInitialEnsembleYear() + 1, getInitialEnsembleYear());
            results.addAll(ens);
            LOG.debug("Done generating forecast ensemble for data type " + dataType + ".");
        }
        return results;
    }

    /**
     * Prepares for and calls {@link #runModel(MEFPFullModelParameters)} within an adapter execution.
     */
    @Override
    protected TimeSeriesArrays executeModel(final TimeSeriesArrays inputTS) throws Exception
    {
        //Setup the source maps and time series.
        populateSourceInformation(inputTS);

        //Prepare the results.
        final TimeSeriesArrays results = new TimeSeriesArrays(DefaultTimeSeriesHeader.class,
                                                              (getLastEnsembleYear() - getInitialEnsembleYear())
                                                                  * _locationToSourceToInfoMap.size());

        //Initialize the random number generator appropriately.
        final Long seed = getTesting();
        if(seed != null)
        {
            LOG.info("Testing mode has been activated with seed " + seed
                + "; initializing the random number generator with the seed.");
            ThreadedRandomTools.initializeRandomNumberGeneratorForTesting(seed);
        }
        else
        {
            ThreadedRandomTools.initializeRandomNumberGenerator();
        }

        //For each identifier and data type (precip and temp).
        for(final Dyad<String, Boolean> locationAndDataType: _locationToSourceToInfoMap.keySet())
        {
            LOG.info("=============== EXECUTING MODEL for " + locationAndDataType._first + " and data type "
                + HEFSTools.determineDataTypeString(locationAndDataType._second) + " =============== ");
            final HStopWatch modelTimer = new HStopWatch();
            final String locationId = locationAndDataType.getFirst();
            final boolean precipitation = locationAndDataType.getSecond();

            //Load the parameters
            LOG.debug("Loading parameters...");
            final HStopWatch parmReadTimer = new HStopWatch();
            MEFPFullModelParameters parameters = null;
            try
            {
                parameters = loadParameters(locationId, locationAndDataType.getSecond());
            }
            catch(final Exception e)
            {
                e.printStackTrace();
                throw new Exception("Unable to load parameters for " + locationId + " and data type "
                    + HEFSTools.determineDataTypeString(precipitation) + ": " + e.getMessage(), e);
            }
            LOG.debug("Parameters loaded in " + (parmReadTimer.getElapsedMillis() / 1000d) + " seconds.");

            //XXX TESTING!!!!!!
//Removing modulation events and base events whose end times exceed some value.  Designed for temperature data (false passed to list constructor).
//            CanonicalEvent evtToRemove = new CanonicalEvent(-1, 31, 60, -1);
//            parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().remove(evtToRemove);
//            parameters.getAlgorithmModelParameters().getBaseCanonicalEvents().remove(evtToRemove);
//            evtToRemove = new CanonicalEvent(-1, 31, 120, -1);
//            parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().remove(evtToRemove);
//            parameters.getAlgorithmModelParameters().getModulationCanonicalEvents().remove(evtToRemove);
//            evtToRemove = new CanonicalEvent(-1, 61, 90, -1);
//            parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().remove(evtToRemove);
//            parameters.getAlgorithmModelParameters().getBaseCanonicalEvents().remove(evtToRemove);
//            evtToRemove = new CanonicalEvent(-1, 61, 150, -1);
//            parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().remove(evtToRemove);
//            parameters.getAlgorithmModelParameters().getModulationCanonicalEvents().remove(evtToRemove);
//            parameters.getAlgorithmModelParameters()
//                      .getFullListOfEventsInOrder()
//                      .removeAll(parameters.getAlgorithmModelParameters().getModulationCanonicalEvents());
//            parameters.getAlgorithmModelParameters().getModulationCanonicalEvents().clear();
//
//            final CanonicalEventList removeList = new CanonicalEventList(false);
//            for(final CanonicalEvent evt: parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder())
//            {
//                if(!evt.equals(new CanonicalEvent(-1, 211, 240, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 241, 270, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 181, 210, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 61, 90, -1))
////                    && !evt.equals(new CanonicalEvent(-1, 31, 120, -1))
////                    && !evt.equals(new CanonicalEvent(-1, 151, 240, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 31, 60, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 91, 120, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 151, 180, -1))
////                    && !evt.equals(new CanonicalEvent(-1, 121, 210, -1))
////                    && !evt.equals(new CanonicalEvent(-1, 61, 150, -1))
//                    && !evt.equals(new CanonicalEvent(-1, 121, 150, -1))
////                    && !evt.equals(new CanonicalEvent(-1, 91, 180, -1))
//                )
//                {
//                    if(evt.getEndLeadPeriod() > 30)
//                    {
//                        System.err.println("####>> REMOVING (base) -- " + evt);
//                        removeList.add(evt);
//                    }
//                }
//            }
//            parameters.getAlgorithmModelParameters().getFullListOfEventsInOrder().removeAll(removeList);
            //XXX DONE--------

            //Instantiate the model to use and set its parameters.
//            final MEFPEnsembleGeneratorModel model = MEFPTools.determineEnsembleGenerationModel(precipitation);
//            model.setModelParameters(parameters);
//            model.setHindcastMode(getHindcastMode());

            results.addAll(runModel(parameters));

            LOG.info("=============== DONE EXECUTING MODEL for " + locationAndDataType._first + " and data type "
                + HEFSTools.determineDataTypeString(locationAndDataType._second) + " in "
                + (modelTimer.getElapsedMillis() / 1000d) + " seconds =============== ");

        }

        return results;
    }

    /**
     * Records information necessary to generate ensembles for a location and one source.
     * 
     * @author hankherr
     */
    @SuppressWarnings("serial")
    private class LocationForecastSourceInformation extends HashMap<ParameterId, TimeSeriesEnsemble>
    {
        private final boolean _precipitation;
        private final MEFPForecastSource _source;
        private final MEFPEnsembleGeneratorModel _model;

        /**
         * Sets {@link #_identifier} and {@link #_source}, which determining {@link #_model} based on {@link #_source}
         * and {@link #_allowedParameters} based on {@link #_model}.
         * 
         * @param identifier Identifier for which to record the ensemble generation information.
         * @param source The source for which information is being stored.
         */
        public LocationForecastSourceInformation(final boolean precipitation, final MEFPForecastSource source)
        {
            _precipitation = precipitation;
            _source = source;
            _model = MEFPTools.determineEnsembleGenerationModel(precipitation);
        }

        public void addTimeSeries(final TimeSeriesArray ts) throws Exception
        {
            //Determine the used parameter id based on the model and change the provided time series to use it.
            ParameterId usedParmId;
            try
            {
                usedParmId = _model.checkForValidityAndConvertToProcessedType(ts.getHeader().getParameterId());
            }
            catch(final Exception e)
            {
                throw new Exception("Error processing time series for location " + ts.getHeader().getLocationId()
                    + ", parameter " + ts.getHeader().getParameterId() + ", and qualifier ids "
                    + Arrays.toString(TimeSeriesArrayTools.getQualifierIds(ts)) + ": " + e.getMessage());
            }
            ((DefaultTimeSeriesHeader)ts.getHeader()).setParameterId(usedParmId.toString());

            //Check the measuring units and convert.
            try
            {
                _model.checkForValidityAndConvertUnits(ts);
            }
            catch(final Exception e)
            {
                throw new Exception("Error in time series for location " + ts.getHeader().getLocationId()
                    + ", parameter " + ts.getHeader().getParameterId() + ", and qualifier ids "
                    + Arrays.toString(TimeSeriesArrayTools.getQualifierIds(ts)) + ": " + e.getMessage());
            }

            //Forcibly set the ts forecast time to one step prior to the start time.  This is used for some validation later
            //involving computing the number of full days spanned.
            try
            {
                ((DefaultTimeSeriesHeader)ts.getHeader()).setForecastTime(ts.getStartTime()
                    - ts.getHeader().getTimeStep().getStepMillis());
            }
            catch(final IllegalStateException e)
            {
                //An illegal state exception occurs if ts is empty.  That will trigger an exception below when the
                //number of forecast days is checked against the number of time series values.
            }

            //Create an ensemble to store time series if one does not already exist.  Add the time
            //to this map.
            TimeSeriesEnsemble ens = get(usedParmId);
            if(ens == null)
            {
                ens = new TimeSeriesEnsemble(new TimeSeriesArrays(ts));
                put(usedParmId, ens);
            }
            else
            {
                ens.addMember(ts);
            }
        }

        /**
         * Checks that the required number of data types were provided to the {@link #_model}. Also checks that for each
         * data type, the required number of time series for the {@link #_source} were provided, checking against
         * {@link MEFPForecastSource#getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration()}.
         * 
         * @throws Exception
         */
        public void validate() throws Exception
        {
            //No time series are expected for climatology sources, so we can skip their validation.
            if(_source.isClimatologySource())
            {
                return;
            }

            //Number of data types
            if(keySet().size() != _model.getProcessedDataTypes().length)
            {
                throw new Exception("For source " + _source + ", the model used for "
                    + HEFSTools.determineDataTypeString(_precipitation) + " must process "
                    + _model.getProcessedDataTypes().length + " data types, but " + keySet().size()
                    + " were provided with non-empty time series (parameterIds: " + keySet().toString() + ").");
            }

            //All data types are covered (FMAP for precip, TFMX and TFMN for temp)
            for(final ParameterId dataType: _model.getProcessedDataTypes())
            {
                if(this.get(dataType) == null)
                {
                    throw new Exception("For source " + _source + ", the model used for "
                        + HEFSTools.determineDataTypeString(_precipitation) + " must process data type " + dataType
                        + ", but the data types provided with non-empty time series did not include it (parameterIds: "
                        + keySet().toString() + ").");
                }
            }

            //For each paramter id, confirm the number of time series, length of time series, and make sure the ts T0
            //is not AFTER the forecast T0.
            for(final ParameterId parmId: this.keySet())
            {
                //Check the time series to see if any is not empty.
                if(TimeSeriesArraysTools.areAllMissing(get(parmId)))
                {
                    throw new Exception("For source " + _source.getSourceId()
                        + ", all time series values provided are missing.");
                }

                //Number of time series
                if(get(parmId).size() != _source.getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration())
                {
                    throw new Exception("The forecast source " + _source.getName() + " requires "
                        + _source.getRequiredNumberOfTimeSeriesPerDataTypeForEnsembleGeneration()
                        + " time series, but " + get(parmId).size() + " time series were provided for parameter "
                        + parmId.toString() + ".");
                }

                //Check ensemble T0 against desired forecast time and warn if needed.
                final TimeSeriesEnsemble forecastTS = this.get(parmId);
                for(final TimeSeriesArray ts: forecastTS)
                {
                    //Warn if the forecast time does not match desired time.
                    if(!NumberTools.nearEquals(ts.getHeader().getForecastTime(),
                                               getForecastTime(),
                                               ts.getHeader().getTimeStep().getStepMillis() + 1))
                    {
                        throw new Exception("For the the time series for the "
                            + _source.getSourceId()
                            + " forecast source with locationId "
                            + ts.getHeader().getLocationId()
                            + ", parameterId "
                            + ts.getHeader().getParameterId()
                            + " and member index "
                            + ts.getHeader().getEnsembleMemberIndex()
                            + " (index of -1 indicates it was a single forecast time series), the forecast time computed as one step before first value, "
                            + HCalendar.buildDateTimeStr(ts.getHeader().getForecastTime())
                            + ", is not within one time series step of the run_info.xml forecast time, "
                            + HCalendar.buildDateTimeStr(getForecastTime()) + ".");
                    }

                    //The time series must encompass all needed days based on the desired forecast time.
                    final int numberOfFullDays = TimeSeriesArrayTools.computeNumberOfFullDaysSpanned(ts,
                                                                                                     getForecastTime());
                    if(numberOfFullDays < getNumberOfForecastDays(ts.getHeader().getLocationId(), _source))
                    {
                        throw new Exception("The number of forecast days for forecast source " + _source.getName()
                            + ", " + getNumberOfForecastDays(ts.getHeader().getLocationId(), _source)
                            + ", is larger than the number of full days of data available in the forecast time series "
                            + "relative to the desired forecast time with locationId " + ts.getHeader().getLocationId()
                            + ", parameterId " + ts.getHeader().getParameterId() + " and member index "
                            + ts.getHeader().getEnsembleMemberIndex()
                            + " (index of -1 indicates it was a single forecast time series), which is "
                            + numberOfFullDays + " days.");
                    }
                }

                //Add to the source missing data time list any times for which data is missing across all time series in forecastTS.  
                for(int checkIndex = 0; checkIndex < (getNumberOfForecastDays(forecastTS.getLocationId(), _source) * 24)
                    / forecastTS.getTimeStepHours(); checkIndex++)
                {
                    boolean missing = true;
                    for(final TimeSeriesArray ts: forecastTS)
                    {
                        if(!TimeSeriesArrayTools.isOHDMissingValue(ts.getValue(checkIndex)))
                        {
                            missing = false;
                            break;
                        }
                    }
                    if(missing)
                    {
                        addSourceMissingDataTime(_source,
                                                 forecastTS.get(0).getTime(checkIndex),
                                                 forecastTS.getTimeStepHours(),
                                                 forecastTS.getParameterId()); //Uses member 0 as a proxy for all.
                    }
                }
            }
        }

        public MEFPForecastSource getSource()
        {
            return _source;
        }
    }

    /**
     * Calls {@link #runModel(MEFPFullModelParameters)} for a single forecast time and generates the output to an output
     * file under the provided base directory. Output is generated to stdout as part of this method.
     * 
     * @param parameters Parameters for which to generate hindcasts.
     * @param forecastTime The desired forecast time.
     * @param outputBaseDir Where to put the output.
     * @return The {@link File} generated containing the hindcast time series.
     */
    public File generateHindcast(final MEFPFullModelParameters parameters,
                                 final long forecastTime,
                                 final File outputBaseDir) throws Exception
    {
        setHindcastMode(true);

        //Put an entry in the source info
        populateSourceInformation(new TimeSeriesArrays(parameters.getHistoricalTimeSeries().get(0)));

        setForecastTime(forecastTime);
        final Calendar forecastCal = HCalendar.computeCalendarFromMilliseconds(getForecastTime());

        System.out.println("####>> Generating hindcast for " + parameters.getIdentifier().buildStringToDisplayInTree()
            + " for T0 of " + HCalendar.buildDateTimeTZStr(forecastCal) + " (free mem = "
            + Runtime.getRuntime().freeMemory() + ")...");
        final String outputFileNamePrefix = "hindcast." + parameters.getIdentifier().getLocationId() + "."
            + parameters.getIdentifier().getParameterId() + "." + HCalendar.buildDateStr(forecastCal, "CCYYMMDDhh");
        final File logFile = FileTools.newFile(outputBaseDir, forecastCal.get(Calendar.YEAR), outputFileNamePrefix
            + ".log");
        final String appenderName = LoggingTools.initializeLogFileLogger(logFile, "ohd.hseb", Level.DEBUG, false);

        final File outputFile = FileTools.newFile(outputBaseDir, forecastCal.get(Calendar.YEAR), outputFileNamePrefix
            + ".xml");

        try
        {
            //Initialize the random number generator appropriately.
            final Long seed = getTesting();
            if(seed != null)
            {
                LOG.info("Testing mode has been activated with seed " + seed
                    + "; initializing the random number generator with the seed.");
                ThreadedRandomTools.initializeRandomNumberGeneratorForTesting(seed);
            }
            else
            {
                ThreadedRandomTools.initializeRandomNumberGenerator();
            }
            
            final HStopWatch timer = new HStopWatch();
            final HStopWatch modelTimer = new HStopWatch();
            final TimeSeriesArrays results = runModel(parameters);
            modelTimer.stop();

            //missing value is -999 for hindcast XML files.  This is for the EVS system.
            TimeSeriesArraysTools.writeToFile(outputFile, results, -999f);

            System.out.println("####>> Successfully generated hindcast in " + (timer.getElapsedMillis() / 1000D)
                + " seconds (model time = " + modelTimer.getElapsedMillis() + ").");
        }
        catch(final Throwable e)
        {
            //            e.printStackTrace();
            LOG.error("Failed to generate hindcast for " + parameters.getIdentifier().buildStringToDisplayInTree()
                + " for time " + HCalendar.buildDateTimeTZStr(getForecastTime()) + ": " + e.getMessage());
            LoggingTools.outputStackTraceAsDebug(LOG, e);
            return null;
        }
        finally
        {
            LoggingTools.removeLoggingAppender(appenderName, "ohd.hseb");
        }
        return outputFile;
    }

    public static void main(final String[] args) throws IOException
    {
        HEFSModelAdapter.runAdapter(args, MEFPEnsembleGeneratorModelAdapter.class.getName());
    }
}
