package ohd.hseb.hefs.mefp.adapter;

import java.io.File;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.Properties;
import java.util.TimeZone;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.DefaultTimeSeriesHeader;
import nl.wldelft.util.timeseries.SimpleEquidistantTimeStep;
import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.pe.tools.TimeSeriesSorter;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;
import ohd.hseb.time.DateTime;
import ohd.hseb.util.fews.Diagnostics;
import ohd.hseb.util.fews.RunInfo;
import ohd.hseb.util.misc.HCalendar;

public class MEFPHindcaster
{

    private static final Logger LOG = LogManager.getLogger(MEFPHindcaster.class);
    
    /**
     * @param minTS
     * @param maxTS
     * @return A 6h FMAT time series based on 24h TFMN and TMFX time series. Diurnal computations are pulled from
     *         CHPS/Fortran.
     */
    public static TimeSeriesArray convertMaxMinToFMAT(final TimeSeriesArray minTS, final TimeSeriesArray maxTS)
    {
        final TimeSeriesArray fmatTS = TimeSeriesArrayTools.prepareTimeSeries(minTS);
        ((DefaultTimeSeriesHeader)fmatTS.getHeader()).setTimeStep(SimpleEquidistantTimeStep.getInstance(6 * HCalendar.MILLIS_IN_HR));
        ((DefaultTimeSeriesHeader)fmatTS.getHeader()).setParameterId("FMAT");

        long minMaxTime = minTS.getStartTime(); //maxTS.getStartTime()); -- Both values should be the same!

        double previousMinValue = Double.NaN;
        while(minMaxTime <= minTS.getEndTime()) //maxTS.getStartTime()); -- Both values should be the same!
        {
            final double minValue = TimeSeriesArrayTools.getValueByTime(minTS, minMaxTime);
            final double maxValue = TimeSeriesArrayTools.getValueByTime(maxTS, minMaxTime);
            final long startOfDay = minMaxTime - minTS.getHeader().getTimeStep().getStepMillis();

            if(!Double.isNaN(previousMinValue))
            {
                fmatTS.put(startOfDay + 1 * fmatTS.getHeader().getTimeStep().getStepMillis(),
                           (float)(0.6D * maxValue + 0.4D * previousMinValue));
                fmatTS.put(startOfDay + 2 * fmatTS.getHeader().getTimeStep().getStepMillis(), (float)(0.925D * maxValue
                    + 0.025D * previousMinValue + 0.05D * minValue));
            }
            else
            {
                fmatTS.put(startOfDay + 1 * fmatTS.getHeader().getTimeStep().getStepMillis(),
                           (float)(0.6D * maxValue + 0.4D * minValue));
                fmatTS.put(startOfDay + 2 * fmatTS.getHeader().getTimeStep().getStepMillis(), (float)(0.925D * maxValue
                    + 0.025D * minValue + 0.05D * minValue));
            }
            fmatTS.put(startOfDay + 3 * fmatTS.getHeader().getTimeStep().getStepMillis(),
                       (float)(0.33D * maxValue + 0.67D * minValue));
            fmatTS.put(startOfDay + 4 * fmatTS.getHeader().getTimeStep().getStepMillis(),
                       (float)(0.95D * minValue + 0.05D * maxValue));

            previousMinValue = minValue;
            minMaxTime += minTS.getHeader().getTimeStep().getStepMillis();
        }

        return fmatTS;
    }

    /**
     * File must have name hindcast.*.MAT.*.xml to be processed.<br>
     * <br>
     * This method was used before when a single run of the conversion was done after all hindcasts were generated.
     */
    @SuppressWarnings("unused")
    private static FileFilter makeTemperatureHindcastFileFilter()
    {
        return new FileFilter()
        {
            @Override
            public boolean accept(final File pathname)
            {
                final String fileName = pathname.getName();
                if(!fileName.startsWith("hindcast."))
                {
                    return false;
                }
                if(!fileName.contains(".MAT."))
                {
                    return false;
                }
                if(!fileName.endsWith(".xml"))
                {
                    return false;
                }
                return true;
            }
        };
    }

    /**
     * Converts the hindcast temperature file provided to a 6-hour data file calling
     * {@link #convertMaxMinToFMAT(TimeSeriesArray, TimeSeriesArray)}. It is meant to be run one time for each file to
     * convert.
     * 
     * @param file The output time series file to convert.
     */
    private static void convertTemperatureFileToFMAT(final File file) throws Exception
    {
        System.out.println("####>> Converting " + file.getAbsolutePath()
            + " containing 24-hour min/max data to 6-hour MAT data.");

        //Read the time series and parse them into min and max lists (two ensembles).
        final TimeSeriesArrays minMaxTS = TimeSeriesArraysTools.readFromFile(file);
        final List<TimeSeriesArray> minTS = TimeSeriesSorter.subListByParameterId(TimeSeriesArraysTools.convertTimeSeriesArraysToList(minMaxTS),
                                                                                  ParameterId.TFMN);
        final List<TimeSeriesArray> maxTS = TimeSeriesSorter.subListByParameterId(TimeSeriesArraysTools.convertTimeSeriesArraysToList(minMaxTS),
                                                                                  ParameterId.TFMX);

        //Populate the fmat list.
        final List<TimeSeriesArray> fmatTS = Lists.newArrayList();
        for(int i = 0; i < minTS.size(); i++)
        {
            fmatTS.add(convertMaxMinToFMAT(minTS.get(i), maxTS.get(i)));
        }

        //Output ensemble to file
        final File fmatFile = new File(file.getAbsolutePath().replace(".MAT.", ".FMAT.6h."));
        TimeSeriesArraysTools.writeToFile(fmatFile, fmatTS);
    }

    public static void main(final String[] args)
    {
        //XXX To run this, you need to right click on this class and select "Run As" >> "Java Application".  It will fail the first 
        //time it is run because you did not provide the four required arguments:
        if(args.length != 5)
        {
            LOG.error("Expected arguments: java -jar <jar file name> <parameter file name> <first year of run> <last year of run> <output directory> <run file properties>\n\n" + 
            "<parameter file name>: name of parameter file to use (relative or absolute name) for hindcasting.\n" +
            "<first year>: First year of hindcasts to generate (assumes Jan 1 of that year for a start date).\n" +
            "<last year>: Last year of hindcasts to generate (assumes Dec 31 of that year as the end date).\n" +
            "<output directory>: Directory to contain generated hindcasts.\n" +
            "<run file properties>: File to be parsed for run options, including the number of forecast days to include for each source,\n" +
            "    initial ensemble member year, last ensemble member year, etc.\n\n" +
            "Additionally, a forecastSourcesDefinition.xml *must* be in your working directory when you run the hindcaster.");
            System.exit(1);
        }
        final String parameterFileName = args[0];
        final String firstYearStr = args[1];
        final String lastYearStr = args[2];
        final String outputDir = args[3];
        final String propertyFileName = args[4];

        //Process the years
        int firstYear = -1;
        int lastYear = -1;
        try
        {
            firstYear = Integer.parseInt(firstYearStr);
        }
        catch(final NumberFormatException e)
        {
            LOG.error("Aborting Run: The first year argument, " + firstYearStr + ", is not an integer.");
            e.printStackTrace();
            System.exit(1);
        }
        try
        {
            lastYear = Integer.parseInt(lastYearStr);
        }
        catch(final NumberFormatException e)
        {
            LOG.error("Aborting Run: The last year argument, " + lastYearStr + ", is not an integer.");
            e.printStackTrace();
            System.exit(1);
        }

        //Compute Calendar objects to contain the first hindcast date and hindcast date based on the
        //just computed firstYear and lastYear.
        final Calendar firstT0 = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        firstT0.set(firstYear, GregorianCalendar.JANUARY, 1, 12, 0, 0);
        firstT0.set(Calendar.MILLISECOND, 0);
        final Calendar lastT0 = Calendar.getInstance(TimeZone.getTimeZone("GMT"));
        lastT0.set(lastYear, GregorianCalendar.DECEMBER, 31, 12, 0, 0);
        lastT0.set(Calendar.MILLISECOND, 0);

        //baseOutputDir is a File object that records the output directory location.  It is used for checking to make sure the directory
        //can be written to.
        final File baseOutputDir = new File(outputDir);
        if(!baseOutputDir.canWrite())
        {
            LOG.error("Aborting Run: The base output directory, " + baseOutputDir.getAbsolutePath()
                + ", does not exist or cannot be written to.");
            System.exit(1);
        }
        
        //Get the properties.
        final Properties properties = new Properties();
        try
        {
            properties.load(new FileReader(propertyFileName));
        }
        catch(final FileNotFoundException e1)
        {
            LOG.error("Aborting Run: " + e1.getMessage());
            e1.printStackTrace();
            System.exit(1);
        }
        catch(final IOException e1)
        {
            LOG.error("Aborting Run: " + e1.getMessage());
            e1.printStackTrace();
            System.exit(1);
        }

        //Read in the parameters.
        final File parameterFile = new File(parameterFileName);
        MEFPFullModelParameters parameters = null;
        try
        {
            parameters = MEFPTools.instantiateMEFPFullModelParametersForReading();
            parameters.readParametersTarArchive(parameterFile);
        }
        catch(final Exception e)
        {
            System.err.println("Failed to load parameters from file " + parameterFile.getAbsolutePath() + ": "
                + e.getMessage());
            System.exit(1);
        }

        //Used to store run-file properties for execution of MEFP ensemble generator.  See the MEFPEnsembleGeneratorModelAdapter for 
        //information on those run file properties.  Each run-file property is specified one at a time and uses the constants
        //at the top of MEFPEnsembleGeneratorModelAdapter in order to define the properties.
//        final Properties properties = new Properties();
//        properties.setProperty("rfc" + MEFPEnsembleGeneratorModelAdapter.NUMBER_OF_FORECAST_DAYS_SUFFIX, "0");
//        properties.setProperty("gfs" + MEFPEnsembleGeneratorModelAdapter.NUMBER_OF_FORECAST_DAYS_SUFFIX, "0");
//        properties.setProperty("gefs" + MEFPEnsembleGeneratorModelAdapter.NUMBER_OF_FORECAST_DAYS_SUFFIX, "15");
//        properties.setProperty("cfsv2" + MEFPEnsembleGeneratorModelAdapter.NUMBER_OF_FORECAST_DAYS_SUFFIX, "0");
//        properties.setProperty("climatology" + MEFPEnsembleGeneratorModelAdapter.NUMBER_OF_FORECAST_DAYS_SUFFIX, "0");
//        properties.setProperty("rfc" + MEFPEnsembleGeneratorModelAdapter.EPT_OPTION_SUFFIX, "true");
//        properties.setProperty("gfs" + MEFPEnsembleGeneratorModelAdapter.EPT_OPTION_SUFFIX, "true");
//        properties.setProperty("gefs" + MEFPEnsembleGeneratorModelAdapter.EPT_OPTION_SUFFIX, "true");
//        properties.setProperty("cfsv2" + MEFPEnsembleGeneratorModelAdapter.EPT_OPTION_SUFFIX, "true");
//        properties.setProperty("climatology" + MEFPEnsembleGeneratorModelAdapter.EPT_OPTION_SUFFIX, "true");
//        properties.setProperty(MEFPEnsembleGeneratorModelAdapter.COND_COEFF_VAR_MAX, "1.5");
//        properties.setProperty(MEFPEnsembleGeneratorModelAdapter.INITIAL_ENSEMBLE_YEAR, "1950");
//        properties.setProperty(MEFPEnsembleGeneratorModelAdapter.LAST_ENSEMBLE_YEAR, "2008");
//        properties.setProperty("eptUseStratifiedSampling", "false");

//        properties.setProperty("climatologyExcludeModulationEvents", "true");
//        properties.setProperty("climatologyExcludeEventsWithDurLessThan", "120");
//        properties.setProperty("climatologyExcludeEventsWithDurMoreThan", "120");

        final RunInfo runInfo = new RunInfo(new Diagnostics());
        runInfo.setProperties(properties);
        runInfo.setWorkDir(".");
        try
        {
            //These are being artificially set to fill in the information with info
            //that can be processed without error.  
            runInfo.setTime0Long(new DateTime("1990-01-01", "12:00:00"));
            runInfo.setRunStartTimeLong(new DateTime("1990-01-01", "18:00:00"));
            runInfo.setRunEndTimeLong(new DateTime("1990-01-10", "12:00:00"));
        }
        catch(final Exception e1)
        {
            //This should never happen.
            e1.printStackTrace();
            System.exit(1);
        }

        //This is where the hindcasts are generated.
        try
        {
            for(long forecastTime = firstT0.getTimeInMillis(); forecastTime <= lastT0.getTimeInMillis(); forecastTime += 24 * HCalendar.MILLIS_IN_HR)
            {
                MEFPEnsembleGeneratorModelAdapter adapter = new MEFPEnsembleGeneratorModelAdapter();
                adapter.loadModuleDataSetsBeforeExtractingRunTimeInformation(runInfo);
                adapter.extractRunInfo(runInfo);
                
//MEFP does not use this, so the MEFPHindcaster cannot see it...   adapter.loadModuleDataSetsAfterExtractingRunTimeInformation(runInfo);

                final File outputTSFile = adapter.generateHindcast(parameters, forecastTime, baseOutputDir);

                //For temperature, we need to convert the files.
                if((parameters.getIdentifier().isTemperatureDataType()) && (outputTSFile != null))
                {
                    System.out.println("####>> Converting 24h MAT (TFMN/TFMX) file to 6h FMAT files... ");
                    convertTemperatureFileToFMAT(outputTSFile);
                }
                adapter = null;
            }
        }
        catch(final Exception e)
        {
//            e.printStackTrace();
            System.out.println("Failed to run model: " + e.getMessage());
            System.exit(1);
        }
        System.exit(0);
    }
}
