package ohd.hseb.hefs.mefp.adapter;

import java.io.File;
import java.util.concurrent.Executors;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPPrecipitationSourceControlOptions;
import ohd.hseb.hefs.pe.estimation.GenericParameterEstimationRunner;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.util.misc.HStopWatch;

public class MEFPPECommandLineGEFSv12Estimator
{
    private static final Logger LOG = LogManager.getLogger(MEFPPECommandLineGEFSv12Estimator.class);

    public static void main(final String args[])
    {
        try
        {
        if (args.length != 3)
        {
            LOG.error("Incorrect command line syntax.  Expected syntax:\n\n" +
                      "java -jar <jar file name> <MEFPPE run area> <location> <alpha option>\n\n" +
                      "where the arguments after the jar file name are:\n\n"+
                      "<MEFPPE run area>: Directory location of the mefppeRunArea directory, usually [region_dir]/Models/hefs/mefppeRunArea.\n"+
                      "<location>: The location ID of the MEFP location within the MEFPPE.\n" +
                      "<alpha>: The alpha parameter.");
            System.exit(1);          
        }
        
        //Argument specifying the location of the MEFPPE run area against which the batch processor will execute.
        //That run area must be pre-populated with all necessary data and existing parameter files.
        //It must also be configured for parameter estimation of the GEFSv12.
        //This application will only be used to execute for one-source.
        final String mefppeRunAreaLocation = args[0];

        //Argument specifying the location
        final String location = args[1];
        
        //The alpha value.
        final float alpha = Float.parseFloat(args[2]);

        //Create the run info associated with that run area.
        MEFPParameterEstimatorRunInfo runInfo = null;
        runInfo = new MEFPParameterEstimatorRunInfo(Executors.newCachedThreadPool(),
                                                    new File(mefppeRunAreaLocation),
                                                    null);

        //Define the location identifier
        final LocationAndDataTypeIdentifier identifier = LocationAndDataTypeIdentifier.get(location,
                                                                                           ParameterId.MAP.name());

        //Find the parameter .tgz file and read in the parameters.
        final File startingParmFile = runInfo.getEstimatedParametersFileHandler().getPrimaryParameterFile(identifier);
        if(!startingParmFile.exists())
        {
            throw new IllegalArgumentException("Cannot find the expected parameter file, "
                + startingParmFile.getAbsolutePath());
        }
        if(!startingParmFile.canRead() || !startingParmFile.canWrite())
        {
            throw new IllegalArgumentException("Cannot either read or write to the expected parameter file, "
                + startingParmFile.getAbsolutePath());
        }
        final FullModelParameters parameters = runInfo.getEstimatedParametersFileHandler().readModelParameters(identifier);
        
        //Get the MEFP estimation control options from the parameter file.  Set the alpha option for the GEFSv12 forecast source.
        //The forecast source is identified through a method available in runInfo.
        final MEFPEstimationControlOptions options = runInfo.getEstimationControlOptions(identifier);
        ((MEFPPrecipitationSourceControlOptions)options.getSourceControlOptions(runInfo.getForecastSource("GEFSv12"))).setEPTAlpha(alpha);

        //THE ABOVE IS SPECIFICALLY DESIGNED FOR ONE-SOURCE GENERATION.  THE RULES CHANGE AND ARE SIMPLIFIED FOR ALL SOURCES.
        //Specifically, you can just prep empty parameters for based on the runInfo instead of loading the existing and
        //setting the alpha option.  In that case, the base options in run info will need to be set to use the desired alpha.  
        //I'm not sure how to do so at this time.  See the comment for GenericParameterEstimationRunner execute method for more info.
        
        //Estimate parameters for the GEFSv12 source, only.  This will backup the existing parameters and write new ones.
        //The options used will match those specified in the options, above, with the alpha set.
        LOG.info("Estimating parameters for the GEFSv12 source and updating parameters file " + startingParmFile);
        final HStopWatch timer = new HStopWatch();
        final GenericParameterEstimationRunner estRunner = new GenericParameterEstimationRunner(identifier, runInfo, -1, true, runInfo.getForecastSource("GEFSv12"));
        estRunner.execute(parameters);
        LOG.info("Estimation of parameters is completed in " + timer.getElapsedMillis() + " milliseconds.");
        }
        catch (final Throwable t)
        {
            LOG.error("RUN FAILED!: " + t.getMessage());
            t.printStackTrace();
            System.exit(1);
        }
        System.exit(0);
    }
}
