package ohd.hseb.hefs.mefp.adapter;

import java.util.Calendar;

import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLVariable;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.SegmentedLine;

/**
 * An {@link XMLVariable} for handling the memberIndexYear run file property of the ensemble model adapter. The
 * {@link #readVariable(String)} method can process the constants {@link #STANDARD_HYDROLOGIC_WATER_YEAR} and
 * {@value #CALENDAR_YEAR}, as well as strings of the format [ccyy]-[mm]-[dd]; ccyy can be replaced a dash: --mm-dd. The
 * {@link XMLVariable} wraps a {@link Calendar} object.
 * 
 * @author hankherr
 */
public class MemberIndexYearXMLVariable extends XMLVariable<Calendar>
{
    public static final String STANDARD_HYDROLOGIC_WATER_YEAR = "standardHydroWaterYear";
    public static final String CALENDAR_YEAR = "calendarYear";

    protected MemberIndexYearXMLVariable(final String tag)
    {
        super(tag);
    }

    protected MemberIndexYearXMLVariable(final String tag, final Calendar value)
    {
        super(tag, value);
    }

    @Override
    protected Calendar readVariable(final String text) throws XMLReaderException
    {
        if(text.equalsIgnoreCase(STANDARD_HYDROLOGIC_WATER_YEAR))
        {
            return TimeSeriesEnsemble.STANDARD_HISTORICAL_WATER_YEAR;
        }
        else if(text.equalsIgnoreCase(CALENDAR_YEAR))
        {
            return null;
        }
        else
        {
            final String usedStr = text.replaceFirst("-", "1990");
            final SegmentedLine segLine = new SegmentedLine(usedStr, "-", SegmentedLine.MODE_ALLOW_EMPTY_SEGS);
            if(segLine.getNumberOfSegments() != 3)
            {
                throw new IllegalArgumentException("Expected three fields in word defining member indexing year: year-mon-day.  "
                    + "The year may be a '-' (ex: --mon-day).  Found "
                    + segLine.getNumberOfSegments()
                    + " fields in '"
                    + text + "'.");
            }
            final Calendar cal = HCalendar.processDate(segLine.getSegment(0) + "-" + segLine.getSegment(1) + "-"
                + segLine.getSegment(2) + " 00:00:00 GMT");

            //If the cal is for Jan 1, return null as if 'calendarYear' were provided.
            if((cal.get(Calendar.MONTH) == 0) && (cal.get(Calendar.DAY_OF_MONTH) == 1))
            {
                return null; //Standard calendar year is used.
            }
            return cal;
        }
    }

}
