package ohd.hseb.hefs.mefp.adapter;

import java.io.File;
import java.io.IOException;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.base.Strings;
import com.google.common.collect.Lists;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPSourceModelParameters;
import ohd.hseb.hefs.mefp.sources.gefs.GEFSDataHandler;
import ohd.hseb.hefs.mefp.sources.gefs.GEFSForecastSource;
import ohd.hseb.hefs.mefp.tools.MEFPTools;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.utils.tools.ParameterId;

/**
 * Code that will, hopefully, update an MEFP parameter so that it includes events updated to the latest GEFS reforecast
 * files without actually modifying parameter values.
 * 
 * @author Hank.Herr
 */
public class NewGEFSParameterFileUpdater
{
    private static final Logger LOG = LogManager.getLogger(NewGEFSParameterFileUpdater.class);

    public static void main(final String[] args) throws IOException
    {
        //The GEFS forecast source.
        final GEFSForecastSource gefsSource = new GEFSForecastSource();

        //Loads the parameters.
        final MEFPFullModelParameters existingParameters = MEFPTools.instantiateMEFPFullModelParametersForReading();
        try
        {
            existingParameters.readParametersTarArchive(new File(args[1]));
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Unexpected exception reading parameter file: " + e.getMessage());
            return;
        }

        //Prepare the data handler and load prepared time series for all data types based on the identifier.
        GEFSDataHandler dataHandler = null;
        try
        {
            dataHandler = new GEFSDataHandler(new File(args[0]));
            dataHandler.loadPreparedTimeSeries(Lists.newArrayList(existingParameters.getIdentifier()));
            if(dataHandler.getAllLoadedForecastTimeSeries().isEmpty())
            {
                LOG.error("No reforecast time series were read in so events cannot be computed.  Aborting.");
                return;
            }
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Unexpected exception preparing data handler and loading data: " + e.getMessage());
            return;
        }

        //Ger the source event values for storing the new events.
        final MEFPSourceModelParameters srcModelParms = existingParameters.getSourceModelParameters(gefsSource);

        //=======================================
        //Precipitation processing...
        //=======================================
        if(existingParameters.getIdentifier().isPrecipitationDataType())
        {
            LOG.info("Recomputing event values for precipitation....");
            //Get the values instance that will be updated.
            final SourceCanonicalEventValues values = srcModelParms.getPrecipitationSourceEventValues();

            //Compute the events and store in values.
            values.clear();
            values.computeCanonicalEventValues(existingParameters.getIdentifier(),
                                               ParameterId.FMAP,
                                               existingParameters.getSingleHistoricalTimeSeries(),
                                               dataHandler);
            if(values.getNumberOfForecastTimesWithValues() == 0)
            {
                System.err.println("No canonical events were found; perhaps the file wasn't found.  Aborting...");
                return;
            }
            LOG.info("Computing done!");
        }

        //=======================================
        //Temperature processing...
        //=======================================
        else
        {
            //The historical ts will include one for min and one for max.  Loop for each one.
            for(final TimeSeriesArray histTS: existingParameters.getHistoricalTimeSeries())
            {
                //Get the values instance that will be updated.
                LOG.info("Recomputing event values for " + histTS.getHeader().getParameterId() + "...");

                //Parameter Id must be the forecast version of the historical parameter.
                //Also identify the canonical event values object to be cleared and populated.
                ParameterId usedId = ParameterId.TFMX;
                SourceCanonicalEventValues values = srcModelParms.getTMAXSourceEventValues();
                if(ParameterId.of(histTS.getHeader()).isMin())
                {
                    usedId = ParameterId.TFMN;
                    values = srcModelParms.getTMINSourceEventValues();
                }

                //Compute the events and store in values.
                values.clear();
                values.computeCanonicalEventValues(existingParameters.getIdentifier(), usedId, histTS, dataHandler);
                if(values.getNumberOfForecastTimesWithValues() == 0)
                {
                    System.err.println("No canonical events were found; perhaps the file wasn't found.  Aborting...");
                    return;
                }
                LOG.info("Computing done!");
            }
        }

        //Removing any existing file and writing out the file.
        try
        {
            final File inFile = new File(args[1]);

            //Determine the base directory for output.
            String baseDir = ".";
            if(!Strings.isNullOrEmpty(inFile.getParent())) //Could occur if directory not included in file name to process.
            {
                baseDir = inFile.getParent();
            }
            final File outFile = new File(baseDir + "/updated." + inFile.getName());

            //Delete and write.
            LOG.info("Removing file " + outFile.getAbsolutePath()
                + " if it exists and writing updated parameter file...");
            outFile.delete();
            existingParameters.writeParametersTarArchive(outFile, true);
            LOG.info("Writing done!");
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            LOG.error("Unexpected exception writing updated MEFP parameter file: " + e.getMessage());
            return;
        }
    }

}
