package ohd.hseb.hefs.mefp.models;

import java.util.List;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.estimation.options.GenericCompositeControlOption;
import ohd.hseb.hefs.pe.estimation.options.IntegerEstimationOption;
import ohd.hseb.hefs.pe.model.OneTypeParameterValues;
import ohd.hseb.hefs.utils.tools.NumberTools;

/**
 * Class specifies parameters present for both MEFP parameter estimator models, precip and temp.
 * 
 * @author hank.herr
 */
public abstract class MEFPBaseModelControlOptions
extends
    GenericCompositeControlOption
{
    protected static final int NUM_PARAMETERS = 4;

    private IntegerEstimationOption _minobs = null;
    private IntegerEstimationOption _minwin = null;
    private IntegerEstimationOption _maxwin = null;
    private IntegerEstimationOption _parInterval = null;

    protected MEFPBaseModelControlOptions(final String xmlTag,
                                          final Iterable<? extends ControlOption> additionalParameters)
    {
        super(xmlTag,
              Iterables.concat(createDefaultParameters(),
                               additionalParameters));
        mapAttributes();
    }

    /**
     * Maps attributes so that they can be accessed via local attributes instead of via the list. Override if you extend
     * this class but add extra attributes. By default it maps {@link #_minobs}, {@link #_minwin}, and {@link #_maxwin}.
     * Any overridden should call super.mapAttributes() to map these.
     */
    protected void mapAttributes()
    {
        _minobs = (IntegerEstimationOption)_parameters.get(0);
        _minwin = (IntegerEstimationOption)_parameters.get(1);
        _maxwin = (IntegerEstimationOption)_parameters.get(2);
        _parInterval = (IntegerEstimationOption)_parameters.get(3);
    }

    private static final List<? extends ControlOption> createDefaultParameters()
    {
        return Lists.newArrayList(new IntegerEstimationOption("minimumNumberOfObservationsRequired",
                                                              "Minimum Required Observations for Param. Est.:", //Fortran: minobs
                                                              0),
                                  new IntegerEstimationOption("minimumWidthOfDataWindowInDays",
                                                              "Minimum Width of Data Window in Days:", //Fortran: minwin
                                                              31),
                                  new IntegerEstimationOption("maximumWidthOfDataWindowInDays",
                                                              "Maximum Width of Data Window in Days:", //Fortran: maxwin
                                                              61),
                                  new IntegerEstimationOption("intervalBetweenDaysForParameterComputation",
                                                              "Interval between days for param. computation:", //Fortran: par_interval
                                                              5,
                                                              1,
                                                              0,
                                                              366));

    }

    public int getMinimumNumberOfObservationsRequired()
    {
        return _minobs.get();
    }

    public void setMinimumNumberOfObservationsRequired(final int minimumNumberOfObservationsRequired)
    {
        _minobs.set(minimumNumberOfObservationsRequired);
    }

    public int getMinimumWidthOfDataWindowInDays()
    {
        return _minwin.get();
    }

    public void setMinimumWidthOfDataWindowInDays(final int minimumWidthOfDataWindowInDays)
    {
        _minwin.set(minimumWidthOfDataWindowInDays);
    }

    public int getMaximumWidthOfDataWindowInDays()
    {
        return _maxwin.get();
    }

    public void setMaximumWidthOfDataWindowInDays(final int maximumWidthOfDataWindowInDays)
    {
        _maxwin.set(maximumWidthOfDataWindowInDays);
    }

    public int getIntervalBetweenDaysForParameterComputation()
    {
        return _parInterval.get();
    }

    public void setIntervalBetweenDaysForParameterComputation(final int intervalBetweenDaysForParameterComputation)
    {
        _parInterval.set(intervalBetweenDaysForParameterComputation);
    }

    /**
     * Computation is done here because it uses {@link #getIntervalBetweenDaysForParameterComputation()}.
     * 
     * @return The days of the year for which to estimate parameters or for which parameters have been estimated. The
     *         returned numbers correspond to the first index in the {@link OneTypeParameterValues#getValue(int, int)}
     *         method and lists the possible values of the first index for which parameters are directly computed (other
     *         values are interpolated somehow).
     */
    public List<Integer> generateDaysOfTheYearForWhichToEstimateParameters()
    {
        return NumberTools.generateNumberSequence(1,
                                                  getIntervalBetweenDaysForParameterComputation(),
                                                  365);
    }

    /**
     * TODO -- This does not belong here. It is only called one time from {@link MEFPFullModelParameters}, so move it.
     * Computation is done here because it uses {@link #getIntervalBetweenDaysForParameterComputation()}.
     * 
     * @param desiredDay The day for which to find the nearest with estimated parameters.
     * @return The nearest day to the desired day based on a mathematical computation; it does not build the list and
     *         search it.
     */
    public int findNearestComputationalDay(final int desiredDay)
    {
        final int value = (int)Math.round((desiredDay - 1.0d)
            / getIntervalBetweenDaysForParameterComputation())
            * getIntervalBetweenDaysForParameterComputation() + 1;
        if(value > 365)
        {
            return 1;
        }
        return value;
    }

    @Override
    public void copyFrom(final ControlOption base)
    {
        super.copyFrom(base);
        mapAttributes();
    }
}
