package ohd.hseb.hefs.mefp.models;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.collect.Maps;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import nl.wldelft.util.timeseries.TimeSeriesArrays;
import ohd.hseb.hefs.mefp.adapter.MEFPEnsembleGeneratorModelAdapter;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventRestrictor;
import ohd.hseb.hefs.mefp.tools.canonical.SourceCanonicalEventValues;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tools.SamplingTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArrayTools;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesEnsemble;
import ohd.hseb.util.misc.HCalendar;
import ohd.hseb.util.misc.HNumber;

/**
 * Super class of the MEFP ensemble generation models. It provides tools to store the estimated model parameters used to
 * compute ensembles, the number of forecast days to apply each source to, the single-valued forecast time series, the
 * generated ensemble, and a flag indicating if climatology should be resampled if it is to be used to extend the
 * generated ensembles. It also contains a general {@link #generateEnsemble(long, int, int)} method that should be
 * applicable to both precipitation and temperature data.<br>
 * <br>
 * When hindcasting, this will attempt to acquire already computed canonical events instead of using the time series
 * stored in {@link #_sourceToForecastTimeSeriesMap}. For that to work, the method
 * {@link #setCurrentSourceCanonicalEventValues(MEFPForecastSource, SourceCanonicalEventValues)} must be called to make
 * the event values available. If the desired hindcast time (i.e., historical forecast time) does not have event values
 * already computed, it will use the forecast time series stored in {@link #_sourceToForecastTimeSeriesMap}. It is up to
 * the caller to populate that map accordingly. As such, the caller should call
 * {@link #areHindcastCanonicalEventsAlreadyComputed(MEFPForecastSource, long)} to determine if it needs to specify the
 * historical forecast time series for a source.
 * 
 * @author hankherr
 */
public abstract class MEFPEnsembleGeneratorModel
{

    /**
     * Enumeration specifying options for how to handle missing canonical event values.
     */
    public static enum BehaviorIfEventMissing
    {
        errorOut, fillMissing, fillClimatology, fillNextSource;
    }

    private static final Logger LOG = LogManager.getLogger(MEFPEnsembleGeneratorModel.class);

    private MEFPFullModelParameters _modelParameters;
    private boolean _precipitationFlag;
    private final LinkedHashMap<MEFPForecastSource, Integer> _sourceToNumberOfForecastDaysMap = Maps.newLinkedHashMap();
    private final ListMultimap<MEFPForecastSource, TimeSeriesArray> _sourceToForecastTimeSeriesMap =
                                                                                                   ArrayListMultimap.create();
    private final LinkedHashMap<MEFPForecastSource, TimeSeriesEnsemble> _sourceToGeneratedEnsembleMap =
                                                                                                      Maps.newLinkedHashMap();
    private final LinkedHashMap<MEFPForecastSource, SourceCanonicalEventValues> _sourceToPrecomputedEventValues =
                                                                                                                Maps.newLinkedHashMap();
    private MEFPForecastSource _climatologySource;
    private int _climatologyNumberOfDays;
    private boolean _hindcastMode = false;
    private BehaviorIfEventMissing _behaviorIfEventMissing;
    private Calendar _memberIndexingCal = null;

    private CanonicalEventRestrictor _canonicalEventRestrictor = null;

    /**
     * Indicates if stratified sampling should be used for the EPT model. Default is false.
     */
    private SamplingTools.SamplingTechnique _stratifiedSampling;

    /**
     * Tracks if a source specific warning has been generated for an event not computing. This flag is reset for each
     * source processed.
     */
    private boolean _alreadyOutputMissingEventMessage = false;

    /**
     * Call before populating the forecast source maps and generating an ensemble if multiple runs are made for
     * different data types using the same instance of this model.
     */
    public void clearSourceMaps()
    {
        _sourceToForecastTimeSeriesMap.clear();
        _sourceToGeneratedEnsembleMap.clear();
        _sourceToNumberOfForecastDaysMap.clear();
        _sourceToPrecomputedEventValues.clear();
        _climatologyNumberOfDays = 0;
    }

    /**
     * Generic method should work for both precipitation and temperature data types.
     * 
     * @param forecastTime The T0 for which to generate a forecasts ensemble.
     * @param ensembleSize The number of members to generate.
     * @param firstMemberHistoricalWaterYear The first members's historical water year. All other members use years
     *            incremented from this starting point.
     * @return A {@link TimeSeriesEnsemble} specifying the forecast ensemble.
     * @throws Exception For many reasons.
     */
    public TimeSeriesEnsemble generateEnsemble(final long forecastTime,
                                               final int ensembleSize,
                                               final int firstMemberHistoricalWaterYear) throws Exception
    {
        LOG.info("Executing ensemble generator model for forecast time " + HCalendar.buildDateTimeTZStr(forecastTime)
            + " generating " + ensembleSize + " many members starting with year " + firstMemberHistoricalWaterYear
            + ".");

        //Update the number of forecast days for all sources based on parameters available.  This will generate log messages as appropriate.
        LOG.info("Checking the number of forecast days specified for all sources and changing them if necessary...");
        updateNumberOfDaysForAllSources();

        initializeDataTypeSpecificParameters();

        //Compute the day of the year for the forecast time. 
        final int parametersDayOfYear =
                                      this.getModelParameters()
                                          .findNearestComputationalDay(HEFSTools.computeDayOfYearWithLeapDayBeingMar1(HCalendar.computeCalendarFromMilliseconds(forecastTime)));

        //Determine the number of desired days (largest number of forecast lead days for any source).
        final int desiredForecastNumberOfDays = getLargestNumberOfForecastDaysForWhichToApplyModel();

        //No missing values can be output.  To ensure this, we make sure that the climatology number of days is equal to the largest
        //number of days.  If the number of days for climatology was 0, meaning it was not be used directly (only to fill in missing), 
        //then we also ensure that straight historical data is used, not resampled data, to fill in those missings.
        //
        //Its not clear to be that this serves any purpose; see the XXX note below in the combine method.  
        boolean useRawClimatologyBase = false;
        if(getSourceNumberOfForecastDays(getClimatologyForecastSource()) == 0) //User implied to not use it. Turn off resampling.
        {
            useRawClimatologyBase = true;
        }
        setSourceNumberOfForecastDays(getClimatologyForecastSource(), desiredForecastNumberOfDays); //We always use it for missings!

        //Generate a climatology based-ensemble to start with.  The total length should be the largest of the forecast source
        //number of forecast days computed above.  Be sure to account for the ensemble size and first member historical 
        //water year.  This ts will be copied for the various forecast sources.
        TimeSeriesEnsemble baseEnsembleFromClimatology = null;
        try
        {
            baseEnsembleFromClimatology = new TimeSeriesEnsemble(getSingleHistoricalTimeSeries(),
                                                                 forecastTime,
                                                                 desiredForecastNumberOfDays
                                                                     * getNumberOfEventValuesPerDay(),
                                                                 firstMemberHistoricalWaterYear,
                                                                 firstMemberHistoricalWaterYear + ensembleSize - 1,
                                                                 _memberIndexingCal,
                                                                 "MEFP");
            baseEnsembleFromClimatology.setParameterId(determineForecastEnsembleParameterId(baseEnsembleFromClimatology.get(0)
                                                                                                                       .getHeader()
                                                                                                                       .getParameterId()).toString());
        }
        catch(final Exception e)
        {
            e.printStackTrace();
            throw new Exception("Unable to construct base climatology ensemble using historical data; aborting ensemble generation: "
                + e.getMessage());
        }

        //For each forecast source, generate the forecast ensemble.
        for(final MEFPForecastSource source: this.getNonClimatologyForecastSources())
        {
            _alreadyOutputMissingEventMessage = false;
            generateSourceForecastEnsemble(source, forecastTime, parametersDayOfYear, baseEnsembleFromClimatology);
        }

        //Check if the user wants to use resampled climatology to extend the ensemble.  Note that if not, the climatology based-ensemble
        //created before already contains the climatology extendsion.  If yes, then...
        if(this.getSourceNumberOfForecastDays(getClimatologyForecastSource()) < desiredForecastNumberOfDays)
        {
            LOG.info("The forecast number of days, " + desiredForecastNumberOfDays
                + ", exceeds the climatology number of days, "
                + getSourceNumberOfForecastDays(getClimatologyForecastSource())
                + "; the forecast ensmeble will not be extended via climatology.");
            addSourceGeneratedEnsemble(getClimatologyForecastSource(), baseEnsembleFromClimatology);
        }
        else
        {
            //The flag useRawClimatologyBase is only true if the number of forecast days for climatology
            //was set to 0.  In that case, we need to trigger the else part of this if, which puts the base
            //ensemble draw from climatology into the source generated ensemble map.  However, if the flag
            //is false, then the number of forecast days for climatology was non zero, so resampled climatology
            //is used.
            if(!useRawClimatologyBase) //Resampled climatology was used.
            {
                LOG.info("Computing resampled climatology ensemble...");
                generateSourceForecastEnsemble(getClimatologyForecastSource(),
                                               forecastTime,
                                               parametersDayOfYear,
                                               baseEnsembleFromClimatology);
            }
            //Otherwise there is no resample climatology, so just record the baseEnsembleFromClimatology for 
            //the climatology source.  This MUST be recorded since climatology is the starting point for the
            //combine, even if number of days for climate is 0.
            else
            {
                addSourceGeneratedEnsemble(getClimatologyForecastSource(), baseEnsembleFromClimatology);
            }
        }

        //Combine the base ensembles for the forecast sources and climatology.  The overall climatology ensemble is looped in order and
        //the values are pulled from the forecast source adjusted value if its number of forecast days encompasses the working indedx.
        //The value can only be pulled from one forecast source.  Hence, the ensemble looks like 
        //<-- RFC --><-- GFS --><-- CFSv2 --><-- climate -->.
        final TimeSeriesEnsemble tss = combineGeneratedEnsembles();

        //Round all values... Make this a utility?
        roundValues(tss);

        LOG.info("Successfully generated ensembles.");
        return tss;
    }

    /**
     * @param baseEnsemble The base ensemble used to initialize the applier.
     * @return An instance of {@link SchaakeShuffleApplier} constructed in a data type specific way.
     */
    protected abstract SchaakeShuffleApplier constructSchaakeShuffleApplier(TimeSeriesEnsemble baseEnsemble);

    /**
     * Generates an ensemble for a single forecast source.
     * 
     * @param source The source.
     * @param forecastTime The forecast time of the ensemble to generate.
     * @param parametersDayOfYear The day of the year for which parameters were acquired, computed based on the forecast
     *            time and passed in here.
     * @param climatologyEnsemble The base climatology ensemble that will be modified to generate the desired ensemble.
     * @throws Exception For various reasons.
     * @return False if the source is to be used, but a forecast ensemble could not be created due to problems computing
     *         the forecast ensembles for each event and using the Schaake shuffle.
     */
    private void generateSourceForecastEnsemble(final MEFPForecastSource source,
                                                final long forecastTime,
                                                final int parametersDayOfYear,
                                                final TimeSeriesEnsemble climatologyEnsemble) throws Exception
    {
        //Used below
        final int periodTimeStepInHours = getCanonicalEventPeriodStepSizeInHours();

        //The source is not applied if the number of days is 0.
        final int sourceNumberOfDays = getSourceNumberOfForecastDays(source);
        if(sourceNumberOfDays == 0)
        {
            return;
        }

        //Ignore the source if no parameters were estimated for the source. 
        if(!getModelParameters().getSourceModelParameters(source).wereParametersEstimatedForSource())
        {
            LOG.warn("SHOULDN'T SEE THIS MESSAGE (problem should have been caught earlier): The source "
                + source.getName() + " has a number of forecast days of " + sourceNumberOfDays
                + " but the no parameters were estimated for the source (or parameter estimation failed; check MEFPPE). "
                + "The source will be ignored (number of forecast lead days set to 0).");
            ignoreNonClimatologyForecastSource(source);
            return;
        }

        LOG.info("Generating forecast ensemble for T0 " + HCalendar.buildDateTimeTZStr(forecastTime)
            + " for the source " + source.getName() + " which has a number of forecast days set to "
            + sourceNumberOfDays + "...");

        //Get the calculator to use.  It will be called repeatedly for each canonical event.
        final ForecastEnsembleCalculator ensCalculator = instantiateCalculator(source, parametersDayOfYear);

        boolean wereAnyEventsSuccessfullyApplied = false;
        try
        {
            //Identify the number of canonical events that apply, given the number of forecast days for which to use the source.
            //Acquire those events.  The event numbers start counting at 0, not 1 (like Fortran), but only for this purpose.
            //Note the sortedEventList cannot be a CanonicalEventList, because that is an already SortedCollection, which uses the 
            //default comparator and cannot be sorted differently.
            final List<CanonicalEvent> sortedEventList = getApplicableEvents(source, sourceNumberOfDays);

            //Sort the events by Rho.  The indices in the event list will store the original order.  Note that those indices will match
            //that used in the full event list, which drives the secondary index used to acquire parameter values.  Counting starts at 0
            //(in fortran it starts at 1).  The sort is only done for non-climatology sources (climatology has no Rho value).
            if(!source.isClimatologySource())
            {
                sortCanonicalEventsByRho(source, parametersDayOfYear, sortedEventList);
            }

            //Generate a base ensemble from the climatology ensemble generated previously.  Note that the start time and
            //end time passed to the constructor are computed such that it could apply to temperature as well, given the right constants.
            final TimeSeriesEnsemble baseEnsemble = new TimeSeriesEnsemble(climatologyEnsemble,
                                                                           forecastTime,
                                                                           forecastTime + periodTimeStepInHours
                                                                               * HCalendar.MILLIS_IN_HR,
                                                                           forecastTime + sourceNumberOfDays
                                                                               * getNumberOfEventValuesPerDay()
                                                                               * periodTimeStepInHours
                                                                               * HCalendar.MILLIS_IN_HR);
            for(int tsindx = 0; tsindx < baseEnsemble.size(); tsindx++)
            {
                if(TimeSeriesArrayTools.containsMissingValues(baseEnsemble.get(tsindx)))
                    throw (new Exception("Base ensmeble constructed as a copy of the raw climatology ensemble "
                        + "(which was constructed from historical data) contains missing values; "
                        + "does the historical time series in the parameters tarfile include missing data?"));
            }

            addSourceGeneratedEnsemble(source, baseEnsemble); //putting it in storage

            //Initialize the Schaake shuffle applier using the base ensemble for this source.
            final SchaakeShuffleApplier shuffleApplier = constructSchaakeShuffleApplier(baseEnsemble);

            //For each canonical event and its forecast value...
            for(final CanonicalEvent event: sortedEventList)
            {
                //Compute/acquire the forecast canonical event value.
                double canonicalEventValue = -99.0D; //This value is used for climatology in Fortran
                if(!source.isClimatologySource())
                {

                    //If either in operational mode or, if we are in hindcast mode appropriate event values cannot be found, then use the time
                    //series added for the source.  In the later case, the thing calling this model must have done the work of loading and prepping
                    //the reforecast time series.  If the time series is not available for the source, an exception is thrown by computeEvent.
                    if(!_hindcastMode || !areHindcastCanonicalEventsAlreadyComputed(source, forecastTime))
                    {
                        canonicalEventValue = event.computeEvent(getSourceForecastTimeSeries(source),
                                                                 forecastTime,
                                                                 periodTimeStepInHours,
                                                                 true, //Always return if missings are found.  Skipping is not an option.
                                                                 true);
                    }
                    //If hindcasting and event values are available, use that event value directly and use the forecast time as is.  
                    //NOTE: Lag adjustment used to be included here, but then decided to store the canonical event values by 
                    //computational time, which represents a T0 that should correspond to the T0 used here (i.e., the lag is 
                    //already accounted for in the computational time).
                    else
                    {
                        canonicalEventValue = getCurrentSourceCanonicalEventValues(source).getForecastValues()
                                                                                          .getEventValue(event,
                                                                                                         forecastTime);
                    }
                    LOG.debug("Generating and applying forecast ensemble for source " + source.getName() + " and event "
                        + event.toString() + " which has value " + canonicalEventValue + " and correlation parameter "
                        + getRhoValue(source, parametersDayOfYear, event)
                        + "(-99 is always used for resampled climatology).");
                }
                else
                {
                    LOG.debug("Generating and applying forecast ensemble for source " + source.getName() + " and event "
                        + event.toString() + " which has value " + canonicalEventValue + ".");
                }

                //Behave based on the new property behaviorIfEventMissing.  This will either except, fillNaNs for the 
                //ensemble, or do nothing (using raw climatology).  
                if(Double.isNaN(canonicalEventValue))
                {
                    if(_modelParameters.getAlgorithmModelParameters().getModulationCanonicalEvents().contains(event))
                    {
                        LOG.debug("The canonical event value for source " + source.getName() + " and event "
                            + event.toString()
                            + " is missing (NaN); since it is a modulation event, it will be skipped.");
                    }
                    else
                    {
                        processMissingEvent(source, event, baseEnsemble, forecastTime, periodTimeStepInHours);
                    }
                }
                else
                {
                    //Process as usual since the value is non-missing.
                    //Check and modify the full parameter values based on limits of the parameters if necessary.
                    checkAndModifyFullModelParameters(source, event, parametersDayOfYear);

                    //Generate a forecast ensemble using the calculator.
                    final double[] forecastEventEnsemble;
                    try
                    {
                        forecastEventEnsemble = ensCalculator.calculateForecastEnsemble(event,
                                                                                        canonicalEventValue,
                                                                                        climatologyEnsemble.size());

                        //Use the SchaakeShuffleApplier to apply the forecast ensemble to the base ensemble for this source.
                        shuffleApplier.applySchaakeShuffle(forecastEventEnsemble, event);

                        wereAnyEventsSuccessfullyApplied = true;
                    }
                    catch(final Exception t)
                    {
                        LOG.debug("Unable to generate forecast ensemble for source " + source.getName()
                            + ", forecast time " + HCalendar.buildDateTimeStr(forecastTime) + ", parameter day of year "
                            + parametersDayOfYear + ", event " + event.toString() + " (event will be skipped):"
                            + t.getMessage());
                    }
                }
            }
        }
        //The only thing that should be caught here are throwables, such as illegal arg exception, and so on.
        catch(final Throwable t)
        {
            final Exception e2 = new Exception("Unable to generate forecast ensemble for source " + source.getName()
                + ", forecast time " + HCalendar.buildDateTimeStr(forecastTime) + ", and parameter day of year "
                + parametersDayOfYear + ":" + t.getMessage());
            e2.setStackTrace(t.getStackTrace());
            throw e2;
        }

        if(wereAnyEventsSuccessfullyApplied)
        {
            LOG.info("Successfully generated forecast ensemble for source " + source.getName() + ".");
        }
        else
        {
            LOG.warn("The source " + source.getName()
                + " did not yield a forecast ensemble due to problems described in earlier log messages. The source will be ignored (number of forecast lead days set to 0).");
            ignoreNonClimatologyForecastSource(source);
            removeSourceGeneratedEnsemble(source);
        }

        //TESTING!!!
        //        TimeSeriesArraysTools.writeToFile(new File("testdata/precipitationEnsembleGenerationModel/testing."
        //            + source.getClass().getSimpleName() + ".xml"), this.getSourceGeneratedEnsemble(source));
    }

    /**
     * Called at the beginning of {@link #generateEnsemble(long, int, int)} after determining the number of forecast
     * days. At this point, data type specific (subclass specific) parameters can be initialized. Override if such
     * parameters exist and need to be initialized. By default, this does nothing.
     */
    protected void initializeDataTypeSpecificParameters()
    {
    }

    /**
     * @return The step size for the canonical event in hours.
     */
    protected abstract int getCanonicalEventPeriodStepSizeInHours();

    /**
     * @param source The source for which the ensembles are being generated.
     * @param dayOfYear The target day of the year.
     * @return A {@link ForecastEnsembleCalculator} to use to generate forecast ensembles. Any data type specific checks
     *         must be done in here.
     * @throws Exception
     */
    protected abstract ForecastEnsembleCalculator instantiateCalculator(final MEFPForecastSource source,
                                                                        final int dayOfYear) throws Exception;

    /**
     * This is called for each event for which a forecast ensemble is being generated. At this point, for the source,
     * event, and day of the year, the parameters can be checked and modified if necessary based on parameter
     * limitations; for example the conditional coefficient of variation has a maximum for precipitation. This does
     * nothing by default.
     * 
     * @param source The source for which a forecast ensemble is about to be generated.
     * @param event The canonical event.
     * @param dayOfYear The day of the year.
     */
    protected void checkAndModifyFullModelParameters(final MEFPForecastSource source,
                                                     final CanonicalEvent event,
                                                     final int dayOfYear)
    {
    }

    /**
     * Do not override this method. Instead, override the method
     * {@link #getRhoValue(MEFPForecastSource, int, CanonicalEvent)}, which this calls to acquire the rho value to use
     * for ordering application of canonical events.
     * 
     * @param source The working forecast source.
     * @param dayOfYear Day of the year.
     * @param evt0 First event for which a correlation coefficient is returned.
     * @param evt1 The second event.
     * @return Two correlation coefficients, rhos, which are compared in order to sort a canonical event list via
     *         {@link #sortCanonicalEventsByRho(MEFPForecastSource, int, List)}. The two rhos correspond to evt0 and
     *         evt1.
     */
    protected double[] computeTwoRhoValuesForSortComparison(final MEFPForecastSource source,
                                                            final int dayOfYear,
                                                            final CanonicalEvent evt0,
                                                            final CanonicalEvent evt1)
    {

        final double[] rhos = new double[2];
        rhos[0] = getRhoValue(source, dayOfYear, evt0);
        rhos[1] = getRhoValue(source, dayOfYear, evt1);
        return rhos;
    }

    /**
     * Used for ordering applications of canonical events and debug log output. Override as needed.
     * 
     * @param source The working forecast source.
     * @param dayOfYear Day of the year.
     * @param evt0 Event for which a correlation coefficient is returned.
     * @return Correlation parameter for the source, day of year, and event; the same parameter used in sorting.
     */
    protected abstract double getRhoValue(final MEFPForecastSource source,
                                          final int dayOfYear,
                                          final CanonicalEvent evt0);

    /**
     * Update the source number of forecast days map, {@link #_sourceToNumberOfForecastDaysMap}, to record the usable
     * number of forecast days based on available parameters.
     */
    protected void updateNumberOfDaysForAllSources() throws Exception
    {
        int maxDaysFound = 0;
        for(final MEFPForecastSource source: this.getNonClimatologyForecastSources())
        {
            final int numberOfDays = determineNumberOfDaysForWhichToApplyModel(source);
            this.setSourceNumberOfForecastDays(source, numberOfDays);
            maxDaysFound = Math.max(maxDaysFound, numberOfDays);
        }

        final int numberOfDays = determineNumberOfDaysForWhichToApplyModel(getClimatologyForecastSource());
        this.setSourceNumberOfForecastDays(getClimatologyForecastSource(), numberOfDays);
        maxDaysFound = Math.max(maxDaysFound, numberOfDays);

        if(maxDaysFound == 0)
        {
            throw new Exception("For location " + _modelParameters.getIdentifier().buildStringToDisplayInTree()
                + ", the largest number of forecast days for any source is 0 so that MEFP cannot generate an ensemble.");
        }
    }

    /**
     * Must be overridden.
     * 
     * @param baseEnsembleHistoricalParameterId The parameter id of the base ensemble constructed from historical data.
     *            It will be a historical observed parameter.
     * @return A forecast parameter id to use instead of the base ensemble id for the forecast ensemble being generated.
     */
    protected abstract ParameterId determineForecastEnsembleParameterId(String baseEnsembleHistoricalParameterId);

    /**
     * @return {@link TimeSeriesArray} containing the one historical time series to use to derive a base ensemble to
     *         which the Schaake shuffle will be applied. It must return only one time series, which means the
     *         temperature version needs to determine if the ts will be for TMAX or TMIN data.
     */
    protected abstract TimeSeriesArray getSingleHistoricalTimeSeries();

    /**
     * Sorts the canonical event list based on Rho (ascending order).
     * 
     * @param source The source
     * @param dayOfYear The day of the year
     * @param eventList The list of events to sort.
     * @param precipitation True for precipitation, false for temperature data.
     * @param eptOrTminFlag True for EPT (for precip) or TMIN data, false for IPT (for precip) or TMAX data.
     */
    private void sortCanonicalEventsByRho(final MEFPForecastSource source,
                                          final int dayOfYear,
                                          final List<CanonicalEvent> eventList)
    {
        Collections.sort(eventList, new Comparator<CanonicalEvent>()
        {
            @Override
            public int compare(final CanonicalEvent evt0, final CanonicalEvent evt1)
            {
                final double[] rhos = computeTwoRhoValuesForSortComparison(source, dayOfYear, evt0, evt1);
                return Double.compare(rhos[0], rhos[1]);
            }
        });
    }

    /**
     * Determines how many days will be used for ensemble generation.
     * 
     * @param source The forecast source to consider.
     * @return The number of forecast days for which to apply the model based on user desired and available parameters.
     * @throws Exception if the number of days requested exceeds the number of days for which parameters were estimated.
     */
    private int determineNumberOfDaysForWhichToApplyModel(final MEFPForecastSource source) throws Exception
    {
        final int numberOfDays = this.getSourceNumberOfForecastDays(source);
        if((numberOfDays > 0) && ((getModelParameters().getSourceModelParameters(source) == null)
            || (!getModelParameters().getSourceModelParameters(source).wereParametersEstimatedForSource())))
        {
            throw new Exception("The forecast source " + source.getName()
                + " is to be applied (desired number of forecast days, " + numberOfDays
                + ", is positive), but no parameters were estimated for that source.");
        }

        final int parameterNumberOfDays = this.getModelParameters()
                                              .getSourceModelParameters(source)
                                              .getNumberOfForecastLeadDays();
        if(numberOfDays > parameterNumberOfDays)
        {
            throw new Exception("For source " + source.getName() + ", the desired number of days, " + numberOfDays
                + ", exceeds the number of days for which parameters were estimated, " + parameterNumberOfDays + ".");
        }
        return numberOfDays;
    }

    /**
     * Rounds all the values in the given time series to the accuracy dictated by
     * {@link MEFPModelTools#FORECAST_ENSEMBLE_NUMBER_OF_DECIMAL_PLACES}.
     * 
     * @param tss {@link TimeSeriesEnsemble} to round.
     */
    private void roundValues(final TimeSeriesEnsemble tss)
    {
        for(int i = 0; i < tss.size(); i++)
        {
            for(int j = 0; j < tss.get(i).size(); j++)
            {
                tss.get(i)
                   .setValue(j,
                             (float)HNumber.roundDouble(tss.get(i).getValue(j),
                                                        MEFPModelTools.FORECAST_ENSEMBLE_NUMBER_OF_DECIMAL_PLACES));
            }
        }
    }

    /**
     * Calls {@link #getCanonicalEventPeriodStepSizeInHours()}.
     * 
     * @return The number of canonical event steps per day.
     */
    private int getNumberOfEventValuesPerDay()
    {
        return (int)(24d / getCanonicalEventPeriodStepSizeInHours());
    }

    /**
     * @return True if canonical event values are available for the desired forecastTime for the purposes of
     *         hindcasting. This check can be called to determine if hindcast time series need to be loaded from
     *         reforecast files associated with the MEFPPE. This checks the {@link SourceCanonicalEventValues} returned
     *         by {@link #getCurrentSourceCanonicalEventValues(MEFPForecastSource)}. If this returns false, then the
     *         outside caller must add a forecast time series with the appropriate forecastTime in order for hindcasting
     *         to run.
     */
    public boolean areHindcastCanonicalEventsAlreadyComputed(final MEFPForecastSource source, final long forecastTime)
    {
        return this.getCurrentSourceCanonicalEventValues(source)
                   .getForecastValues()
                   .getEventValues(computeLagAdjustedHindcastForecastTime(source, forecastTime)) != null;
    }

    /**
     * @param forecastTime The desired historical forecast time.
     * @return The forecast time to use when looking for events for a hindcast. It adjusts the forecast time by the
     *         {@link MEFPForecastSource#getOperationalLagInHoursWhenAcquringReforecast(LocationAndDataTypeIdentifier)}
     *         lag amount.
     */
    public long computeLagAdjustedHindcastForecastTime(final MEFPForecastSource source, final long forecastTime)
    {
        final LocationAndDataTypeIdentifier identifier = getModelParameters().getIdentifier();
        return forecastTime
            - source.getOperationalLagInHoursWhenAcquringReforecast(identifier) * HCalendar.MILLIS_IN_HR;
    }

    public SamplingTools.SamplingTechnique getStratifiedSampling()
    {
        return _stratifiedSampling;
    }

    /**
     * @param eptStratifiedSampling Null is allowed and will be ignored (current value will remain unchanged).
     */
    public void setStratifiedSampling(final SamplingTools.SamplingTechnique stratifiedSampling)
    {
        if(stratifiedSampling != null)
        {
            _stratifiedSampling = stratifiedSampling;
        }
    }

    public MEFPFullModelParameters getModelParameters()
    {
        return _modelParameters;
    }

    /**
     * Sets the {@link #_modelParameters} and initializes the sources, setting the
     * {@link #_sourceToNumberOfForecastDaysMap} map to 0 for all sources.
     * 
     * @param modelParameters
     */
    public void setModelParameters(final MEFPFullModelParameters modelParameters)
    {
        _modelParameters = modelParameters;
        _precipitationFlag = _modelParameters.getIdentifier().isPrecipitationDataType();
        for(final MEFPForecastSource source: modelParameters.getOrderedForecastSources())
        {
            if(source.isClimatologySource())
            {
                setClimatologySource(source, 0);
            }
            else
            {
                addNonClimatologyForecastSource(source, 0);
            }
        }
    }

    public boolean getHindcastMode()
    {
        return _hindcastMode;
    }

    public void setHindcastMode(final boolean b)
    {
        _hindcastMode = b;
    }

    public boolean getPrecipitationFlag()
    {
        return _precipitationFlag;
    }

    public Set<MEFPForecastSource> getNonClimatologyForecastSources()
    {
        return _sourceToNumberOfForecastDaysMap.keySet();
    }

    private void addNonClimatologyForecastSource(final MEFPForecastSource source, final int numberOfForecastDays)
    {
        _sourceToNumberOfForecastDaysMap.put(source, numberOfForecastDays);
    }

    /**
     * Forces the number of forecast days for the source to 0.
     */
    private void ignoreNonClimatologyForecastSource(final MEFPForecastSource source)
    {
        _sourceToNumberOfForecastDaysMap.put(source, 0);
    }

    private void setClimatologySource(final MEFPForecastSource source, final int numberOfForecastDays)
    {
        _climatologySource = source;
        _climatologyNumberOfDays = numberOfForecastDays;
    }

    public MEFPForecastSource getClimatologyForecastSource()
    {
        return _climatologySource;
    }

    public void setCanonicalEventRestrictor(final CanonicalEventRestrictor restrictor)
    {
        _canonicalEventRestrictor = restrictor;
    }

    /**
     * Sets the number of forecast days for any sources, including both climatology and non-climatology.
     * 
     * @param source
     * @param numberOfForecastDays
     * @return The number of forecast days for the source.
     */
    public void setSourceNumberOfForecastDays(final MEFPForecastSource source, final int numberOfForecastDays)
    {
        if(_climatologySource.equals(source))
        {
            _climatologyNumberOfDays = numberOfForecastDays;
        }
        else
        {
            _sourceToNumberOfForecastDaysMap.put(source, numberOfForecastDays);
        }
    }

    /**
     * Checks all sources, including climatology.
     * 
     * @param source
     * @return The number of forecast days for the source.
     */
    public int getSourceNumberOfForecastDays(final MEFPForecastSource source)
    {
        if(_climatologySource == source)
        {
            return _climatologyNumberOfDays;
        }
        return _sourceToNumberOfForecastDaysMap.get(source);
    }

    /**
     * Used for all forecast sources, including climatology.
     * 
     * @param source The source for which to set the time series.
     * @param forecastTimeSeries The deterministic (single-valued) or ensemble (e.g., CFSv2 lagged ensemble) forecast
     *            time series from which an ensemble is to be estimated.
     */
    public void addSourceForecastTimeSeries(final MEFPForecastSource source, final TimeSeriesArrays forecastTimeSeries)
    {
        for(int i = 0; i < forecastTimeSeries.size(); i++)
        {
            _sourceToForecastTimeSeriesMap.put(source, forecastTimeSeries.get(i));
        }
    }

    /**
     * Used for all forecast sources, including climatology.
     * 
     * @param source The source for which to set the time series.
     * @param forecastTimeSeries The deterministic (single-valued) forecast time series from which an ensemble is to be
     *            estimated.
     */
    public void addSourceForecastTimeSeries(final MEFPForecastSource source, final TimeSeriesArray forecastTimeSeries)
    {
        addSourceForecastTimeSeries(source, new TimeSeriesArrays(forecastTimeSeries));
    }

    /**
     * Used for all forcast sources, including climatology.
     * 
     * @param source The source for which to get the time series.
     * @return The single-valued forecast time series.
     */
    public List<TimeSeriesArray> getSourceForecastTimeSeries(final MEFPForecastSource source)
    {
        return _sourceToForecastTimeSeriesMap.get(source);
    }

    /**
     * Adds the generated ensemble for a source to storage.
     * 
     * @param source The source for which the ensemble was generated.
     * @param ensemble The ensemble.
     */
    public void addSourceGeneratedEnsemble(final MEFPForecastSource source, final TimeSeriesEnsemble ensemble)
    {
        this._sourceToGeneratedEnsembleMap.put(source, ensemble);
    }

    public void removeSourceGeneratedEnsemble(final MEFPForecastSource source)
    {
        _sourceToGeneratedEnsembleMap.remove(source);
    }

    public TimeSeriesEnsemble getSourceGeneratedEnsemble(final MEFPForecastSource source)
    {
        return this._sourceToGeneratedEnsembleMap.get(source);
    }

    public void setCurrentSourceCanonicalEventValues(final MEFPForecastSource source,
                                                     final SourceCanonicalEventValues values)
    {
        _sourceToPrecomputedEventValues.put(source, values);
    }

    public SourceCanonicalEventValues getCurrentSourceCanonicalEventValues(final MEFPForecastSource source)
    {
        return _sourceToPrecomputedEventValues.get(source);
    }

    public void setBehavoirIfEventMissing(final BehaviorIfEventMissing behavior)
    {
        _behaviorIfEventMissing = behavior;
    }

    public BehaviorIfEventMissing getBehavoirIfEventMissing(final MEFPForecastSource source)
    {
        return this._behaviorIfEventMissing;
    }

    public Calendar getMemberIndexingCal()
    {
        return _memberIndexingCal;
    }

    public void setMemberIndexingCal(final Calendar memberIndexingCal)
    {
        _memberIndexingCal = memberIndexingCal;
    }

    /**
     * @return {@link Collection} of {@link TimeSeriesArray} containing the historical data, MAP for precipitation (for
     *         one time series, call), TMIN and TMAX for temperature.
     * @throws Exception If a problem occurs.
     */
    public Collection<TimeSeriesArray> getHistoricalTimeSeries()
    {
        return getModelParameters().getHistoricalTimeSeries();
    }

    /**
     * Called by the adapter when processing input time series. This checks the provided parameter for validity and
     * returns a new {@link ParameterId} which is the provided one converted to the appropriate processed data type
     * returned by {@link #getProcessedDataTypes()}.<br>
     * <br>
     * For example, if the model processes forecast precip but the data type is MAPX, it should still be usable. Hence,
     * this would convert it to an acceptable forecast precipitation data type for usage.
     * 
     * @param inputParameter Parameter id of the data as a String.
     * @param unitStr Units of the data (degc, mm, etc)
     * @return
     */
    public abstract ParameterId checkForValidityAndConvertToProcessedType(String inputParameterStr) throws Exception;

    /**
     * Called by the adapter when processing input time series. This checks the provided units for validity and converts
     * the units to the expected unit for the model. For example, precipitation data is expected to be of measurement
     * type length (in, mm, etc.) and must be converted to mm before ensemble generation.
     * 
     * @param inputTS
     * @throws Exception If the units are not valid for conversion to mm.
     */
    public abstract void checkForValidityAndConvertUnits(TimeSeriesArray inputTS) throws Exception;

    /**
     * @return The data types that should be processed by the given implementation of this class. Precipitation expects
     *         to process only FMAP, which temperature expects to process TFMX and TFMN.
     */
    public abstract ParameterId[] getProcessedDataTypes();

    /**
     * @param numberOfDays Desired forecast number of days.
     * @return COPY of the canonical events indexed according to how it would be used to access parameters. If
     *         {@link #_externallySpecifiedEvents} is not null, that list is used to further remove events. The event
     *         number for each event provides the index of the event in the full list.
     */
    public List<CanonicalEvent> getApplicableEvents(final MEFPForecastSource source, final int numberOfDays)
    {
        final CanonicalEventList computedEvents = _modelParameters.getSourceModelParameters(source).getComputedEvents();
        final List<CanonicalEvent> results = new ArrayList<CanonicalEvent>();
        for(final CanonicalEvent evt: computedEvents.subList(computedEvents.determineNumberOfApplicableEvents(numberOfDays)))
        {
            results.add(new CanonicalEvent(0,
                                           evt.getStartLeadPeriod(),
                                           evt.getEndLeadPeriod(),
                                           evt.getNumberOfLaggedEnsembleMembers()));
        }
        for(int i = 0; i < results.size(); i++)
        {
            results.get(i).setEventNumber(i);
        }

        //Only include events also listed in the externally specified list, if that list is provided.
        if(_canonicalEventRestrictor != null)
        {
            for(int i = results.size() - 1; i >= 0; i--)
            {
                if(!_canonicalEventRestrictor.useEvent(source, results.get(i)))
                {
                    results.remove(i);
                }
            }
        }
        return results;
    }

    /**
     * @return The max of {@link #getSourceNumberOfForecastDays(MEFPForecastSource)} for all sources, including
     *         climatology.
     */
    public int getLargestNumberOfForecastDaysForWhichToApplyModel()
    {
        int max = getSourceNumberOfForecastDays(_climatologySource);
        for(final MEFPForecastSource source: getNonClimatologyForecastSources())
        {
            final int days = getSourceNumberOfForecastDays(source);
            if(days > max)
            {
                max = days;
            }
        }
        return max;
    }

    /**
     * Always starts with whatever generated ensemble is stored for the {@link #getClimatologyForecastSource()} via
     * {@link #addSourceGeneratedEnsemble(MEFPForecastSource, TimeSeriesEnsemble)}. This means you should always put
     * your starting point as the generated ensemble for climatology; even if climatology is not used, this is still
     * needed.<br>
     * <br>
     * XXX I don't know that I agree with this. The length of the forecastEnsemble returned by
     * getSourceGeneratedEnsemble will never be longer than the longest source number of forecast days. So, if the
     * climatology source number of days is set to 0, the length will be driven by CFSv2 or GEFS or RFC. That means that
     * when the values are copied from those sources into forecastEnsemble, it will override the climatology starting
     * point. Hence, the starting point could be all missing and the same results would be acquired.<br>
     * <br>
     * Still, since it does not appear to do any harm, I'll leave it in place.
     */
    public TimeSeriesEnsemble combineGeneratedEnsembles()
    {
        //Get the starting point ensemble.  This should be the climatology ensemble, if it is used.  If not, then 
        //it should use the generated ensemble for any source that is the longest.  
        final TimeSeriesEnsemble forecastEnsemble = getSourceGeneratedEnsemble(getClimatologyForecastSource());

        //I need the member index below because it is used to pair up the ensemble time series.  Can use iterable for loop here.
        for(int memberIndex = 0; memberIndex < forecastEnsemble.size(); memberIndex++)
        {
            final TimeSeriesArray member = forecastEnsemble.get(memberIndex);
            for(int i = 0; i < forecastEnsemble.get(memberIndex).size(); i++)
            {
                for(final MEFPForecastSource source: this.getNonClimatologyForecastSources())
                {
                    if(getSourceGeneratedEnsemble(source) != null)
                    {
                        final TimeSeriesArray sourceMember = getSourceGeneratedEnsemble(source).get(memberIndex);

                        //This is looking at millis, so it uses <=.  For the source, the maximum time for which its values
                        //should be used is the forecast number of days in milliseconds.  Break out of this source searching loop
                        //as soon as we find the first match.
                        if(member.getTime(i)
                            - member.getHeader().getForecastTime() <= getSourceNumberOfForecastDays(source) * 24
                                * HCalendar.MILLIS_IN_HR)
                        {
                            //MIN_VALUE is used to mark points where fillNextSource needs to be applied.  When MIN_VALUE is found,
                            //skip the point and do not allow it to break.  This will cause the source loop above to continue to the next source.
                            if(sourceMember.getValue(i) != Float.MIN_VALUE)
                            {
                                member.setValue(i, sourceMember.getValue(i));
                                break;
                            }
                        }
                    }
                }
            }
        }
        return forecastEnsemble;
    }

    /**
     * When a canonical event's forecast value is missing, if the value of the property "behaviorIfEventMissing" is:
     * "errorOut" -- throw an exception; "fillMissing" -- put missing values in the locations within the baseEnsemble
     * covered by the event; "fillClimatology" or undefined -- skip the event and produce debug messages.
     */
    public void processMissingEvent(final MEFPForecastSource source,
                                    final CanonicalEvent event,
                                    final TimeSeriesEnsemble baseEnsemble,
                                    final long t0,
                                    final int periodTimeStepInHours) throws Exception
    {
        if(!_alreadyOutputMissingEventMessage)
        {
            LOG.info("For location " + getModelParameters().getIdentifier().buildStringToDisplayInTree()
                + " and source " + source.getSourceId()
                + ", at least one canonical event could not be computed likely due to missing data; the "
                + MEFPEnsembleGeneratorModelAdapter.BEHAVIOR_IF_EVENT_MISSING + " run property was set to "
                + _behaviorIfEventMissing + " and will be applied (run in debug mode for more information).");
            _alreadyOutputMissingEventMessage = true;
        }

        if(_behaviorIfEventMissing.equals(BehaviorIfEventMissing.errorOut))
        {
            throw new Exception("The canonical event value for source " + source.getName() + " and event "
                + event.toString() + " is missing (NaN): stop.");
        }
        else if(_behaviorIfEventMissing.equals(BehaviorIfEventMissing.fillMissing))
        {
            for(final TimeSeriesArray ts: baseEnsemble)
            {
                TimeSeriesArrayTools.fillNaNs(ts,
                                              event.computeStartTime(t0, periodTimeStepInHours),
                                              event.computeEndTime(t0, periodTimeStepInHours));
            }
            LOG.debug("The canonical event value for source " + source.getName() + " and event " + event.toString()
                + " is missing (NaN), BehaviorIfEventMissing is fillMissing: filling the ensemble with NaN for the appropriate times.");
            LOG.debug("This happens sometimes during "
                + "hindcasting if the forecast time does not line up exacctly on one of the available hindcast dates.");
        }
        else if(_behaviorIfEventMissing.equals(BehaviorIfEventMissing.fillClimatology))
        {
            LOG.debug("The canonical event value for source " + source.getName() + " and event " + event.toString()
                + " is missing (NaN), BehaviorIfEventMissing is fillClimatology: using climatology for the appropriate times.");
            LOG.debug("This happens sometimes during "
                + "hindcasting if the forecast time does not line up exacctly on one of the available hindcast dates.");
        }
        else if(_behaviorIfEventMissing.equals(BehaviorIfEventMissing.fillNextSource))
        {
            for(final TimeSeriesArray ts: baseEnsemble)
            {
                TimeSeriesArrayTools.fillMinValue(ts,
                                                  event.computeStartTime(t0, periodTimeStepInHours),
                                                  event.computeEndTime(t0, periodTimeStepInHours));
            }
            LOG.debug("The canonical event value for source " + source.getName() + " and event " + event.toString()
                + " is missing (NaN), BehaviorIfEventMissing is fillNextSource: filling the ensemble with MIN_VALUE for the appropriate times to mark for fill from next source later.");
            LOG.debug("This happens sometimes during "
                + "hindcasting if the forecast time does not line up exacctly on one of the available hindcast dates.");
        }
    }
}
