package ohd.hseb.hefs.mefp.models;

import java.util.ArrayList;
import java.util.List;

import nl.wldelft.util.timeseries.TimeSeriesArray;
import ohd.hseb.hefs.mefp.models.parameters.MEFPAlgorithmModelParameters;
import ohd.hseb.hefs.mefp.models.parameters.MEFPFullModelParameters;
import ohd.hseb.hefs.mefp.pe.core.MEFPParameterEstimatorRunInfo;
import ohd.hseb.hefs.mefp.pe.estimation.MEFPEstimationControlOptions;
import ohd.hseb.hefs.mefp.sources.MEFPForecastSource;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.core.ParameterEstimatorRunInfo;
import ohd.hseb.hefs.pe.estimation.options.ControlOption;
import ohd.hseb.hefs.pe.model.FullModelParameters;
import ohd.hseb.hefs.pe.model.ParameterEstimationException;
import ohd.hseb.hefs.pe.model.ParameterEstimationModel;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.jobs.JobMessenger;
import ohd.hseb.hefs.utils.tools.ParameterId;
import ohd.hseb.hefs.utils.tsarrays.TimeSeriesArraysTools;

import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.LogManager;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;

/**
 * Super class of all MEFP based {@link ParameterEstimationModel} implementations. It provides a top-level
 * {@link #estimateParameters(FullModelParameters, ParameterEstimatorRunInfo)} method that is usable for both precip and
 * temp, but requires subclasses to override the source specific
 * {@link #estimateParametersForSource(MEFPForecastSource)}. It also provides several attributes used for both precip
 * and temp.
 * 
 * @author hankherr
 */
public abstract class MEFPParameterEstimationModel implements ParameterEstimationModel
{
    private static final Logger LOG = LogManager.getLogger(MEFPParameterEstimationModel.class);

    /**
     * Stores the historical time series to be used in parameter estimation for MEFP.
     */
    private final List<TimeSeriesArray> _historicalTimeSeries = new ArrayList<TimeSeriesArray>();

    private MEFPParameterEstimatorRunInfo _mefpRunInfo = null;

    /**
     * The identifier for which parameters are being estimated.
     */
    private LocationAndDataTypeIdentifier _estimatedIdentifier = null;

    /**
     * Algorithm model parameters used for both reading (specifying the required model parameter types) and estimation.
     * It is reset during estimation, since I'm not sure if it will equal that provided in
     * {@link #_estimatedMefpModelParameters}.
     */
    private MEFPAlgorithmModelParameters _algorithmModelParameters = null;

    /**
     * The {@link MEFPFullModelParameters} being computed/estimated.
     */
    private MEFPFullModelParameters _estimatedMefpModelParameters = null;

    /**
     * The control parameters being used for the estimation.
     */
    private MEFPEstimationControlOptions _estimationCtlParms = null;

    /**
     * The model specific control parmaeters being used for estimation, pulled from {@link #_estimationCtlParms}.
     */
    private MEFPBaseModelControlOptions _estimationModelCtlParms = null;

    @Override
    public void prepareEstimationOptions(final FullModelParameters modelParameters,
                                         final ParameterEstimatorRunInfo runInfo,
                                         final ForecastSource forecastSource)
    {
        //Prepare the estimation options stored within these parameters and set them herein.
        if(forecastSource != null)
        {
            modelParameters.prepareEstimationOptionsForComputationOfOneSource(runInfo.getEstimationControlOptions(modelParameters.getIdentifier()),
                                                                              forecastSource);
        }

        //If there is ever anything else to do, insert it here.
    }

    @Override
    public void estimateParameters(final FullModelParameters modelParameters,
                                   final ParameterEstimatorRunInfo runInfo,
                                   final ForecastSource oneSource) throws ParameterEstimationException
    {
        JobMessenger.madeProgress("Preparing for estimation of parameters...");
        modelParameters.setModel(this);

        //Map attributes to available objects.  These mappings are done only for convenience.
        setMEFPRunInfo((MEFPParameterEstimatorRunInfo)runInfo);
        setEstimatedMEFPModelParameters((MEFPFullModelParameters)modelParameters);
        setAlgorithmModelParameters(getEstimatedMEFPModelParameters().getAlgorithmModelParameters());
        setEstimatedIdentifier(getEstimatedMEFPModelParameters().getIdentifier());
        setEstimationCtlOptions(((MEFPFullModelParameters)modelParameters).getEstimationControlOptions()); //This must use the version in the parameters, not run info.
        this._historicalTimeSeries.clear();

        LOG.info("Estimating parameters for location " + getEstimatedIdentifier().buildStringToDisplayInTree() + "...");

        //Build canonical event lists for getAlgorithmModelParameters() from the _mefpRunInfo canonical events mgr.

        //When estimating for one source, canonical events will already be in place and will match those in the existing
        //parameter file.  When estimating for all sources, which will trigger this if, gather the canonical events from
        //those specified in the run-time information (i.e., current user specified).  
        if(oneSource == null)
        {
            getAlgorithmModelParameters().gatherCanonicalEvents(getEstimatedIdentifier(),
                                                                getMEFPRunInfo().getCanonicalEventsMgr());
        }

        //In either case, make sure at this point that the events are ready for use in the algorithm model parameters.
        if(getAlgorithmModelParameters().getFullListOfEventsInOrder().isEmpty())
        {
            throw new ParameterEstimationException("No canonical events were specified (check the Setup Panel); cannot estimate parameters.");
        }

        //Check canonical events for validity.
        try
        {
            getAlgorithmModelParameters().getFullListOfEventsInOrder().validate();
        }
        catch(final Exception e)
        {
            final ParameterEstimationException newE = new ParameterEstimationException(e.getMessage());
            newE.setStackTrace(e.getStackTrace());
            throw newE;
        }

//XXX Override canonical event list to use 1 pd events for 365 days.
//        getAlgorithmModelParameters().getBaseCanonicalEvents().clear();
//        getAlgorithmModelParameters().getModulationCanonicalEvents().clear();
//        getAlgorithmModelParameters().getFullListOfEventsInOrder().clear();
//        for(int i = 1; i <= 365 * 4; i++)
//        {
//            getAlgorithmModelParameters().getBaseCanonicalEvents().add(new CanonicalEvent(i, i, i, 1)); //The number of members must be set for CFSv2???
//            getAlgorithmModelParameters().getFullListOfEventsInOrder().add(new CanonicalEvent(i, i, i, 1));
//        }
//        System.err.println("####>> JUST HARDWIRED EVENTS!!!");

        //For one forecast source, do the following: ============================================
        if(oneSource != null)
        {
            //Prepare the parameters appropriately for the run time list of sources and source to estimate.
            getEstimatedMEFPModelParameters().prepareParametersForComputationOfOneSource(getMEFPRunInfo().getForecastSources(),
                                                                                         (MEFPForecastSource)oneSource);

            //Use the historical data already stored with the parameters.
            _historicalTimeSeries.clear();
            _historicalTimeSeries.addAll(getEstimatedMEFPModelParameters().getHistoricalTimeSeries());

            //Estimate the parameters for the source.
            JobMessenger.madeProgress("Estimating parameters for source " + oneSource.getName() + "...");
            estimateParametersForSource((MEFPForecastSource)oneSource);
        }
        //For all source, do the following: ======================================================
        else
        {
            //Populate the aglorithm  model parameters from the control parameters.
            populateAlgorithmModelParameters();

            //Prepare the model parameters for computation
            getEstimatedMEFPModelParameters().prepareParametersForComputation();

            //Load the historical data.
            loadHistoricalData();
            getEstimatedMEFPModelParameters().setHistoricalTimeSeries(TimeSeriesArraysTools.convertListOfTimeSeriesToTimeSeriesArrays(getHistoricalTimeSeries()));

            //Loop through each forecast source
            for(final MEFPForecastSource source: getMEFPRunInfo().getForecastSources())
            {
                JobMessenger.madeProgress("Estimating parameters for source " + source.getName() + "...");
                try
                {
                    estimateParametersForSource(source);
                }
                catch(final Throwable t)
                {
                    throw new ParameterEstimationException("Problems estimating parameters for source "
                        + source.getName() + ": " + t.getMessage(), t);
                }
            }
        }
        LOG.info("Done estimating parameters for " + getEstimatedIdentifier().buildStringToDisplayInTree() + ".");
    }

    /**
     * Subclasses should call this to add questionable log messages as a one-liner: this will handle checking the
     * message for null or empty before adding.
     */
    protected void addQuestionableParameterLogMessage(final MEFPForecastSource source,
                                                      final CanonicalEvent event,
                                                      final int dayOfYear,
                                                      final String message)
    {
        if(!Strings.isNullOrEmpty(message))
        {
            getEstimatedMEFPModelParameters().getQuestionableParameterLog().addEntry(source, event, dayOfYear, message);
        }
    }

    public MEFPParameterEstimatorRunInfo getMEFPRunInfo()
    {
        return _mefpRunInfo;
    }

    public void setMEFPRunInfo(final MEFPParameterEstimatorRunInfo mefpRunInfo)
    {
        _mefpRunInfo = mefpRunInfo;
    }

    public LocationAndDataTypeIdentifier getEstimatedIdentifier()
    {
        return _estimatedIdentifier;
    }

    public void setEstimatedIdentifier(final LocationAndDataTypeIdentifier estimatedIdentifier)
    {
        _estimatedIdentifier = estimatedIdentifier;
    }

    @Override
    public MEFPAlgorithmModelParameters getAlgorithmModelParameters()
    {
        return _algorithmModelParameters;
    }

    public void setAlgorithmModelParameters(final MEFPAlgorithmModelParameters algoModelParameters)
    {
        _algorithmModelParameters = algoModelParameters;
    }

    public MEFPFullModelParameters getEstimatedMEFPModelParameters()
    {
        return _estimatedMefpModelParameters;
    }

    public void setEstimatedMEFPModelParameters(final MEFPFullModelParameters estimatedMefpModelParameters)
    {
        _estimatedMefpModelParameters = estimatedMefpModelParameters;
    }

    public MEFPEstimationControlOptions getEstimationCtlParms()
    {
        return _estimationCtlParms;
    }

    /**
     * Sets both {@link #_estimationModelCtlParms} and {@link #_estimatedMefpModelParameters}.
     * 
     * @param estimationCtlParms
     */
    public void setEstimationCtlOptions(final MEFPEstimationControlOptions estimationCtlParms)
    {
        _estimationCtlParms = estimationCtlParms;
        _estimationModelCtlParms = _estimationCtlParms.getModelParameters();
    }

    public MEFPBaseModelControlOptions getEstimationModelCtlParms()
    {
        return _estimationModelCtlParms;
    }

    /**
     * @return The one time series.
     * @throws {@link IllegalStateException} if more than one time series exists.
     */
    public TimeSeriesArray getSingleHistoricalTimeSeries()
    {
        if(_historicalTimeSeries.size() > 1)
        {
            throw new IllegalStateException("Attempt to acquire a single historical time series when there are "
                + _historicalTimeSeries.size() + " many.");
        }
        return _historicalTimeSeries.get(0);
    }

    /**
     * @return The full list of historical time series. For precpitation, there will be only one time series. For
     *         temperature
     */
    public List<TimeSeriesArray> getHistoricalTimeSeries()
    {
        return _historicalTimeSeries;
    }

    /**
     * @param historicalTimeSeries Time series to add to the list.
     */
    protected void addHistoricalTimeSeries(final TimeSeriesArray historicalTimeSeries)
    {
        _historicalTimeSeries.add(historicalTimeSeries);
    }

    /**
     * Populate the algorithm model parameters returned by {@link #getAlgorithmModelParameters()} from the control
     * parameters returned by {@link #getEstimationModelCtlParms()}.
     */
    protected void populateAlgorithmModelParameters()
    {
    }

    /**
     * @return Loads the historical data to use for parameter estimation via the historical data handler. For
     *         precipitation, this returns only the historical MAP time series returned by the handler. For temperature,
     *         those time series are enhanced by the RFC observed time series, if present, filling in missing values and
     *         extending the period of record. The returned composite time series is used to compute observed canonical
     *         event values for all sources and will consist of TMIN in slot 0 and TMAX in slot 1.
     * @throws ParameterEstimationException
     */
    protected abstract void loadHistoricalData() throws ParameterEstimationException;

    /**
     * @param source The forecast source for which to estimate parameters.
     * @return True if any processing was done, false if not (the source is not enabled or the number of days is 0). I
     *         don't know if the return will be useful, but just in case, its available.
     * @throws ParameterEstimationException
     */
    protected abstract boolean estimateParametersForSource(final MEFPForecastSource source) throws ParameterEstimationException;

    /**
     * @return the data type that this model deals with
     */
    protected abstract ParameterId.Type getDataType();

    /**
     * Since this is limited to one data type, we don't need the usual argument.
     * 
     * @return a new set of control parameters for this model
     */
    public abstract ControlOption createControlOptions();

    /**
     * Throws an exception if {@code type} is not equal to {@link #getDataType()}.
     */
    @Override
    public ControlOption createControlOptions(final ParameterId.Type type)
    {
        Preconditions.checkArgument(type.equals(getDataType()), "Argument %s must be %s.", type, getDataType());
        return createControlOptions();
    }
}
