package ohd.hseb.hefs.mefp.models.log;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.pe.sources.ForecastSource;
import ohd.hseb.hefs.utils.Triad;
import ohd.hseb.hefs.utils.effect.Effect;
import ohd.hseb.hefs.utils.xml.CollectionXMLWriter;
import ohd.hseb.hefs.utils.xml.ListXMLReader;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReader;
import ohd.hseb.hefs.utils.xml.XMLReaderFactory;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.XMLWriter;
import ohd.hseb.hefs.utils.xml.vars.XMLInteger;
import ohd.hseb.hefs.utils.xml.vars.XMLString;

import com.google.common.collect.LinkedHashMultimap;
import com.google.common.collect.Lists;

/**
 * Maintains a log of questionable parameter messages.
 * 
 * @author hankherr
 */
public class QuestionableLog implements XMLReadable, XMLWritable
{

    private final LinkedHashMultimap<Triad<ForecastSource, CanonicalEvent, Integer>, String> _logs = LinkedHashMultimap.create();

    /**
     * {@link List} of {@link ForecastSource} instances for which questionable log messages will be stored.
     */
    private final List<ForecastSource> _availableSources;

    /**
     * @param availableSources {@link List} of {@link ForecastSource} instances for which questionable log messages will
     *            be stored.
     */
    public QuestionableLog(final List<? extends ForecastSource> availableSources)
    {
        _availableSources = new ArrayList<>();
        _availableSources.addAll(availableSources);
    }

    /**
     * @param sourceId
     * @return {@link ForecastSource} from within {@link #_availableSources} that has the provided id, or null if not
     *         found.
     */
    public ForecastSource findSource(final String sourceId)
    {
        for(final ForecastSource source: _availableSources)
        {
            if(source.getSourceId().equals(sourceId))
            {
                return source;
            }
        }
        return null;
    }

    /**
     * Remoaves all of the entries for the provided source from the {@link #_logs}.
     * 
     * @param source Source to remove.
     */
    public void removeAllEntriesForSource(final ForecastSource source)
    {
        //Get a list of the keys to remove.
        final List<Triad<ForecastSource, CanonicalEvent, Integer>> keysToRemove = new ArrayList<>();
        for(final Triad<ForecastSource, CanonicalEvent, Integer> keyTriad: _logs.keySet())
        {
            if(keyTriad._first.equals(source))
            {
                keysToRemove.add(keyTriad);
            }
        }

        //Remove each key.  Not that removeAll removes all of the items associated with that key.
        //There is no way I could find to remove the entire list for a bunch of keys in one call.
        for(final Triad<ForecastSource, CanonicalEvent, Integer> keyTriad: keysToRemove)
        {
            _logs.removeAll(keyTriad);
        }
    }

    /**
     * Adds an entry to {@link #_logs}.
     * 
     * @param reason Message explaining why the parameters are questionable for source, event, and day of year.
     */
    public void addEntry(final ForecastSource source,
                         final CanonicalEvent event,
                         final int dayOfYear,
                         final String reason)
    {
        _logs.put(new Triad(source, event, dayOfYear), reason);
    }

    public Set<String> getEntries(final ForecastSource source, final CanonicalEvent event, final int dayOfYear)
    {
        return _logs.get(new Triad(source, event, dayOfYear));
    }

    @Override
    public XMLReader getReader()
    {
        return new ListXMLReader("questionableParameterLog", new XMLReaderFactory<XMLString>()
        {
            @Override
            public XMLString get()
            {
                final XMLString element = new XMLString("questionableMessage");
                element.addAttribute("source", XMLString.class, true);
                element.addAttribute("event", XMLString.class, true);
                element.addAttribute("day", XMLInteger.class, true);
                return element;
            }
        }, new Effect()
        {

            @Override
            public void perform(final Object input)
            {
                _logs.clear();
                for(final XMLString element: (List<XMLString>)input)
                {
                    final ForecastSource source;
                    source = findSource((String)element.getAttributeValue("source"));
                    if(source == null)
                    {
                        // Source no longer exists
                        return;
                    }
                    final CanonicalEvent event = CanonicalEvent.parseXMLAttributeString((String)element.getAttributeValue("event"));
                    final int dayOfYear = (Integer)element.getAttributeValue("day");
                    addEntry(source, event, dayOfYear, element.get());
                }
            }
        });
    }

    @Override
    public XMLWriter getWriter()
    {
        final List<XMLString> elements = Lists.newArrayList();
        for(final Triad<ForecastSource, CanonicalEvent, Integer> logKey: _logs.keySet())
        {
            final Set<String> messages = getEntries(logKey.getFirst(), logKey.getSecond(), logKey.getThird());
            for(final String message: messages)
            {
                final XMLString element = new XMLString("questionableMessage", message);
                element.addAttribute("source", XMLString.class, true).set(logKey.getFirst().getSourceId());
                element.addAttribute("event", XMLString.class, true)
                       .set(CanonicalEvent.createXMLAttributeString(logKey.getSecond()));
                element.addAttribute("day", XMLInteger.class, true).set(logKey.getThird());
                elements.add(element);
            }
        }
        return new CollectionXMLWriter("questionableParameterLog", elements);
    }
}
