package ohd.hseb.hefs.mefp.models.parameters;

import java.util.ArrayList;
import java.util.List;

import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEvent;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventList;
import ohd.hseb.hefs.mefp.tools.canonical.CanonicalEventsManager;
import ohd.hseb.hefs.pe.model.AlgorithmModelParameters;
import ohd.hseb.hefs.pe.tools.HEFSTools;
import ohd.hseb.hefs.pe.tools.LocationAndDataTypeIdentifier;
import ohd.hseb.hefs.utils.xml.CompositeXMLReader;
import ohd.hseb.hefs.utils.xml.CompositeXMLWriter;
import ohd.hseb.hefs.utils.xml.XMLReadable;
import ohd.hseb.hefs.utils.xml.XMLReaderException;
import ohd.hseb.hefs.utils.xml.XMLWritable;
import ohd.hseb.hefs.utils.xml.vars.XMLString;
import ohd.hseb.util.io.EndianConvertingInputStream;

/**
 * Subclasses of this class store parameters required for the model algorithm to execute which are independent of the
 * data sources. There should be sets and gets for all parameters provide here or in subclasses. Additionally, there are
 * three methods to override, which are javadoc'd below.<br>
 * <br>
 * Note that, for the first versions of this class, some parameters are related to the data sources, which is due to the
 * poor design of the parameter I/O mechanism. This class needs to be redesigned in the future.<br>
 * <br>
 * The three storage CanonicalEventList attributes, _baseCanonicalEvents, _modulationCanonicalEvents, and
 * _fullListOfEventsInOrder are all initialized to be for precipitation, having a period unit of 6 hours. Temperature
 * requires 24 hour units, which should be set for these event lists within the constructor for the temperature subclass
 * of this (just call the setPeriodUnitsInHours method on the three attributes).
 * 
 * @author hank.herr
 */
public class MEFPAlgorithmModelParameters extends AlgorithmModelParameters
{
    private final CanonicalEventList _baseCanonicalEvents;
    private final CanonicalEventList _modulationCanonicalEvents;

    /**
     * The full list is necessary in order to know which events parameter values correspond to. It contains both base
     * and modulation events intermingled.
     */
    private final CanonicalEventList _fullListOfEventsInOrder;

    private final boolean _precipitation;

    /**
     * Records the precip flag and sets the period units for {@link #_baseCanonicalEvents},
     * {@link #_modulationCanonicalEvents}, and {@link #_fullListOfEventsInOrder}.
     * 
     * @param precipitation Boolean indicating if the data is precipitation (true) or temperature (false).
     */
    public MEFPAlgorithmModelParameters(final boolean precipitation)
    {
        _precipitation = precipitation;

        //Make sure the lists are appropriately initialized for proper reading of parameters.  Note the flags passed in.
        _baseCanonicalEvents = new CanonicalEventList(_precipitation);
        _baseCanonicalEvents.setToBaseEventType();
        _modulationCanonicalEvents = new CanonicalEventList(_precipitation);
        _modulationCanonicalEvents.setToModulationEventType();
        _fullListOfEventsInOrder = new CanonicalEventList(_precipitation);
    }

    /**
     * Builds {@link #_fullListOfEventsInOrder} from {@link #_baseCanonicalEvents} and
     * {@link #_modulationCanonicalEvents}.
     */
    public void gatherCanonicalEvents(final LocationAndDataTypeIdentifier identifier, final CanonicalEventsManager mgr)
    {
        getBaseCanonicalEvents().clear();
        getModulationCanonicalEvents().clear();
        getFullListOfEventsInOrder().clear();
        getBaseCanonicalEvents().addAll(mgr.getBaseCanonicalEventList(identifier));
        getModulationCanonicalEvents().addAll(mgr.getModulationCanonicalEventList(identifier));
        if(identifier.isPrecipitationDataType())
        {
            getFullListOfEventsInOrder().addAll(mgr.buildFullPrecipitationCanonicalEventList());
        }
        else if(identifier.isTemperatureDataType())
        {
            getFullListOfEventsInOrder().addAll(mgr.buildFullTemperatureCanonicalEventList());
        }
        else
        {
            throw new IllegalArgumentException("Identifier " + identifier.buildStringToDisplayInTree()
                + " is neither precipitation nor temperature.");
        }
    }

    public CanonicalEventList getBaseCanonicalEvents()
    {
        return _baseCanonicalEvents;
    }

    public CanonicalEventList getModulationCanonicalEvents()
    {
        return _modulationCanonicalEvents;
    }

    public CanonicalEventList getFullListOfEventsInOrder()
    {
        return this._fullListOfEventsInOrder;
    }

    public void readCanonicalEventsFromModelParameterFile(final EndianConvertingInputStream stream,
                                                          final int numberOfEvents,
                                                          final boolean isBaseIdIndexColumnIncluded) throws Exception
    {
        //Determine the number of bytes per event so that I can check on the byteCount below.
        int numberOfIntsPerEvent = 4;
        if(isBaseIdIndexColumnIncluded)
        {
            numberOfIntsPerEvent = 5;
        }

        //Prefix for data from Fortran.  20 comes from 5 numbers per event and 4 bytes per number.
        int byteCount = stream.readIntSwap();
        if(byteCount != numberOfIntsPerEvent * 4 * numberOfEvents)
        {
            throw new ParameterIOException("Canonical event byte count found, " + byteCount + ", was not "
                + numberOfIntsPerEvent + " * 4 * the number of events declared previously, " + numberOfEvents + ".");
        }

        _baseCanonicalEvents.clear();
        _modulationCanonicalEvents.clear();
        _fullListOfEventsInOrder.clear();
        for(int i = 0; i < numberOfEvents; i++)
        {
            final int initialPeriod = stream.readIntSwap();
            final int lastPeriod = stream.readIntSwap();
            stream.readIntSwap(); //thrown away -- duration is not needed in the object
            final int numberOfMembers = stream.readIntSwap();
            int base = -1;
            if(isBaseIdIndexColumnIncluded)
            {
                base = stream.readIntSwap();
            }

            final CanonicalEvent event = new CanonicalEvent();
            event.setStartLeadPeriod(initialPeriod);
            event.setEndLeadPeriod(lastPeriod);
            event.setNumberOfLaggedEnsembleMembers(numberOfMembers);

            if(base == 0)
            {
                _modulationCanonicalEvents.add(event);
            }
            else
            //DEFAULT TO BASE EVENT -- temperature parameter files does not include the base/mod indicator in file
            {
                _baseCanonicalEvents.add(event);
            }
            _fullListOfEventsInOrder.add(event);
        }

        //Assigns the event numbers.
        _baseCanonicalEvents.indexElements();
        _modulationCanonicalEvents.indexElements();

        //Suffix from Fortran
        byteCount = stream.readIntSwap();
    }

    @Override
    public void validateParameters() throws Exception
    {
        // TODO What validation should be done?
        throw new Exception("Cannot validate MEFP algorithm parameters yet!");
    }

    @Override
    public CompositeXMLWriter getWriter()
    {
        _baseCanonicalEvents.setXMLTagName("baseCanonicalEventList");
        _modulationCanonicalEvents.setXMLTagName("modulationCanonicalEventList");
        final List<XMLWritable> writables = new ArrayList<XMLWritable>();

        writables.add(_baseCanonicalEvents);
        writables.add(_modulationCanonicalEvents);

        final CompositeXMLWriter writer = super.getWriter();
        writer.addComponents(writables);

        writer.addAttribute(new XMLString("algorithmType", HEFSTools.determineDataTypeString(_precipitation)), true);

        return writer;
    }

    @Override
    public CompositeXMLReader getReader()
    {
        _baseCanonicalEvents.setXMLTagName("baseCanonicalEventList");
        _modulationCanonicalEvents.setXMLTagName("modulationCanonicalEventList");
        final List<XMLReadable> readables = new ArrayList<XMLReadable>();

        readables.add(_baseCanonicalEvents);
        readables.add(_modulationCanonicalEvents);

        final CompositeXMLReader superReader = super.getReader();
        final CompositeXMLReader reader = new CompositeXMLReader(superReader.getXMLTagName(),
                                                                 superReader.getComponents())
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                //Full list must be populated after the algorithm parameters are read in.
                _baseCanonicalEvents.indexElements();
                _modulationCanonicalEvents.indexElements();
                _fullListOfEventsInOrder.clear();
                _fullListOfEventsInOrder.addAll(_baseCanonicalEvents);
                _fullListOfEventsInOrder.addAll(_modulationCanonicalEvents);
                //CanonicalEventList is maintained in sorted order, so no sorting is needed here!
            }
        };
        reader.addComponents(readables);
        reader.getAttributes().addAll(superReader.getAttributes());

        reader.addAttribute(new XMLString("algorithmType")
        {
            @Override
            public void finalizeReading() throws XMLReaderException
            {
                if(!get().equals(HEFSTools.determineDataTypeString(_precipitation)))
                {
                    throw new XMLReaderException("Attribute algorithmType, '" + get() + "', was not as expected, '"
                        + HEFSTools.determineDataTypeString(_precipitation) + "'.");
                }
            }
        }, true);
        return reader;
    }
}
